﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cmath>
#include <limits>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Assert.h>

namespace nnt{ namespace grc{

    class AudioSinWave
    {
    public:
        AudioSinWave(double mag, double freq, int channelCount, int64_t sampleRate) NN_NOEXCEPT
        {
            m_Magnitude = mag;
            m_Frequency = freq;
            m_ChannelCount = channelCount;
            m_SampleRate = static_cast<double>(sampleRate);
            m_TotalSampleCount = 0;
        }

        template<typename ChannelType>
        void Generate(ChannelType* buffer, int length) NN_NOEXCEPT
        {
            static const double Pi = 3.1415926535897932384626433832795;

            NN_ASSERT(length % (sizeof(ChannelType) * m_ChannelCount) == 0);
            int sampleCount = length / m_ChannelCount;

            for(int k = 0; k < sampleCount; k++)
            {
                double time = (m_TotalSampleCount + k) / m_SampleRate;
                double ang = (2 * Pi) * m_Frequency * time;
                double val = m_Magnitude * std::sin(ang);
                ChannelType v = {};
                if(val >= std::numeric_limits<ChannelType>::max())
                {
                    v = std::numeric_limits<ChannelType>::max();
                }
                else if(val <= std::numeric_limits<ChannelType>::min())
                {
                    v = std::numeric_limits<ChannelType>::min();
                }
                else
                {
                    v = static_cast<ChannelType>(val);
                }

                for(int c = 0; c < m_ChannelCount; c++)
                {
                    buffer[k * m_ChannelCount + c] = v;
                }
            }

            m_TotalSampleCount += sampleCount;
        }

    private:
        double m_Magnitude;
        double m_Frequency; // Hz
        int m_ChannelCount;
        double m_SampleRate; // samples/sec


        int64_t m_TotalSampleCount;
    };

}}
