﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testPerf_Main.h"
#include <nnt.h>

nns::gfx::GraphicsFramework gfw;
nn::os::MutexType   g_Mutex;
char nullName[128];


void InitializeGraphicsFramework() NN_NOEXCEPT
{
    const size_t graphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(graphicsSystemMemorySize);
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, 32 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Data, 32 * 1024 * 1024);
    gfw.Initialize(fwInfo);
}

void FinalizeGraphicsFramework() NN_NOEXCEPT
{
    gfw.Finalize();
}

nns::gfx::GraphicsFramework* GetGfw() NN_NOEXCEPT
{
    return &gfw;
}

const char* GetNullName() NN_NOEXCEPT
{
    return nullName;
}

nns::gfx::GraphicsFramework::CommandBuffer g_CommandBuffer[3];
void InitializeSubCommandBuffer() NN_NOEXCEPT
{
    // 赤青緑オブジェクト描画コマンド用のコマンドバッファを初期化
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
    info.SetCommandBufferType(nn::gfx::CommandBufferType_Nested);
    size_t commandMemorySize = 1024 * 1024 * 2;
    size_t controlMemorySize = 256;

    gfw.InitializeCommandBuffer(&g_CommandBuffer[0], info, commandMemorySize, controlMemorySize);
    gfw.InitializeCommandBuffer(&g_CommandBuffer[1], info, commandMemorySize, controlMemorySize);
    gfw.InitializeCommandBuffer(&g_CommandBuffer[2], info, commandMemorySize, controlMemorySize);
}

void FinalizeSubCommandBuffer() NN_NOEXCEPT
{
    gfw.FinalizeCommandBuffer(&g_CommandBuffer[0]);
    gfw.FinalizeCommandBuffer(&g_CommandBuffer[1]);
    gfw.FinalizeCommandBuffer(&g_CommandBuffer[2]);
}

nns::gfx::GraphicsFramework::CommandBuffer* GetSubCommandBuffer(int index) NN_NOEXCEPT
{
    return &g_CommandBuffer[index];
}

void Sleep() NN_NOEXCEPT
{
    nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(5));
}

void SleepGpu() NN_NOEXCEPT
{
    nn::os::LockMutex( &g_Mutex );
    nn::gfx::ColorTargetView* pTarget = gfw.GetColorTargetView();
    gfw.GetRootCommandBuffer(0)->ClearColor(pTarget, 0, 255.0f, 255.0f, 1.0f, NULL);
    nn::os::UnlockMutex( &g_Mutex );
}

void SleepGpu(int index) NN_NOEXCEPT
{
    nn::os::LockMutex( &g_Mutex );
    nn::gfx::ColorTargetView* pTarget = gfw.GetColorTargetView();
    GetSubCommandBuffer(index)->object.ClearColor(pTarget, 0, 255.0f, 255.0f, 1.0f, NULL);
    nn::os::UnlockMutex( &g_Mutex );
}

void CpuMeterManager::Initialize(int coreCount, int bufferCount, int sectionCountMax) NN_NOEXCEPT
{
    nn::perf::LoadMeterInfo info;
    info.SetCoreCount(coreCount);
    info.SetBufferCount(bufferCount);
    info.SetSectionCountMax(sectionCountMax);

    size_t memorySize = nn::perf::CpuMeter::CalculateBufferSize(info);
    size_t memoryAlignment = nn::perf::CpuMeter::GetBufferAlignment();
    m_Memory = gfw.AllocateMemory(memorySize, memoryAlignment);

    m_CpuMeter.Initialize(info, m_Memory, memorySize);
}

void CpuMeterManager::Finalize() NN_NOEXCEPT
{
    m_CpuMeter.Finalize();
    gfw.FreeMemory(m_Memory);
}

nn::perf::CpuMeter* CpuMeterManager::GetMeter() NN_NOEXCEPT
{
    return &m_CpuMeter;
}

void GpuMeterManager::Initialize(int coreCount, int bufferCount, int sectionCountMax) NN_NOEXCEPT
{
    nn::perf::LoadMeterInfo info;
    info.SetCoreCount(coreCount);
    info.SetBufferCount(bufferCount);
    info.SetSectionCountMax(sectionCountMax);

    size_t memorySize = nn::perf::GpuMeter::CalculateBufferSize(info);
    size_t memoryAlignment = nn::perf::GpuMeter::GetBufferAlignment();
    m_Memory = gfw.AllocateMemory(memorySize, memoryAlignment);

    size_t memoryPoolSize = nn::perf::GpuMeter::CalculateQueryBufferSize(gfw.GetDevice(), info);
    m_MemoryPoolOffset = gfw.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, memoryPoolSize, nn::perf::GpuMeter::GetQueryBufferAlignment(gfw.GetDevice(), info));
    nn::gfx::MemoryPool* meterMemoryPool = gfw.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_Data);

    m_GpuMeter.Initialize(gfw.GetDevice(), info, m_Memory, memorySize, meterMemoryPool, m_MemoryPoolOffset, memoryPoolSize);
}

void GpuMeterManager::Finalize() NN_NOEXCEPT
{
    m_GpuMeter.Finalize(gfw.GetDevice());
    gfw.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, m_MemoryPoolOffset);
    gfw.FreeMemory(m_Memory);
}

nn::perf::GpuMeter* GpuMeterManager::GetMeter() NN_NOEXCEPT
{
    return &m_GpuMeter;
}

void LoadMeterCenterManager::Initialize(bool isGpuEnabled, int coreCount, int userMeterCount, int cpuBufferCount, int gpuBufferCount, int sectionCountMax) NN_NOEXCEPT
{
    LoadMeterCenterManager::GetInstance().m_IsGpuEnabled = isGpuEnabled;
    nn::perf::LoadMeterCenterInfo info;
    info.SetCoreCount(coreCount);
    info.SetUserMeterCount(userMeterCount);
    info.SetCpuBufferCount(cpuBufferCount);
    info.SetCpuSectionCountMax(sectionCountMax);

    if(isGpuEnabled)
    {
        info.SetGpuBufferCount(gpuBufferCount);
        info.SetGpuSectionCountMax(sectionCountMax);
    }
    else
    {
        NN_UNUSED(gpuBufferCount);
    }
    // 計測で使用するメモリの確保
    size_t memorySize = NN_PERF_GET_BUFFER_SIZE(info);
    size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
    LoadMeterCenterManager::GetInstance().m_Memory = gfw.AllocateMemory(memorySize, memoryAlignment);

    if(isGpuEnabled)
    {
        // メモリプールの確保
        size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(gfw.GetDevice(), info);
        LoadMeterCenterManager::GetInstance().m_MemoryPoolOffset = gfw.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, memoryPoolSize, NN_PERF_GET_MEMORY_POOL_ALIGNMENT(gfw.GetDevice(), info));
        nn::gfx::MemoryPool* memoryPool = gfw.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_Data);

        NN_PERF_INITIALIZE_METER(gfw.GetDevice(), info, LoadMeterCenterManager::GetInstance().m_Memory, memorySize, memoryPool, LoadMeterCenterManager::GetInstance().m_MemoryPoolOffset, memoryPoolSize);
        NN_ASSERT(LoadMeterCenterManager::GetInstance().m_MemoryPoolOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset, "memoryPool shortage");
    }
    else
    {
        NN_PERF_INITIALIZE_METER_CPU_ONLY(info, LoadMeterCenterManager::GetInstance().m_Memory, memorySize);
    }
}

void LoadMeterCenterManager::Finalize() NN_NOEXCEPT
{
    if(LoadMeterCenterManager::GetInstance().m_IsGpuEnabled)
    {
        NN_PERF_FINALIZE_METER(gfw.GetDevice());
        gfw.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, LoadMeterCenterManager::GetInstance().m_MemoryPoolOffset);
    }
    else
    {
        NN_PERF_FINALIZE_METER_CPU_ONLY();
    }
    gfw.FreeMemory(LoadMeterCenterManager::GetInstance().m_Memory);
}

extern "C" void nnMain()
{
    int argc = ::nnt::GetHostArgc();
    char** argv = ::nnt::GetHostArgv();

    InitializeGraphicsFramework();
    nn::os::InitializeMutex( &g_Mutex, false, 0 );
    memset(nullName, 0, 128);

    ::testing::InitGoogleTest(&argc, argv);

    int result = RUN_ALL_TESTS();

    nn::os::FinalizeMutex( &g_Mutex );
    FinalizeGraphicsFramework();

    ::nnt::Exit(result);
}
