﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <cstdio>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>
#include <map>
#ifdef _WIN32
    #ifndef WIN32_LEAN_AND_MEAN
    #define WIN32_LEAN_AND_MEAN
    #endif
    #ifndef NOMINMAX
    #define NOMINMAX
    #endif
    #include <nn/nn_Windows.h>
#endif
#include <nvngdSupport/TutorialBaseClass.h>
#include <nvngdSupport/ShaderHeaders/GraphicsObjectCubeShaderDataHelper.h>
#include <nvngdSupport/AssetFileLoadingHelper.h>
#include <nvngdSupport/TutorialUtil.h>
#include <nvngdSupport/UniformBuffer.h>
#include <nvngdSupport/UniformBufferManager.h>
#include <nvngdSupport/RenderThreadPool.h>
#include <nvngdSupport/GraphicsObjectCube.h>
#include <nvngdSupport/MemoryPool.h>
#include <nvngdSupport/TextureIDManager.h>
#include <nvngdSupport/FrameBufferManager.h>

static const int    g_NumChunks = 2;
static const size_t g_CommandMemoryChunkSize = 32 * 1024;
static const size_t g_ControlMemoryChunkSize = 4 * 1024;

static const int g_NumColorBuffers = 2;

static const float g_Pi = 3.14159f;

class MultiThreadedRenderer : public TutorialBaseClass
{
    NN_DISALLOW_COPY(MultiThreadedRenderer);

    public:
        MultiThreadedRenderer();
        virtual ~MultiThreadedRenderer();
        virtual void Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow);
        virtual void Shutdown();

        virtual void Draw(uint64_t millisec);
        virtual void Resize(int width, int height);

    private:
        static void NVNAPIENTRY DebugLayerCallback(
            NVNdebugCallbackSource source,
            NVNdebugCallbackType type,
            int id,
            NVNdebugCallbackSeverity severity,
            const char* message,
            void* user
            );

        int UpdateRenderTargets();

        void UpdateObjectData(uint32_t frame);

        NVNdevice                    m_Device;
        NVNqueue                     m_Queue;
        void*                        m_pQueueMemory;

        NVNtextureBuilder            m_RenderTargetBuilder;
        NVNtexture*                  m_RenderTargets[g_NumColorBuffers];
        NVNtexture*                  m_pDepthTarget;
        NVNsamplerBuilder            m_SamplerBuilder;
        NVNbufferBuilder             m_BufferBuilder;

        int                          m_ScreenWidth;
        int                          m_ScreenHeight;

        MemoryPool*                  m_pRenderTargetMemoryPool;

        TextureIDManager*            m_pTextureIDManager;
        AssetFileLoadingHelper*      m_pAssetLoader;
        AssetFileDataHolder*         m_pDataHolder;

        ManagedCommandBuffer*        m_pRenderTargetManagedCommandBuffer;
        NVNcommandHandle             m_RenderTargetCommandHandle;

        NVNwindow*                   m_pWindow;
        NVNwindowBuilder             m_WindowBuilder;
        int                          m_CurrentWindowIndex;
        NVNsync                      m_WindowSync;

        UniformBufferManager*        m_pUniformBufferManager;

        FrameBufferedSyncManager*    m_pFrameBufferedSyncManager;

        RenderThreadPool*            m_pRenderThreadPool;
        std::vector<GraphicsObject*> m_Objects;

        size_t                       m_ColorTargetSize;
        size_t                       m_DepthTargetSize;
};

/*
 * MultiThreadedRenderer Constructor
 * ---------------------------------
 * Sets up default values for member data.
 */
MultiThreadedRenderer::MultiThreadedRenderer() :
    m_pQueueMemory(NULL),
    m_pDepthTarget(NULL),
    m_ScreenWidth(0),
    m_ScreenHeight(0),
    m_pRenderTargetMemoryPool(NULL),
    m_pTextureIDManager(NULL),
    m_pAssetLoader(NULL),
    m_pDataHolder(NULL),
    m_pRenderTargetManagedCommandBuffer(NULL),
    m_RenderTargetCommandHandle(0),
    m_pWindow(NULL),
    m_CurrentWindowIndex(-1),
    m_pUniformBufferManager(NULL),
    m_pFrameBufferedSyncManager(NULL),
    m_pRenderThreadPool(NULL),
    m_ColorTargetSize(0),
    m_DepthTargetSize(0)
{
    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
        m_RenderTargets[i] = NULL;
    }
}

/*
 * MultiThreadedRenderer Destructor
 * --------------------------------
 * Empty destructor.
 */
MultiThreadedRenderer::~MultiThreadedRenderer()
{
}

#ifndef _WIN32
//#define NVN_GRAPHICS_DEBUGGER
#endif

#ifdef NVN_GRAPHICS_DEBUGGER
extern "C" PFNNVNGENERICFUNCPTRPROC nvnDeviceGetToolsProcAddress(NVNdevice *device, const char* name);
#endif

/*!
* AssetFileLoading::Init
* ----------------------
* Initialize NVN, load asset files, and create objects needed for the
* application to run.
*/
void MultiThreadedRenderer::Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow)
{
#ifdef NVN_GRAPHICS_DEBUGGER
    /* Load procs from NVNGD instead of from the driver, this must be called once prior to nvnDeviceInitialize. */
    nvnLoadCProcs(nullptr, (PFNNVNDEVICEGETPROCADDRESSPROC)nvnDeviceGetToolsProcAddress);

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    int deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT;
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

    nvnDeviceSetDebugLabel(&m_Device, "Tutorial07_Device");

    /* Never call nvnLoadCProcs after nvnDeviceInitialize. */
#else /* NVN_GRAPHICS_DEBUGGER */
    pfnc_nvnDeviceInitialize = reinterpret_cast<PFNNVNDEVICEINITIALIZEPROC>((*pLoader)("nvnDeviceInitialize"));
    pfnc_nvnDeviceGetProcAddress = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>((*pLoader)("nvnDeviceGetProcAddress"));
    if (!pfnc_nvnDeviceInitialize)
    {
        /* This can happen if an NVN driver is not installed on a Windows PC. */
        NN_ASSERT(0, "BootstrapLoader failed to find nvnDeviceInitialize");
    }

    nvnLoadCProcs(NULL, pfnc_nvnDeviceGetProcAddress);

    int MajorVersion, MinorVersion;
    nvnDeviceGetInteger(NULL, NVN_DEVICE_INFO_API_MAJOR_VERSION, &MajorVersion);
    nvnDeviceGetInteger(NULL, NVN_DEVICE_INFO_API_MINOR_VERSION, &MinorVersion);

    if (MajorVersion != NVN_API_MAJOR_VERSION || MinorVersion < NVN_API_MINOR_VERSION)
    {
        NN_ASSERT(0, "NVN SDK not supported by current driver.");
    }

    /* If debug or develop is enabled, turn on NVN's debug layer. */
    int deviceFlags = 0;
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT;
#endif

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
        /*
        * This can fail for a few reasons; the most likely on Horizon is
        * insufficent device memory.
        */
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress);
#endif /* NVN_GRAPHICS_DEBUGGER */

        /* Setup the debug callback for the debug layer. */
    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
            );
    }

        /* Initialize the queue. */
    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDevice(&queueBuilder, &m_Device);
    nvnQueueBuilderSetDefaults(&queueBuilder);
    nvnQueueBuilderSetComputeMemorySize(&queueBuilder, 0);

    int minQueueCommandMemorySize = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_QUEUE_COMMAND_MEMORY_MIN_SIZE, &minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandMemorySize(&queueBuilder, minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandFlushThreshold(&queueBuilder, minQueueCommandMemorySize);

    size_t neededQueueMemorySize = nvnQueueBuilderGetQueueMemorySize(&queueBuilder);

    if ((neededQueueMemorySize % NVN_MEMORY_POOL_STORAGE_GRANULARITY) != 0)
    {
        NN_ASSERT(0, "Memory size reported for queue is not the proper granularity");
    }

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    m_pQueueMemory = NULL;
#else
    m_pQueueMemory = AlignedAllocate(neededQueueMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
#endif

    nvnQueueBuilderSetQueueMemory(&queueBuilder, m_pQueueMemory, neededQueueMemorySize);

    if (nvnQueueInitialize(&m_Queue, &queueBuilder) == false)
    {
        NN_ASSERT(0, "nvnQueueInitialize failed");
    }

        /* Create a managed Command Buffer */
    m_pRenderTargetManagedCommandBuffer = new ManagedCommandBuffer(&m_Device, g_CommandMemoryChunkSize, g_ControlMemoryChunkSize, g_NumChunks);

        /* Builders */
    nvnSamplerBuilderSetDevice(&m_SamplerBuilder, &m_Device);
    nvnSamplerBuilderSetDefaults(&m_SamplerBuilder);

    nvnBufferBuilderSetDevice(&m_BufferBuilder, &m_Device);
    nvnBufferBuilderSetDefaults(&m_BufferBuilder);

        /* Set up the texture builder for the render target. */
    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_DEPTH32F);
    m_DepthTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

        /* Allocate the render target memory. */
    m_pRenderTargetMemoryPool = new MemoryPool();
    m_pRenderTargetMemoryPool->Init(
        NULL,
        m_ColorTargetSize * g_NumColorBuffers + m_DepthTargetSize,
        NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
        &m_Device);

    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);

        /* Create the texture ID manager. */
    m_pTextureIDManager = new TextureIDManager(&m_Device);

        /* Create the asset loader. */
    m_pAssetLoader = new AssetFileLoadingHelper(&m_Device, m_pTextureIDManager);

    size_t cacheSize = 0;
    nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT( result.IsSuccess() );

    char* mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

    result = nn::fs::MountRom("rom", mountRomCacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        /*! Initialize the window sync. */
    if (!nvnSyncInitialize(&m_WindowSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize window sync");
    }

        /* Load assets. */
    m_pDataHolder = m_pAssetLoader->LoadAssetFile("graphicsObjectCube.out");
    m_pDataHolder->SetupAttributeStatesNVN(GraphicsObjectCubeShader::Attributes::GetAttributeLocation);
    m_pDataHolder->SetupTextureSamplerHandle(&m_Device, m_pTextureIDManager, &m_SamplerBuilder);

        /* Setup the window builder. */
    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);

    m_pUniformBufferManager = new UniformBufferManager(&m_Device, 32 * 1024);

    m_pFrameBufferedSyncManager = new FrameBufferedSyncManager(&m_Device, &m_Queue);
    m_pFrameBufferedSyncManager->RegisterMemoryManager(m_pUniformBufferManager);
    m_pFrameBufferedSyncManager->RegisterMemoryManager(m_pRenderTargetManagedCommandBuffer);

    m_pRenderThreadPool = new RenderThreadPool();
    m_pRenderThreadPool->Init(4, &m_Device, &m_Queue);

        /* Create the objects to be renderer by the thread pool. */
    for(int i = 0; i < 4; ++i)
    {
        for(int j = 0; j < 4; ++j)
        {
            for(int k = 0; k < 4; ++k)
            {
                void* place = AlignedAllocate(sizeof(GraphicsObjectCube), NN_ALIGNOF(GraphicsObjectCube));
                GraphicsObjectCube* obj = new(place)GraphicsObjectCube;

                obj->Init((GraphicsObjectCube::StateType)((i + j) % 2), m_pDataHolder);

                obj->SetScale(0.5f, 0.5f, 0.5f);
                obj->SetTranslate(-1.5f + i, -1.5f + k, -4.0f - j);
                obj->SetRotate(0.0f, 0.0f, 0.0f);

                obj->SetupUniforms(m_pUniformBufferManager);
                m_Objects.push_back(obj);
            }
        }
    }

    nn::fs::Unmount("rom");
    delete[] mountRomCacheBuffer;
}//NOLINT(impl/function_size)

/*
 * MultiThreadedRenderer::UpdateObjectData
 * ---------------------------------------
 * Updates the transform data of the object for a
 * given frame.
 */
void MultiThreadedRenderer::UpdateObjectData(uint32_t frame)
{
    for(size_t i = 0; i < m_Objects.size(); ++i)
    {
        reinterpret_cast<GraphicsObjectCube*>(m_Objects[i])->SetRotate(sin(frame * g_Pi / 360.0f) * 30.0f, static_cast<float>(frame % 360), 0.0f);
    }
}

/*
 * MultiThreadedRenderer::UpdateRenderTargets
 * ------------------------------------------
 * Gets the index of the current render target from the NVNwindow
 * and records a command buffer that sets it up to be rendered to
 * and clears it.
 */
int MultiThreadedRenderer::UpdateRenderTargets()
{
        /* Get next render target to be used */
    NVNwindowAcquireTextureResult result = nvnWindowAcquireTexture(m_pWindow, &m_WindowSync, &m_CurrentWindowIndex);

    NN_ASSERT(result == NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);

        /* Record the command buffer to set and clear the target. */
    m_pRenderTargetManagedCommandBuffer->BeginRecording();
    {
        NVNcommandBuffer* commandBuffer = m_pRenderTargetManagedCommandBuffer->GetCommandBuffer();

        nvnCommandBufferSetRenderTargets(commandBuffer, 1, &m_RenderTargets[m_CurrentWindowIndex], NULL, m_pDepthTarget, NULL);

        nvnCommandBufferSetScissor(commandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);
        nvnCommandBufferSetViewport(commandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);

        float clear_color[4] = { 0.4f, 0.55f, 0.6f, 1.0f };
        nvnCommandBufferClearColor(commandBuffer, 0, clear_color, NVN_CLEAR_COLOR_MASK_RGBA);
        nvnCommandBufferClearDepthStencil(commandBuffer, 1.0, NVN_TRUE, 0, 0);
    }
    m_RenderTargetCommandHandle = m_pRenderTargetManagedCommandBuffer->EndRecording();

        /* Return the index. */
    return m_CurrentWindowIndex;
}

/*
 * MultiThreadedRenderer::::Shutdown
 * ---------------------------------
 * This method cleans up all nvn objects and dynamically allocated memory.
 */
void MultiThreadedRenderer::Shutdown()
{
    nvnQueueFinish(&m_Queue);
    nvnSyncFinalize(&m_WindowSync);

    if(m_pRenderTargetManagedCommandBuffer)
    {
        delete m_pRenderTargetManagedCommandBuffer;
        m_pRenderTargetManagedCommandBuffer = NULL;
    }

    if (m_pWindow != NULL)
    {
        nvnWindowFinalize(m_pWindow);
        delete m_pWindow;
        m_pWindow = NULL;
    }

    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        if (m_RenderTargets[i] != NULL)
        {
            nvnTextureFinalize(m_RenderTargets[i]);
            delete m_RenderTargets[i];
            m_RenderTargets[i] = NULL;
        }
    }

    if (m_pDepthTarget != NULL)
    {
        nvnTextureFinalize(m_pDepthTarget);
        delete m_pDepthTarget;
        m_pDepthTarget = NULL;
    }

    if (m_pDataHolder)
    {
        delete m_pDataHolder;
        m_pDataHolder = NULL;
    }

    if (m_pAssetLoader != NULL)
    {
        delete m_pAssetLoader;
        m_pAssetLoader = NULL;
    }

    if (m_pTextureIDManager != NULL)
    {
        delete m_pTextureIDManager;
        m_pTextureIDManager = NULL;
    }

    for (size_t i = 0; i < m_Objects.size(); ++i)
    {
        AlignedDeallocate(m_Objects[i]);
    }

    m_Objects.clear();

    if(m_pRenderThreadPool)
    {
        m_pRenderThreadPool->Clean();
        delete m_pRenderThreadPool;
        m_pRenderThreadPool = NULL;
    }

    if(m_pRenderTargetMemoryPool)
    {
        m_pRenderTargetMemoryPool->Shutdown();
        delete m_pRenderTargetMemoryPool;
        m_pRenderTargetMemoryPool = NULL;
    }

    if(m_pUniformBufferManager)
    {
        delete m_pUniformBufferManager;
        m_pUniformBufferManager = NULL;
    }

    if (m_pFrameBufferedSyncManager != NULL)
    {
        delete m_pFrameBufferedSyncManager;
        m_pFrameBufferedSyncManager = NULL;
    }

    nvnQueueFinalize(&m_Queue);
    if (m_pQueueMemory)
    {
        AlignedDeallocate(m_pQueueMemory);
        m_pQueueMemory = NULL;
    }

    nvnDeviceFinalize(&m_Device);
}

/*
 * MultiThreadedRenderer::Draw
 * ---------------------------
 * This function runs the threads in the render thread
 * pool to generate the command buffers to draw the list
 * of objects and submits them to the queue. It also presents
 * the current render target to the screen.
 */
void MultiThreadedRenderer::Draw(uint64_t millisec)
{
    NN_UNUSED(millisec);
    static uint32_t s_Frame = 0;

    ++s_Frame;

        /* Get the current render target and setup/submit a command buffer to set it. */
    int index = UpdateRenderTargets();

        /* Update the transform data for the objects to be drawn. */
    UpdateObjectData(s_Frame);

        /*
         * Run the threads in the thread pool to generate
         * command buffers to render the objects in the work
         * list.
         */
    std::vector<NVNcommandHandle> outHandles;
    m_pRenderThreadPool->Run(&outHandles, &m_Objects);
    m_pRenderThreadPool->Wait();
    m_pRenderThreadPool->SwapCommandMemory();

        /* Insert a fence in the sync manager. */
    m_pFrameBufferedSyncManager->InsertFence();

        /*!
         * Wait on sync that was received in UpdateRenderTargets now that we are
         * actually ready to use the render target
         */
    nvnSyncWait(&m_WindowSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* Submit command buffer for switching render targets */
    nvnQueueSubmitCommands(&m_Queue, 1, &m_RenderTargetCommandHandle);

        /* Submit the command buffers generated by the thread pool. */
    nvnQueueSubmitCommands(&m_Queue, static_cast<int>(outHandles.size()), &outHandles[0]);

        /* Present the texture to the screen. */
    nvnQueuePresentTexture(&m_Queue, m_pWindow, index);

    m_pFrameBufferedSyncManager->SwapPools();
}

/*
 * MultiThreadedRenderer::Resize
 * -----------------------------
 * This method is called everytime the window is resized and is passed
 * the new size of the window. It frees the old render target and creates a new
 * render target with the new screen size.
 */
void MultiThreadedRenderer::Resize(int width, int height)
{
    m_ScreenWidth = width;
    m_ScreenHeight = height;

        /* Check for the window being minimized or having no visible surface. */
    if (width == 0 || height == 0)
    {
        return;
    }

        /* If it's the first time Resize is called, allocate the NVNwindow. */
    if (m_pWindow == NULL)
    {
        m_pWindow = new NVNwindow;
    }
        /*
         * Otherwise finalize (free) the NVNwindow used for the previous window size.
         * The NVNWindow must be finalized before a render target it owns is finalized.
         */
    else
    {
        nvnWindowFinalize(m_pWindow);
    }

        /* Set up the builder for the render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);

    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
            /* If it's the first time Resize is called, allocate the render target. */
        if(m_RenderTargets[i] == NULL)
        {
            m_RenderTargets[i] = new NVNtexture;
        }
            /* Otherwise finalize (free) the render target used for the previous window size. */
        else
        {
            nvnTextureFinalize(m_RenderTargets[i]);
        }

        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * i);

            /* Create the texture using the current state of the texture builder. */
        nvnTextureInitialize(m_RenderTargets[i], &m_RenderTargetBuilder);
    }

    if (m_pDepthTarget == NULL)
    {
        m_pDepthTarget = new NVNtexture;
    }
    else
    {
        nvnTextureFinalize(m_pDepthTarget);
    }

        /* Initialize depth buffer for render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_DEPTH32F);
    nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * g_NumColorBuffers);
    nvnTextureInitialize(m_pDepthTarget, &m_RenderTargetBuilder);

        /* Pass off the render targets to the window. */
    nvnWindowBuilderSetTextures(&m_WindowBuilder, g_NumColorBuffers, m_RenderTargets);
    nvnWindowInitialize(m_pWindow, &m_WindowBuilder);

        /* Remake the projection matrix with the new screen size. */
    nn::util::Matrix4x4fType projMat44;
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projMat44, 60.0f * 3.14159f / 180.0f, static_cast<float>(m_ScreenWidth) / static_cast<float>(m_ScreenHeight), 1.0f, 1000.0f);
    nn::util::Matrix4x4fType camMat44;
    nn::util::MatrixIdentity(&camMat44);
    for (size_t i = 0; i < m_Objects.size(); ++i)
    {
        reinterpret_cast<GraphicsObjectCube*>(m_Objects[i])->SetProjectionMatrix(projMat44);
        reinterpret_cast<GraphicsObjectCube*>(m_Objects[i])->SetCameraMatrix(camMat44);
    }
}

void MultiThreadedRenderer::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char* message,
    void* user
    )
{
    NN_ASSERT(user == NULL);

    NN_LOG("NVN Debug Layer Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n",     message);

    NN_ASSERT(0);
}

TutorialBaseClass* t()
{
    static MultiThreadedRenderer tut;
    return (&tut);
}

extern "C" void nnMain()
{
    TutorialRun();
}
