﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{StatTracker.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a helper class that sets up command buffers to report GPU
 *  statistic counters and reset those counters. It also provides functionality
 *  to print those counters to the screen using a given DebugTextRenderer.
 */

#include <nvn/nvn_FuncPtrInline.h>
#include <nn/nn_Assert.h>
#include <nvngdSupport/StatTracker.h>
#include <nvngdSupport/MemoryPool.h>
#include <nvngdSupport/TutorialUtil.h>

static const size_t g_CommandMemorySize = 1024;
static const size_t g_ControlMemorySize = 1024;

/*
 * StatTracker Constructor
 * -----------------------
 * Setup some default values for the tracker.
 */
StatTracker::StatTracker() : m_pDevice(NULL),
                             m_CounterAlignment(0),
                             m_CounterFlags(CounterFlags_All_Counters),
                             m_pManagedCommandBuffer(NULL),
                             m_pResetControlMemoryPool(NULL),
                             m_pStatCounterManager(NULL)
{
}

/*
 * StatTracker::Init
 * -----------------
 * Allocates memory the command buffers that report and reset
 * counter data.  A StatCounterManager is used to manage the
 * counter buffers through multi buffering.
 */
void StatTracker::Init(NVNdevice* pDevice, int flags, FrameBufferedSyncManager* pSyncManager, unsigned numCounters /* = 2 */)
{
    m_pDevice = pDevice;
    m_CounterFlags = flags;

        /* Query the device for the command and control memory alignment. */
    int commandBufferCommandAlignment = 0;
    int commandBufferControlAlignment = 0;
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandBufferCommandAlignment);
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &commandBufferControlAlignment);

    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_COUNTER_ALIGNMENT, &m_CounterAlignment);

        /* Allocate the command and control memory. */
    m_MemoryPool.Init(NULL,
                      Align(g_CommandMemorySize, commandBufferCommandAlignment),
                      NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
                      m_pDevice);

    m_pResetControlMemoryPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

        /*
         * If more than one chunk is requested, then the stat counter manager is
         * initialized with 2 sync counters and registered with the sync manager
         * which handles properly swapping the counters.  Otherwise, only one
         * counter is used and no swapping occurs; a sync is used to ensure that
         * the counter data is reported properly and ready for use instead of
         * sampling them in the middle.
         */
    if (numCounters == 1)
    {
        m_pStatCounterManager = new StatCounterManager(pDevice, 1);

            /* Initialize the sync object. */
        nvnSyncInitialize(&m_CommandBufferSync, m_pDevice);
    }
    else
    {
        m_pStatCounterManager = new StatCounterManager(pDevice);

        pSyncManager->RegisterMemoryManager(m_pStatCounterManager);
    }

        /* Initialize the command buffers. */
    nvnCommandBufferInitialize(&m_ResetCommandBuffer, m_pDevice);

        /* Add the command and control memory to the command buffers. */
    size_t offset = m_MemoryPool.GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

    offset = m_MemoryPool.GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);
    nvnCommandBufferAddCommandMemory(&m_ResetCommandBuffer, m_MemoryPool.GetMemoryPool(), offset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_ResetCommandBuffer, m_pResetControlMemoryPool, g_ControlMemorySize);

        /* Setup an NVNbuffer for the counters to be reported into. */
    m_pManagedCommandBuffer = new ManagedCommandBuffer(m_pDevice, g_CommandMemorySize, g_ControlMemorySize);
    pSyncManager->RegisterMemoryManager(m_pManagedCommandBuffer);

        /* Record the counter reset command buffer. */
    nvnCommandBufferBeginRecording(&m_ResetCommandBuffer);
    {
        unsigned end = NVN_COUNTER_TYPE_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN;
        unsigned flagChecker = CounterFlags_Samples_Passed;

            /* Check for each potential counter and reset it if needed. */
        for (unsigned i = 1; i <= end; ++i)
        {
            if (m_CounterFlags & flagChecker)
            {
                nvnCommandBufferResetCounter(&m_ResetCommandBuffer, (NVNcounterType)i);
            }

            flagChecker <<= 1;
        }
    }
    m_ResetHandle = nvnCommandBufferEndRecording(&m_ResetCommandBuffer);
}

/*
 * StatTracker::GetCounters
 * ------------------------
 * Uses a multi-buffered StatCounter to grab counter data
 * without using a sync
 */
void StatTracker::GetCounters(NVNqueue* pQueue)
{
    if (m_pStatCounterManager->GetNumChunks() == 1)
    {
        NN_ASSERT(pQueue != NULL);

            /* Wait for the counters to be reported. */
        nvnQueueFenceSync(pQueue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, NVN_SYNC_FLAG_FLUSH_FOR_CPU_BIT);
        nvnQueueFlush(pQueue);
        NVNsyncWaitResult waitRes = nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);
        NN_ASSERT(waitRes != NVNsyncWaitResult::NVN_SYNC_WAIT_RESULT_FAILED || waitRes != NVNsyncWaitResult::NVN_SYNC_WAIT_RESULT_FAILED,
            "Failed to wait on StatTracker sync\n");
    }

    unsigned end = NVN_COUNTER_TYPE_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN;
    StatCounters* pStatCounters = m_pStatCounterManager->GetCurrentStatCounter();
    uint64_t* counterArray = reinterpret_cast<uint64_t*>(pStatCounters);

    NVNcounterData* mappedBuffer = reinterpret_cast<NVNcounterData*>(m_pStatCounterManager->GetCounterBufferPointer());

        /* Grab the timestamp from the bottom of the frame first. */
    NVNcounterData* counterData = mappedBuffer;
    counterArray[0] = counterData->timestamp;

        /* Write the rest of the counters into the output struct. */
    for(unsigned i = 1; i <= end; ++i)
    {
        counterData = mappedBuffer + i;
        counterArray[i] = counterData->counter;
    }
}

/*
 * StatTracker::ReportCounters
 * ---------------------------
 * Starts recording a command buffer to report the
 * counters.
 */
void StatTracker::ReportCounters(NVNqueue* pQueue)
{
        /* Record the counter reporting command buffer. */
    nvnCommandBufferBeginRecording(m_pManagedCommandBuffer->GetCommandBuffer());
    {
        unsigned end = NVN_COUNTER_TYPE_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN;
        unsigned commandOffset = 0;
        unsigned flagChecker = CounterFlags_Timestamp;
        NVNbufferAddress address = m_pStatCounterManager->GetCounterBufferAddress();

            /* Check for each potential counter and report it if needed. */
        for (unsigned i = 0; i <= end; ++i)
        {
            if (m_CounterFlags & flagChecker)
            {
                nvnCommandBufferReportCounter(m_pManagedCommandBuffer->GetCommandBuffer(), (NVNcounterType)i, address + commandOffset);
            }

            flagChecker <<= 1;
            commandOffset += m_CounterAlignment;
        }
    }
    m_ReportHandle = nvnCommandBufferEndRecording(m_pManagedCommandBuffer->GetCommandBuffer());

    nvnQueueSubmitCommands(pQueue, 1, &m_ReportHandle);
}

/*
 * StatTracker::ResetCounters
 * --------------------------
 * Submit the previously recorded command buffer
 * to reset the counter values.
 */
void StatTracker::ResetCounters(NVNqueue* pQueue)
{
    nvnQueueSubmitCommands(pQueue, 1, &m_ResetHandle);
}

/*
 * StatTracker::PrintCounters
 * --------------------------
 * Print out the previously queried counters on
 * the screen using the provided debug text renderer.
 */
void StatTracker::PrintCounters(DebugTextRenderer* textRenderer, float column, float startingLine)
{
    StatCounters* pCounters = m_pStatCounterManager->GetCounterForPrint();

    unsigned flagChecker = CounterFlags_Timestamp;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "Timestamp:                %llu", pCounters->m_Timestamp);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "SamplesPassed:            %llu", pCounters->m_SamplesPassed);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "Input Vertices:           %llu", pCounters->m_InputVertices);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "InputPrimitives:          %llu", pCounters->m_InputPrimitives);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "VertexShader Invocations: %llu", pCounters->m_VertexShaderInvocations);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "TesCntrlShdr Invocations: %llu", pCounters->m_TessControlShaderInvocations);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "TesEvalShdr Invocations:  %llu", pCounters->m_TessEvalShaderInvocations);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "GeomShdr Invocations:     %llu", pCounters->m_GeometryShaderInvocations);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "FragShader Invocations:   %llu", pCounters->m_FragmentShaderInvocations);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "TesEvalShdr Primitives :  %llu", pCounters->m_TessEvaluationShaderPrimitives);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "GeomShdr Primitives:      %llu", pCounters->m_GeometryShaderPrimitives);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "ClipperInput Primitives:  %llu", pCounters->m_ClipperInputPrimitives);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "ClipperOutput Primitives: %llu", pCounters->m_ClipperOutputPrimitives);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "Primitives Generated:     %llu", pCounters->m_PrimitivesGenerated);
    }
    flagChecker <<= 1;

    if(m_CounterFlags & flagChecker)
    {
        textRenderer->Printf(column, startingLine++, "TrnsfFeedback Prims Wrtn: %llu", pCounters->m_TransformFeedbackPrimitivesWritten);
    }
}

/*
 * StatTracker::Shutdown
 * ---------------------
 * Clean up allocated memory and NVN data.
 */
void StatTracker::Shutdown()
{
    if (m_pManagedCommandBuffer)
    {
        delete m_pManagedCommandBuffer;
        m_pManagedCommandBuffer = NULL;
    }

    if (m_pStatCounterManager->GetNumChunks() == 1)
    {
        nvnSyncFinalize(&m_CommandBufferSync);
    }

    if (m_pResetControlMemoryPool)
    {
        AlignedDeallocate(m_pResetControlMemoryPool);
        m_pResetControlMemoryPool = NULL;
    }

    nvnCommandBufferFinalize(&m_ResetCommandBuffer);

    m_MemoryPool.Shutdown();

    if (m_pStatCounterManager)
    {
        delete m_pStatCounterManager;
    }
}

/*
 * StatTracker Destructor
 * ----------------------
 * Empty destructor.
 */
StatTracker::~StatTracker()
{
}

/*
 * StatCounterManager Constructor
 * ------------------------------
 * Initializes array of Stat counters and multi
 * buffers the NVN buffers that receive counter
 * data
 */
StatCounterManager::StatCounterManager(NVNdevice* pDevice, int numChunks /* = 2 */) :
    m_pDevice(pDevice),
    m_pStatCounters(NULL),
    m_NumChunks(numChunks),
    m_CurrentChunk(0),
    m_PreviousChunk(numChunks - 1)
{
    NN_ASSERT(numChunks > 0);

        /* Query the device for the counter alignment. */
    int counterAlignment;
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_COUNTER_ALIGNMENT, &counterAlignment);

        /*
         * Each individual counter is a struct of two 64-bit integers, one is a timestamp and
         * the other is the value of the counter.  As a result, the size of the buffer that
         * is needed is essentially double that of the StatCounters struct that is used to
         * hold the sampled counter data.
         */
    size_t counterSize = (sizeof(StatCounters) / sizeof(uint64_t)) * sizeof(NVNcounterData);

    m_pStatCounters = new StatCounters[numChunks];
    memset(m_pStatCounters, 0, sizeof(StatCounters) * numChunks);

    m_BufferMemoryPool.Init(
        NULL,
        NVN_MEMORY_POOL_STORAGE_GRANULARITY,
        NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
        m_pDevice);

    NVNbufferBuilder bufferBuilder;

    nvnBufferBuilderSetDefaults(&bufferBuilder);
    nvnBufferBuilderSetDevice(&bufferBuilder, m_pDevice);
    nvnBufferBuilderSetStorage(&bufferBuilder, m_BufferMemoryPool.GetMemoryPool(), 0, NVN_MEMORY_POOL_STORAGE_GRANULARITY);

    if (!nvnBufferInitialize(&m_Buffer, &bufferBuilder))
    {
        NN_ASSERT(0, "Failed to initialize counter's uniform buffer");
    }

    ptrdiff_t offset = 0;
    for (int i = 0; i < m_NumChunks; ++i)
    {
        m_Offsets.push_back(offset);

        offset += counterSize;
        offset = Align(static_cast<size_t>(offset), counterAlignment);
    }
}

/*
 * StatCounterManager Destructor
 * -----------------------------
 * Cleans up stat counter manager
 */
StatCounterManager::~StatCounterManager()
{
    if (m_pStatCounters)
    {
        delete [] m_pStatCounters;
        m_pStatCounters = NULL;
    }

    nvnBufferFinalize(&m_Buffer);

    m_BufferMemoryPool.Shutdown();
}

/*
 * StatCounterManager::SwapPools
 * -----------------------------
 * Moves to next StatCounter struct so the
 * next one we use is clear and not using
 * old values
 */
void StatCounterManager::SwapPools()
{
        /* If there is only undivided chunk in the memory pool, then there is nothing to swap */
    if (m_NumChunks == 1)
    {
        return;
    }

    m_CurrentChunk = (m_CurrentChunk + 1) % m_NumChunks;
    m_PreviousChunk = (m_PreviousChunk + 1) % m_NumChunks;
    memset(&m_pStatCounters[m_CurrentChunk], 0, sizeof(StatCounters));
}

/*
 * StatCounterManager::GetNumChunks
 * --------------------------------
 * Returns number of StatCounters in array.
 * If a SyncManager manager is used then
 * the number of chunks must equal the
 * number of syncs in the manager.
 */
int StatCounterManager::GetNumChunks()
{
    return m_NumChunks;
}

/*
 * StatCounterManager::GetCounterBufferPointer
 * -------------------------------------------
 * Gets pointer to the mapped NVN buffer
 */
char* StatCounterManager::GetCounterBufferPointer()
{
    char* ptr = reinterpret_cast<char*>(nvnBufferMap(&m_Buffer));

    return ptr + m_Offsets[m_CurrentChunk];
}

/*
 * StatCounterManager::GetCounterBufferAddress
 * -------------------------------------------
 * Gets device address of the NVN buffer
 */
NVNbufferAddress StatCounterManager::GetCounterBufferAddress()
{
    NVNbufferAddress addr = nvnBufferGetAddress(&m_Buffer);

    return addr + m_Offsets[m_CurrentChunk];
}

/*
 * StatCounterManager::GetCurrentStatCounter
 * -----------------------------------------
 * Returns current StatCounter to be used this
 * frame to receive counter data
 */
StatCounters* StatCounterManager::GetCurrentStatCounter()
{
    return &m_pStatCounters[m_CurrentChunk];
}

/*
 * StatCounterManager::GetCounterForPrint
 * --------------------------------------
 * Gets a stat counter that has been properly filled
 * with counter data.  If more than two counters
 * are available, the one that was filled last
 * frame is used to print to guarantee that the
 * report has gone through the command buffer
 * properly.
 */
StatCounters* StatCounterManager::GetCounterForPrint()
{
    if (m_NumChunks == 1)
    {
        return &m_pStatCounters[0];
    }

    return &m_pStatCounters[m_PreviousChunk];
}
