﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{MultiThreadedAssetFileLoadingHelper.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a helper class to load asset
 *  files in parallel on multiple threads.
 */

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>

#include <nvngdSupport/MultiThreadedAssetFileLoadingHelper.h>
#include <nvngdSupport/OutputFileHeaders.h>
#include <nvngdSupport/TutorialUtil.h>
#include <nvngdSupport/TextureIDManager.h>
#include <nvngdSupport/MemoryPool.h>

/*
 * MultiThreadedAssetFileLoadingHelper Constructor
 * -----------------------------------------------
 * Default constructor that sets up initial values for members. Takes
 * an AssetLoaderArg that contains relevent nvn data and the file name.
 */
MultiThreadedAssetFileLoadingHelper::MultiThreadedAssetFileLoadingHelper(AssetLoaderArg* pArg) : m_pDataHolder(NULL),
                                                                                                 m_pArg(pArg),
                                                                                                 m_pFileHead(NULL),
                                                                                                 m_FileSize(0)
{
}

/*
 * MultiThreadedAssetFileLoadingHelper Destructor
 * ----------------------------------------------
 * Destructor
 */
MultiThreadedAssetFileLoadingHelper::~MultiThreadedAssetFileLoadingHelper()
{
}

/*
 * MultiThreadedAssetFileLoadingHelper::LoadAssetFile
 * --------------------------------------------------
 * Loads the binary asset file, checks for each section of
 * the file, and interprets the data for the sections that
 * exist.
 */
void MultiThreadedAssetFileLoadingHelper::LoadAssetFile()
{
        /* Reads in the binary file and grabs a pointer to its head. */
    nn::fs::FileHandle fileHandle;

    std::string rom("rom:/");
    std::string fileName(m_pArg->GetConfigFileName());
    nn::Result res = nn::fs::OpenFile(&fileHandle, (rom + fileName).c_str(), nn::fs::OpenMode_Read);
    NN_ASSERT(res.IsSuccess());

    res = nn::fs::GetFileSize(&m_FileSize, fileHandle);
    NN_ASSERT(res.IsSuccess());

    m_pFileHead = reinterpret_cast<char*>(AlignedAllocate(static_cast<size_t>(m_FileSize + 1), 8));
    memset(m_pFileHead, 0, static_cast<size_t>(m_FileSize + 1));
    size_t out;
    res = nn::fs::ReadFile(&out, fileHandle, 0, m_pFileHead, static_cast<size_t>(m_FileSize));
    NN_ASSERT(res.IsSuccess());

    OutputFileHeader* pFileHeader = reinterpret_cast<OutputFileHeader*>(m_pFileHead);
    NN_ASSERT(pFileHeader != NULL, "Failed to read file header")

        /* Creates a default buffer builder. */
    nvnBufferBuilderSetDevice(&m_BufferBuilder, m_pArg->GetDevice());
    nvnBufferBuilderSetDefaults(&m_BufferBuilder);

        /* Creates a default texture builder. */
    nvnTextureBuilderSetDevice(&m_TextureBuilder, m_pArg->GetDevice());
    nvnTextureBuilderSetDefaults(&m_TextureBuilder);

        /* Creates a default sampler builder. */
    nvnSamplerBuilderSetDevice(&m_SamplerBuilder, m_pArg->GetDevice());
    nvnSamplerBuilderSetDefaults(&m_SamplerBuilder);

    m_pDataHolder = new AssetFileDataHolder();

        /* Check for the shader section of the file and interpret the data. */
    if (pFileHeader->m_ShaderBlockOffset > 0)
    {
        char* pShaderBlockHead = m_pFileHead + pFileHeader->m_ShaderBlockOffset;
        LoadShaders(pShaderBlockHead);
    }

        /* Check for the texture section of the file and interpret the data. */
    if (pFileHeader->m_TextureBlockOffset > 0)
    {
        char* pTextureBlockHead = m_pFileHead + pFileHeader->m_TextureBlockOffset;
        LoadTextures(pTextureBlockHead);
    }

        /* Check for the model section of the file and interpret the data. */
    if (pFileHeader->m_ModelBlockOffset > 0)
    {
        char* pModelBlockHead = m_pFileHead + pFileHeader->m_ModelBlockOffset;
        LoadModels(pModelBlockHead);
    }

    m_pArg->LockMemoryPoolMutex();
    nn::fs::CloseFile(fileHandle);
    m_pArg->UnlockMemoryPoolMutex();

    AlignedDeallocate(m_pFileHead);
    m_pFileHead = NULL;
}

/*
 * MultiThreadedAssetFileLoadingHelper::LoadShaders
 * ------------------------------------------------
 * Runs through the shaders listed by the loaded file and
 * creates a shader program for each.
 */
void MultiThreadedAssetFileLoadingHelper::LoadShaders(const char* pShaderBlockHead)
{
    NN_ASSERT(pShaderBlockHead != NULL, "Shader block header pointer NULL");

        /* Grabs the number of shaders and their offsets. */
    const ShaderBlockHeader* pShaderBlockHeader = reinterpret_cast<const ShaderBlockHeader*>(pShaderBlockHead);
    uint32_t numPrograms = pShaderBlockHeader->m_NumShaderPrograms;

    const uint64_t* pShaderProgramHeaderOffsets = reinterpret_cast<const uint64_t*>((pShaderBlockHead + sizeof(uint32_t) * 2));

        /* Creates a program for each shader. */
    for(uint32_t i = 0; i < numPrograms; ++i)
    {
        NVNProgramData* pProgramData = CreateShaderProgram(m_pFileHead + pShaderProgramHeaderOffsets[i]);
        m_pDataHolder->AddProgramData(pProgramData);
    }
}

/*
 * MultiThreadedAssetFileLoadingHelper::CreateShaderProgram
 * --------------------------------------------------------
 * Saves the shader metadata in an NVNProgramData object
 * and creates an NVNprogram object from the compiled
 * shader binary.
 */
NVNProgramData* MultiThreadedAssetFileLoadingHelper::CreateShaderProgram(const char* pShaderProgramHead)
{
    NN_ASSERT(pShaderProgramHead != NULL, "Shader program header pointer NULL");

        /* Grabs the metadata and the list of offsets to individual shader stages. */
    const ShaderProgramHeader* pShaderProgramHeader = reinterpret_cast<const ShaderProgramHeader*>(pShaderProgramHead);

    uint32_t        numShaderStages = pShaderProgramHeader->m_NumShaderStages;
    const uint64_t* pShaderStageOffsets = reinterpret_cast<const uint64_t*>(pShaderProgramHead + sizeof(uint32_t) * 2);
    const char*     pProgramName = pShaderProgramHead + sizeof(uint32_t) * 2 + sizeof(uint64_t) * numShaderStages;

    if (pShaderProgramHeader == NULL ||
        pShaderProgramHeader->m_ProgramNameLength == 0 ||
        pProgramName == NULL ||
        numShaderStages == 0 ||
        pShaderStageOffsets == NULL )
    {
        NN_ASSERT(0, "Failed to read ShaderProgramHeader");
    }

    NVNProgramData* pProgramData = new NVNProgramData;
    pProgramData->Initialize();
    pProgramData->SetName(pProgramName, pShaderProgramHeader->m_ProgramNameLength);
    pProgramData->m_ShaderType = ShaderTypes::GetShaderTypeEnum(std::string(pProgramData->m_pProgramName));

    std::vector<ShaderStageHeader> shaderStageHeaders;
    shaderStageHeaders.resize(numShaderStages, ShaderStageHeader());

    std::vector<size_t> alignmentOffsets;
    alignmentOffsets.resize(numShaderStages, 0);

        /*
         * Grab the stage metadata and pointers to the control and data section
         * of the compiled shader.
         */
    size_t shaderDataBufferMemoryPoolSize = 0;
    int bufferAlignment = 0;
    nvnDeviceGetInteger(m_pArg->GetDevice(), NVN_DEVICE_INFO_UNIFORM_BUFFER_ALIGNMENT, &bufferAlignment);
    for (uint32_t i = 0; i < numShaderStages; ++i)
    {
        ShaderStageHeader* pShaderStageHeader = reinterpret_cast<ShaderStageHeader*>(m_pFileHead + pShaderStageOffsets[i]);

        shaderStageHeaders[i].m_ShaderStage         = pShaderStageHeader->m_ShaderStage;
        shaderStageHeaders[i].m_ShaderDataSize      = pShaderStageHeader->m_ShaderDataSize;
        shaderStageHeaders[i].m_ShaderControlSize   = pShaderStageHeader->m_ShaderControlSize;
        shaderStageHeaders[i].m_ShaderControlOffset = pShaderStageHeader->m_ShaderControlOffset;
        shaderStageHeaders[i].m_pShaderData         = m_pFileHead + pShaderStageOffsets[i] + 4 * sizeof(uint32_t);
        shaderStageHeaders[i].m_pShaderControl      = m_pFileHead + shaderStageHeaders[i].m_ShaderControlOffset;

            /* Store offsets for filling the memory pool and initializing buffers. */
        alignmentOffsets[i] = shaderDataBufferMemoryPoolSize;
        shaderDataBufferMemoryPoolSize += shaderStageHeaders[i].m_ShaderDataSize;

            /* Make sure the size is at the correct alignment. */
        shaderDataBufferMemoryPoolSize = Align(shaderDataBufferMemoryPoolSize, bufferAlignment);
    }

        /*
         * Shader code is not allowed to be in the last 1024 bytes of a memory pool,
         * additional padding is added to ensure that does not happen.
         */
    shaderDataBufferMemoryPoolSize += 1024;

        /* Copy the shader data into a contiguous block of memory to wrap a pool around. */
    size_t memoryPoolAlignedSize;

    if (shaderDataBufferMemoryPoolSize < g_MinimumPoolSize)
    {
        memoryPoolAlignedSize = g_MinimumPoolSize;
    }
    else
    {
        memoryPoolAlignedSize = Align(shaderDataBufferMemoryPoolSize, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    }

    m_pArg->LockMemoryPoolMutex();
    char* pShaderDataArray = reinterpret_cast<char*>(AlignedAllocate(memoryPoolAlignedSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT));
    m_pArg->UnlockMemoryPoolMutex();
    memset(pShaderDataArray, 0, memoryPoolAlignedSize);

    for (uint32_t i = 0; i < shaderStageHeaders.size(); ++i)
    {
        memcpy(pShaderDataArray + alignmentOffsets[i], shaderStageHeaders[i].m_pShaderData, shaderStageHeaders[i].m_ShaderDataSize);
    }

    m_pArg->LockMemoryPoolMutex();

        /* Create new memory pool around the shader data. */
    pProgramData->m_ShaderMemoryPool.Init(pShaderDataArray,
                                          memoryPoolAlignedSize,
                                          NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_SHADER_CODE_BIT,
                                          m_pArg->GetDevice());

    m_pArg->UnlockMemoryPoolMutex();

    std::vector<NVNshaderData> nvnShaderData;
    nvnShaderData.resize(numShaderStages, NVNshaderData());
    pProgramData->m_ShaderBuffers.resize(numShaderStages, NULL);

        /*
         * Create NVNbuffer objects out of the memory pool for each shader stage.
         * The NVNbufferAddress's for these buffers are placed in a list with the
         * pointers to the corresponding shader control sections.
         */
    for (uint32_t i = 0; i < shaderStageHeaders.size(); ++i)
    {
        pProgramData->m_ShaderStages |= ConvertNVNStageToBitField(static_cast<NVNshaderStage>(shaderStageHeaders[i].m_ShaderStage));
        pProgramData->m_ShaderBuffers[i] = new NVNbuffer;

        nvnBufferBuilderSetDefaults(&m_BufferBuilder);
        nvnBufferBuilderSetStorage(&m_BufferBuilder, pProgramData->m_ShaderMemoryPool.GetMemoryPool(), alignmentOffsets[i], shaderStageHeaders[i].m_ShaderDataSize);

        if (!nvnBufferInitialize(pProgramData->m_ShaderBuffers[i], &m_BufferBuilder))
        {
            NN_ASSERT(0, "Failed to initialize shader buffer storage");
        }

        nvnShaderData[i].control    = shaderStageHeaders[i].m_pShaderControl;
        nvnShaderData[i].data       = nvnBufferGetAddress(pProgramData->m_ShaderBuffers[i]);
    }

        /* Initialize the shader program and provide it with the compiled shader data. */
    nvnProgramInitialize(&pProgramData->m_Program, m_pArg->GetDevice());

    if (!nvnProgramSetShaders(&pProgramData->m_Program, numShaderStages, &nvnShaderData[0]))
    {
        NN_ASSERT(0, "Failed to set pre-compiled headers");
    }

    return pProgramData;
}

/*
 * MultiThreadedAssetFileLoadingHelper::LoadTextures
 * -------------------------------------------------
 * Grabs the texture block header of the asset file and loads
 * the texture data at the specified offsets.
 */
void MultiThreadedAssetFileLoadingHelper::LoadTextures(const char* pTextureBlockHead)
{
    NN_ASSERT(pTextureBlockHead != NULL, "Texture block header pointer NULL");

    const TextureBlockHeader* pTextureBlockHeader = reinterpret_cast<const TextureBlockHeader*>(pTextureBlockHead);
    const uint64_t* pTextureOffsets = reinterpret_cast<const uint64_t*>(pTextureBlockHead + sizeof(uint32_t) * 2);

    for (uint32_t i = 0; i < pTextureBlockHeader->m_NumTextures; ++i)
    {
        NVNTextureData* pTextureData = LoadTextureData(m_pFileHead + pTextureOffsets[i]);
        m_pDataHolder->AddTextureData(pTextureData);
    }
}

/*
 * MultiThreadedAssetFileLoadingHelper::LoadTextureData
 * ----------------------------------------------------
 * Loads the packaged texture data and metadata from the texture header.
 */
NVNTextureData* MultiThreadedAssetFileLoadingHelper::LoadTextureData(const char* pTextureDataHead)
{
    NN_ASSERT(pTextureDataHead != NULL, "Texture data pointer NULL");

    NVNTextureData* pNvnTextureData = new NVNTextureData;
    pNvnTextureData->Initialize();

    uint32_t currentDataEntryOffset = 0;

    /* Grab the texture's metadata. */
    pNvnTextureData->m_TextureDataSize  = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint64_t);
    pNvnTextureData->m_GpuVersion       = *reinterpret_cast<const int*>(pTextureDataHead + currentDataEntryOffset);               currentDataEntryOffset += sizeof(int);
    pNvnTextureData->m_Alignment        = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    pNvnTextureData->m_Width            = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    pNvnTextureData->m_Height           = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    pNvnTextureData->m_Depth            = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    pNvnTextureData->m_NvnTextureTarget = *reinterpret_cast<const NVNtextureTarget*>(pTextureDataHead + currentDataEntryOffset);  currentDataEntryOffset += sizeof(uint32_t);
    pNvnTextureData->m_NvnFormat        = *reinterpret_cast<const NVNformat*>(pTextureDataHead + currentDataEntryOffset);         currentDataEntryOffset += sizeof(uint32_t);
    pNvnTextureData->m_MipLevels        = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    uint32_t textureNameLength          = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    uint32_t textureDataOffset          = *reinterpret_cast<const uint32_t*>(pTextureDataHead + currentDataEntryOffset);          currentDataEntryOffset += sizeof(uint32_t);
    const char* pTextureName            = reinterpret_cast<const char*>(pTextureDataHead + currentDataEntryOffset);               currentDataEntryOffset += sizeof(char) * textureNameLength;
    void* pTextureData                  = (m_pFileHead + textureDataOffset);

    if(!textureNameLength ||
       pTextureName == NULL ||
       !pNvnTextureData->m_Width ||
       !pNvnTextureData->m_Height ||
       !pNvnTextureData->m_Depth ||
       pNvnTextureData->m_NvnTextureTarget > NVN_TEXTURE_TARGET_BUFFER ||
       !pNvnTextureData->m_NvnFormat ||
       !pNvnTextureData->m_MipLevels ||
       pTextureData == NULL)
    {
        NN_ASSERT(0, "Failed to read texture data");
    }

    pNvnTextureData->SetName(pTextureName, textureNameLength);

    nvnTextureBuilderSetDefaults(&m_TextureBuilder);
    nvnSamplerBuilderSetDefaults(&m_SamplerBuilder);

        /* Set up the texture builder based on the loaded metadata. */
    nvnTextureBuilderSetTarget(&m_TextureBuilder, pNvnTextureData->m_NvnTextureTarget);
    nvnTextureBuilderSetFormat(&m_TextureBuilder, pNvnTextureData->m_NvnFormat);
    nvnTextureBuilderSetSize2D(&m_TextureBuilder, pNvnTextureData->m_Width, pNvnTextureData->m_Height);
    nvnTextureBuilderSetDepth(&m_TextureBuilder, pNvnTextureData->m_Depth);
    nvnTextureBuilderSetLevels(&m_TextureBuilder, pNvnTextureData->m_MipLevels);
    uint32_t size = static_cast<uint32_t>(nvnTextureBuilderGetStorageSize(&m_TextureBuilder));
    NN_ASSERT(size == pNvnTextureData->m_TextureDataSize);

    if(m_pArg->GetUseMipmaps())
    {
            /*
             * Sets option in sampler for mip map filtering
             * These options cause a hard cutoff between mip
             * levels, use of linear filters would allow
             * for interpolation between levels.
             */
        nvnSamplerBuilderSetMinMagFilter(&m_SamplerBuilder, NVNminFilter::NVN_MIN_FILTER_NEAREST_MIPMAP_NEAREST, NVNmagFilter::NVN_MAG_FILTER_NEAREST);
    }
    else if(m_pArg->GetUseCubemap())
    {
        if ((pNvnTextureData->m_Width != pNvnTextureData->m_Height))
        {
            NN_ASSERT(0, "Texture's dimensions are not equal, cubemap's sides are not square");
        }
    }

    size_t memoryPoolSize;

    if (pNvnTextureData->m_TextureDataSize < g_MinimumPoolSize)
    {
        memoryPoolSize = g_MinimumPoolSize;
    }
    else
    {
        memoryPoolSize = Align(static_cast<uint32_t>(pNvnTextureData->m_TextureDataSize), NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    }

    m_pArg->LockMemoryPoolMutex();

    void* memoryPoolData = AlignedAllocate(memoryPoolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);

    memcpy(memoryPoolData, pTextureData, static_cast<uint32_t>(pNvnTextureData->m_TextureDataSize));

        /*
        * On Windows, the packaged texture data cannot be used by the GPU as
        * it will not be compatible with the format used by the device. To
        * remedy this, it can be provided to the texture builder through the
        * SetPackagedTextureData function. The windows driver will decode the
        * texture data and re-encode it to the proper format. On the device, the
        * function is a no-op; the pointer is ignored and its contents unmodified.
        */
    nvnTextureBuilderSetPackagedTextureData(&m_TextureBuilder, pTextureData);

        /* Wrap a memory pool around the texture data. */
    pNvnTextureData->m_TextureMemoryPool.Init(memoryPoolData,
                                              memoryPoolSize,
                                              NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
                                              m_pArg->GetDevice());

        /* Set the storage for the texture. */
    nvnTextureBuilderSetStorage(&m_TextureBuilder, pNvnTextureData->m_TextureMemoryPool.GetMemoryPool(), 0);

        /* Initialize the texture with the builder's settings. */
    if (nvnTextureInitialize(&pNvnTextureData->m_Texture, &m_TextureBuilder) == NVN_FALSE)
    {
        pNvnTextureData->Finalize();
        delete pNvnTextureData;
        NN_ASSERT(0, "Failed to initialize texture");
        return NULL;
    }
    else
    {
        pNvnTextureData->m_TextureInitialized = true;
    }

    m_pArg->UnlockMemoryPoolMutex();

        /* Initialize the sampler. */
    if (nvnSamplerInitialize(&pNvnTextureData->m_Sampler, &m_SamplerBuilder) == NVN_FALSE)
    {
        pNvnTextureData->Finalize();
        delete pNvnTextureData;
        NN_ASSERT(0, "Failed to initialize sampler");
        return NULL;
    }
    else
    {
        pNvnTextureData->m_SamplerInitialized = true;
    }

        /* Register the texture/sampler and hold on to their IDs. */
    pNvnTextureData->m_TextureID = m_pArg->GetTextureIDManager()->RegisterTexture(&pNvnTextureData->m_Texture);
    pNvnTextureData->m_SamplerID = m_pArg->GetTextureIDManager()->RegisterSampler(&pNvnTextureData->m_Sampler);

        /* Grab the combined texture handle. */
    pNvnTextureData->m_TextureHandle = nvnDeviceGetTextureHandle(m_pArg->GetDevice(), pNvnTextureData->m_TextureID, pNvnTextureData->m_SamplerID);
    if (!pNvnTextureData->m_TextureHandle)
    {
        pNvnTextureData->Finalize();
        delete pNvnTextureData;
        NN_ASSERT(0, "Failed to get handle to texture");
        return NULL;
    }
    return pNvnTextureData;
}//NOLINT(impl/function_size)

/*
 * MultiThreadedAssetFileLoadingHelper::LoadModels
 * -----------------------------------------------
 * Grabs the model block header and loads the model data
 * at the specified offsets.
 */
void MultiThreadedAssetFileLoadingHelper::LoadModels(const char* pModelBlockHead)
{
    NN_ASSERT(pModelBlockHead != NULL, "Model block header pointer NULL");

    uint32_t pNumModels = *reinterpret_cast<const uint32_t*>(pModelBlockHead);

    const uint64_t* pModelOffsets = reinterpret_cast<const uint64_t*>(pModelBlockHead + sizeof(uint32_t) * 2);

    for (uint32_t i = 0; i < pNumModels; ++i)
    {
        NVNModelData* pModelData = LoadModelData(m_pFileHead + pModelOffsets[i]);
        m_pDataHolder->AddModelData(pModelData);
    }
}

/*
 * MultiThreadedAssetFileLoadingHelper::LoadModelData
 * --------------------------------------------------
 * Loads in the model vertex attributes and index buffer as well
 * as some data describing the model.
 */
NVNModelData* MultiThreadedAssetFileLoadingHelper::LoadModelData(const char* pModelHead)
{
    NN_ASSERT(pModelHead != NULL, "Model data pointer NULL");

    /* Grab the model's metadata. */
    uint32_t            currentDataEntryOffset = 0;
    uint32_t            numPrimitives           = *reinterpret_cast<const uint32_t*>(pModelHead + currentDataEntryOffset);              currentDataEntryOffset += sizeof(uint32_t);
    NVNdrawPrimitive    nvnDrawPrimitiveType    = *reinterpret_cast<const NVNdrawPrimitive*>(pModelHead + currentDataEntryOffset);      currentDataEntryOffset += sizeof(uint32_t);
    uint32_t            numVertexAttributes     = *reinterpret_cast<const uint32_t*>(pModelHead + currentDataEntryOffset);              currentDataEntryOffset += sizeof(uint32_t);
    uint32_t            modelNameLength         = *reinterpret_cast<const uint32_t*>(pModelHead + currentDataEntryOffset);              currentDataEntryOffset += sizeof(uint32_t);
    uint64_t            indexBufferOffset       = *reinterpret_cast<const uint64_t*>(pModelHead + currentDataEntryOffset);              currentDataEntryOffset += sizeof(uint64_t);
    const uint64_t*     pVertexAttributeOffsets = reinterpret_cast<const uint64_t*>(pModelHead + currentDataEntryOffset);               currentDataEntryOffset += sizeof(uint64_t) * numVertexAttributes;
    const char*         pModelName              = pModelHead + currentDataEntryOffset;                                                  currentDataEntryOffset += sizeof(char) * modelNameLength;

    if(modelNameLength == 0 ||
       pModelName == NULL ||
       !numPrimitives ||
       nvnDrawPrimitiveType > NVN_DRAW_PRIMITIVE_PATCHES ||
       !numVertexAttributes ||
       pVertexAttributeOffsets == NULL ||
       !indexBufferOffset)
    {
        NN_ASSERT(0, "Failed to read model data");
    }

    NVNModelData* pModelData = new NVNModelData;
    pModelData->Initialize();
    pModelData->SetName(pModelName, modelNameLength);

    Model* pModel = &pModelData->m_Model;
    pModel->m_NumVertexAttributes = numVertexAttributes;
    pModel->m_NumPrimitives = numPrimitives;
    pModel->m_NvnDrawPrimitiveType = nvnDrawPrimitiveType;

        /* Grab the data for the individual vertex attributes. */
    for (uint32_t i = 0; i < numVertexAttributes; ++i)
    {
        VertexAttributeHeader header;
        char* pAttributeHeader = m_pFileHead + pVertexAttributeOffsets[i];
        NN_ASSERT(pAttributeHeader);

        uint32_t currVertexEntryOffset = 0;

        header.m_AttributeStride        = *reinterpret_cast<const uint32_t*>(pAttributeHeader + currVertexEntryOffset);     currVertexEntryOffset += sizeof(uint32_t);
        header.m_NvnFormat              = *reinterpret_cast<const uint32_t*>(pAttributeHeader + currVertexEntryOffset);     currVertexEntryOffset += sizeof(uint32_t);
        header.m_AttributeDataSize      = *reinterpret_cast<const uint32_t*>(pAttributeHeader + currVertexEntryOffset);     currVertexEntryOffset += sizeof(uint32_t);
        header.m_AttributeNameLength    = *reinterpret_cast<const uint32_t*>(pAttributeHeader + currVertexEntryOffset);     currVertexEntryOffset += sizeof(uint32_t);
        header.m_AttributeDataOffset    = *reinterpret_cast<const uint32_t*>(pAttributeHeader + currVertexEntryOffset);     currVertexEntryOffset += sizeof(uint32_t);
        header.m_MagicPadding           = *reinterpret_cast<const uint32_t*>(pAttributeHeader + currVertexEntryOffset);     currVertexEntryOffset += sizeof(uint32_t);
        header.m_pAttributeName         = pAttributeHeader + currVertexEntryOffset;                                         currVertexEntryOffset += sizeof(char) * header.m_AttributeNameLength;
        header.m_AttributeData          = m_pFileHead + header.m_AttributeDataOffset;

        pModelData->AddVertexAttribute(header);
        pModelData->m_VertexBufferSize += header.m_AttributeDataSize;
    }

        /* Grab the index buffer data. */
    char* pIndexBufferHead = m_pFileHead + indexBufferOffset;
    NN_ASSERT(pIndexBufferHead);

    uint32_t currIndexEntryOffset = 0;

    uint32_t indexBufferSize    = *reinterpret_cast<uint32_t*>(pIndexBufferHead + currIndexEntryOffset);     currIndexEntryOffset += sizeof(uint32_t);
    uint32_t numIndices         = *reinterpret_cast<uint32_t*>(pIndexBufferHead + currIndexEntryOffset);     currIndexEntryOffset += sizeof(uint32_t);
    uint32_t indexType          = *reinterpret_cast<uint32_t*>(pIndexBufferHead + currIndexEntryOffset);     currIndexEntryOffset += sizeof(uint32_t);
    currIndexEntryOffset        += sizeof(uint32_t); /* Account for padding in struct */
    void*    pIndexData         = (pIndexBufferHead + currIndexEntryOffset);

    if(!indexBufferSize     ||
       !numIndices          ||
       pIndexData == NULL)
    {
        NN_ASSERT(0, "Failed to read index buffer data");
    }

        /* Align the index buffer according to its type. */
    size_t memoryPoolSize = pModelData->m_VertexBufferSize;
    uint32_t indexBufferAlignment = 0;
    switch((NVNindexType)indexType)
    {
        case NVN_INDEX_TYPE_UNSIGNED_BYTE:
            indexBufferAlignment = sizeof(uint8_t);
            break;
        case NVN_INDEX_TYPE_UNSIGNED_SHORT:
            indexBufferAlignment = sizeof(uint16_t);
            break;
        case NVN_INDEX_TYPE_UNSIGNED_INT:
            indexBufferAlignment = sizeof(uint32_t);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

        /* Get the aligned pool size. */
    memoryPoolSize = Align(memoryPoolSize, indexBufferAlignment);
    size_t indexBufferAlignedLocation = memoryPoolSize;
    memoryPoolSize += indexBufferSize;

    pModel->m_IndexData.m_IndexType     = indexType;
    pModel->m_IndexData.m_DataSize      = indexBufferSize;
    pModel->m_IndexData.m_Stride        = indexBufferAlignment;
    pModel->m_IndexData.m_pData         = new char[indexBufferSize];

    memcpy(pModel->m_IndexData.m_pData, pIndexData, indexBufferSize);

    if (memoryPoolSize < g_MinimumPoolSize)
    {
        memoryPoolSize = g_MinimumPoolSize;
    }
    else
    {
        memoryPoolSize = Align(memoryPoolSize, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    }

    m_pArg->LockMemoryPoolMutex();
    char* pModelDataBuffer = reinterpret_cast<char*>(AlignedAllocate(memoryPoolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT));
    m_pArg->UnlockMemoryPoolMutex();
    memset(pModelDataBuffer, 0, memoryPoolSize);

    uint32_t currentBufferOffset = 0;

        /* Copy in the vertex data. */
    for (uint32_t i = 0; i < pModel->m_VertexAttributes.size(); ++i)
    {
        memcpy(pModelDataBuffer + currentBufferOffset, pModel->m_VertexAttributes[i].m_pData, pModel->m_VertexAttributes[i].m_DataSize);
        pModelData->m_VertexAttributeBufferOffsets.push_back(currentBufferOffset);
        currentBufferOffset += pModel->m_VertexAttributes[i].m_DataSize;
    }

        /* Copy the index data. */
    memcpy(pModelDataBuffer + indexBufferAlignedLocation, pIndexData, indexBufferSize);


    m_pArg->LockMemoryPoolMutex();

        /* Create a memory pool around the model data. */
    pModelData->m_BufferMemoryPool.Init(pModelDataBuffer,
                                        memoryPoolSize,
                                        NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
                                        m_pArg->GetDevice());
    m_pArg->UnlockMemoryPoolMutex();

        /* Create the vertex buffer from the memory pool at the correct offset. */
    nvnBufferBuilderSetDefaults(&m_BufferBuilder);
    nvnBufferBuilderSetStorage(&m_BufferBuilder, pModelData->m_BufferMemoryPool.GetMemoryPool(), 0, pModelData->m_VertexBufferSize);

    if (!nvnBufferInitialize(&pModelData->m_VertexBuffer, &m_BufferBuilder))
    {
        pModelData->Finalize();
        NN_ASSERT(0, "Failed to initialize vertex buffer storage");
    }

        /* Create the index buffer from the memory pool at the correct offset. */
    nvnBufferBuilderSetDefaults(&m_BufferBuilder);
    nvnBufferBuilderSetStorage(&m_BufferBuilder, pModelData->m_BufferMemoryPool.GetMemoryPool(), indexBufferAlignedLocation, indexBufferSize);

    if (!nvnBufferInitialize(&pModelData->m_IndexBuffer, &m_BufferBuilder))
    {
        pModelData->Finalize();
        NN_ASSERT(0, "Failed to initialize index buffer storage");
    }

    return pModelData;
}//NOLINT(impl/function_size)

/*
 * MultiThreadedAssetFileLoadingHelper::GetAssetFileDataHolder
 * -----------------------------------------------------------
 * Gets the data holder associated with this loader.
 */
AssetFileDataHolder* MultiThreadedAssetFileLoadingHelper::GetAssetFileDataHolder()
{
    return m_pDataHolder;
}

/*
 * AssetLoaderArg Constructor
 * --------------------------
 * Constructor for the AssetLoaderArg class. Contains
 * initilization data for the multi threaded loading
 * helper class.
 */
AssetLoaderArg::AssetLoaderArg(NVNdevice* pDevice,
                               const char* pConfigFileName,
                               nn::os::MutexType* pMemoryPoolMutex,
                               TextureIDManager* pTextureIDManager,
                               bool useMipmaps,
                               bool useCubemap)
                               :
                               m_pDevice(pDevice),
                               m_pMemoryPoolMutex(pMemoryPoolMutex),
                               m_pConfigFileName(pConfigFileName),
                               m_useMipmap(useMipmaps),
                               m_useCubeMap(useCubemap),
                               m_pTextureIDManager(pTextureIDManager)
{
}

/*
 * AssetLoaderArg::GetDevice
 * -------------------------
 * Get a pointer to the nvn device.
 */
NVNdevice* AssetLoaderArg::GetDevice()
{
    return m_pDevice;
}

/*
 * AssetLoaderArg::GetTextureIDManager
 * -----------------------------------
 * Get a pointer to the texture id manager.
 */
TextureIDManager* AssetLoaderArg::GetTextureIDManager()
{
    return m_pTextureIDManager;
}

/*
 * AssetLoaderArg::GetConfigFileName
 * ---------------------------------
 * Get the asset's file name.
 */
const char* AssetLoaderArg::GetConfigFileName()
{
    return m_pConfigFileName;
}

/*
 * AssetLoaderArg::GetUseMipmaps
 * -----------------------------
 * Get whether to use mipmaps.
 */
bool AssetLoaderArg::GetUseMipmaps()
{
    return m_useMipmap;
}

/*
 * AssetLoaderArg::GetUseCubemap
 * -----------------------------
 * Get whether to use a cube map.
 */
bool AssetLoaderArg::GetUseCubemap()
{
    return m_useCubeMap;
}

/*
 * AssetLoaderArg::LockMemoryPoolMutex
 * -----------------------------------
 * Lock the memory pool mutex.
 */
void AssetLoaderArg::LockMemoryPoolMutex()
{
    nn::os::LockMutex(m_pMemoryPoolMutex);
}

/*
 * AssetLoaderArg::UnlockMemoryPoolMutex
 * -------------------------------------
 * Unlock the memory pool mutex.
 */
void AssetLoaderArg::UnlockMemoryPoolMutex()
{
    nn::os::UnlockMutex(m_pMemoryPoolMutex);
}
