﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{ManagedCommandBuffer.h,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a class that manages memory
 *  for command buffers by using multiple chunks of
 *  of GPU memory. The active chunk is written to and
 *  generates a command handle at rendering end. At the
 *  end of the frame, the next chunk is provided to the
 *  command buffer which then becomes active and writable
 *  for the next frame. Proper syncing to ensure that
 *  memory is not overwritten as it is being used is
 *  necessary; the provided FrameBufferedSyncManager class
 *  handles the syncing of all classes that derive from
 *  the FrameBufferedMemoryManager class.
 *
 *  Requesting a single chunk for the memory pool will provide
 *  an undivided block of memory for the command buffer that is
 *  good for command sets that will never change, though the user
 *  must now handle proper syncing if they do wish to update the
 *  memory.
 */

#pragma once

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <vector>
#include "TutorialUtil.h"
#include "MemoryPool.h"
#include "FrameBufferManager.h"

class ManagedCommandBuffer : public FrameBufferedMemoryManager
{
public:
    ManagedCommandBuffer(NVNdevice* pDevice, size_t commandChunkSize, size_t controlChunkSize, int numChunks = 2);
    virtual ~ManagedCommandBuffer();

    virtual void SwapPools();
    virtual int GetNumChunks();

    void BeginRecording();
    NVNcommandHandle EndRecording();

    NVNcommandBuffer* GetCommandBuffer();

    static inline void NVNAPIENTRY outOfMemory(NVNcommandBuffer *cmdBuf, NVNcommandBufferMemoryEvent event, size_t minSize, void *callbackData);

private:
    NVNcommandBuffer    m_CommandBuffer;

    int                 m_NumChunks;
    MemoryPool          m_CommandMemoryPool;
    char*               m_ControlMemoryPool;
    size_t              m_CommandMemoryChunkSize;
    size_t              m_ControlMemoryChunkSize;
    size_t              m_CommandMemoryPoolSize;
    size_t              m_ControlMemoryPoolSize;

    unsigned            m_CurrentChunk;

    NVNdevice*          m_pDevice;
};
