﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NvnTutorial01.cpp,PageSampleNvnTutorial01}
 *
 * @brief
 *  This tutorial demonstrates how to setup an application to use the NVN graphics
 *  API and performs some basic operations with it.
 */

/**
 * @page PageSampleNvnTutorial01 NVN Tutorial 01: Getting Started With NVN
 * @tableofcontents
 *
 * @brief
 *  This tutorial demonstrates how to setup an application to use the NVN graphics
 *  API and performs some basic operations with it.
 *
 * @section PageSampleNvnTutorial01_SectionBrief Overview
 *
 * @subsection PageSampleNvnTutorial01_SectionIntroduction Getting Started With NVN
 *  Welcome to NVN! It's a new graphics library, and it's pretty great. This
 *  tutorial shows the most basic initialization and cleanup steps, how to
 *  allocate frame buffers, clear, and display.
 *
 *  One of the interesting features of the NVN API is it's ability to run
 *  both on the target device, but also on Windows PCs. In order to run on
 *  Windows, the API must be initialized via a "donor" OpenGL context.
 *
 *  In this series, the applications will use a small abstraction base class
 *  to hide the details necessary to write applications that run on both systems.
 *  (Such as OpenGL handling for Windows.) For more information on this class,
 *  see the TutorialBaseClass.
 *
 * @subsection PageSampleNvnTutorial01_SectionLinking Header Files And Link Libraries
 *  NVN is a fully dynamically linked library. This means that all NVN functions
 *  are called through function pointers. Including nvn.h in a source file brings
 *  enums, constants, macros, and definitions of all of the various function
 *  pointer types that define the parameters and return types of each NVN
 *  function.
 *
 *  An application may implement it's own function loading system, or it may use
 *  the nvn_FuncPtrInline system that is included with the NVN headers. This system
 *  defines a series of inline functions that wrap calls to function pointers
 *  loaded with the nvnLoadProcs function. In order to use this system, include
 *  nvn_FuncPtrInline.h and nvn_FuncPtrImpl instead of nvn.h in source files.
 *
 *  NVN also includes a further set of wrappers that implement a C++-style
 *  interface.
 *
 *  As NVN is dynamically linked, there is no special link library requirement on
 *  Windows (though OpenGL32.dll must be linked.)
 *
 * @subsection PageSampleNvnTutorial01_SectionExpectedOutput Expected Output
 * @image html NvnTutorial01.png
 *
 * @section PageSampleNvnTutorial01_SectionFileStructure File Structure
 *  The main tutorial file and Visual Studio solutions can be found at
 *  @link ../../../Samples/Sources/Applications/NvnTutorial01GettingStartedWithNVN Samples/Sources/Applications/NvnTutorial01GettingStartedWithNVN @endlink
 *
 *  The tutorial library which contains common code shared by the tutorials can be found at
 *  Samples/Sources/Libraries/NvnTutorial
 *
 * @section PageSampleNvnTutorial01_SectionNecessaryEnvironment System Requirements
 *  No extra system requirements.
 *
 * @section PageSampleNvnTutorial01_SectionHowToOperate Operation Procedure
 *  This sample runs on its own without any input.
 *
 * @section PageSampleNvnTutorial01_SectionPrecaution Precautions
 *  None.
 *
 * @section PageSampleNvnTutorial01_SectionHowToExecute Execution Procedure
 *  Build the Visual Solution in the desired configuration and run it.
 *
 * @section PageSampleNvnTutorial01_SectionDetail Description
 *
 * @subsection PageSampleNvnTutorial01_SectionSampleProgram Sample Program
 *  Below is the source code for the main tutorial file for this sample.
 *
 *  NvnTutorial01.cpp
 *  @includelineno NvnTutorial01.cpp
 *
 * @subsection PageSampleNvnTutorial01_SectionSampleDetail Sample Program Description
 *  This tutorial demonstrates how to setup an application to use the NVN graphics
 *  API and performs some basic operations with it. It walks through how to load the
 *  function pointers for the API, setting up the nvnDevice and nvnQueue objects, and
 *  creates command buffers for setting a render target and clearing quadrants of the
 *  screen.
 *
 */

#include <cmath>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>
#include <nvntutorial/TutorialBaseClass.h>
#include <nvntutorial/MemoryPool.h>
#include <nvntutorial/TutorialUtil.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>

#include <nnt.h>
#include <nnt/graphics/testGraphics_Path.h>
#include <nnt/graphics/testGraphics_GetHostExecutableFilepath.h>
#include <nnt/graphics/testGraphics_Png.h>
#include <nnt/graphics/testGraphics_CreateDirectories.h>
#include <nnt/graphics/testGraphics_InitializeAllocatorFunctionForStandardAllocator.h>

static const size_t g_CommandMemorySize = 512;
static const size_t g_ControlMemorySize = 512;
static const int    g_NumColorBuffers = 2;

class GettingStartedWithNVN : public TutorialBaseClass
{
    public:
        GettingStartedWithNVN() ;
        virtual ~GettingStartedWithNVN() ;
        virtual void Init(PFNNVNBOOTSTRAPLOADERPROC loader, NVNnativeWindow nativeWindow);
        virtual void Shutdown();

        virtual void Draw(uint64_t millisec) ;
        virtual void Resize(int width, int height) ;

    private:
        static void NVNAPIENTRY DebugLayerCallback(
            NVNdebugCallbackSource source,
            NVNdebugCallbackType type,
            int id,
            NVNdebugCallbackSeverity severity,
            const char* message,
            void* pUser
            ) ;

        int UpdateRenderTargets() ;

        NVNdevice         m_Device;
        NVNqueue          m_Queue;

        MemoryPool*       m_pCommandMemoryPool;

        ptrdiff_t         m_CommandPoolOffset;
        void*             m_pControlPool;
        NVNcommandBuffer  m_CommandBuffer;
        NVNcommandHandle  m_CommandHandle;

        ptrdiff_t         m_RenderTargetCommandPoolOffset;
        void*             m_pRenderTargetControlPool;
        NVNcommandBuffer  m_RenderTargetCommandBuffer;
        NVNcommandHandle  m_RenderTargetCommandHandle;

        NVNtextureBuilder m_RenderTargetBuilder;
        NVNtexture*       m_RenderTargets[g_NumColorBuffers];

        NVNsync           m_CommandBufferSync;

        MemoryPool*       m_pRenderTargetMemoryPool;

        NVNwindow*        m_pWindow;
        NVNwindowBuilder  m_WindowBuilder;
        size_t            m_ColorTargetSize;
        int               m_CurrentIndex;
        int               m_Width;
        int               m_Height;

        NVNbuffer         m_CopyBuffer;
        MemoryPool*       m_CopyMemoryPool;
};

/*
 * GettingStartedWithNVN Constructor
 * ----------------------------
 * Sets up default values for the members of the class.
 */
GettingStartedWithNVN::GettingStartedWithNVN() :
    m_pCommandMemoryPool(NULL),
    m_pControlPool(NULL),
    m_CommandHandle(NULL),
    m_pRenderTargetControlPool(NULL),
    m_RenderTargetCommandHandle(NULL),
    m_pRenderTargetMemoryPool(NULL),
    m_pWindow(NULL),
    m_CurrentIndex(0)
{
    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        m_RenderTargets[i] = NULL;
    }
}

/*
 * GettingStartedWithNVN Destructor
 * --------------------------------
 * Empty destructor.
 */
GettingStartedWithNVN::~GettingStartedWithNVN()
{
}

/*
 * GettingStartedWithNVN::Init
 * ---------------------------
 * This method demonstrates basic NVN initialization. The bootstrap loader
 * function pointer is obtained in the platform-specific base class.
 */
void GettingStartedWithNVN::Init(PFNNVNBOOTSTRAPLOADERPROC loader, NVNnativeWindow nativeWindow)
{
    NN_ASSERT(loader != NULL, "Bootstrap loader function pointer is NULL\n");

    /*
     * NVN Device
     * ----------
     * The NVN device object is intended to be a per-process singleton; it is
     * used to establish communication with the GPU, and a parent for all of
     * the other NVN object types.
     *
     * Device Memory
     * -------------
     * The loader function pointer allows access to the NVN device initialization
     * function. This function takes a structure pointer allowing the app to
     * set relevant parameters. One parameter is the "device memory", a block
     * of memory the NVN driver should use for it's internal data structures.
     *
     * Debug Layer
     * -----------
     * It is possible to enable the debug layer via device initialization
     * parameters. The debug layer adds error checking to many NVN entry points,
     * but has no overhead at all when not enabled. The debug layer also will
     * generate error and warning messages when various programming errors are
     * detected, so be sure to provide a message handler callback. See below for
     * an example of how to do this.
     *
     * Device Thread Safety
     * --------------------
     * There is no need to provide synchronization for the NVNdevice object;
     * it may be passed in to any NVN function that requires it from any thread.
     * The exception to this rule is device initialization and finalization;
     * threads may not begin to use the NVNdevice until nvnDeviceInitialize has
     * returned sucessfully, and must cease using the device before
     * nvnDeviceFinalize is called.
     */

    pfnc_nvnDeviceInitialize = reinterpret_cast<PFNNVNDEVICEINITIALIZEPROC>((*loader)("nvnDeviceInitialize"));
    pfnc_nvnDeviceGetProcAddress = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>((*loader)("nvnDeviceGetProcAddress"));
    if (!pfnc_nvnDeviceInitialize)
    {
            /* This can happen if an NVN driver is not installed on a Windows PC. */
        NN_ASSERT(0, "BootstrapLoader failed to find nvnDeviceInitialize");
    }

        /*
         * Load function pointers for creating nvn device.  NULL
         * is passed here at first, the function is called again
         * after device initialization to ensure all the function
         * pointers work properly.
         */
    nvnLoadCProcs( NULL, pfnc_nvnDeviceGetProcAddress );

        /* If debug is enabled, turn on NVN's debug layer. */
    int deviceFlags = 0;
#ifdef _DEBUG
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_BIT | NVN_DEVICE_FLAG_DEBUG_SKIP_CALLS_ON_ERROR_BIT;
#endif

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
            /*
             * This can fail for a few reasons; the most likely on Horizon is
             * insufficent device memory.
             */
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

        /*
         * API Loading
         * -----------
         * Once the device has been initialized, it is now possible to load all of the
         * remaining NVN API entry points. The bootstrap loader is used again to
         * locate the "real" loader function. The real loader may be passed along
         * with the device into nvnLoadProcs if using nvn_FuncPtrInline.h. It is also
         * completely valid for applications to load with their own entry points.
         */
    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress );

        /*
         * Version Check
         * -------------
         * Check for API version mismatches. This is a comparison between the version
         * support reported by the driver, and the version of the NVN headers this
         * source file is compiled with. Exit with an error if the major version
         * mismatches (major revisions are backward-incompatible) or if the driver
         * reports a lower minor version.
         */
    int MajorVersion, MinorVersion;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_API_MAJOR_VERSION, &MajorVersion);
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_API_MINOR_VERSION, &MinorVersion);

    if (MajorVersion != NVN_API_MAJOR_VERSION || MinorVersion < NVN_API_MINOR_VERSION)
    {
        NN_ASSERT(0, "NVN SDK not supported by current driver.");
    }

        /*
         * Debug Layer Callback
         * --------------------
         * Install the debug layer callback if the debug layer was enabled during
         * device initialization. It is possible to pass a pointer to the NVN API
         * to remember and pass back through the debug callback.
         */
    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
            );
    }

        /*
         * NVN Queue
         * ---------
         * The NVNqueue object represent a GPU context and is used to submit command buffers
         * to the GPU. The queue must be associated with an NVNdevice through the Initialize
         * function before being used. Commands submitted to the queue through the command
         * buffers are processed in the order that they are submitted. Commands submitted
         * to one queue are processed independently from commands submittted to other queues.
         * The order in which commands in multiple queues are executed relative to each other
         * is undefined unless a synchronization primitive is used. Each queue has it's own
         * NVN state, meaning that state changes in one queue will not affect rendering in another.
         */
    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDevice(&queueBuilder, &m_Device);
    nvnQueueBuilderSetDefaults(&queueBuilder);
    if(nvnQueueInitialize(&m_Queue, &queueBuilder) == false)
    {
        NN_ASSERT(0, "nvnQueueInitialize failed");
    }

        /*
         * NVN Command Buffer
         * ------------------
         * The NVNcommandBuffer object is used to record a list of commands to be run
         * by the GPU. The command buffer is populated by calling nvnCommandBuffer***
         * methods between calls to the BeginRecording and EndRecording methods. The
         * EndRecording method returns a NVNcommandBufferHandle that can be submitted
         * to the queue to run the commands in the command buffer in the order they
         * were recorded.
         *
         * A command buffer requires two blocks of memory: command memory and control
         * memory. Command memory is used to record the GPU commands. The command
         * memory is an NVNmemoryPool object and should CPU coherent or CPU non-coherent.
         * Control memory is used to store additional information/data needed for the
         * GPU commands to be processed. This memory is simply allocated by the CPU and
         * is not an NVNmemoryPool.
         */
    if(!nvnCommandBufferInitialize(&m_CommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize");
    }

        /*
         * Queries the device for the proper control and command
         * memory alignment for a command buffer.
         */
    int commandBufferCommandAlignment = 0;
    int commandBufferControlAlignment = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandBufferCommandAlignment);
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &commandBufferControlAlignment);

        /* Setup the command buffer memory pool. */
    m_pCommandMemoryPool = new MemoryPool();
    m_pCommandMemoryPool->Init(
        NULL,
        Align(g_CommandMemorySize, commandBufferCommandAlignment) * 2,
        NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
        &m_Device);

        /* Setup the control memory with the proper alignment. */
    m_pControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

        /* Grab the offset from the memory pool. */
    m_CommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

        /* Provides the command buffer with the command and control memory blocks. */
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

        /* Render Target Setting Command Buffer */
    if(!nvnCommandBufferInitialize(&m_RenderTargetCommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize");
    }

        /* Setup the control memory with the proper alignment. */
    m_pRenderTargetControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

        /* Grab the offset from the memory pool. */
    m_RenderTargetCommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

        /*
         * NVN Sync Object
         * ---------------
         * The NVNsync object is a synchronization primitive that can be used to
         * synchronize between queue or between a queue and the CPU. The condition
         * that causes the sync to fire is set by the nvnQueueFenceSync command.
         * The condition can be:
         *
         * ALL_GPU_COMMANDS_COMPLETE     - Signals when all preceding GPU commands have
         *                                 been executed.
         *
         * GRAPHICS_WORLD_SPACE_COMPLETE - Signals when the vertices from the previous
         *                                 commands have been processed, meaning that the
         *                                 vertex, tessellation, and geometry shader stages
         *                                 and the transform feedback stage have all been
         *                                 fully completed.
         */
    if(!nvnSyncInitialize(&m_CommandBufferSync, &m_Device))
    {
        NN_ASSERT(0, "nvnSyncInitialize");
    }

        /*
         * NVN Builder Objects
         * -------------------
         * Some objects in NVN require a builder object to initialize them (buffers, textures, etc).
         * These builder objects are used to setup the state of the object to be created. Once
         * an object has been created through a builder, it can't be edited. Before using a builder
         * object, the SetDefaults method should be called to avoid having unitialized/undefined
         * state.
         */

        /*
         * NVN Texture
         * -----------
         * The NVNtexture object represents a block of GPU memory used to hold image data. It also
         * holds data to describe the image and how the image is allowed to be accessed.
         */

        /*
         * NVN Texture Builder
         * -------------------
         * The NVNtextureBuilder object is used to initialize an NVNtexture with
         * a specific set of states. States that can be set:
         *
         * SetFlags                 - Set flags indentifying special properties of the texture.
         *
         * SetTarget                - Type of the texture (1D, 2D, Array, etc).
         *
         * SetWidth                 - Width of the texture.
         *
         * SetHeight                - Height of the texture.
         *
         * SetDepth                 - Depth of the texture.
         *
         * SetSize*D                - Set width, height, and/or depth together in one call.
         *
         * SetLevels                - Number of mip map levels.
         *
         * SetFormat                - Pixel format.
         *
         * SetSamples               - Number of samples. Must be 0 for non multisample, 2/4/8/16 for multisample.
         *
         * SetSwizzle               - Specifies what is output from a texture lookup.
         *
         * SetDepthStencilMode      - Identifies whether texture lookups will return depth or stencil values.
         *
         * SetStorage               - Set the location in a memory pool where the texture will be allocated from.
         *
         * SetPackagedTextureData   - (Windows only) Pointer to texture data that has been run through the texture packager.
         *                            Converts the texture data from the packaged format back to something usable on Windows.
         *                            This is ignored on the target device as the packaged texture data may be used directly.
         *
         * SetPackagedTextureLayout - Specify the layout of packaged texture data used for texture storage.
         *
         * SetStride                - Specifies the stride for linear textures.
         */
    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);

        /* Render targets that need to be displayed to the screen need both the display access and compressible bit. */
    nvnTextureBuilderSetFlags (&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

        /*
         * Setup the render target memory pool.  Render targets are required to be compressible
         * textures allocated from memory pools with the NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT
         * bit set.
         *
         * Vsync is enforced on the device, so if an application takes too long in a frame to
         * maintain 60 FPS it will drop to 30 FPS.  The tutorials provide 2 render targets to
         * the NVNwindow to swap between but more intensive applications may require more.  An
         * NVNwindow can support up to 6 render targets.
         *
         * For more information on memory pools see MemoryPool.cpp/.h
         */
    m_pRenderTargetMemoryPool = new MemoryPool();
    m_pRenderTargetMemoryPool->Init(
        NULL,
        m_ColorTargetSize * g_NumColorBuffers,
        NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
        &m_Device);

        /*
         * NVN Window
         * ----------
         * The NVNwindow is used to present a render target to the screen. It manages
         * an array of render targets and associates them with an NVNnativeWindow. On
         * Horizon NVNnativeWindow defined as ANativeWindow* and on Windows it's defined
         * as HWND. Each frame, the NVNwindow needs to be queried for the active render
         * target for that frame.
         *
         * Window Builder Settings
         * -----------------------
         * SetTextures        - Sets the list of render targets to be used by the window.
         *                      Textures used by the NVNwindow must be initialized with the
         *                      display access bit. A window can be given a max of 6 targets.
         *
         * SetNativeWindow    - Sets the handle to the window.
         *
         * SetPresentInterval - Sets interval for how often the window is presented to.
         *                      Legal values:
         *                      - 0: Update without waiting for next refresh, tearing likely
         *                           to occur
         *                      - 1: Updates display on next refresh, frame rate limited to
         *                           display's refresh rate
         *                      - 2: Updates display on next refresh, no sooner than two since
         *                           previous update.  Halves display's refresh rate.
         *                      This api is not supported on Windows
         */
    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);
}

/*
 * GettingStartedWithNVN::Shutdown
 * -------------------------------
 * This method cleans up all nvn objects and dynamically allocated memory.
 */
void GettingStartedWithNVN::Shutdown()
{
    nvnQueueFinish(&m_Queue);

    nvnSyncFinalize(&m_CommandBufferSync);

    nvnCommandBufferFinalize(&m_RenderTargetCommandBuffer);

    if(m_pRenderTargetControlPool)
    {
        AlignedDeallocate(m_pRenderTargetControlPool);
        m_pRenderTargetControlPool = NULL;
    }

    nvnCommandBufferFinalize(&m_CommandBuffer);

    if(m_pCommandMemoryPool)
    {
        m_pCommandMemoryPool->Shutdown();
        delete m_pCommandMemoryPool;
        m_pCommandMemoryPool = NULL;
    }

    if(m_pControlPool)
    {
        AlignedDeallocate(m_pControlPool);
        m_pControlPool = NULL;
    }

    if (m_pWindow)
    {
        nvnWindowFinalize(m_pWindow);
        delete m_pWindow;
        m_pWindow = NULL;
    }

    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
        if (m_RenderTargets[i])
        {
            nvnTextureFinalize(m_RenderTargets[i]);
            delete m_RenderTargets[i];
            m_RenderTargets[i] = NULL;
        }
    }

    if(m_pRenderTargetMemoryPool)
    {
        m_pRenderTargetMemoryPool->Shutdown();
        delete m_pRenderTargetMemoryPool;
        m_pRenderTargetMemoryPool = NULL;
    }

    nvnQueueFinalize(&m_Queue);

    nvnDeviceFinalize(&m_Device);
}

/*
 * GettingStartedWithNVN::Resize
 * -----------------------------
 * This method is called everytime the window is resized and is passed
 * the new size of the window. It frees the old render target and creates a new
 * render target with the new screen size. A new set of commands for the command buffer
 * is recorded that performs four scissored clears of different colors on the target.
 */
void GettingStartedWithNVN::Resize(int width, int height)
{
        /* Check for the window being minimized or having no visible surface. */
    if (width == 0 || height == 0)
    {
        return;
    }

    m_Width = width;
    m_Height = height;

        /*
         * Before recording a new command buffer, make sure all previous GPU
         * commands have finished by setting up a fence and waiting for it to
         * be signalled. This so that the memory the command buffer uses isn't
         * overwritten before the GPU is done using it.
         */
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);

        /*
         * A sync placed into the queue as above must first be flushed to the
         * GPU before nvnSyncWait can be called on it.  If not, then a dead
         * lock happens where the CPU is waiting for the GPU to signal a
         * sync it does not know about yet.  With the debug layer active
         * this causes the callback function to be called if the sync is not
         * flushed.
         */
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* If it's the first time Resize is called, allocate the NVNwindow. */
    if (m_pWindow == NULL)
    {
        m_pWindow = new NVNwindow;
    }
        /*
         * Otherwise finalize (free) the NVNwindow used for the previous window size.
         * The NVNWindow must be finalized before a render target it owns is finalized.
         */
    else
    {
        nvnWindowFinalize(m_pWindow);
    }

        /*
         * Set the size of the texture to be created by the texture builder.
         * Implicitly sets depth to one.
         */
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, width, height);

    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
            /* If it's the first time Resize is called, allocate the texture. */
        if (!m_RenderTargets[i])
        {
            m_RenderTargets[i] = new NVNtexture;
        }
            /*
             * Otherwise, finalize (free) the texture from the previous frame.
             * A texture belonging to an NVNwindow must be finalized after the
             * window that owns it is finalized.
             */
        else
        {
            nvnTextureFinalize(m_RenderTargets[i]);
        }

            /* Set the memory pool the texture is to be allocated from and offset to where it will be located. */
        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * i);

            /* Create the texture using the current state of the texture builder. */
        nvnTextureInitialize(m_RenderTargets[i], &m_RenderTargetBuilder);
    }

        /*
         *  Sets the number of render targets that the NVNwindow will have available and
         *  gives a pointer to an array of those targets.
         */
    nvnWindowBuilderSetTextures(&m_WindowBuilder, g_NumColorBuffers, m_RenderTargets);
    nvnWindowInitialize(m_pWindow, &m_WindowBuilder);

        /* Clear Colors */
    float black[4] = { 0.0f, 0.0f, 0.0f, 1.0f };
    float   red[4] = { 1.0f, 0.0f, 0.0f, 1.0f };
    float green[4] = { 0.0f, 1.0f, 0.0f, 1.0f };
    float  blue[4] = { 0.0f, 0.0f, 1.0f, 1.0f };

        /* Temporary variables for size/position of cleared rects. */
    int halfWidth  = static_cast<int>(ceil(width / 2.0f));
    int halfHeight = static_cast<int>(ceil(height / 2.0f));

        /*
         * Reset the command and control memory pointers back to the beginning
         * of their blocks of memory. This causes the next recording to overwrite
         * whatever memory was there.  This is only safe to do if a sync is
         * used to ensure that the command memory about to be overwritten
         * has already been used by the gpu.
         */
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

        /* Starts the recording of a new set of commands for the given command buffer. */
    nvnCommandBufferBeginRecording(&m_CommandBuffer);

    {
            /*
             * Lower left quadrant - black
             * Sets the scissor rectangle by the coordinates of its lower left
             * and its width/height.
             */
        nvnCommandBufferSetScissor(&m_CommandBuffer, 0, 0, halfWidth, halfHeight);

            /*
             * Clears the currently set render target at a given index.
             * Channels to be cleared are set through the clear color mask.
             */
        nvnCommandBufferClearColor(&m_CommandBuffer, 0, black, NVN_CLEAR_COLOR_MASK_RGBA);

            /* Lower right quadrant - red */
        nvnCommandBufferSetScissor(&m_CommandBuffer, halfWidth, 0, width - halfWidth, halfHeight);
        nvnCommandBufferClearColor(&m_CommandBuffer, 0, red, NVN_CLEAR_COLOR_MASK_RGBA);

            /* Upper right quadrant - green */
        nvnCommandBufferSetScissor(&m_CommandBuffer, halfWidth, halfHeight, width - halfWidth, height - halfHeight);
        nvnCommandBufferClearColor(&m_CommandBuffer, 0, green, NVN_CLEAR_COLOR_MASK_RGBA);

            /* Upper left quadrant - blue */
        nvnCommandBufferSetScissor(&m_CommandBuffer, 0, halfHeight, halfWidth, height - halfHeight);
        nvnCommandBufferClearColor(&m_CommandBuffer, 0, blue, NVN_CLEAR_COLOR_MASK_RGBA);
    }

        /*
         * Ends the recording of commands for the command buffer. Returns a
         * command handle to the compiled set of commands that can be submitted
         * to a queue or used as a part of another command buffer.
         */
    m_CommandHandle = nvnCommandBufferEndRecording(&m_CommandBuffer);

#ifdef CHECK_COMMAND_BUFFER_MEMORY_USAGE
        /*
         * In order to pare down the memory used for command/control memory,
         * the command buffer can be queried for the amount of memory that is
         * free or used. This is useful to figure out how much memory to
         * allocate initially for small, reuasble command buffers.
         *
         * More information about these functions can be found in RingBufferManagers.h
         * with regards to command buffer out-of-memory events.
         */
    size_t commandMemoryUsed = nvnCommandBufferGetCommandMemoryUsed(&m_CommandBuffer);
    size_t controlMemoryUsed = nvnCommandBufferGetControlMemoryUsed(&m_CommandBuffer);
#endif
}

/*
 * GettingStartedWithNVN::Draw
 * ---------------------------
 * This method submits the commands recorded in the command buffers to the queue
 * and presents the current render target to the screen.
 */
void GettingStartedWithNVN::Draw(uint64_t /*millisec*/)
{
        /*
         * Update the command buffer to set the current render target and grab
         * the index to that target.
         */
    m_CurrentIndex = UpdateRenderTargets();

        /*
         * Submits the recorded commands in the command buffer to the queue
         * to be processed by the GPU. Multiple command buffers can be
         * submitted with one call by passing the number of command buffers
         * to be submitted and an array of command handles.
         */

        /* Render target setting command buffer. */
    nvnQueueSubmitCommands(&m_Queue, 1, &m_RenderTargetCommandHandle);

        /* Scissored clears command buffer, created/update by the Resize function. */
    nvnQueueSubmitCommands(&m_Queue, 1, &m_CommandHandle);

        /* Displays the render target at the given index to the screen. */
    nvnQueuePresentTexture(&m_Queue, m_pWindow, m_CurrentIndex);
    static bool outputTexture = false;
    if (!outputTexture)
    {
        outputTexture = true;

        nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
        nvnQueueFlush(&m_Queue);
        nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        m_CopyMemoryPool = new MemoryPool();
        int bufferSize = m_Width * m_Height * 4;
        void* copyBuffer = AlignedAllocate(bufferSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
        m_CopyMemoryPool->Init(
            copyBuffer,
            bufferSize,
            NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_UNCACHED_BIT,
            &m_Device);

        NVNbufferBuilder bufferBuilder;
        nvnBufferBuilderSetDefaults(&bufferBuilder);
        nvnBufferBuilderSetDevice(&bufferBuilder, &m_Device);
        nvnBufferBuilderSetStorage(&bufferBuilder, m_CopyMemoryPool->GetMemoryPool(), 0, bufferSize);

        if (!nvnBufferInitialize(&m_CopyBuffer, &bufferBuilder))
        {
            FAIL();
        }

        NVNbufferAddress bufferAddress = nvnBufferGetAddress(&m_CopyBuffer);

        NVNcopyRegion copyRegion;
        memset(&copyRegion, 0, sizeof(NVNcopyRegion));
        copyRegion.width = m_Width;
        copyRegion.height = m_Height;
        copyRegion.depth = 1;

        nvnCommandBufferBeginRecording(&m_CommandBuffer);
        {
            nvnCommandBufferCopyTextureToBuffer(
                &m_CommandBuffer,
                m_RenderTargets[m_CurrentIndex],
                NULL,
                &copyRegion,
                bufferAddress,
                0);
        }
        NVNcommandHandle commandHandle = nvnCommandBufferEndRecording(&m_CommandBuffer);

        nvnQueueSubmitCommands(&m_Queue, 1, &commandHandle);

        nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
        nvnQueueFlush(&m_Queue);
        nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        nn::Result result = nn::fs::MountHostRoot();
        NN_ASSERT(result.IsSuccess());

        nnt::graphics::Path outputsPath(nnt::graphics::GetHostExecutableFilepath());
        outputsPath.CombineAssign("../Outputs");
        outputsPath.NormalizeAssign();

        result = nnt::graphics::CreateDirectories(outputsPath);
        NN_ASSERT(result.IsSuccess());
        result = nn::fs::MountHost("Outputs", outputsPath.GetString());
        NN_ASSERT(result.IsSuccess());

        nnt::graphics::PngIhdr ihdr;
        ihdr.SetDefault();
        ihdr.bitDepth = 8;
        ihdr.channels = 4;
        ihdr.colorType = nnt::graphics::PngColorType_RgbAlpha;
        ihdr.height = 720;
        ihdr.width = 1280;

        nn::mem::StandardAllocator testGraphicsStandardAllocator;
        nnt::graphics::AllocatorFunction testGraphicsAllocatorFunction;

        static const size_t HeapSize = 16 * 1024 * 1024;
        testGraphicsStandardAllocator.Initialize(malloc(HeapSize), HeapSize);
        nnt::graphics::InitializeAllocatorFunctionForStandardAllocator::Initialize(
            &testGraphicsAllocatorFunction,
            &testGraphicsStandardAllocator);

        uint8_t* pMappedBuffer = reinterpret_cast<uint8_t*>(nvnBufferMap(&m_CopyBuffer));

        nnt::graphics::PngIO::Write(
            nnt::graphics::Path("Outputs:/testNVN_Tutorial01_Raw_Output.png").GetString(),
            pMappedBuffer,
            bufferSize,
            &ihdr,
            &testGraphicsAllocatorFunction);

        nn::fs::Unmount("Outputs");
        nn::fs::UnmountHostRoot();

        nvnBufferFinalize(&m_CopyBuffer);
        delete m_CopyMemoryPool;
    }
}

/*
 * GettingStartedWithNVN::UpdateRenderTargets
 * ------------------------------------------
 * Gets the index of the current render target from the NVNwindow
 * and records a command buffer that sets it up to be rendered to.
 */
int GettingStartedWithNVN::UpdateRenderTargets()
{
    int bufferIndex;

        /*
         * Asks for an index to a render target from the NVNwindow. Returns an index into
         * the array of NVNtexture objects that the window was given at initialization
         * through the "textureIndex" parameter.
         */
    nvnQueueAcquireTexture(&m_Queue, m_pWindow, &bufferIndex);

        /*
         * Set the sync object to signal when all gpu commands are complete and wait on the
         * sync object before overwriting the command buffer.
         */
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /*
         * Reset the command and control memory pointers back to the beginning
         * of their blocks of memory. This causes the next recording to overwrite
         * whatever memory was there.
         */
    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

        /*
         * Records a command buffer that sets the render target at the index given
         * by the NVNwindow.
         */
    nvnCommandBufferBeginRecording(&m_RenderTargetCommandBuffer);
    nvnCommandBufferSetRenderTargets(&m_RenderTargetCommandBuffer, 1, &m_RenderTargets[bufferIndex], NULL, NULL, NULL);
    m_RenderTargetCommandHandle = nvnCommandBufferEndRecording(&m_RenderTargetCommandBuffer);

    /* Return the index acquired from the NVNwindow so it can be used in the next present call. */
    return bufferIndex;
}

// static
void GettingStartedWithNVN::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char* message,
    void* pUser
    )
{
    NN_ASSERT(pUser == NULL);

    NN_LOG("NVN Debug Layer Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n",     message);

    NN_ASSERT(0, "NVN Debug layer callback hit");
}

TutorialBaseClass* t()
{
    static GettingStartedWithNVN tut;
    return (&tut);
}

void EndFrameCallbackFunction(bool* pOutExitLoop, const CallbackFunctionType::EndFrameCallbackParameter* param, void*)
{
    //NN_LOG("frame: %d\n", param->currentFrame);
    if(param->currentFrame >= 100)
    {
        *pOutExitLoop = true;
    }
}

TEST(NvnTutorial01, Run)
{
    //NN_LOG("Start test\n");
    CallbackFunctionType cbFunctions;
    cbFunctions.endFrameCallbackFunction = EndFrameCallbackFunction;
    cbFunctions.endFrameCallbackUserParam = NULL;
    TutorialRun(&cbFunctions);
    //NN_LOG("End test\n");
    SUCCEED();
}
