﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/hid.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>
#ifdef _WIN32
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#endif
#include <nvngdSupport/TutorialBaseClass.h>
#include <nvngdSupport/ShaderHeaders/SimpleTexturedModelDataHelper.h>
#include <nvngdSupport/AssetFileLoadingHelper.h>
#include <nvngdSupport/TutorialUtil.h>
#include <nvngdSupport/TextureIDManager.h>
#include <nvngdSupport/UniformBuffer.h>
#include <nvngdSupport/StatTracker.h>
#include <nvngdSupport/DebugTextRenderer.h>

#include <lopProfiler/LOP_Profiler.h>

#include "icosphere.h"

typedef enum RenderingPhase
{
    Render_All = 0,
    Render_Small = 1,
    Render_Big = 2
} RenderingPhase;

static const int    g_NumColorBuffers = 2;
static const int    g_SpheresToRender = 9;
static const size_t g_CommandMemoryChunkSize = 32 * 1024;
static const size_t g_ControlMemoryChunkSize = 4 * 1024;
static const int    g_NumChunks = 2;
static const float  g_Pi = 3.14159f;
static const float  g_ToRad = g_Pi / 180.0f;
RenderingPhase      g_RenderPhase = Render_Small;

typedef std::vector<std::pair<std::string, double> > MetricHolder;

#define TEST_LOP

class GeneratedSphere
{
public:
    GeneratedSphere() :
        m_VertexBuffer(NULL),
        m_IndexBuffer(NULL),
        m_MemoryPool(NULL),
        m_VertexSize(0),
        m_IndexSize(0),
        m_VertexOffset(0),
        m_IndexOffset(0),
        m_MemoryPoolData(NULL),
        m_Initialized(false)
    {
    }

    void Init(NVNdevice* pDevice, int subdivisions = 8)
    {
        if (m_Initialized)
        {
            return;
        }

        // Generate mesh data
        Icosphere::IndexedMesh* mesh = new Icosphere::IndexedMesh;

        Icosphere::makeIcosphere(subdivisions, mesh);

        // Initialize memory pool
        Icosphere::VertexList& verts = mesh->first;
        Icosphere::TriangleList& tris = mesh->second;

        m_VertexSize = verts.size() * 12;//sizeof(nn::util::Vector3fType);
        m_IndexSize = tris.size() * sizeof(Icosphere::Triangle);

        size_t poolSize = Align(Align(m_VertexSize, 16) + m_IndexSize, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
        m_MemoryPoolData = reinterpret_cast<uint8_t*>(AlignedAllocate(poolSize, 16));
        memset(m_MemoryPoolData, 0, poolSize);

        for (size_t i = 0; i < verts.size(); ++i)
        {
            memcpy(m_MemoryPoolData + m_IndexOffset, &verts[i], 12);
            m_IndexOffset += 12;
        }

        m_VertexOffset = 0;
        m_IndexOffset = Align(m_IndexOffset, 4);

        memcpy(m_MemoryPoolData + m_IndexOffset, &tris[0], m_IndexSize);

        m_MemoryPool = new MemoryPool();

        m_MemoryPool->Init(m_MemoryPoolData, poolSize, NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, pDevice);

        nvnMemoryPoolSetDebugLabel(m_MemoryPool->GetMemoryPool(), "Sphere_data");
        // Initialize buffers to hold mesh data
        m_VertexBuffer = new NVNbuffer;
        m_IndexBuffer = new NVNbuffer;

        NVNbufferBuilder bufferBuilder;
        nvnBufferBuilderSetDefaults(&bufferBuilder);
        nvnBufferBuilderSetDevice(&bufferBuilder, pDevice);
        nvnBufferBuilderSetStorage(&bufferBuilder, m_MemoryPool->GetMemoryPool(), m_VertexOffset, m_VertexSize);
        nvnBufferInitialize(m_VertexBuffer, &bufferBuilder);
        nvnBufferSetDebugLabel(m_VertexBuffer, "Vertex_Buffer");

        nvnBufferBuilderSetDefaults(&bufferBuilder);
        nvnBufferBuilderSetDevice(&bufferBuilder, pDevice);
        nvnBufferBuilderSetStorage(&bufferBuilder, m_MemoryPool->GetMemoryPool(), m_IndexOffset, m_IndexSize);
        nvnBufferInitialize(m_IndexBuffer, &bufferBuilder);
        nvnBufferSetDebugLabel(m_IndexBuffer, "Index_Buffer");

        // Cleanup
        delete mesh;
        mesh = NULL;
        m_Initialized = true;
    }

    void Cleanup()
    {
        if (m_VertexBuffer)
        {
            nvnBufferFinalize(m_VertexBuffer);
            delete m_VertexBuffer;
            m_VertexBuffer = NULL;
        }

        if (m_IndexBuffer)
        {
            nvnBufferFinalize(m_IndexBuffer);
            delete m_IndexBuffer;
            m_IndexBuffer = NULL;
        }

        if (m_MemoryPool)
        {
            m_MemoryPool->Shutdown();
            delete m_MemoryPool;
            m_MemoryPool = NULL;
        }
    }

    NVNbuffer*  m_VertexBuffer;
    NVNbuffer*  m_IndexBuffer;
    MemoryPool* m_MemoryPool;

    size_t      m_VertexSize;
    size_t      m_IndexSize;

    size_t      m_VertexOffset;
    size_t      m_IndexOffset;

private:
    uint8_t*    m_MemoryPoolData;
    bool        m_Initialized;
};

class TestLop_Basic3D : public TutorialBaseClass
{
    NN_DISALLOW_COPY(TestLop_Basic3D);

public:
    TestLop_Basic3D();
    virtual ~TestLop_Basic3D();
    virtual void Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow);
    virtual void Shutdown();

    virtual void Draw(uint64_t millisec);
    virtual bool TestDraw(uint64_t millisec);
    virtual void Resize(int width, int height);

    void SetContinueRendering(bool c)
    {
        m_ContinueRendering = c;
    }

    bool ContinueRendering()
    {
        return m_ContinueRendering;
    }

    MetricHolder& GetMetricValues()
    {
        return m_MetricValues;
    }

private:
    static void NVNAPIENTRY DebugLayerCallback(
        NVNdebugCallbackSource source,
        NVNdebugCallbackType type,
        int id,
        NVNdebugCallbackSeverity severity,
        const char* message,
        void* pUser
    );

    void SetupUniformBlock();
    void UpdateUniformBlock(
        ManagedUniformBuffer* pVertexUniformBuffer,
        ManagedUniformBuffer* pFragmentUniformBuffer,
        nn::util::Matrix4x4fType& modelMatrix,
        nn::util::Matrix4x4fType& projectionMatrix,
        NVNtextureHandle handle);

    int UpdateRenderTargets();
    bool UpdateTesselation();
    int PopulateCommandBuffer();

    NVNdevice                       m_Device;
    NVNqueue                        m_Queue;
    void*                           m_pQueueMemory;

    ManagedCommandBuffer*           m_pManagedCommandBuffer;
    NVNcommandHandle                m_CommandHandle;

    NVNtextureBuilder               m_RenderTargetBuilder;
    NVNtexture*                     m_pRenderTargets[g_NumColorBuffers];
    NVNtexture*                     m_pDepthBuffer;
    NVNsamplerBuilder               m_SamplerBuilder;
    NVNbufferBuilder                m_BufferBuilder;

    NVNsync                         m_CommandBufferSync;
    AssetFileLoadingHelper*         m_pAssetLoader;
    AssetFileDataHolder*            m_pDataSphereHolder;

    NVNblendState                   m_BlendState;
    NVNchannelMaskState             m_ChannelMaskState;
    NVNcolorState                   m_ColorState;
    NVNdepthStencilState            m_DepthStencilState;
    NVNmultisampleState             m_MultisampleState;
    NVNpolygonState                 m_PolygonState;

    UniformBufferManager*           m_pUniformBufferManager;
    ManagedUniformBuffer*           m_pVertexUniformBlock[g_SpheresToRender];
    ManagedUniformBuffer*           m_pFragmentUniformBlock[g_SpheresToRender];

    ManagedUniformBuffer*           m_pBigVertexUniformBlock;
    ManagedUniformBuffer*           m_pBigFragmentUniformBlock;

    TextureIDManager*               m_pTextureIDManager;

    MemoryPool*                     m_pRenderTargetMemoryPool;

    ptrdiff_t                       m_RenderTargetCommandPoolOffset;

    NVNwindow*                      m_pWindow;
    NVNwindowBuilder                m_WindowBuilder;
    int                             m_CurrentWindowIndex;
    NVNsync                         m_WindowSync;

    DebugTextRenderer*              m_pDebugTextRenderer;
    FrameBufferedSyncManager*       m_pFrameBufferedSyncManager;

    int                             m_CommandMemoryUsed;
    int                             m_ControlMemoryUsed;

    int                             m_ScreenWidth;
    int                             m_ScreenHeight;
    float                           m_RotY;

    size_t                          m_ColorTargetSize;
    size_t                          m_DepthTargetSize;

    GeneratedSphere                 m_Sphere[2];

#ifdef USE_LOP
    lop::LOP_Profiler*              m_pProfiler;
    std::vector<uint8_t>            m_ConfigImage;
    std::vector<lop::MetricSpec>    m_SelectedMetrics;
    MetricHolder                    m_MetricValues;
    bool                            m_ContinueRendering;
#endif
};

/*
* RangeProfiler_Vertex Constructor
* ----------------------------
* Sets up default values for the members of the class.
*/

TestLop_Basic3D::TestLop_Basic3D() :
    m_pQueueMemory(NULL),
    m_pManagedCommandBuffer(NULL),
    m_CommandHandle(0),
    m_pDepthBuffer(NULL),
    m_pAssetLoader(NULL),
    m_pDataSphereHolder(NULL),
    m_pUniformBufferManager(NULL),
    m_pBigVertexUniformBlock(NULL),
    m_pBigFragmentUniformBlock(NULL),
    m_pTextureIDManager(NULL),
    m_pRenderTargetMemoryPool(NULL),
    m_RenderTargetCommandPoolOffset(0),
    m_pWindow(NULL),
    m_CurrentWindowIndex(-1),
    m_pDebugTextRenderer(NULL),
    m_pFrameBufferedSyncManager(NULL),
    m_CommandMemoryUsed(0),
    m_ControlMemoryUsed(0),
    m_ScreenWidth(0),
    m_ScreenHeight(0),
    m_RotY(45.0f),
    m_ColorTargetSize(0),
    m_DepthTargetSize(0)
#ifdef USE_LOP
    , m_pProfiler(NULL)
    , m_ContinueRendering(true)
#endif
{
    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        m_pRenderTargets[i] = NULL;
    }

    for (int i = 0; i < g_SpheresToRender; ++i)
    {
        m_pVertexUniformBlock[i] = NULL;
        m_pFragmentUniformBlock[i] = NULL;
    }
}

/*
* RangeProfiler_Vertex Destructor
* ---------------------------
* Empty destructor.
*/
TestLop_Basic3D::~TestLop_Basic3D()
{
}

/*
* RangeProfiler_Vertex::SetupUniformBlock
* -----------------------------------
* Initializes the uniform buffers used for the
* shader program.
*/
void TestLop_Basic3D::SetupUniformBlock()
{
    for (int i = 0; i < g_SpheresToRender; ++i)
    {
        m_pVertexUniformBlock[i] = m_pUniformBufferManager->CreateUniformBuffer(sizeof(SimpleTexturedModel::BlockVSUniformBlockData));
        m_pFragmentUniformBlock[i] = m_pUniformBufferManager->CreateUniformBuffer(sizeof(SimpleTexturedModel::BlockFSUniformBlockData));
    }

    m_pBigVertexUniformBlock = m_pUniformBufferManager->CreateUniformBuffer(sizeof(SimpleTexturedModel::BlockVSUniformBlockData));
    m_pBigFragmentUniformBlock = m_pUniformBufferManager->CreateUniformBuffer(sizeof(SimpleTexturedModel::BlockFSUniformBlockData));
}

/*
* RangeProfiler_Vertex::UpdateUniformBlock
* ------------------------------------
* Updates the uniform blocks with the given data.
*/
void TestLop_Basic3D::UpdateUniformBlock(ManagedUniformBuffer* pVertexUniformBuffer, ManagedUniformBuffer* pFragmentUniformBuffer, nn::util::Matrix4x4fType& modelMatrix, nn::util::Matrix4x4fType& projectionMatrix, NVNtextureHandle handle)
{
    /* Update Vertex shader uniform buffer. */

    /* Get the current mapped point fom the managed buffer. */
    void* vertexPoolMap = pVertexUniformBuffer->GetMappedPointer();

    SimpleTexturedModel::BlockVSUniformBlockData vertexShaderData;

    nn::util::Float4x4 temp;
    nn::util::MatrixStore(&temp, modelMatrix);
    vertexShaderData.SetUniform_u_modelMtx(*reinterpret_cast<float(*)[16]>(&temp));

    nn::util::Matrix4x4fType cameraMat;
    nn::util::MatrixIdentity(&cameraMat);
    nn::util::MatrixStore(&temp, cameraMat);
    vertexShaderData.SetUniform_u_viewMtx(*reinterpret_cast<float(*)[16]>(&temp));

    nn::util::MatrixStore(&temp, projectionMatrix);
    vertexShaderData.SetUniform_u_projMtx(*reinterpret_cast<float(*)[16]>(&temp));

    /* Copy in the uniform data. */
    memcpy(vertexPoolMap, &vertexShaderData, sizeof(SimpleTexturedModel::BlockVSUniformBlockData));

    /* Update Fragment shader uniform buffer. */

    /* Get the current mapped point fom the managed buffer. */
    void* fragmentPoolMap = pFragmentUniformBuffer->GetMappedPointer();

    SimpleTexturedModel::BlockFSUniformBlockData fragmentShaderData;
    fragmentShaderData.SetUniform_u_bindlessTex(handle);

    /* Copy in the uniform data. */
    memcpy(fragmentPoolMap, &fragmentShaderData, sizeof(SimpleTexturedModel::BlockFSUniformBlockData));
}

#ifndef _WIN32
//#define NVN_GRAPHICS_DEBUGGER
#endif

#ifdef NVN_GRAPHICS_DEBUGGER
extern "C" PFNNVNGENERICFUNCPTRPROC nvnDeviceGetToolsProcAddress(NVNdevice *device, const char* name);
#endif

/*!
* RangeProfiler_Vertex::Init
* ----------------------
* Initialize NVN, load asset files, and create objects needed for the
* application to run.
*/
void TestLop_Basic3D::Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow)
{
#ifdef NVN_GRAPHICS_DEBUGGER
    NN_UNUSED(pLoader);
    /* Load procs from NVNGD instead of from the driver, this must be called once prior to nvnDeviceInitialize. */
    nvnLoadCProcs(nullptr, (PFNNVNDEVICEGETPROCADDRESSPROC)nvnDeviceGetToolsProcAddress);

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    int deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT;
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

    nvnDeviceSetDebugLabel(&m_Device, "RangeProfiler_Vertex_Device");

    /* Never call nvnLoadCProcs after nvnDeviceInitialize. */
#else /* NVN_GRAPHICS_DEBUGGER */
    pfnc_nvnDeviceInitialize = reinterpret_cast<PFNNVNDEVICEINITIALIZEPROC>((*pLoader)("nvnDeviceInitialize"));
    pfnc_nvnDeviceGetProcAddress = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>((*pLoader)("nvnDeviceGetProcAddress"));
    if (!pfnc_nvnDeviceInitialize)
    {
        /* This can happen if an NVN driver is not installed on a Windows PC. */
        NN_ASSERT(0, "BootstrapLoader failed to find nvnDeviceInitialize");
    }

    nvnLoadCProcs(NULL, pfnc_nvnDeviceGetProcAddress);

    int MajorVersion, MinorVersion;
    nvnDeviceGetInteger(NULL, NVN_DEVICE_INFO_API_MAJOR_VERSION, &MajorVersion);
    nvnDeviceGetInteger(NULL, NVN_DEVICE_INFO_API_MINOR_VERSION, &MinorVersion);

    if (MajorVersion != NVN_API_MAJOR_VERSION || MinorVersion < NVN_API_MINOR_VERSION)
    {
        NN_ASSERT(0, "NVN SDK not supported by current driver.");
    }

    /* If debug or develop is enabled, turn on NVN's debug layer. */
    int deviceFlags = 0;
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT;
#endif

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
        /*
        * This can fail for a few reasons; the most likely on Horizon is
        * insufficent device memory.
        */
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress);
#endif /* NVN_GRAPHICS_DEBUGGER */

    /*! Setup the debug callback for the debug layer. */
    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
        );
    }

    /* Initialize the queue. */
    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDevice(&queueBuilder, &m_Device);
    nvnQueueBuilderSetDefaults(&queueBuilder);
    nvnQueueBuilderSetComputeMemorySize(&queueBuilder, 0);

    int minQueueCommandMemorySize = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_QUEUE_COMMAND_MEMORY_MIN_SIZE, &minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandMemorySize(&queueBuilder, minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandFlushThreshold(&queueBuilder, minQueueCommandMemorySize);

    size_t neededQueueMemorySize = nvnQueueBuilderGetQueueMemorySize(&queueBuilder);

    if ((neededQueueMemorySize % NVN_MEMORY_POOL_STORAGE_GRANULARITY) != 0)
    {
        NN_ASSERT(0, "Memory size reported for queue is not the proper granularity");
    }

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    m_pQueueMemory = NULL;
#else
    m_pQueueMemory = AlignedAllocate(neededQueueMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
#endif

    nvnQueueBuilderSetQueueMemory(&queueBuilder, m_pQueueMemory, neededQueueMemorySize);

    if (nvnQueueInitialize(&m_Queue, &queueBuilder) == false)
    {
        NN_ASSERT(0, "nvnQueueInitialize failed");
    }

    nvnQueueSetDebugLabel(&m_Queue, "RangeProfiler_Vertex_Queue");

    m_pManagedCommandBuffer = new ManagedCommandBuffer(&m_Device, g_CommandMemoryChunkSize, g_ControlMemoryChunkSize, g_NumChunks);

    /*! Initialize the rendering commad buffer sync. */
    if (!nvnSyncInitialize(&m_CommandBufferSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize command buffer sync");
    }

    nvnSyncSetDebugLabel(&m_CommandBufferSync, "RangeProfiler_Vertex_CommandBufferSync");

    /*! Initialize the window sync. */
    if (!nvnSyncInitialize(&m_WindowSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize window sync");
    }

    nvnSyncSetDebugLabel(&m_WindowSync, "RangeProfiler_Vertex_WindowSync");

    /* State */
    nvnBlendStateSetDefaults(&m_BlendState);

    nvnChannelMaskStateSetDefaults(&m_ChannelMaskState);

    nvnColorStateSetDefaults(&m_ColorState);

    nvnDepthStencilStateSetDefaults(&m_DepthStencilState);
    nvnDepthStencilStateSetDepthTestEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthWriteEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthFunc(&m_DepthStencilState, NVN_DEPTH_FUNC_LESS);

    nvnMultisampleStateSetDefaults(&m_MultisampleState);

    nvnPolygonStateSetDefaults(&m_PolygonState);
    nvnPolygonStateSetFrontFace(&m_PolygonState, NVNfrontFace::NVN_FRONT_FACE_CW);
    nvnPolygonStateSetCullFace(&m_PolygonState, NVNface::NVN_FACE_BACK);
    nvnPolygonStateSetPolygonMode(&m_PolygonState, NVNpolygonMode::NVN_POLYGON_MODE_FILL);

    nvnSamplerBuilderSetDevice(&m_SamplerBuilder, &m_Device);
    nvnSamplerBuilderSetDefaults(&m_SamplerBuilder);

    nvnBufferBuilderSetDevice(&m_BufferBuilder, &m_Device);
    nvnBufferBuilderSetDefaults(&m_BufferBuilder);

    /*! Set up the texture builder for the render target. */
    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1280, 720);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1280, 720);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_DEPTH32F);
    m_DepthTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    /*! Set up the render target memory pool. */
    m_pRenderTargetMemoryPool = new MemoryPool();
    m_pRenderTargetMemoryPool->Init(
        NULL,
        m_ColorTargetSize * g_NumColorBuffers + m_DepthTargetSize,
        NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
        &m_Device);

    nvnMemoryPoolSetDebugLabel(m_pRenderTargetMemoryPool->GetMemoryPool(), "RangeProfiler_Vertex_RenderTargetMemoryPool");

    m_pTextureIDManager = new TextureIDManager(&m_Device);
    m_pUniformBufferManager = new UniformBufferManager(&m_Device, 32 * 1024);

    SetupUniformBlock();

    m_pFrameBufferedSyncManager = new FrameBufferedSyncManager(&m_Device, &m_Queue);
    m_pFrameBufferedSyncManager->RegisterMemoryManager(m_pUniformBufferManager);
    m_pFrameBufferedSyncManager->RegisterMemoryManager(m_pManagedCommandBuffer);

    /////////////////////////////
    /////////////////////////////
    // Loading
    /////////////////////////////
    /////////////////////////////
    size_t cacheSize = 0;
    nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT(result.IsSuccess());

    char* mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

    result = nn::fs::MountRom("rom", mountRomCacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    /*! Create a new asset loader. */
    m_pAssetLoader = new AssetFileLoadingHelper(&m_Device, m_pTextureIDManager);

    /*! Load in the asset file for the tutorial. */
    m_pDataSphereHolder = m_pAssetLoader->LoadAssetFile("lopBasic_3D.out");
    m_pDataSphereHolder->SetupAttributeStatesNVN(SimpleTexturedModel::Attributes::GetAttributeLocation);
    m_pDataSphereHolder->SetupTextureSamplerHandle(&m_Device, m_pTextureIDManager, &m_SamplerBuilder);

    /////////////////////////////
    /////////////////////////////
    // Debug Text Renderer
    /////////////////////////////
    /////////////////////////////

    m_pDebugTextRenderer = DebugTextRenderer::GetInstance();

    m_pDebugTextRenderer->Init(&m_Device, &m_Queue, m_pTextureIDManager, m_pFrameBufferedSyncManager);
    m_pDebugTextRenderer->SetDrawEnable(true);
    m_pDebugTextRenderer->SetSpacing(false);
    m_pDebugTextRenderer->SetGridSize(120.0f, 36.0f);

#ifdef USE_LOP
    /////////////////////////////
    /////////////////////////////
    // Profiler
    /////////////////////////////
    /////////////////////////////

    void* place = AlignedAllocate(sizeof(lop::LOP_Profiler), NN_ALIGNOF(lop::LOP_Profiler));
    m_pProfiler = new(place) lop::LOP_Profiler();
    m_pProfiler->Initialize(&m_Device, &m_Queue);
    m_pProfiler->InitializeChipDesc();

    // Basic 3D Activity
    static const char* const ppMetricNames[] = {
        "gr__cycles_active",
        "pda__cycles_active",
        "pda__input_prims",

        "mmu__cycles_active",
        "scc__cycles_active",
        "crop__cycles_active",
        "zrop__cycles_active",
        "vpc__cycles_active",

        "raster__zcull_input_tiles",
        "raster__frstr_output_subtiles",
        "prop__zrop_output_active",

        "vaf__alpha_cycles_active",
        "sm__cycles_active",
        "sm__cycles_active_3d",
        "sm__cycles_active_3d_ps",
        nullptr
    };

    int metricPrint[] = {
        lop::MetricPrint_Avg,                                               // gr__cycles_active
        lop::MetricPrint_Avg | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // pda__cycles_active
        lop::MetricPrint_Sum,                                               // pda__input_prims

        lop::MetricPrint_Avg | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // mmu__cycles_active
        lop::MetricPrint_Avg | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // scc__cycles_active
        lop::MetricPrint_Avg | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // crop__cycles_active
        lop::MetricPrint_Avg | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // zrop__cycles_active
        lop::MetricPrint_Avg | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // vpc__cycles_active

        lop::MetricPrint_Sum | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // raster__zcull_input_tiles
        lop::MetricPrint_Sum | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // raster__frstr_output_subtiles
        lop::MetricPrint_Sum | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // prop__zrop_output_active

        lop::MetricPrint_Avg | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // vaf__alpha_cycles_active
        lop::MetricPrint_Avg | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // sm__cycles_active
        lop::MetricPrint_Avg | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // sm__cycles_active_3d
        lop::MetricPrint_Avg | lop::MetricPrint_Pct_Peak_Sustained_Elapsed, // sm__cycles_active_3d_ps
    };

    NN_ASSERT(m_pProfiler->GenerateMetricSpecs(ppMetricNames, metricPrint, m_SelectedMetrics));
    m_pProfiler->PrepareOfflineData(m_ConfigImage, m_SelectedMetrics);

    m_pProfiler->BeginSession(m_ConfigImage);

    /////////////////////////////
    /////////////////////////////
    // Profiler
    /////////////////////////////
    /////////////////////////////
#endif

    nn::fs::Unmount("rom");
    delete[] mountRomCacheBuffer;

    /*! Set up the window builder. */
    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);

    m_Sphere[0].Init(&m_Device, 2);
    m_Sphere[1].Init(&m_Device, 8);
}//NOLINT(impl/function_size)

 /*!
 * RangeProfiler_Vertex::Shutdown
 * --------------------------
 * This method cleans up all nvn objects and dynamically allocated memory.
 */
void TestLop_Basic3D::Shutdown()
{
#ifdef USE_LOP
    if (m_pProfiler)
    {
        m_pProfiler->EndSession();
        AlignedDeallocate(m_pProfiler);
    }
#endif

    if (m_pManagedCommandBuffer != NULL)
    {
        delete m_pManagedCommandBuffer;
        m_pManagedCommandBuffer = NULL;
    }

    m_Sphere[0].Cleanup();
    m_Sphere[1].Cleanup();

    nvnQueueFinish(&m_Queue);
    nvnSyncFinalize(&m_CommandBufferSync);
    nvnSyncFinalize(&m_WindowSync);

    if (m_pDebugTextRenderer != NULL)
    {
        m_pDebugTextRenderer->CleanUp();
        m_pDebugTextRenderer = NULL;
    }

    if (m_pWindow)
    {
        nvnWindowFinalize(m_pWindow);
        delete m_pWindow;
        m_pWindow = NULL;
    }

    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        if (m_pRenderTargets[i])
        {
            nvnTextureFinalize(m_pRenderTargets[i]);
            delete m_pRenderTargets[i];
            m_pRenderTargets[i] = NULL;
        }
    }

    if (m_pDepthBuffer)
    {
        nvnTextureFinalize(m_pDepthBuffer);
        delete m_pDepthBuffer;
        m_pDepthBuffer = NULL;
    }

    if (m_pUniformBufferManager != NULL)
    {
        delete m_pUniformBufferManager;
        m_pUniformBufferManager = NULL;
    }

    if (m_pFrameBufferedSyncManager != NULL)
    {
        delete m_pFrameBufferedSyncManager;
        m_pFrameBufferedSyncManager = NULL;
    }

    if (m_pDataSphereHolder)
    {
        delete m_pDataSphereHolder;
        m_pDataSphereHolder = NULL;
    }

    if (m_pAssetLoader)
    {
        delete m_pAssetLoader;
        m_pAssetLoader = NULL;
    }

    if (m_pRenderTargetMemoryPool)
    {
        m_pRenderTargetMemoryPool->Shutdown();
        delete m_pRenderTargetMemoryPool;
        m_pRenderTargetMemoryPool = NULL;
    }

    if (m_pTextureIDManager)
    {
        delete m_pTextureIDManager;
        m_pTextureIDManager = NULL;
    }

    nvnQueueFinalize(&m_Queue);
    if (m_pQueueMemory)
    {
        AlignedDeallocate(m_pQueueMemory);
        m_pQueueMemory = NULL;
    }

    nvnDeviceFinalize(&m_Device);
}

/*!
* RangeProfiler_Vertex::Draw
* --------------------------
* This method updates uniform data and submits the commands
* recorded in the command buffer to the queue and presents
*/
void TestLop_Basic3D::Draw(uint64_t /*millisec*/)
{
    UpdateTesselation();
    int currentRenderTargetIndex = PopulateCommandBuffer();
#ifdef TEST_LOP
    static float xPos[] = { -2.20f, 0.0f, 2.20f, -2.20f, 0.0f, 2.20f, -2.20f, 0.0f, 2.20f };
    static float yPos[] = { 1.0f, 1.0f, 1.0f, 0.0f, 0.0f, 0.0f, -1.0f, -1.0f, -1.0f };
#else
    static float xPos[] = { 0.20f, 1.20f, 2.20f, 0.20f, 1.20f, 2.20f, 0.20f, 1.20f, 2.20f };
    static float yPos[] = { 1.0f, 1.0f, 1.0f, 0.0f, 0.0f, 0.0f, -1.0f, -1.0f, -1.0f };
#endif

    nn::util::Matrix4x4fType projMat44;
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projMat44, 60.0f * g_ToRad, static_cast<float>(m_ScreenWidth) / static_cast<float>(m_ScreenHeight), 1.0f, 1000.0f);

    std::vector<NVNTextureData*>& textureDataSimple = m_pDataSphereHolder->GetTextureData();

    if (g_RenderPhase == RenderingPhase::Render_All || g_RenderPhase == RenderingPhase::Render_Small)
    {
        for (int i = 0; i < g_SpheresToRender; ++i)
        {
            /*! Update the uniform data. */
            nn::util::Matrix4x3fType simpleTranslateMat;
            nn::util::MatrixIdentity(&simpleTranslateMat);
            nn::util::Vector3fType simpleTranslateVector;
            nn::util::VectorSet(&simpleTranslateVector, xPos[i], yPos[i], -3.0f);
            nn::util::MatrixSetTranslate(&simpleTranslateMat, simpleTranslateVector);

            nn::util::Matrix4x3fType simpleRotateXMat;
            nn::util::MatrixIdentity(&simpleRotateXMat);
            nn::util::Vector3fType simpleRotateXVector;
            nn::util::VectorSet(&simpleRotateXVector, 30.0f * g_ToRad, 0.0f, 0.0f);
            nn::util::MatrixSetRotateXyz(&simpleRotateXMat, simpleRotateXVector);

            nn::util::Matrix4x3fType simpleRotateYMat;
            nn::util::MatrixIdentity(&simpleRotateYMat);
            nn::util::Vector3fType simpleRotateYVector;
            nn::util::VectorSet(&simpleRotateYVector, 0.0f, m_RotY * (i + 1) * g_ToRad, 0.0f);
            nn::util::MatrixSetRotateXyz(&simpleRotateYMat, simpleRotateYVector);

            nn::util::Matrix4x3fType simpleScaleMat;
            nn::util::MatrixIdentity(&simpleScaleMat);
            nn::util::Vector3fType simpleScaleVector;
            nn::util::VectorSet(&simpleScaleVector, 0.45f, 0.45f, 0.45f);
            nn::util::MatrixSetScale(&simpleScaleMat, simpleScaleVector);

            nn::util::Matrix4x3fType simpleModelMat;
            nn::util::Matrix4x3fType tempMat0;
            nn::util::Matrix4x3fType tempMat1;
            nn::util::MatrixMultiply(&tempMat0, simpleScaleMat, simpleRotateXMat);
            nn::util::MatrixMultiply(&tempMat1, simpleRotateYMat, tempMat0);
            nn::util::MatrixMultiply(&simpleModelMat, tempMat1, simpleTranslateMat);

            nn::util::Matrix4x4fType simpleModelMat44;
            nn::util::MatrixConvert(&simpleModelMat44, simpleModelMat);

            UpdateUniformBlock(m_pVertexUniformBlock[i], m_pFragmentUniformBlock[i], simpleModelMat44, projMat44, textureDataSimple[0]->m_TextureHandle);
        }
    }

    if (g_RenderPhase == RenderingPhase::Render_All || g_RenderPhase == RenderingPhase::Render_Big)
    {
        /*! Update the uniform data. */
        nn::util::Matrix4x3fType simpleTranslateMat;
        nn::util::MatrixIdentity(&simpleTranslateMat);
        nn::util::Vector3fType simpleTranslateVector;
        nn::util::VectorSet(&simpleTranslateVector, 3.0f, 0.0f, -10.0f);
        nn::util::MatrixSetTranslate(&simpleTranslateMat, simpleTranslateVector);

        nn::util::Matrix4x3fType simpleRotateXMat;
        nn::util::MatrixIdentity(&simpleRotateXMat);
        nn::util::Vector3fType simpleRotateXVector;
        nn::util::VectorSet(&simpleRotateXVector, 30.0f * g_ToRad, 0.0f, 0.0f);
        nn::util::MatrixSetRotateXyz(&simpleRotateXMat, simpleRotateXVector);

        nn::util::Matrix4x3fType simpleRotateYMat;
        nn::util::MatrixIdentity(&simpleRotateYMat);
        nn::util::Vector3fType simpleRotateYVector;
        nn::util::VectorSet(&simpleRotateYVector, 0.0f, m_RotY * g_ToRad, 0.0f);
        nn::util::MatrixSetRotateXyz(&simpleRotateYMat, simpleRotateYVector);

        nn::util::Matrix4x3fType simpleScaleMat;
        nn::util::MatrixIdentity(&simpleScaleMat);
        nn::util::Vector3fType simpleScaleVector;
        nn::util::VectorSet(&simpleScaleVector, 4.0f, 4.0f, 4.0f);
        nn::util::MatrixSetScale(&simpleScaleMat, simpleScaleVector);

        nn::util::Matrix4x3fType simpleModelMat;
        nn::util::Matrix4x3fType tempMat0;
        nn::util::Matrix4x3fType tempMat1;
        nn::util::MatrixMultiply(&tempMat0, simpleScaleMat, simpleRotateXMat);
        nn::util::MatrixMultiply(&tempMat1, simpleRotateYMat, tempMat0);
        nn::util::MatrixMultiply(&simpleModelMat, tempMat1, simpleTranslateMat);

        nn::util::Matrix4x4fType simpleModelMat44;
        nn::util::MatrixConvert(&simpleModelMat44, simpleModelMat);
        UpdateUniformBlock(m_pBigVertexUniformBlock, m_pBigFragmentUniformBlock, simpleModelMat44, projMat44, textureDataSimple[0]->m_TextureHandle);
    }

    m_pFrameBufferedSyncManager->InsertFence();

    /*!
    * Wait on sync that was received in UpdateRenderTargets now that we are
    * actually ready to use the render target
    */
    nvnSyncWait(&m_WindowSync, NVN_WAIT_TIMEOUT_MAXIMUM);

    /*! Submit the command buffer to render the cube. */
#ifdef USE_LOP
    NVPA_Bool allPassesDecoded = false;
    static bool first = true;
    int runs = 0;
    do
    {
        m_pProfiler->BeginPass();
        {
            /* Submit the commands to the queue. */
            nvnQueueSubmitCommands(&m_Queue, 1, &m_CommandHandle);
        }
        m_pProfiler->EndPass();
        nvnQueueFinish(&m_Queue);
        m_pProfiler->DecodeCounters(m_ConfigImage, allPassesDecoded);

        ++runs;

        if (allPassesDecoded)
        {
#ifdef TEST_LOP
            m_pProfiler->UnpackRawMetrics_Test(m_SelectedMetrics, m_MetricValues);
            m_ContinueRendering = false;
#else
            m_pDebugTextRenderer->SetColor(1.0f, 1.0f, 1.0f, 1.0f);
            m_pDebugTextRenderer->Printf(0, 0, "Runs: %i", runs);

            std::string outputString = "%-35s = %8.0f\n";
            m_pProfiler->UnpackRawMetrics(m_SelectedMetrics, false, m_pDebugTextRenderer, outputString, 1, false);
#endif
            m_pProfiler->ProfilerInitializeCounterData();
            break;
        }
    } while (!allPassesDecoded && !first);

    first = false;
#ifndef TEST_LOP
    m_pDebugTextRenderer->Draw(m_pRenderTargets[currentRenderTargetIndex], m_ScreenWidth, m_ScreenHeight);
#endif
#else
    nvnQueueSubmitCommands(&m_Queue, 1, &m_CommandHandle);
#endif

    /*! Present the target to the screen. */
    nvnQueuePresentTexture(&m_Queue, m_pWindow, currentRenderTargetIndex);

    /* Swap the multi buffered memory. */
    m_pFrameBufferedSyncManager->SwapPools();
}//NOLINT(impl/function_size)

void DrawHelper(TestLop_Basic3D* bas)
{
    std::vector<std::pair<std::string, double> >& metricValues = bas->GetMetricValues();
    metricValues.clear();

    while (bas->ContinueRendering())
    {
        bas->Draw(0);
    }

    bas->SetContinueRendering(true);
}

double CycleRatioHelper(double small, double big, double all)
{
    double cyclesDiff = abs(all - (small + big));
    return cyclesDiff / all;
}

void WithinRange(double testValue, double lowerBound, double upperBound, double* variance)
{
    if (lowerBound > testValue)
    {
        (*variance) += std::min(2.5, (lowerBound - testValue) / (upperBound - lowerBound));
    }

    else if (upperBound < testValue)
    {
        (*variance) += std::min(2.5, (testValue - upperBound) / (upperBound - lowerBound));
    }
}

/*
 * Test is set to run on NX only.  Values being tested below would
 * need to be determined for every graphics card that this test
 * could be run on.
 */
#if defined ( NN_BUILD_TARGET_PLATFORM_OS_NN )
//#define OUTPUT_METRIC_VALUES
TEST(Basic3DTest, test_VarianceRange)
{
    TestLop_Basic3D* bas = reinterpret_cast<TestLop_Basic3D*>(t());
    MetricHolder& metricValues = bas->GetMetricValues();

        // Renders 9 small spheres on the right side of the screen
    g_RenderPhase = Render_Small;

    std::vector<double> averagedValues(28, 0.0);

    int loops = 100;

    std::vector<MetricHolder> metrics;
    metrics.reserve(loops);

        // Render 100 times and store metric values for each frame
        // for variance and average values for range testing
    for (int i = 0; i < loops; ++i)
    {
        DrawHelper(bas);

        metrics.push_back(metricValues);
        for (int j = 0; j < metricValues.size(); ++j)
        {
            averagedValues[j] += (metricValues[j].second);
        }
    }

        // Get average values for the runs
    for (int i = 0; i < averagedValues.size(); ++i)
    {
        averagedValues[i] = averagedValues[i] / static_cast<double>(loops);
    }

        // Checks if average values are within acceptable ranges
    EXPECT_LT(4750.0, averagedValues[1]);       EXPECT_LT(averagedValues[1], 6000.0);

    EXPECT_EQ(averagedValues[3], 2880.0);
    EXPECT_LT(1600.0,   averagedValues[4]);     EXPECT_LT(averagedValues[4], 1800.0);
    EXPECT_LT(0.5,      averagedValues[5]);     EXPECT_LT(averagedValues[5], 1.25);
    EXPECT_LT(1650.0,   averagedValues[6]);     EXPECT_LT(averagedValues[6], 1950);
    EXPECT_LT(0.5,      averagedValues[7]);     EXPECT_LT(averagedValues[7], 1.25);
    EXPECT_LT(130000.0, averagedValues[8]);     EXPECT_LT(averagedValues[8], 160000.0);
    EXPECT_LT(55.0,     averagedValues[9]);     EXPECT_LT(averagedValues[9], 75.0);
    EXPECT_LT(73000.0,  averagedValues[10]);    EXPECT_LT(averagedValues[10], 83000.0);
    EXPECT_LT(27.5,     averagedValues[11]);    EXPECT_LT(averagedValues[11], 37.5);
    EXPECT_LT(5000.0,   averagedValues[12]);    EXPECT_LT(averagedValues[12], 7000.0);
    EXPECT_LT(1.5,      averagedValues[13]);    EXPECT_LT(averagedValues[13], 3.5);
    EXPECT_LT(12500.0,  averagedValues[14]);    EXPECT_LT(averagedValues[14], 13750.0);
    EXPECT_LT(4.0,      averagedValues[15]);    EXPECT_LT(averagedValues[15], 6.5);
    EXPECT_LT(18500.0,  averagedValues[16]);    EXPECT_LT(averagedValues[16], 20000.0);
    EXPECT_LT(6.5,      averagedValues[17]);    EXPECT_LT(averagedValues[17], 9.5);
    EXPECT_LT(15000.0,  averagedValues[18]);    EXPECT_LT(averagedValues[18], 16500.0);
    EXPECT_LT(5.5,      averagedValues[19]);    EXPECT_LT(averagedValues[19], 8.5);
    EXPECT_LT(4300.0,   averagedValues[20]);    EXPECT_LT(averagedValues[20], 4700.0);
    EXPECT_LT(1.5,      averagedValues[21]);    EXPECT_LT(averagedValues[21], 2.5);

    EXPECT_LT(60.0,     averagedValues[23]);    EXPECT_LT(averagedValues[23], 75.0);

    EXPECT_LT(60.0,     averagedValues[25]);    EXPECT_LT(averagedValues[25], 75.0);

    EXPECT_LT(58.0,     averagedValues[27]);    EXPECT_LT(averagedValues[27], 70.0);

    std::vector<double> variance(28, 0.0);

        // Checks how often values leave acceptable range
    for (MetricHolder& metricHolder : metrics)
    {
        WithinRange(metricHolder[1].second, 4750.0, 6000.0, &variance[1]);

        WithinRange(metricHolder[3].second, 2875.0, 2885.0, &variance[3]);
        WithinRange(metricHolder[4].second, 1600.0, 1800.0, &variance[4]);
        WithinRange(metricHolder[5].second, 0.5, 1.25, &variance[5]);
        WithinRange(metricHolder[6].second, 1650.0, 1950, &variance[6]);
        WithinRange(metricHolder[7].second, 0.5, 1.25, &variance[7]);

        WithinRange(metricHolder[9].second, 55.0, 75.0, &variance[9]);
        WithinRange(metricHolder[10].second, 73000.0, 83000.0, &variance[10]);
        WithinRange(metricHolder[11].second, 27.5, 37.5, &variance[11]);
        WithinRange(metricHolder[12].second, 5000.0, 7000.0, &variance[12]);
        WithinRange(metricHolder[13].second, 1.5, 3.5, &variance[13]);
        WithinRange(metricHolder[14].second, 12500.0, 13750.0, &variance[14]);
        WithinRange(metricHolder[15].second, 4.0, 6.5, &variance[15]);
        WithinRange(metricHolder[16].second, 18500.0, 20000.0, &variance[16]);
        WithinRange(metricHolder[17].second, 6.5, 9.5, &variance[17]);
        WithinRange(metricHolder[18].second, 15000.0, 16500.0, &variance[18]);
        WithinRange(metricHolder[19].second, 5.5, 8.5, &variance[19]);
        WithinRange(metricHolder[20].second, 4300.0, 4700.0, &variance[20]);
        WithinRange(metricHolder[21].second, 1.5, 2.5, &variance[21]);

        WithinRange(metricHolder[23].second, 60.0, 75.0, &variance[23]);

        WithinRange(metricHolder[25].second, 60.0, 75.0, &variance[25]);

        WithinRange(metricHolder[27].second, 58.0, 70.0, &variance[27]);
    }

    for (int i = 0; i < variance.size(); ++i)
    {
#ifdef OUTPUT_METRIC_VALUES
        if (variance[i] > 10.0)
        {
            NN_LOG("[%i] %s\n", i, metrics[0][i].first.c_str());
            NN_LOG("[%i] Variance: %f \n", i, variance[i]);
            for (MetricHolder& metricHolder : metrics)
            {
                NN_LOG("  %f\n", metricHolder[i].second);
            }
            EXPECT_LE(variance[i], 10.0);
        }
#else
        EXPECT_LE(variance[i], 10.0);
#endif
    }
}

TEST(Basic3DTest, test_Basic3D)
{
    TestLop_Basic3D* bas = reinterpret_cast<TestLop_Basic3D*>(t());
    MetricHolder& metricValues = bas->GetMetricValues();

    bool rerun = false;

    do
    {
        rerun = false;

        ///////////
        // Small //
        ///////////
        g_RenderPhase = Render_Small;
        DrawHelper(bas);

        auto smallValues = metricValues;

        /////////
        // Big //
        /////////
        g_RenderPhase = Render_Big;
        DrawHelper(bas);

        auto bigValues = metricValues;

        /////////
        // All //
        /////////
        g_RenderPhase = Render_All;
        DrawHelper(bas);

        auto allValues = metricValues;

        EXPECT_LT(smallValues[2].second, 20.0);                         // pda__cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(bigValues[2].second, 90.0);                           // pda__cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(allValues[2].second, 90.0);                           // pda__cycles_active.Pct_Peak_Sustained_Elapsed

        EXPECT_EQ(smallValues[3].second + bigValues[3].second, allValues[3].second); // pda__input_prims.Sum

        double mmuCyclesDiffPct = CycleRatioHelper(smallValues[4].second, bigValues[4].second, allValues[4].second);
        EXPECT_LT(mmuCyclesDiffPct, 0.01);                              // mmu__cycles_active.Avg

        EXPECT_LT(smallValues[5].second, 10.0);                         // mmu__cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(bigValues[5].second, 10.0);                           // mmu__cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_LT(bigValues[5].second, 25.0);                           // mmu__cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(allValues[5].second, 10.0);                           // mmu__cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_LT(allValues[5].second, 25.0);                           // mmu__cycles_active.Pct_Peak_Sustained_Elapsed

        double sccCyclesDiffPct = CycleRatioHelper(smallValues[6].second, bigValues[6].second, allValues[6].second);
        EXPECT_LT(sccCyclesDiffPct, 0.1);                               // scc_cycles_active.Avg

        EXPECT_LT(smallValues[7].second, 10.0);                         // scc_cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(bigValues[7].second, 30.0);                           // scc_cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_LT(bigValues[7].second, 50.0);                           // scc_cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(allValues[7].second, 30.0);                           // scc_cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_LT(allValues[7].second, 50.0);                           // scc_cycles_active.Pct_Peak_Sustained_Elapsed

        EXPECT_LT(smallValues[8].second, allValues[8].second);          // crop__cycles_active.Avg
        EXPECT_LT(allValues[8].second, bigValues[8].second);            // crop__cycles_active.Avg

        EXPECT_LT(bigValues[9].second, 15.0);                           // crop__cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_LT(allValues[9].second, 10.0);                           // crop__cycles_active.Pct_Peak_Sustained_Elapsed

        EXPECT_LT(smallValues[10].second, allValues[10].second);        // zrop__cycles_active.Avg
        EXPECT_LT(allValues[10].second, bigValues[10].second);          // zrop__cycles_active.Avg

        EXPECT_LT(smallValues[11].second, 50.0);                        // zrop__cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_LT(bigValues[11].second, 20.0);                          // zrop__cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(bigValues[11].second, 10.0);                          // zrop__cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_LT(allValues[11].second, 10.0);                          // zrop__cycles_active.Pct_Peak_Sustained_Elapsed

        EXPECT_GT(bigValues[13].second, 95.0);                          // vpc__cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(allValues[13].second, 95.0);                          // vpc__cycles_active.Pct_Peak_Sustained_Elapsed

        EXPECT_GT(smallValues[14].second + bigValues[14].second, allValues[14].second); // raster__zcull_input_tiles.Sum

        EXPECT_LT(smallValues[16].second, allValues[16].second);        // raster__frstr_output_subtiles.Sum
        EXPECT_LT(allValues[16].second, bigValues[16].second);          // raster__frstr_output_subtiles.Sum

        EXPECT_LT(smallValues[18].second, allValues[18].second);        // prop__zrop_output_active.Sum
        EXPECT_LT(allValues[18].second, bigValues[18].second);          // prop__zrop_output_active.Sum

        EXPECT_LT(smallValues[21].second, 65.0);                        // vaf__alpha_cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(bigValues[21].second, 95.0);                          // vaf__alpha_cycles_active.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(allValues[21].second, 95.0);                          // vaf__alpha_cycles_active.Pct_Peak_Sustained_Elapsed

        EXPECT_LT(smallValues[25].second, bigValues[25].second);        // sm__cycles_active_3d.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(bigValues[25].second, 90.0);                          // sm__cycles_active_3d.Pct_Peak_Sustained_Elapsed

        EXPECT_LT(smallValues[26].second, allValues[26].second);        // sm__cycles_active_3d_ps.Avg
        EXPECT_LT(allValues[26].second, bigValues[26].second);          // sm__cycles_active_3d_ps.Avg

        EXPECT_LT(bigValues[27].second, 45.0);                          // sm__cycles_active_3d_ps.Pct_Peak_Sustained_Elapsed
        EXPECT_LT(allValues[27].second, 25.0);                          // sm__cycles_active_3d_ps.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(smallValues[27].second, bigValues[27].second);        // sm__cycles_active_3d_ps.Pct_Peak_Sustained_Elapsed
        EXPECT_GT(bigValues[27].second, allValues[27].second);          // sm__cycles_active_3d_ps.Pct_Peak_Sustained_Elapsed
    } while (rerun);
}
#endif

bool TestLop_Basic3D::TestDraw(uint64_t /*millisec*/)
{
#ifdef TEST_LOP
    int result = RUN_ALL_TESTS();
    m_pProfiler->EndSession();
    nnt::Exit(result);
#else
    Draw(0);
#endif
    return true;
}

/*
* RangeProfiler_Vertex::UpdateRenderTargets
* -------------------------------------
* Gets the index of the current render target from the NVNwindow
* and records a command buffer that sets it up to be rendered to.
*/
int TestLop_Basic3D::UpdateRenderTargets()
{
    /* Get next render target to be used */
    NVNwindowAcquireTextureResult result = nvnWindowAcquireTexture(m_pWindow, &m_WindowSync, &m_CurrentWindowIndex);

    NN_ASSERT(result == NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS && m_CurrentWindowIndex != -1);

    /* Record the command buffer to set the target. */
    nvnCommandBufferSetRenderTargets(m_pManagedCommandBuffer->GetCommandBuffer(), 1, &m_pRenderTargets[m_CurrentWindowIndex], NULL, m_pDepthBuffer, NULL);

    /* Return the index. */
    return m_CurrentWindowIndex;
}

bool TestLop_Basic3D::UpdateTesselation()
{
#if defined ( NN_BUILD_TARGET_PLATFORM_OS_WIN )
#ifndef TEST_LOP
    static int cooldown = 10;
    if (cooldown < 120)
    {
        ++cooldown;
    }
    else
    {
        g_RenderPhase = static_cast<RenderingPhase>((g_RenderPhase + 1) % 3);
        cooldown = 0;
    }

    return false;
#else
    return false;
#endif
#else
#ifndef TEST_LOP
    static int cooldown = 10;
    if (cooldown < 10)
    {
        ++cooldown;
    }
    else
    {
        nn::hid::TouchScreenState<3> touchScreenState;

        int touchCount = nn::hid::GetTouchScreenStates(&touchScreenState, 1);
        if (touchScreenState.count > 0 && touchCount)
        {
            if (cooldown < 10)
            {
                return false;
            }
            g_RenderPhase = static_cast<RenderingPhase>((g_RenderPhase + 1) % 3);
            cooldown = 0;
            return true;
        }
    }
    return false;
#else
    return false;
#endif
#endif
}

/*!
* RangeProfiler_Vertex::Resize
* -----------------------------
* This method is called everytime the window is resized and is passed
* the new size of the window. It frees the old render target and creates a new
* render target with the new screen size. A new set of commands for the command buffer
* is recorded.
*/
void TestLop_Basic3D::Resize(int width, int height)
{
    /*! Check for the window being minimized or having no visible surface. */
    if (width == 0 || height == 0)
    {
        return;
    }

    /*! Save the current height and width of the screen. */
    m_ScreenHeight = height;
    m_ScreenWidth = width;

    /*! Sync to safely rewrite the command buffer. */
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

    /*! If it's the first time Resize is called, allocate the NVNwindow. */
    if (m_pWindow == NULL)
    {
        m_pWindow = new NVNwindow;
    }

    /*
    * Otherwise finalize (free) the NVNwindow used for the previous window size.
    * The NVNWindow must be finalized before a render target it owns is finalized.
    */
    else
    {
        nvnWindowFinalize(m_pWindow);
    }

    /*! Set up the builder for the render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);

    std::string renderTargetName = "RangeProfiler_Vertex_RenderTarget_";
    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        /*! If it's the first time Resize is called, allocate the render target. */
        if (m_pRenderTargets[i] == NULL)
        {
            m_pRenderTargets[i] = new NVNtexture;
        }
        /*! Otherwise finalize (free) the render target used for the previous window size. */
        else
        {
            nvnTextureFinalize(m_pRenderTargets[i]);
        }

        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * i);

        /*! Create the texture using the current state of the texture builder. */
        if (!nvnTextureInitialize(m_pRenderTargets[i], &m_RenderTargetBuilder))
        {
            NN_ASSERT(0, "Failed to initialize render target");
        }

        char indexAsChar = static_cast<char>(i + '0');
        std::string updatedTargetName = renderTargetName + indexAsChar;
        nvnTextureSetDebugLabel(m_pRenderTargets[i], updatedTargetName.c_str());
    }

    /*! If it's the first time Resize is called, allocate the depth buffer. */
    if (m_pDepthBuffer == NULL)
    {
        m_pDepthBuffer = new NVNtexture;
    }
    /*! Otherwise finalize (free) the depth buffer used for the previous window size. */
    else
    {
        nvnTextureFinalize(m_pDepthBuffer);
    }

    /*! Initialize depth buffer for render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_DEPTH32F);
    nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * g_NumColorBuffers);

    if (!nvnTextureInitialize(m_pDepthBuffer, &m_RenderTargetBuilder))
    {
        NN_ASSERT(0, "Failed to initialize depth target");
    }

    nvnTextureSetDebugLabel(m_pDepthBuffer, "RangeProfiler_Vertex_DepthBuffer");

    /*! Pass off the render targets to the window. */
    nvnWindowBuilderSetTextures(&m_WindowBuilder, g_NumColorBuffers, m_pRenderTargets);
    nvnWindowInitialize(m_pWindow, &m_WindowBuilder);

    nvnWindowSetDebugLabel(m_pWindow, "RangeProfiler_Vertex_Window");
}//NOLINT(impl/function_size)

int TestLop_Basic3D::PopulateCommandBuffer()
{
    int currentRenderTargetIndex = -1;

    /*! Starts the recording of a new set of commands for the given command buffer. */
    m_pManagedCommandBuffer->BeginRecording();
    {
        NVNcommandBuffer* commandBuffer = m_pManagedCommandBuffer->GetCommandBuffer();
        nvnCommandBufferPushDebugGroup(m_pManagedCommandBuffer->GetCommandBuffer(), "Draw");

        currentRenderTargetIndex = UpdateRenderTargets();

        /*! Bind the texture and sampler descriptor pools. */
        m_pTextureIDManager->SetSamplerPool(commandBuffer);
        m_pTextureIDManager->SetTexturePool(commandBuffer);

        // Sets the scissor rectangle and viewport to the full screen
        nvnCommandBufferSetScissor(commandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);
        nvnCommandBufferSetViewport(commandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);

        /*! Clears the currently set render target at a given index. */
        float clear_color[4] = { 0.4f, 0.55f, 0.6f, 1.0f };
        nvnCommandBufferClearColor(commandBuffer, 0, clear_color, NVN_CLEAR_COLOR_MASK_RGBA);
        nvnCommandBufferClearDepthStencil(commandBuffer, 1.0, NVN_TRUE, 0, 0);

        std::vector<NVNProgramData*>&   programData = m_pDataSphereHolder->GetProgramData();

        nvnProgramSetDebugLabel(&programData[0]->m_Program, "RangeProfiler_Vertex_Program");

        /*! Bind the render state objects. */
        nvnCommandBufferBindBlendState(commandBuffer, &m_BlendState);
        nvnCommandBufferBindChannelMaskState(commandBuffer, &m_ChannelMaskState);
        nvnCommandBufferBindColorState(commandBuffer, &m_ColorState);
        nvnCommandBufferBindDepthStencilState(commandBuffer, &m_DepthStencilState);
        nvnCommandBufferBindMultisampleState(commandBuffer, &m_MultisampleState);
        nvnCommandBufferBindPolygonState(commandBuffer, &m_PolygonState);
        nvnCommandBufferSetSampleMask(commandBuffer, static_cast<uint32_t>(~0));

        /*! Bind the shader program. */
        nvnCommandBufferBindProgram(commandBuffer, &programData[0]->m_Program, programData[0]->m_ShaderStages);

        // Render small spheres
        if (g_RenderPhase == RenderingPhase::Render_All || g_RenderPhase == RenderingPhase::Render_Small)
        {
            NVNbufferAddress smallSphereVbo = nvnBufferGetAddress(m_Sphere[0].m_VertexBuffer);
            nvnCommandBufferBindVertexBuffer(commandBuffer, 0, smallSphereVbo, m_Sphere[0].m_VertexSize);

            for (int i = 0; i < g_SpheresToRender; ++i)
            {
                /*! Bind the uniform buffer for the vertex shader. */
                uint32_t vertBindingLoc = SimpleTexturedModel::BlockVSUniformBlockData::GetBinding(NVN_SHADER_STAGE_VERTEX);
                nvnCommandBufferBindUniformBuffer(commandBuffer, NVN_SHADER_STAGE_VERTEX, vertBindingLoc, m_pVertexUniformBlock[i]->GetCurrentBufferAddress(), sizeof(SimpleTexturedModel::BlockVSUniformBlockData));

                /*! Bind the uniform buffer for the fragment shader. */
                uint32_t fragBindingLoc = SimpleTexturedModel::BlockFSUniformBlockData::GetBinding(NVN_SHADER_STAGE_FRAGMENT);
                nvnCommandBufferBindUniformBuffer(commandBuffer, NVN_SHADER_STAGE_FRAGMENT, fragBindingLoc, m_pFragmentUniformBlock[i]->GetCurrentBufferAddress(), sizeof(SimpleTexturedModel::BlockFSUniformBlockData));

                /*! Bind the vertex attribute states and vertex stream states. */
                NVNvertexAttribState attribState;
                nvnVertexAttribStateSetDefaults(&attribState);
                nvnVertexAttribStateSetFormat(&attribState, NVN_FORMAT_RGB32F, 0);
                nvnVertexAttribStateSetStreamIndex(&attribState, 0);

                nvnCommandBufferBindVertexAttribState(commandBuffer, 1, &attribState);

                NVNvertexStreamState streamState;
                nvnVertexStreamStateSetDefaults(&streamState);
                nvnVertexStreamStateSetStride(&streamState, 12);

                nvnCommandBufferBindVertexStreamState(commandBuffer, 1, &streamState);

                size_t numIndices = m_Sphere[0].m_IndexSize / 4;

                nvnCommandBufferDrawElements(commandBuffer,
                    NVN_DRAW_PRIMITIVE_TRIANGLES,
                    NVN_INDEX_TYPE_UNSIGNED_INT,
                    static_cast<int>(numIndices),
                    nvnBufferGetAddress(m_Sphere[0].m_IndexBuffer));
            }
        }

        if (g_RenderPhase == RenderingPhase::Render_All || g_RenderPhase == RenderingPhase::Render_Big)
        {
            // Render big sphere
            NVNbufferAddress bigSphereVbo = nvnBufferGetAddress(m_Sphere[1].m_VertexBuffer);
            nvnCommandBufferBindVertexBuffer(commandBuffer, 0, bigSphereVbo, m_Sphere[1].m_VertexSize);

            uint32_t vertBindingLoc = SimpleTexturedModel::BlockVSUniformBlockData::GetBinding(NVN_SHADER_STAGE_VERTEX);
            nvnCommandBufferBindUniformBuffer(commandBuffer, NVN_SHADER_STAGE_VERTEX, vertBindingLoc, m_pBigVertexUniformBlock->GetCurrentBufferAddress(), sizeof(SimpleTexturedModel::BlockVSUniformBlockData));

            uint32_t fragBindingLoc = SimpleTexturedModel::BlockFSUniformBlockData::GetBinding(NVN_SHADER_STAGE_FRAGMENT);
            nvnCommandBufferBindUniformBuffer(commandBuffer, NVN_SHADER_STAGE_FRAGMENT, fragBindingLoc, m_pBigFragmentUniformBlock->GetCurrentBufferAddress(), sizeof(SimpleTexturedModel::BlockFSUniformBlockData));

            NVNvertexAttribState attribState;
            nvnVertexAttribStateSetDefaults(&attribState);
            nvnVertexAttribStateSetFormat(&attribState, NVN_FORMAT_RGB32F, 0);
            nvnVertexAttribStateSetStreamIndex(&attribState, 0);

            nvnCommandBufferBindVertexAttribState(commandBuffer, 1, &attribState);

            NVNvertexStreamState streamState;
            nvnVertexStreamStateSetDefaults(&streamState);
            nvnVertexStreamStateSetStride(&streamState, 12);

            nvnCommandBufferBindVertexStreamState(commandBuffer, 1, &streamState);

            size_t numIndices = m_Sphere[1].m_IndexSize / 4;

            nvnCommandBufferDrawElements(commandBuffer,
                NVN_DRAW_PRIMITIVE_TRIANGLES,
                NVN_INDEX_TYPE_UNSIGNED_INT,
                static_cast<int>(numIndices),
                nvnBufferGetAddress(m_Sphere[1].m_IndexBuffer));
        }
        nvnCommandBufferPopDebugGroup(commandBuffer);
    }

    /*! End the command buffer recording. */
    m_CommandHandle = m_pManagedCommandBuffer->EndRecording();
#ifdef CHECK_COMMAND_BUFFER_MEMORY_USAGE
    m_CommandMemoryUsed = nvnCommandBufferGetCommandMemoryUsed(&m_CommandBuffer);
    m_ControlMemoryUsed = nvnCommandBufferGetControlMemoryUsed(&m_CommandBuffer);
#endif

    return currentRenderTargetIndex;
}

// static
void TestLop_Basic3D::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char* message,
    void* pUser
)
{
    NN_ASSERT(pUser == NULL);

    NN_LOG("NVN Debug Layer Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n", message);
}

TutorialBaseClass* t()
{
    static TestLop_Basic3D s_Tut;
    return (&s_Tut);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    TutorialRun(true);
}
