﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// 全スロットが未確保の状態から初めて
// 複数スレッドから同時に AcquireIndexRange() するテスト

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nnt.h>
#include <cstdlib>
#include <algorithm>

#include <nn/gfx/util/detail/gfx_IndexRingBuffer.h>
#include "testGfxUtil_WorkerThreads.h"

namespace {
    struct ThreadParameter
    {
        nn::gfx::util::detail::IndexRingBuffer* pRing;
        int acquireCount;
        int rangeSize;
        int* acquiredIndices;
    };
    void ThreadFunction(int tid, ThreadParameter* pParam)
    {
        NN_UNUSED(tid);
        for(int i = 0; i < pParam->acquireCount; i++)
        {
            pParam->acquiredIndices[i] = pParam->pRing->AcquireIndexRange(pParam->rangeSize);
        }
    }
}

TEST(IndexRingBuffer, MultiThreadRange)
{
    using namespace nnt::gfxUtil;
    static const int AcquireCount = 128;
    static const int RangeSizeMax = 10;
    static const int RangeSizeMin = 1;
    static const int StackSize = 1024 * 1024;
    static const int MaxNumberOfThreads = 64;
    static const int TrialCount = 1000;
    NN_STATIC_ASSERT(StackSize % nn::os::ThreadStackAlignment == 0);

    for(int trial = 0; trial < TrialCount; trial++)
    {
        const int InvalidIndex = nn::gfx::util::detail::IndexRingBuffer::InvalidIndex;
        nn::gfx::util::detail::IndexRingBuffer ring;

        // ワーカースレッドを用意
        WorkerThreads<ThreadParameter> wthreads;
        wthreads.Initialize(ThreadFunction, StackSize);
        const int numThreads = wthreads.GetNumberOfThreads();
        // パラメータを設定
        int sumAcquireSlotCount = 0;
        ThreadParameter params[MaxNumberOfThreads];
        for(int t = 0; t < numThreads; t++)
        {
            ThreadParameter& param = params[t];
            param.pRing = &ring;
            param.rangeSize = std::max(RangeSizeMax - t, RangeSizeMin);
            param.acquireCount = AcquireCount;
            param.acquiredIndices = static_cast<int*>(AlignedAllocate(sizeof(int) * AcquireCount + 1,  NN_ALIGNOF(int)));
            for(int j = 0; j < AcquireCount + 1; j++)
            {
                param.acquiredIndices[j] = InvalidIndex;
            }
            wthreads.SetThreadParameter(t, &param);
            sumAcquireSlotCount += param.rangeSize * param.acquireCount;
        }
        wthreads.PrepareWorker();

        int ringSize = sumAcquireSlotCount + RangeSizeMax;
        ring.Initialize(0, ringSize);
        int firstIndex;
        {
            // head の位置を真ん中あたりに移動しておく
            nn::gfx::util::detail::IndexRange range;
            ring.Begin();
            ring.AcquireIndexRange(ringSize / 2);
            ring.End(&range);
            firstIndex = range.base + range.count;
            ring.ReleaseIndexRange(&range);
        }
        ring.Begin();
        {
            wthreads.StartWorker();
            wthreads.WaitWorkerComplete();
        }
        nn::gfx::util::detail::IndexRange range;
        ring.End(&range);

        // 正しく確保されているか確認
        {
            int foundCount = 0;
            int skipCount = 0;
            // 確保された内容が正しいか確認
            {
                int* nextIndices[MaxNumberOfThreads];
                for(int t = 0; t < numThreads; t++)
                {
                    nextIndices[t] = params[t].acquiredIndices;
                }

                // 前半
                for(int i = firstIndex; i < static_cast<int>(ringSize); )
                {
                    bool found = false;
                    for(int t = 0; t < numThreads; t++)
                    {
                        int value = *nextIndices[t];
                        if(value == i)
                        {
                            found = true;
                            nextIndices[t]++;
                            i += static_cast<int>(params[t].rangeSize);
                            foundCount += params[t].rangeSize;
                            break;
                        }
                    }
                    if(!found)
                    {
                        skipCount = ringSize - i;
                        break;
                    }
                }
                // 後半
                for(int i = 0; i < firstIndex; )
                {
                    bool found = false;
                    for(int t = 0; t < numThreads; t++)
                    {
                        int value = *nextIndices[t];
                        if(value == i)
                        {
                            found = true;
                            nextIndices[t]++;
                            i += static_cast<int>(params[t].rangeSize);
                            foundCount += params[t].rangeSize;
                            break;
                        }
                    }
                    if(!found)
                    {
                        break;
                    }
                }
                EXPECT_EQ(sumAcquireSlotCount, foundCount);
                EXPECT_EQ(range.count, foundCount + skipCount);
            }
        }

        // 後始末
        for(int t = 0; t < numThreads; t++)
        {
            AlignedFree(params[t].acquiredIndices);
        }
        wthreads.Finalize();
    }
}// NOLINT(impl/function_size)
