﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_DebugPad.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NPadJoyDual.h>

#include "testGfxUtil_ControllerInput.h"

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#include <Windows.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#endif

namespace
{

nn::hid::DebugPadState g_DebugPadState = {};
nn::hid::KeyboardState g_HidKeyboardState = {};
#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
nn::hid::NpadHandheldState g_NpadHandheldState = {};
nn::hid::NpadJoyDualState g_NpadJoyDualState = {};
#endif


unsigned int ConvertNpadButtonSet(const nn::hid::NpadButtonSet& npadButtonSet)
{
    unsigned int resultMask = 0;

    if (npadButtonSet.Test<::nn::hid::NpadButton::Left>())
    {
        resultMask |= UserInput_Left;
    }

    if (npadButtonSet.Test<::nn::hid::NpadButton::Right>())
    {
        resultMask |= UserInput_Right;
    }

    if (npadButtonSet.Test<::nn::hid::NpadButton::Up>())
    {
        resultMask |= UserInput_Up;
    }

    if (npadButtonSet.Test<::nn::hid::NpadButton::Down>())
    {
        resultMask |= UserInput_Down;
    }

    if (npadButtonSet.Test<::nn::hid::NpadButton::Plus>())
    {
        resultMask |= UserInput_Exit;
    }

    if (npadButtonSet.Test<::nn::hid::NpadButton::A>())
    {
        resultMask |= UserInput_Validate;
        resultMask |= UserInput_A;
    }

    if (npadButtonSet.Test<::nn::hid::NpadButton::B>())
    {
        resultMask |= UserInput_Cancel;
        resultMask |= UserInput_B;
    }

    if (npadButtonSet.Test<::nn::hid::NpadButton::X>())
    {
        resultMask |= UserInput_X;
    }

    if (npadButtonSet.Test<::nn::hid::NpadButton::Y>())
    {
        resultMask |= UserInput_Y;
    }

    if (npadButtonSet.Test<::nn::hid::NpadButton::L>())
    {
        resultMask |= UserInput_L;
    }

    if (npadButtonSet.Test<::nn::hid::NpadButton::R>())
    {
        resultMask |= UserInput_R;
    }

    return resultMask;
}

} // anonymous namespace

void InitializeControllerInput()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeKeyboard();

#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleHandheld::Mask | nn::hid::NpadStyleJoyDual::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));
#endif
}

void FinalizeControllerInput()
{

}

void ClearInputState()
{
    memset(&g_DebugPadState, 0, sizeof(g_DebugPadState));
    memset(&g_HidKeyboardState, 0, sizeof(g_HidKeyboardState));
#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
    memset(&g_NpadHandheldState, 0, sizeof(g_NpadHandheldState));
    memset(&g_NpadJoyDualState, 0, sizeof(g_NpadJoyDualState));
#endif
}


unsigned int UpdateDebugPadInput()
{
    unsigned int resultMask = 0;

    nn::hid::DebugPadButtonSet debugPadButtonCurrent = {};
    nn::hid::DebugPadButtonSet debugPadButtonDown = {};
    nn::hid::DebugPadState state;
    nn::hid::GetDebugPadState(&state);
    debugPadButtonCurrent |= state.buttons;
    debugPadButtonDown |= state.buttons & ~g_DebugPadState.buttons;
    g_DebugPadState = state;

    if (debugPadButtonDown.Test<::nn::hid::DebugPadButton::Left>())
    {
        resultMask |= UserInput_Left;
    }

    if (debugPadButtonDown.Test<::nn::hid::DebugPadButton::Right>())
    {
        resultMask |= UserInput_Right;
    }

    if (debugPadButtonDown.Test<::nn::hid::DebugPadButton::Up>())
    {
        resultMask |= UserInput_Up;
    }

    if (debugPadButtonDown.Test<::nn::hid::DebugPadButton::Down>())
    {
        resultMask |= UserInput_Down;
    }

    if (debugPadButtonDown.Test<::nn::hid::DebugPadButton::Start>())
    {
        resultMask |= UserInput_Exit;
    }

    if (debugPadButtonDown.Test<::nn::hid::DebugPadButton::A>())
    {
        resultMask |= UserInput_Validate;
        resultMask |= UserInput_A;
    }

    if (debugPadButtonDown.Test<::nn::hid::DebugPadButton::B>())
    {
        resultMask |= UserInput_Cancel;
        resultMask |= UserInput_B;
    }

    if (debugPadButtonDown.Test<::nn::hid::DebugPadButton::X>())
    {
        resultMask |= UserInput_X;
    }

    if (debugPadButtonDown.Test<::nn::hid::DebugPadButton::Y>())
    {
        resultMask |= UserInput_Y;
    }

    if (debugPadButtonDown.Test<::nn::hid::DebugPadButton::L>())
    {
        resultMask |= UserInput_L;
    }

    if (debugPadButtonDown.Test<::nn::hid::DebugPadButton::R>())
    {
        resultMask |= UserInput_R;
    }

    return resultMask;
}

unsigned int UpdateControllerKeyboardInput()
{
    unsigned int resultMask = 0;

    nn::hid::KeyboardState keyboardState = {};
    nn::hid::GetKeyboardState(&keyboardState);


    if (keyboardState.attributes.Test<nn::hid::KeyboardAttribute::IsConnected>())
    {
        nn::hid::KeyboardKeySet pressedKeySet = keyboardState.keys & ~g_HidKeyboardState.keys;

        if (pressedKeySet.Test<nn::hid::KeyboardKey::UpArrow>())
        {
            resultMask |= UserInput_Up;
        }

        if (pressedKeySet.Test<nn::hid::KeyboardKey::DownArrow>())
        {
            resultMask |= UserInput_Down;
        }

        if (pressedKeySet.Test<nn::hid::KeyboardKey::LeftArrow>())
        {
            resultMask |= UserInput_Left;
        }

        if (pressedKeySet.Test<nn::hid::KeyboardKey::RightArrow>())
        {
            resultMask |= UserInput_Right;
        }

        if (pressedKeySet.Test<::nn::hid::KeyboardKey::Return>())
        {
            resultMask |= UserInput_Validate;
        }

        if (pressedKeySet.Test<::nn::hid::KeyboardKey::Escape>())
        {
            resultMask |= UserInput_Exit;
        }

        if (pressedKeySet.Test<::nn::hid::KeyboardKey::A>())
        {
            resultMask |= UserInput_A;
        }

        if (pressedKeySet.Test<::nn::hid::KeyboardKey::B>())
        {
            resultMask |= UserInput_B;
        }

        if (pressedKeySet.Test<::nn::hid::KeyboardKey::X>())
        {
            resultMask |= UserInput_X;
        }

        if (pressedKeySet.Test<::nn::hid::KeyboardKey::Y>())
        {
            resultMask |= UserInput_Y;
        }

        if (pressedKeySet.Test<::nn::hid::KeyboardKey::L>())
        {
            resultMask |= UserInput_L;
        }

        if (pressedKeySet.Test<::nn::hid::KeyboardKey::R>())
        {
            resultMask |= UserInput_R;
        }

        g_HidKeyboardState = keyboardState;
    }

    return resultMask;
}


#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
unsigned int UpdateControllerNpadInput()
{
    nn::hid::NpadButtonSet npadButtonCurrent = {};
    nn::hid::NpadButtonSet npadButtonDown = {};


    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
    {
        nn::hid::NpadHandheldState state;
        nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
        npadButtonCurrent = state.buttons;
        npadButtonDown = state.buttons & ~g_NpadHandheldState.buttons;
        g_NpadHandheldState = state;
    }

    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleJoyDual>())
    {
        nn::hid::NpadJoyDualState state;
        nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
        npadButtonCurrent = state.buttons;
        npadButtonDown = state.buttons & ~g_NpadJoyDualState.buttons;
        g_NpadJoyDualState = state;
    }

    return ConvertNpadButtonSet(npadButtonDown);
}
#endif

unsigned int UpdateControllerInput(void* pHostContext)
{
    uint32_t resultMask = 0;
    bool lostFocus = false;

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    nns::gfx::GraphicsFramework* pGfw = reinterpret_cast<nns::gfx::GraphicsFramework*>(pHostContext);
    nn::vi::NativeWindowHandle nativeWindowHandle;
    nn::vi::GetNativeWindow(&nativeWindowHandle, pGfw->GetLayer());
    HWND hWnd = GetForegroundWindow();
    if (hWnd != static_cast<HWND>(nativeWindowHandle))
    {
        lostFocus = true;
    }
#else
    NN_UNUSED(pHostContext);
#endif

    if (lostFocus)
    {
        ClearInputState();
    }
    else
    {
        resultMask |= UpdateDebugPadInput();
        resultMask |= UpdateControllerKeyboardInput();

#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
        resultMask |= UpdateControllerNpadInput();
#endif
    }

    return resultMask;
}
