﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#if defined(NNT_GFX_UTIL_ENABLE_LOP)

#include <nn/nn_Log.h>

#include "testGfxUtil_Application.h"
#include "testGfxUtil_LopIntegration.h"
#include "testGfxUtil_ControllerInput.h"

#include "gfxUtilGpuBenchmark_Property.h"
#include "gfxUtilGpuBenchmark_Factory.h"
#include "gfxUtilGpuBenchmark_PlatformId.h"
#include "gfxUtilGpuBenchmark_JsonStreamer.h"
#include "gfxUtilGpuBenchmark_ResourceAllocator.h"

namespace {

const char* g_LopUnitNameArray[] =
{
    "cdp",
    "crop",
    "fbp",
    "gpc",
    "gpu",
    "gr",
    "host",
    "lts",
    "pda",
    "prop",
    "raster",
    "sm",
    "smsp",
    "stri",
    "tex",
    "tpc",
    "vaf",
    "vpc",
    "zrop",
};
int g_LopUnitNameArrayCount = sizeof(g_LopUnitNameArray) / sizeof(g_LopUnitNameArray[0]);

const int g_WarmupCount = 1;

void RecordGpuBenchmarkCommandListWithLop(
    nnt::gfx::util::GpuBenchmark* pGpuBenchmark, nnt::gfx::util::RuntimeGfxObjects* pRuntimeGfxObjects,
    int warmupRunCount, int measureRunCount)
{
    NN_ASSERT(warmupRunCount >= 0);
    NN_ASSERT(measureRunCount > 0);

    nnt::gfx::util::CommandBufferData* pCommandBufferData = &pRuntimeGfxObjects->benchmarkCommandBuffer;

    nn::gfx::CommandBuffer* pCommandBuffer = &pCommandBufferData->commandBuffer;

    pCommandBuffer->Reset();

    pCommandBuffer->AddCommandMemory(
        pCommandBufferData->pCommandMemoryPool,
        pCommandBufferData->commandMemoryOffset, pCommandBufferData->commandMemorySize);
    pCommandBuffer->AddControlMemory(
        pCommandBufferData->pControlMemory, pCommandBufferData->controlMemorySize);

    pCommandBuffer->Begin();

    pCommandBuffer->InvalidateMemory(
        nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor |
        nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer |
        nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ColorBuffer);

    pCommandBuffer->SetDescriptorPool(pRuntimeGfxObjects->pBufferViewDescriptorPool);
    pCommandBuffer->SetDescriptorPool(pRuntimeGfxObjects->pTextureViewDescriptorPool);
    pCommandBuffer->SetDescriptorPool(pRuntimeGfxObjects->pSamplerDescriptorPool);

    pCommandBuffer->SetVertexState(&pRuntimeGfxObjects->defaultVertexState);
    pCommandBuffer->SetBlendState(&pRuntimeGfxObjects->defaultBlendState);
    pCommandBuffer->SetRasterizerState(&pRuntimeGfxObjects->defaultRasterizerState);
    pCommandBuffer->SetDepthStencilState(&pRuntimeGfxObjects->defaultDepthStencilState);

    pGpuBenchmark->PreBenchmark(pCommandBuffer);

    if (warmupRunCount > 0)
    {
        pGpuBenchmark->DoBenchmark(pCommandBuffer, warmupRunCount);
    }

    profiler::PushProfilerDebugGroup(pCommandBuffer, "benchmark");

    pGpuBenchmark->DoBenchmark(pCommandBuffer, measureRunCount);

    profiler::PopProfilerDebugGroup(pCommandBuffer);

    pCommandBuffer->End();
}

struct MenuData
{
    int displayedLineCount;
    int moduleFilterIndex;

    int selectedResultSection;
    int firstLineMetricIndex;

    int menuIndex;
    int benchmarkRepeatCount;

    bool exitRequested;

    nnt::gfx::util::GpuBenchmark* pGpuBenchmark;

    profiler::ProfilerResult profilerResult;
    int profilerResultRepeatCount;

    profiler::ProfilerResult savedProfilerResult;
    int savedProfilerResultRepeatCount;



    profiler::Profiler profiler;
};

bool MatchModuleFilter(const char* moduleFilter, const char* metricName)
{
    if (moduleFilter == nullptr)
        return true;

    if (strstr(metricName, moduleFilter) == metricName)
    {
        int moduleFilterLen = static_cast<int>(strlen(moduleFilter));
        int metricNameLen = static_cast<int>(strlen(metricName));
        if ((metricNameLen > moduleFilterLen)
            && (metricName[moduleFilterLen] == '_'))
        {
            return true;
        }
    }

    return false;
}



enum FindDirection
{
    FindDirection_Forward = 1,
    FindDirection_Backward = -1,
};

int FindFirstMetricIndex(
    const profiler::Profiler* pProfiler, const char* moduleFilter,
    int startingIndex, FindDirection direction)
{
    int totalMetricCount = pProfiler->GetMetricCount();
    int metricIndex = startingIndex;

    while (((metricIndex >= 0) && (metricIndex < totalMetricCount))
        && (!MatchModuleFilter(moduleFilter, pProfiler->GetMetricName(metricIndex))))
    {
        metricIndex += direction;
    }

    return metricIndex;
}

int CountRemainingMetric(
    const profiler::Profiler* pProfiler, const char* moduleFilter,
    int startingIndex, FindDirection direction)
{
    int totalMetricCount = pProfiler->GetMetricCount();
    int metricIndex = startingIndex;
    int count = 0;

    while ((metricIndex >= 0) && (metricIndex < totalMetricCount))
    {
        if (MatchModuleFilter(moduleFilter, pProfiler->GetMetricName(metricIndex)))
        {
            count++;
        }
        metricIndex += direction;
    }

    return count;
}

int ComputeSelectedMetricCount(const profiler::Profiler* pProfiler, const char* moduleFilter)
{
    int totalMetricCount = pProfiler->GetMetricCount();
    int selectedMetricCount = 0;

    for (int metricIndex = 0; metricIndex < totalMetricCount; ++metricIndex)
    {
        if (MatchModuleFilter(
            moduleFilter,
            pProfiler->GetMetricName(metricIndex)))
        {
            selectedMetricCount++;
        }
    }

    return selectedMetricCount;
}

void DoProfileModePrintProfilerResult(
    ApplicationTestData* pTestData,
    const profiler::Profiler* pProfiler,
    const profiler::ProfilerResult* pProfilerResult,
    const profiler::ProfilerResult* pSavedProfilerResult,
    const MenuData* pMenuData)
{
    float scale = 0.8f;
    pTestData->debugFontWriter.SetFixedWidthEnabled(true);
    pTestData->debugFontWriter.SetScale(scale, scale);
    pTestData->debugFontWriter.SetFixedWidth(11 * scale);

    const char* moduleFilter = g_LopUnitNameArray[pMenuData->moduleFilterIndex];

    int selectedMetricCount = ComputeSelectedMetricCount(pProfiler, moduleFilter);

    int totalMetricCount = pProfiler->GetMetricCount();
    int metricIndex = pMenuData->firstLineMetricIndex;

    NN_ASSERT(
        (selectedMetricCount == 0)
        || MatchModuleFilter(moduleFilter, pProfiler->GetMetricName(metricIndex)));

    int resultArrayOffset =
        pProfilerResult->sectionResultArray[pMenuData->selectedResultSection].resultArrayOffset;

    int remaingCount =
        CountRemainingMetric(pProfiler, moduleFilter, metricIndex, FindDirection_Forward);

    pTestData->debugFontWriter.Print("Filter %s (%d/%d)\n",
        moduleFilter,
        selectedMetricCount - remaingCount + 1,
        selectedMetricCount);

    for (int lineIndex = 0; lineIndex < pMenuData->displayedLineCount; ++lineIndex)
    {
        if (metricIndex < totalMetricCount)
        {
            double totalTime = pProfilerResult->metricResultArray[resultArrayOffset + metricIndex];
            double timePerRepeat = totalTime / static_cast<double>(pMenuData->profilerResultRepeatCount);

            double savedTotalTime = pSavedProfilerResult->metricResultArray[resultArrayOffset + metricIndex];
            double savedTimePerRepeat = savedTotalTime / static_cast<double>(pMenuData->savedProfilerResultRepeatCount);


            char bufferTest[64];
            sprintf(bufferTest, "%8.6f", totalTime);
            char bufferTest2[64];
            sprintf(bufferTest2, "%8.6f", timePerRepeat);
            char bufferTest3[64];
            sprintf(bufferTest3, "%8.6f", savedTimePerRepeat);

            pTestData->debugFontWriter.Print("%-50s: %14s %14s %14s\n",
                pProfiler->GetMetricName(metricIndex),
                bufferTest,
                bufferTest2,
                bufferTest3);
        }
        else
        {
            pTestData->debugFontWriter.Print("\n");
        }

        metricIndex = FindFirstMetricIndex(
            pProfiler, moduleFilter,
            metricIndex + 1, FindDirection_Forward);
    }

    pTestData->debugFontWriter.SetFixedWidthEnabled(false);
    pTestData->debugFontWriter.SetScale(1.0f, 1.0f);
}


void InitializeGpuBenchmark(
    MenuData* pMenuData,
    ApplicationTestData* pTestData,
    nnt::gfx::util::BenchmarkType benchmarkType)
{
    NN_ASSERT(pMenuData->pGpuBenchmark == nullptr);

    const char* benchmarkName = GetBenchmarkNameFromType(benchmarkType);
    pMenuData->pGpuBenchmark = InitializeBenchmarkFromConfiguration(pTestData, benchmarkName, nullptr, 0);
}


void FinalizeGpuBenchmark(MenuData* pMenuData, ApplicationTestData* pTestData)
{
    FinalizeBenchmark(pTestData, pMenuData->pGpuBenchmark);
    pMenuData->pGpuBenchmark = nullptr;
}


enum MenuItem
{
    MenuItem_RepeatCount,
    MenuItem_SelectBenchmark,
    MenuItem_BenchmarkPropertyFirst,
};

enum SelectionDirection
{
    SelectionDirection_Left,
    SelectionDirection_Right
};


void UpdateSelection(MenuData* pMenuData, ApplicationTestData* pTestData, SelectionDirection direction)
{
    pTestData->benchmarkQueue.Flush();
    pTestData->benchmarkQueue.Sync();


    bool rebuildCommandList = false;
    const int powerUpdateIgnoreCount = 32;

    if (pMenuData->menuIndex == MenuItem_RepeatCount)
    {
        int step = (pMenuData->benchmarkRepeatCount <= 10) ? 1 : 10;

        if (direction == SelectionDirection_Left)
        {
            if (pMenuData->benchmarkRepeatCount > step)
                pMenuData->benchmarkRepeatCount -= step;
        }
        else
        {
            pMenuData->benchmarkRepeatCount += step;
        }

        rebuildCommandList = true;
    }
    else  if (pMenuData->menuIndex == MenuItem_SelectBenchmark)
    {
        int previousBenchmarkIndex = pMenuData->pGpuBenchmark->GetType();
        int currentBenchmarkIndex = previousBenchmarkIndex;

        if (direction == SelectionDirection_Left)
        {
            if (currentBenchmarkIndex == 0)
                currentBenchmarkIndex = nnt::gfx::util::BenchmarkType_Max;
            currentBenchmarkIndex = currentBenchmarkIndex - 1;
        }
        else
        {
            currentBenchmarkIndex = currentBenchmarkIndex + 1;
            if (currentBenchmarkIndex == nnt::gfx::util::BenchmarkType_Max)
                currentBenchmarkIndex = 0;
        }

        pTestData->platformMeasurementTracker.ResetAndIgnoreNextUpdates(powerUpdateIgnoreCount);

        FinalizeGpuBenchmark(pMenuData, pTestData);

        nnt::gfx::util::BenchmarkType currentBenchmarkType = static_cast<nnt::gfx::util::BenchmarkType>(currentBenchmarkIndex);
        InitializeGpuBenchmark(pMenuData, pTestData, currentBenchmarkType);

        pMenuData->profiler.ResetCounterValues();
        rebuildCommandList = true;
    }
    else
    {
#if defined(NN_SDK_BUILD_DEBUG)
        pTestData->pResourceAllocator->PopAndCompareMemoryPoolAllocatorStatus();
#endif
        int propertyIndex = pMenuData->menuIndex - MenuItem_BenchmarkPropertyFirst;
        nnt::gfx::util::GpuBenchmark* pBenchmark = pMenuData->pGpuBenchmark;
        nnt::gfx::util::GpuBenchmarkPropertyHolder* pProperty = pBenchmark->GetPropertyByIndex(propertyIndex);
        NN_ASSERT(pProperty != nullptr);
        pBenchmark->FinalizeGfxObjects(pTestData->pResourceAllocator, pTestData->pDevice);
        if (direction == SelectionDirection_Left)
            pProperty->Decrease();
        else
            pProperty->Increase();
        pBenchmark->InitializeGfxObjects(pTestData->pResourceAllocator, pTestData->pDevice);


#if defined(NN_SDK_BUILD_DEBUG)
        pTestData->pResourceAllocator->PushMemoryPoolAllocatorStatus();
        pTestData->pResourceAllocator->UpdateMemoryPoolAllocatorMaxUsage();
#endif
        pMenuData->profiler.ResetCounterValues();

        rebuildCommandList = true;
    }

    if (rebuildCommandList)
    {
        RecordGpuBenchmarkCommandListWithLop(
            pMenuData->pGpuBenchmark, &pTestData->runtimeGfxObjects,
            g_WarmupCount, pMenuData->benchmarkRepeatCount);
    }
}

void UpdateInput(MenuData* pMenuData, ApplicationTestData* pTestData)
{
    unsigned int buttonMask = UpdateControllerInput(pTestData->pHostContext);
    int menuCount = pMenuData->pGpuBenchmark->GetPropertyCount() + MenuItem_BenchmarkPropertyFirst;

    if (buttonMask & UserInput_Left)
    {
        UpdateSelection(pMenuData, pTestData, SelectionDirection_Left);
    }

    if (buttonMask & UserInput_Right)
    {
        UpdateSelection(pMenuData, pTestData, SelectionDirection_Right);
    }

    if (buttonMask & UserInput_Up)
    {
        pMenuData->menuIndex = pMenuData->menuIndex - 1;
        if (pMenuData->menuIndex < 0)
        {
            pMenuData->menuIndex = menuCount - 1;
        }
    }

    if (buttonMask & UserInput_Down)
    {
        pMenuData->menuIndex = pMenuData->menuIndex + 1;
        if (pMenuData->menuIndex >= menuCount)
        {
            pMenuData->menuIndex = 0;
        }

    }

    if (buttonMask & UserInput_X)
    {
        if (pMenuData->firstLineMetricIndex > 0)
        {
            const char* moduleFilter = g_LopUnitNameArray[pMenuData->moduleFilterIndex];
            int metricIndex = FindFirstMetricIndex(
                &pMenuData->profiler, moduleFilter,
                pMenuData->firstLineMetricIndex - 1, FindDirection_Backward);

            if (metricIndex >= 0)
            {
                pMenuData->firstLineMetricIndex = metricIndex;
            }
        }
    }

    if (buttonMask & UserInput_Y)
    {
        int metricCount = pMenuData->profiler.GetMetricCount();

        if (pMenuData->firstLineMetricIndex < (metricCount - 1))
        {
            const char* moduleFilter = g_LopUnitNameArray[pMenuData->moduleFilterIndex];
            int metricIndex = FindFirstMetricIndex(
                &pMenuData->profiler, moduleFilter,
                pMenuData->firstLineMetricIndex + 1, FindDirection_Forward);

            if (metricIndex <= (metricCount - 1))
            {
                pMenuData->firstLineMetricIndex = metricIndex;
            }
        }
    }

    if (buttonMask & UserInput_B)
    {
        int filterIndex = pMenuData->moduleFilterIndex;
        filterIndex = filterIndex - 1;
        if (filterIndex < 0)
        {
            filterIndex = g_LopUnitNameArrayCount - 1;
        }

        pMenuData->firstLineMetricIndex = FindFirstMetricIndex(
            &pMenuData->profiler, g_LopUnitNameArray[filterIndex],
            0, FindDirection_Forward);

        pMenuData->moduleFilterIndex = filterIndex;

    }

    if (buttonMask & UserInput_A)
    {
        int filterIndex = pMenuData->moduleFilterIndex;
        filterIndex = filterIndex + 1;
        if (filterIndex >= g_LopUnitNameArrayCount)
        {
            filterIndex = 0;
        }

        pMenuData->firstLineMetricIndex = FindFirstMetricIndex(
            &pMenuData->profiler, g_LopUnitNameArray[filterIndex],
            0, FindDirection_Forward);

        pMenuData->moduleFilterIndex = filterIndex;
    }

    if (buttonMask & UserInput_L)
    {
        pMenuData->savedProfilerResult = pMenuData->profilerResult;
        pMenuData->savedProfilerResultRepeatCount = pMenuData->profilerResultRepeatCount;
    }

    if (buttonMask & UserInput_Exit)
    {
        pMenuData->exitRequested = true;
    }
}


void PrintPowerUsage(ApplicationTestData* pTestData)
{
    pTestData->debugFontWriter.Print("Average Power Usage Cpu:%4d Gpu:%4d Ddr:%4d VsysAp:%4d\n",
        pTestData->platformMeasurementTracker.GetAverage(PlatformMeasuringPoint_PowerCpu),
        pTestData->platformMeasurementTracker.GetAverage(PlatformMeasuringPoint_PowerGpu),
        pTestData->platformMeasurementTracker.GetAverage(PlatformMeasuringPoint_PowerDdr),
        pTestData->platformMeasurementTracker.GetAverage(PlatformMeasuringPoint_PowerVsysAp));
    pTestData->debugFontWriter.Print("Average temperature internal:%4d external:%4d\n",
        pTestData->platformMeasurementTracker.GetAverage(PlatformMeasuringPoint_ThermalSensorInternal),
        pTestData->platformMeasurementTracker.GetAverage(PlatformMeasuringPoint_ThermalSensorExternal));
}

void DoProfileModeDrawFrame(ApplicationTestData* pTestData, const MenuData* pMenuData)
{
    pTestData->debugFontWriter.SetCursor(0.0f, 0.0f);
    pTestData->debugFontWriter.SetScale(1.0f, 1.0f);

    PrintPowerUsage(pTestData);

    char buffer[256];
    snprintf(buffer, sizeof(buffer), "RepeatCount: %d\n", pMenuData->benchmarkRepeatCount);
    PrintWithSelectionMarker(
        &pTestData->debugFontWriter,
        pMenuData->menuIndex == MenuItem_RepeatCount,
        buffer);

    PrintBenchmarkInformation(
        &pTestData->debugFontWriter,
        pMenuData->pGpuBenchmark,
        pMenuData->menuIndex - MenuItem_BenchmarkPropertyFirst);

    DoProfileModePrintProfilerResult(
        pTestData, &pMenuData->profiler,
        &pMenuData->profilerResult,
        &pMenuData->savedProfilerResult,
        pMenuData);

    DrawFrame(pTestData, pMenuData->pGpuBenchmark);
}



} // anonymous namespace


void DoProfileMode(ApplicationTestData* pTestData, nnt::gfx::util::BenchmarkType benchmarkType)
{
    MenuData menuData;

    profiler::ProfilerOptions profilerOptions;
    profilerOptions.SetDefaults();
    bool initializationSuccess = menuData.profiler.Initialize(
        pTestData->pDevice, &pTestData->benchmarkQueue,
        pTestData->pResourceAllocator,
        profilerOptions);
    if (!initializationSuccess)
    {
        NN_LOG("Profiler initialization failed\n");
        return;
    }

    InitializeControllerInput();

    menuData.displayedLineCount = 20;
    menuData.moduleFilterIndex = 0;
    menuData.selectedResultSection = 0;
    menuData.benchmarkRepeatCount = 1;

    memset(&menuData.profilerResult, 0, sizeof(menuData.profilerResult));
    menuData.profilerResultRepeatCount = 0;

    memset(&menuData.savedProfilerResult, 0, sizeof(menuData.savedProfilerResult));
    menuData.savedProfilerResultRepeatCount = 0;

    menuData.firstLineMetricIndex = FindFirstMetricIndex(
        &menuData.profiler, g_LopUnitNameArray[menuData.moduleFilterIndex],
        0, FindDirection_Forward);
    menuData.menuIndex = MenuItem_SelectBenchmark;
    menuData.exitRequested = false;
    menuData.pGpuBenchmark = nullptr;

    InitializeGpuBenchmark(&menuData, pTestData, benchmarkType);

    RecordGpuBenchmarkCommandList(
        menuData.pGpuBenchmark, &pTestData->runtimeGfxObjects,
        g_WarmupCount, menuData.benchmarkRepeatCount);

    while (true)
    {
        pTestData->platformMeasurementTracker.Update();

        UpdateInput(&menuData, pTestData);

        if (menuData.exitRequested)
        {
            break;
        }

       menuData.profiler.BeginPass(&pTestData->benchmarkQueue);

       pTestData->benchmarkQueue.ExecuteCommand(
           &pTestData->runtimeGfxObjects.benchmarkCommandBuffer.commandBuffer, nullptr);

        menuData.profiler.EndPass(&pTestData->benchmarkQueue);

        DoProfileModeDrawFrame(pTestData, &menuData);

        profiler::ProfilerUpdateResult profilerResult =
            menuData.profiler.Update(&menuData.profilerResult);
        if (profilerResult == profiler::ProfilerUpdateResult_NewResultReady)
        {
            menuData.profilerResultRepeatCount = menuData.benchmarkRepeatCount;
        }
    }

    pTestData->benchmarkQueue.Flush();
    pTestData->benchmarkQueue.Sync();

    FinalizeGpuBenchmark(&menuData, pTestData);

    menuData.profiler.Finalize();

    FinalizeControllerInput();
}

#endif
