﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Log.h>

#include "testGfxUtil_Application.h"
#include "testGfxUtil_ControllerInput.h"

#include "gfxUtilGpuBenchmark_JsonStreamer.h"
#include "gfxUtilGpuBenchmark_Property.h"
#include "gfxUtilGpuBenchmark_Factory.h"
#include "gfxUtilGpuBenchmark_PlatformId.h"
#include "gfxUtilGpuBenchmark_ResourceAllocator.h"

struct FindMaxPowerRecurseState
{
    static const int            WarmUpCount = 4;
    static const nn::TimeSpan   FinMaxPowerTargetDuration;
    static const nn::TimeSpan   TestCaseTargetDuration;
    static const int            RetryCount = 4;

    nnt::gfx::util::GpuBenchmark* pGpuBenchmark;
    nnt::gfx::util::GpuBenchmarkPropertyHolder** pPropertyArray;
    int propertyArraySize;
    int* propertyValueArray;
    int propertyValueArraySize;
    int outMaxGpuPower;
    bool mustExit;
};

const nn::TimeSpan FindMaxPowerRecurseState::FinMaxPowerTargetDuration = nn::TimeSpan::FromMilliSeconds(500);
const nn::TimeSpan FindMaxPowerRecurseState::TestCaseTargetDuration = nn::TimeSpan::FromMilliSeconds(15);


void GetPropertyIterationRange(
    int* pOutMin, int* pOutMax, int* pOutStep,
    const nnt::gfx::util::GpuBenchmarkPropertyHolder* pProperty)
{
    nnt::gfx::util::PropertyType type = pProperty->GetType();
    int min = 0;
    int max = 0;
    int step = 1;

    switch (type)
    {
    case nnt::gfx::util::PropertyType_Enumeration:
        {
            min = 0;
            max = pProperty->ToEnum()->GetCount() - 1;
            step = 1;
        }
        break;
    case nnt::gfx::util::PropertyType_IntegerRange:
        {
            min = pProperty->ToIntegerRange()->GetMin();
            max = pProperty->ToIntegerRange()->GetMax();
            step = pProperty->ToIntegerRange()->GetStep();
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    *pOutMin = min;
    *pOutMax = max;
    *pOutStep = step;
}

void DoFindMaxPowerDrawFrame(
    ApplicationTestData* pTestData, nnt::gfx::util::GpuBenchmark* pGpuBenchmark,
    int currentGpuPower, int maxGpuPower, nn::TimeSpan testDuration)
{
    pTestData->debugFontWriter.SetCursor(0.0f, 0.0f);
    pTestData->debugFontWriter.SetScale(1.0f, 1.0f);
    pTestData->debugFontWriter.Print("Max power: %d\n", maxGpuPower);
    pTestData->debugFontWriter.Print("Previous test power: %d\n", currentGpuPower);
    pTestData->debugFontWriter.Print("Test duration: %llums\n", testDuration.GetMilliSeconds());

    PrintBenchmarkInformation(
        &pTestData->debugFontWriter, pGpuBenchmark,
        g_DisableDrawSelection);

    DrawFrame(pTestData, pGpuBenchmark);
}

void DoFindMaxPowerRecurse(
    ApplicationTestData* pTestData,
    FindMaxPowerRecurseState* pState,
    int propertyIndex)
{
    NN_ASSERT(pState->propertyArraySize == pState->pGpuBenchmark->GetPropertyCount());
    NN_ASSERT(pState->propertyArraySize == pState->propertyValueArraySize);
    NN_ASSERT(propertyIndex <= pState->propertyArraySize);

    if (propertyIndex == pState->propertyArraySize)
    {
        pState->pGpuBenchmark->InitializeGfxObjects(pTestData->pResourceAllocator, pTestData->pDevice);

#if defined(NN_SDK_BUILD_DEBUG)
        pTestData->pResourceAllocator->UpdateMemoryPoolAllocatorMaxUsage();
#endif
        pTestData->platformMeasurementTracker.Reset();

        nn::TimeSpan        testDuration;
        int                 testRunCount;

        RunGpuBenchmarkForDurationWithRetry(
            &testDuration, &testRunCount,
            pTestData, pState->pGpuBenchmark,
            FindMaxPowerRecurseState::WarmUpCount,
            FindMaxPowerRecurseState::FinMaxPowerTargetDuration,
            FindMaxPowerRecurseState::RetryCount);

        int currentGpuPower = pTestData->platformMeasurementTracker.GetCurrent(PlatformMeasuringPoint_PowerGpu);
        if (currentGpuPower > pState->outMaxGpuPower)
        {
            NN_LOG("Found new maximum power usage: %d\n", currentGpuPower);
            LogBenchmarkInformation(pState->pGpuBenchmark);

            pState->outMaxGpuPower = currentGpuPower;

            for (int i = 0; i < pState->propertyArraySize; ++i)
            {
                pState->propertyValueArray[i] = pState->pPropertyArray[i]->Get();
            }
        }

        DoFindMaxPowerDrawFrame(
            pTestData, pState->pGpuBenchmark,
            currentGpuPower, pState->outMaxGpuPower, testDuration);

        pState->pGpuBenchmark->FinalizeGfxObjects(pTestData->pResourceAllocator, pTestData->pDevice);

        unsigned int buttonMask = UpdateControllerInput(pTestData->pHostContext);

        if (buttonMask & UserInput_Exit)
            pState->mustExit = true;

        return;
    }
    else
    {
        nnt::gfx::util::GpuBenchmarkPropertyHolder* pProperty = pState->pPropertyArray[propertyIndex];
        int min = 0;
        int max = 0;
        int step = 0;
        GetPropertyIterationRange(&min, &max, &step, pProperty);

        for (int i = min; i <= max; i += step)
        {
            pProperty->Set(i);
            DoFindMaxPowerRecurse(pTestData, pState, propertyIndex + 1);

            if (pState->mustExit)
            {
                break;
            }
        }
    }
}

void DoFindMaxPowerStartRecurse(ApplicationTestData* pTestData, nnt::gfx::util::GpuBenchmark* pGpuBenchmark)
{
    NN_LOG("Start search for benchmark %s\n", pGpuBenchmark->GetName());

    int propertyCount = pGpuBenchmark->GetPropertyCount();
    size_t propertyArraySize = propertyCount * sizeof(nnt::gfx::util::GpuBenchmarkPropertyHolder**);
    nnt::gfx::util::GpuBenchmarkPropertyHolder** pPropertyArray =
        reinterpret_cast<nnt::gfx::util::GpuBenchmarkPropertyHolder**>(alloca(propertyArraySize));
    pGpuBenchmark->FillPropertyList(pPropertyArray, propertyCount);

    size_t propertyValueArraySize = propertyCount * sizeof(int);
    int* propertyValueArray = reinterpret_cast<int*>(alloca(propertyValueArraySize));
    memset(propertyValueArray, 0, propertyValueArraySize);

    FindMaxPowerRecurseState recurseState;
    recurseState.pGpuBenchmark = pGpuBenchmark;
    recurseState.pPropertyArray = pPropertyArray;
    recurseState.propertyArraySize = propertyCount;
    recurseState.propertyValueArray = propertyValueArray;
    recurseState.propertyValueArraySize = propertyCount;
    recurseState.outMaxGpuPower = -1;
    recurseState.mustExit = false;

    DoFindMaxPowerRecurse(pTestData, &recurseState, 0);

    for (int i = 0; i < propertyCount; ++i)
    {
        pPropertyArray[i]->Set(propertyValueArray[i]);
    }
}

void DoFindMaxPowerMode(ApplicationTestData* pTestData, FindMaxPowerUpdateMode updateMode, const char* outputFilePath, int benchmarkMask)
{
    const int TestIdBufferSize = 256;
    char testIdBuffer[TestIdBufferSize];

    InitializeControllerInput();

    nnt::gfx::util::json::Document* pJsonDocument = nullptr;
    nnt::gfx::util::json::Create(&pJsonDocument, nullptr);

    if ((updateMode != FindMaxPowerUpdateMode_CreateNew) && TestFilePathMountPointExist(outputFilePath))
        nnt::gfx::util::json::LoadFromFile(pJsonDocument, outputFilePath);

    for (int gpuBenchmarkTypeIndex = 0; gpuBenchmarkTypeIndex < nnt::gfx::util::BenchmarkType_Max; ++gpuBenchmarkTypeIndex)
    {
        if ((benchmarkMask & (1 << gpuBenchmarkTypeIndex)) == 0)
            continue;

        nnt::gfx::util::BenchmarkType gpuBenchmarkType = static_cast<nnt::gfx::util::BenchmarkType>(gpuBenchmarkTypeIndex);
        nnt::gfx::util::GpuBenchmark* pGpuBenchmark = CreateBenchmarkFromType(gpuBenchmarkType, pTestData->pResourceAllocator);
        pGpuBenchmark->Initialize(pTestData->pResourceAllocator);

        MakeTestId(testIdBuffer, TestIdBufferSize, pGpuBenchmark);

        for (int i = 0; i < 30; ++i)
            DoFindMaxPowerDrawFrame(pTestData, pGpuBenchmark, 0, 0, 0);

        DoFindMaxPowerStartRecurse(pTestData, pGpuBenchmark);

        pGpuBenchmark->InitializeGfxObjects(pTestData->pResourceAllocator, pTestData->pDevice);
        pTestData->platformMeasurementTracker.Reset();

        nn::TimeSpan testDuration;
        int testRunCount = 0;
        RunGpuBenchmarkForDurationWithRetry(
            &testDuration, &testRunCount,
            pTestData, pGpuBenchmark,
            FindMaxPowerRecurseState::WarmUpCount,
            FindMaxPowerRecurseState::TestCaseTargetDuration,
            FindMaxPowerRecurseState::RetryCount);

        CreateTestCaseData(
            pTestData, pGpuBenchmark,
            FindMaxPowerRecurseState::WarmUpCount,
            testRunCount,
            pJsonDocument, testIdBuffer);

        pGpuBenchmark->FinalizeGfxObjects(pTestData->pResourceAllocator, pTestData->pDevice);
        pGpuBenchmark->Finalize(pTestData->pResourceAllocator);
        DestroyBenchmark(pGpuBenchmark, pTestData->pResourceAllocator);
        pGpuBenchmark = nullptr;
    }

    if (TestFilePathMountPointExist(outputFilePath))
    {
        nnt::gfx::util::json::SaveToFile(pJsonDocument, outputFilePath);
    }

    nnt::gfx::util::json::Finalize(pJsonDocument);

    FinalizeControllerInput();
}
