﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/gfx/util/gfxUtil_AgingTestApi.h>

#include <nn/nn_Result.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/fs.h>
#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
#include <nn/oe.h>
#endif

#include <cstdio>

#define HOST_MOUNT_POINT "host"
#define CONTENTS_MOUNT_POINT "contents"

bool g_HostMountCompleted = false;

void MountContents()
{
    static const size_t MountRomCacheBufferSize = 4 * 1024;
    static char s_MountRomCacheBuffer[MountRomCacheBufferSize];
    size_t mountRomCacheUseSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize));
    NN_ASSERT(mountRomCacheUseSize <= MountRomCacheBufferSize);

    NN_LOG("Mount Contents\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom(CONTENTS_MOUNT_POINT, s_MountRomCacheBuffer, MountRomCacheBufferSize));
    nn::fs::DirectoryEntry entryList[100];
    nn::fs::DirectoryHandle h = {};
    nn::fs::OpenDirectory(&h, CONTENTS_MOUNT_POINT":/", nn::fs::OpenDirectoryMode_All);
    int64_t n = 0;
    nn::fs::ReadDirectory(&n, entryList, h, 100);
    NN_LOG("%d entry found.\n", static_cast<int>(n));
    for (int64_t i = 0; i < n; i++)
    {
        auto& e = entryList[i];
        NN_LOG("  %s%s\n", e.name, (e.directoryEntryType == nn::fs::DirectoryEntryType_Directory ? "/" : ""));
    }
    nn::fs::CloseDirectory(h);

    nn::Result moutHostResult = nn::fs::MountHost(HOST_MOUNT_POINT, "./");
    if (moutHostResult.IsSuccess())
    {
        g_HostMountCompleted = true;
    }
}

void UnmountContents()
{
    if (g_HostMountCompleted)
    {
        nn::fs::Unmount(HOST_MOUNT_POINT);

    }
    nn::fs::Unmount(CONTENTS_MOUNT_POINT);
}

void DrawFps()
{
    const size_t printBufferSize = 256;
    char printBuffer[printBufferSize];

    float minFps = 0.0f;
    float maxFps = 0.0f;
    float averageFps = 0.0f;

    nnt::gfx::util::agingtest::GetFps(&minFps, &maxFps, &averageFps);

    snprintf(printBuffer, printBufferSize, "FPS:\nmin %3.5f\nmax %3.5f\navg %3.5f", minFps, maxFps, averageFps);
    nnt::gfx::util::agingtest::DrawFont(printBuffer, 960, 20, 1.0f, nnt::gfx::util::agingtest::DefaultColor::White);
}

void DrawTestResult(
    const char* text,
    const nnt::gfx::util::agingtest::Color& color,
    nnt::gfx::util::agingtest::TextureHandle textureHandle)
{
    const nnt::gfx::util::agingtest::Color FillColor = { 60, 60, 60, 255 };
    nnt::gfx::util::agingtest::Draw2DRect(950, 200, 200, 100, FillColor);
    nnt::gfx::util::agingtest::Draw2DFrame(950, 200, 200, 100, nnt::gfx::util::agingtest::DefaultColor::White);
    nnt::gfx::util::agingtest::Draw2DRectTexture(1010, 210, 80, 80, textureHandle);
    nnt::gfx::util::agingtest::DrawFont(text, 960, 175, 1.0f, color);
}


struct TestResources
{
    nnt::gfx::util::agingtest::TextureHandle maruTextureHandle;
    nnt::gfx::util::agingtest::TextureHandle batsuTextureHandle;
    nnt::gfx::util::agingtest::TextureHandle updaterIconV1TextureHandle;
    nnt::gfx::util::agingtest::TextureHandle updaterIconV4TextureHandle;
    nnt::gfx::util::agingtest::TextureHandle nintendoLogoV5TextureHandle;
};


const char* GetResultMessage(int testResultFlags)
{
    static const char* g_ResultMessage[]
    {
        "PASSED",
        "FAILED (CPU)",
        "FAILED (GPU)",
        "FAILED (CPU, GPU)",
        "FAILED (FB)",
        "FAILED (FB, CPU)",
        "FAILED (FB, GPU)",
        "FAILED (FB, CPU, GPU)",
    };

    NN_ASSERT(testResultFlags < (sizeof(g_ResultMessage) / sizeof(g_ResultMessage[0])));
    return g_ResultMessage[testResultFlags];
}

void DrawTestFrame(
    nnt::gfx::util::agingtest::TestSuiteHandle handle,
    bool needMoreSamples,
    int testResultFlags,
    const TestResources* pTestResources)
{
    int frameBufferWidth = nnt::gfx::util::agingtest::GetFrameBufferWidth();
    int frameBufferHeight = nnt::gfx::util::agingtest::GetFrameBufferHeight();

    nnt::gfx::util::agingtest::StartFrame();

    nnt::gfx::util::agingtest::DrawFont(
        "Aging test - prototype demo",
        20, 20, 1.0f, nnt::gfx::util::agingtest::DefaultColor::White);

    nnt::gfx::util::agingtest::Color backgroundColor = { 25, 25, 25, 255 };
    nnt::gfx::util::agingtest::Draw2DRect(10, 10, frameBufferWidth - 400, frameBufferHeight - 200, backgroundColor);
    nnt::gfx::util::agingtest::Draw2DFrame(10, 10, frameBufferWidth - 400, frameBufferHeight - 200,
        nnt::gfx::util::agingtest::DefaultColor::White);

    DrawFps();

    int iconV1Width = nnt::gfx::util::agingtest::GetTextureWidth(pTestResources->updaterIconV1TextureHandle);
    int iconV1Height = nnt::gfx::util::agingtest::GetTextureHeight(pTestResources->updaterIconV1TextureHandle);
    nnt::gfx::util::agingtest::Draw2DRectTexture(
        frameBufferWidth - iconV1Width, frameBufferHeight - iconV1Height,
        iconV1Width, iconV1Height, pTestResources->updaterIconV1TextureHandle);

    int iconV4Width = nnt::gfx::util::agingtest::GetTextureWidth(pTestResources->updaterIconV4TextureHandle);
    int iconV4Height = nnt::gfx::util::agingtest::GetTextureHeight(pTestResources->updaterIconV4TextureHandle);
    nnt::gfx::util::agingtest::Draw2DRectTexture(
        frameBufferWidth - iconV4Width, 0,
        iconV4Width, iconV4Height, pTestResources->updaterIconV4TextureHandle);


    nnt::gfx::util::agingtest::DrawTestResults(
        handle, 20, 50, 1.0f,
        nnt::gfx::util::agingtest::DefaultColor::White);

    int nintendoLogoWidth = nnt::gfx::util::agingtest::GetTextureWidth(pTestResources->nintendoLogoV5TextureHandle);
    int nintendoLogoHeight = nnt::gfx::util::agingtest::GetTextureHeight(pTestResources->nintendoLogoV5TextureHandle);
    nnt::gfx::util::agingtest::Draw2DRectTexture(
        10 + frameBufferWidth - 400 - 10 - nintendoLogoWidth,
        10 + frameBufferHeight - 200 - 10 - nintendoLogoHeight,
        nintendoLogoWidth, nintendoLogoHeight, pTestResources->nintendoLogoV5TextureHandle);

    if (!needMoreSamples)
    {
        const char* resultMessage = GetResultMessage(testResultFlags);
        if (testResultFlags == 0)
        {
            DrawTestResult(resultMessage, nnt::gfx::util::agingtest::DefaultColor::Green, pTestResources->maruTextureHandle);
        }
        else
        {
            DrawTestResult(resultMessage, nnt::gfx::util::agingtest::DefaultColor::Red, pTestResources->batsuTextureHandle);
        }
    }

#if defined(NN_SDK_BUILD_DEBUG)
    nnt::gfx::util::agingtest::DrawTestDebugInformation(handle);
#endif

    nnt::gfx::util::agingtest::EndFrame();
}

bool RunTest(
    nnt::gfx::util::agingtest::TestSuiteHandle handle,
    const TestResources* pTestResources)
{
    int testResultFlags = 0;
    bool needMoreSamples = true;

    nnt::gfx::util::agingtest::HashComparisonMode hashComparisonMode =
        nnt::gfx::util::agingtest::HashComparisonMode_LastRunOnly;

    while (needMoreSamples)
    {
        needMoreSamples = nnt::gfx::util::agingtest::RunTest(&testResultFlags, hashComparisonMode, handle);
        DrawTestFrame(handle, needMoreSamples, testResultFlags, pTestResources);
    }

    int displayResultFrameCount = 128;
    for (int displayResultFrameIndex = 0; displayResultFrameIndex < displayResultFrameCount; displayResultFrameIndex++)
    {
        DrawTestFrame(handle, needMoreSamples, testResultFlags, pTestResources);
    }

    const char* testName = nnt::gfx::util::agingtest::GetTestName(handle);
    const char* testId = nnt::gfx::util::agingtest::GetTestCaseId(handle);

    const int printBufferSize = 256;
    char printBuffer[printBufferSize];
    snprintf(printBuffer, printBufferSize, "Test result: %s (%s)", testName, testId);

    const char* resultMessage = GetResultMessage(testResultFlags);
    NN_LOG("%-55s -> %s\n", printBuffer, resultMessage);

    return (testResultFlags == 0);
}

void ParseApplicationParameters(
    const char** pOutTestCasesFilePath,
    int* pOutStartIndex, int* pOutMaxCount,
    const char** pOutFilter)
{
    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    if (argc >= 2)
    {
        *pOutTestCasesFilePath = argv[1];
    }

    if (argc >= 3)
    {
        *pOutStartIndex = atoi(argv[2]);
    }

    if (argc >= 4)
    {
        *pOutMaxCount = atoi(argv[3]);
    }

    if (argc >= 5)
    {
        *pOutFilter = argv[4];
    }
}

const char* defaultTestCasesFilePath = CONTENTS_MOUNT_POINT":/MaxPowerTestCases.json";

extern "C" void nnMain()
{
    const char* filter = nullptr;
    const char* testCasesFilePath = defaultTestCasesFilePath;
    int startIndex = 0;
    int maxCount = nnt::gfx::util::agingtest::MaxCountAllTests;

    ParseApplicationParameters(&testCasesFilePath, &startIndex, &maxCount, &filter);

    nn::Result result;

    MountContents();
    nnt::gfx::util::agingtest::Initialize();
    nnt::gfx::util::agingtest::StartFrame();
    nnt::gfx::util::agingtest::EndFrame();

    TestResources testResources;

    result = nnt::gfx::util::agingtest::CreateTextureFromResourceFile(&testResources.maruTextureHandle, CONTENTS_MOUNT_POINT":/Maru.bntx");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nnt::gfx::util::agingtest::CreateTextureFromResourceFile(&testResources.batsuTextureHandle, CONTENTS_MOUNT_POINT":/Batsu.bntx");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nnt::gfx::util::agingtest::CreateTextureFromBmpFile(&testResources.updaterIconV1TextureHandle, CONTENTS_MOUNT_POINT":/UpdaterIcon_v1.bmp");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nnt::gfx::util::agingtest::CreateTextureFromBmpFile(&testResources.updaterIconV4TextureHandle, CONTENTS_MOUNT_POINT":/UpdaterIcon_v4.bmp");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nnt::gfx::util::agingtest::CreateTextureFromBmpFile(&testResources.nintendoLogoV5TextureHandle, CONTENTS_MOUNT_POINT":/NintendoLogo_v5.bmp");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_LOG("Load test suite file %s\n", testCasesFilePath);
    nnt::gfx::util::agingtest::TestSuiteHandle handle;
    result = nnt::gfx::util::agingtest::LoadTestSuiteFromFile(&handle, testCasesFilePath);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    const int testPlatformIdSize = 128;
    char testPlatformId[testPlatformIdSize];
    nnt::gfx::util::agingtest::GetTestPlatformId(testPlatformId, testPlatformIdSize);
    NN_LOG("PlatformId: %s\n", testPlatformId);

    if (filter == nullptr)
    {
        filter = testPlatformId;
    }

    int testFailureCount = 0;
    const int repeatCount = 3;
    int repeatIndex = 0;

    while (repeatIndex < repeatCount)
    {
        int testCount = nnt::gfx::util::agingtest::InitializeTestSuite(
            handle, filter, startIndex, maxCount);

        NN_LOG("%d tests found\n", testCount);

        while (nnt::gfx::util::agingtest::BeginNextTest(handle))
        {
            bool testSucceded = RunTest(handle, &testResources);
            if (!testSucceded)
            {
                testFailureCount++;
            }
        }

        nnt::gfx::util::agingtest::FinalizeTestSuite(handle);

        repeatIndex++;
    }

    if (testFailureCount == 0)
    {
        NN_LOG("All tests passed\n");
    }
    else
    {
        NN_LOG("%d test failure(s)\n", testFailureCount);
    }

    nnt::gfx::util::agingtest::UnloadTestSuite(handle);

    nnt::gfx::util::agingtest::FreeTextureResource(testResources.maruTextureHandle);
    nnt::gfx::util::agingtest::FreeTextureResource(testResources.batsuTextureHandle);
    nnt::gfx::util::agingtest::FreeTextureResource(testResources.updaterIconV1TextureHandle);
    nnt::gfx::util::agingtest::FreeTextureResource(testResources.updaterIconV4TextureHandle);
    nnt::gfx::util::agingtest::FreeTextureResource(testResources.nintendoLogoV5TextureHandle);

    nnt::gfx::util::agingtest::Finalize();

    UnmountContents();
}
