﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include "gfxUtilGpuBenchmark_GpuBenchmarkTextureFetch.h"

#include "gfxUtilGpuBenchmark_ResHelpers.h"
#include "gfxUtilGpuBenchmark_PropertyMacros.h"
#include "gfxUtilGpuBenchmark_ResourceAllocator.h"
#include "gfxUtilGpuBenchmark_ComputeTextureFetchShaderVariationIndex.h"
#include "gfxUtilGpuBenchmark_ComputeRenderQuadShaderVariationIndex.h"

#include "gfxUtilGpuBenchmark_BuiltinTextureFetchShader.h"
#include "gfxUtilGpuBenchmark_BuiltinRenderQuadShader.h"

#define PROPERTY_TEXTURE_FORMAT_LIST                                        \
    "R8_G8_B8_A8_Unorm", nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm,            \
    "Bc1_Unorm", nn::gfx::ImageFormat_Bc1_Unorm,                            \
    "Bc2_Unorm", nn::gfx::ImageFormat_Bc2_Unorm

#define PROPERTY_TEXTURE_SIZE_LIST                              \
    "256", TextureSize_256,                                     \
    "512", TextureSize_512,                                     \
    "1024", TextureSize_1024

#define PROPERTY_TEXTURE(_n)                                                                                                \
    GpuBenchmarkPropertyHolder* pPropertyTexture ## _n ## Format = m_PropertyArray.Get(Property_Texture ## _n ## Format);   \
    BENCHMARK_PROPERTY_ENUM_DEFINITION_INDEXED(                                                                             \
        pPropertyTexture ## _n ## Format, "Texture" #_n "Format",                                                           \
        nn::gfx::ImageFormat, int, _n,                                                                                      \
        GpuBenchmarkTextureFetch::GetTextureFormat,                                                                         \
        GpuBenchmarkTextureFetch::SetTextureFormat,                                                                         \
        PROPERTY_TEXTURE_FORMAT_LIST);                                                                                      \
    GpuBenchmarkPropertyHolder* pPropertyTexture ## _n ## Size = m_PropertyArray.Get(Property_Texture ## _n ## Size);       \
    BENCHMARK_PROPERTY_ENUM_DEFINITION_INDEXED(                                                                             \
        pPropertyTexture ## _n ## Size, "Texture" #_n "Size",                                                               \
        TextureSize, int,_n,                                                                                                \
        GpuBenchmarkTextureFetch::GetTextureSize,                                                                           \
        GpuBenchmarkTextureFetch::SetTextureSize,                                                                           \
        PROPERTY_TEXTURE_SIZE_LIST);                                                                                        \


namespace nnt { namespace gfx { namespace util {

const char* GpuBenchmarkTextureFetch::ClassName = "TextureFetch";

GpuBenchmarkTextureFetch::GpuBenchmarkTextureFetch()
: m_TextureCount(1)
, m_TextureFilterMode(nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint)
, m_TextureFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm)
, m_TextureSize(TextureSize_256)
, m_RenderTexture()
, m_RenderTextureTargetView()
, m_RenderTextureViewportScissorState()
, m_TextureSampler()
, m_TextureSamplerSlotIndex(-1)
, m_TextureSamplerDescriptorSlot()
, m_TextureDataArray()
, m_VertexBuffer()
{
}

GpuBenchmarkTextureFetch::~GpuBenchmarkTextureFetch()
{
}

void GpuBenchmarkTextureFetch::Initialize(ResourceAllocator* pResourceAllocator)
{
    NN_UNUSED(pResourceAllocator);

    GpuBenchmarkPropertyHolder* pPropertyTextureFilterMode = m_PropertyArray.Get(Property_TextureFilterMode);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyTextureFilterMode, "TextureFilterMode",
        nn::gfx::FilterMode, pResourceAllocator,
        GpuBenchmarkTextureFetch::GetTextureFilterMode,
        GpuBenchmarkTextureFetch::SetTextureFilterMode,
        "MinPoint_MagPoint_MipPoint", nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint,
        "MinLinear_MagLinear_MipPoint", nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint,
        "MinLinear_MagLinear_MipLinear", nn::gfx::FilterMode_MinLinear_MagLinear_MipLinear,
        "Anisotropic", nn::gfx::FilterMode_Anisotropic);

    GpuBenchmarkPropertyHolder* pPropertyTextureCount = m_PropertyArray.Get(Property_TextureCount);
    BENCHMARK_PROPERTY_INTEGER_RANGE_DEFINITION(
        pPropertyTextureCount, "TextureCount",
        GpuBenchmarkTextureFetch::GetTextureCount,
        GpuBenchmarkTextureFetch::SetTextureCount,
        1, m_MaxTextureCount, 1);

    GpuBenchmarkPropertyHolder* pPropertyTextureFormat = m_PropertyArray.Get(Property_TextureFormat);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyTextureFormat, "TextureFormat",
        nn::gfx::ImageFormat, pResourceAllocator,
        GpuBenchmarkTextureFetch::GetTextureFormat,
        GpuBenchmarkTextureFetch::SetTextureFormat,
        PROPERTY_TEXTURE_FORMAT_LIST);

    GpuBenchmarkPropertyHolder* pPropertyTextureSize = m_PropertyArray.Get(Property_TextureSize);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyTextureSize, "TextureSize",
        TextureSize, pResourceAllocator,
        GpuBenchmarkTextureFetch::GetTextureSize,
        GpuBenchmarkTextureFetch::SetTextureSize,
        PROPERTY_TEXTURE_SIZE_LIST);
}

void GpuBenchmarkTextureFetch::Finalize(ResourceAllocator* pResourceAllocator)
{
    NN_UNUSED(pResourceAllocator);

    for (int i = 0; i < m_PropertyArray.GetCount(); ++i)
    {
        m_PropertyArray.Get(i)->Finalize();
    }
}

void GpuBenchmarkTextureFetch::InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    GpuBenchmark::InitializeGfxObjects(pResourceAllocator, pDevice);

    m_OutputCopyBufferSize = InitializeColorRenderTarget(
        &m_RenderTexture, &m_OutputCopyBuffer,
        &m_RenderTextureTargetView, &m_RenderTextureViewportScissorState,
        m_RenderSize, m_RenderSize, m_RenderFormat, m_RenderTileMode,
        pResourceAllocator, pDevice);

    for (int textureIndex = 0; textureIndex < m_TextureCount; ++textureIndex)
    {
        TextureData* pTextureData = &m_TextureDataArray[textureIndex];
        int textureDimension = GetDimensionInPixelFromSize(m_TextureSize);
        int mipCount = 5;

        nn::gfx::Texture::InfoType textureInfo;
        {
            textureInfo.SetDefault();
            textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
            textureInfo.SetHeight(textureDimension);
            textureInfo.SetWidth(textureDimension);
            textureInfo.SetImageFormat(m_TextureFormat);
            textureInfo.SetTileMode(nn::gfx::TileMode_Optimal);
            textureInfo.SetMipCount(mipCount);
        }
        InitializeTexture(&pTextureData->texture, textureInfo, pResourceAllocator, MemoryPoolType_Data, pDevice);

        nn::gfx::TextureView::InfoType textureViewInfo;
        {
            textureViewInfo.SetDefault();
            textureViewInfo.SetTexturePtr(&pTextureData->texture);
            textureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
            textureViewInfo.SetImageFormat(m_TextureFormat);
            textureViewInfo.EditSubresourceRange().EditMipRange().SetMipCount(mipCount);
        }

        pTextureData->textureView.Initialize(pDevice, textureViewInfo);

        pTextureData->slotIndex = pResourceAllocator->AllocateAndSetTextureViewToDescriptorPool(
            &pTextureData->textureView, &pTextureData->descriptorSlot);
    }

    nn::gfx::Sampler::InfoType textureSamplerInfo;
    {
        textureSamplerInfo.SetDefault();
        textureSamplerInfo.SetFilterMode(m_TextureFilterMode);
        textureSamplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToEdge);
        textureSamplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToEdge);
        textureSamplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
    }
    m_TextureSampler.Initialize(pDevice, textureSamplerInfo);
    m_TextureSamplerSlotIndex = pResourceAllocator->AllocateAndSetSamplerToDescriptorPool(
        &m_TextureSampler, &m_TextureSamplerDescriptorSlot);

    InitializeResShader(&m_ResShader, g_TextureFetchShaderData, sizeof(g_TextureFetchShaderData), pResourceAllocator, pDevice);

    InitializeFullScreenQuadVertexBuffer(&m_VertexBuffer, pResourceAllocator, pDevice);

#if defined(NN_GFXUTIL_GPUBENCHMARK_TEXTURE_FETCH_DEBUG)
    InitializeResShader(&m_RenderQuadResShader, g_RenderQuadShaderData, sizeof(g_RenderQuadShaderData), pResourceAllocator, pDevice);

    nn::gfx::TextureView::InfoType renderTextureViewInfo;
    {
        renderTextureViewInfo.SetDefault();
        renderTextureViewInfo.SetTexturePtr(&m_RenderTexture);
        renderTextureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        renderTextureViewInfo.SetImageFormat(m_RenderFormat);
        renderTextureViewInfo.EditSubresourceRange().EditMipRange().SetMipCount(1);
    }
    m_RenderTextureView.Initialize(pDevice, renderTextureViewInfo);

    m_RenderTextureViewDescriptorSlotIndex = pResourceAllocator->AllocateAndSetTextureViewToDescriptorPool(
        &m_RenderTextureView, &m_RenderTextureViewDescriptorSlot);


    nn::gfx::Sampler::InfoType renderTextureSamplerInfo;
    {
        renderTextureSamplerInfo.SetDefault();
        renderTextureSamplerInfo.SetFilterMode(nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint);
        renderTextureSamplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToEdge);
        renderTextureSamplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToEdge);
        renderTextureSamplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
    }
    m_RenderTextureSampler.Initialize(pDevice, renderTextureSamplerInfo);
    m_RenderTextureSamplerDescriptorSlotIndex = pResourceAllocator->AllocateAndSetSamplerToDescriptorPool(
        &m_RenderTextureSampler, &m_RenderTextureSamplerDescriptorSlot);
#endif
}

void GpuBenchmarkTextureFetch::FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
#if defined(NN_GFXUTIL_GPUBENCHMARK_TEXTURE_FETCH_DEBUG)
    FinalizeResShader(&m_RenderQuadResShader, pResourceAllocator, pDevice);
    m_RenderTextureView.Finalize(pDevice);
    pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_TextureView, m_RenderTextureViewDescriptorSlotIndex);
    m_RenderTextureViewDescriptorSlotIndex = -1;

    m_RenderTextureSampler.Finalize(pDevice);
    pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_Sampler, m_RenderTextureSamplerDescriptorSlotIndex);
    m_RenderTextureSamplerDescriptorSlotIndex = -1;
#endif

    FinalizeBuffer(&m_VertexBuffer, pResourceAllocator, pDevice);

    FinalizeResShader(&m_ResShader, pResourceAllocator, pDevice);

    pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_Sampler, m_TextureSamplerSlotIndex);
    m_TextureSamplerSlotIndex = -1;
    m_TextureSampler.Finalize(pDevice);

    for (int textureIndex = 0; textureIndex < m_TextureCount; ++textureIndex)
    {
        TextureData* pTextureData = &m_TextureDataArray[textureIndex];
        pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_TextureView, pTextureData->slotIndex);

        pTextureData->textureView.Finalize(pDevice);
        FinalizeTexture(&pTextureData->texture, pResourceAllocator, pDevice);

        pTextureData->slotIndex = -1;
    }

    FinalizeColorRenderTarget(
        &m_RenderTexture, &m_OutputCopyBuffer,
        &m_RenderTextureTargetView, &m_RenderTextureViewportScissorState,
        pResourceAllocator, pDevice);

    GpuBenchmark::FinalizeGfxObjects(pResourceAllocator, pDevice);
}

void GpuBenchmarkTextureFetch::PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer)
{
    nn::gfx::ColorTargetView* pTestTarget = &m_RenderTextureTargetView;

    pTestCommandBuffer->ClearColor(pTestTarget, 0.5f, 0.5f, 0.5f, 1.0f, nullptr);

    pTestCommandBuffer->SetRenderTargets(1, &pTestTarget, nullptr);
    pTestCommandBuffer->SetViewportScissorState(&m_RenderTextureViewportScissorState);

    int shaderVariationIndex = ComputeTextureFetchShaderVariationIndex(m_TextureCount - 1);
    nn::gfx::ResShaderProgram* pResShaderProgram = m_ResShader.pResShaderContainer->GetResShaderVariation(shaderVariationIndex)->GetResShaderProgram(m_ResShader.codeType);

#if defined(NN_GFXUTIL_GPUBENCHMARK_TEXTURE_FETCH_DEBUG)
    nn::util::BinTPtr<nn::gfx::ResShaderReflectionData> pPtrResShaderReflectionData = pResShaderProgram->ToData().pShaderReflection;
    nn::util::BinTPtr<nn::gfx::ResShaderReflectionStageData > pPtrResShaderReflectionStageData = pPtrResShaderReflectionData.Get()->pPixelReflection;
    nn::util::BinTPtr<nn::util::ResDic> pPtrSamplerDic = pPtrResShaderReflectionStageData.Get()->pSamplerDic;
    NN_ASSERT(pPtrSamplerDic.Get()->GetCount() == m_TextureCount);
#endif

    nn::gfx::Shader* pShader = pResShaderProgram->GetShader();
    pTestCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);

    nn::gfx::GpuAddress vertexBufferGpuAddress;
    m_VertexBuffer.GetGpuAddress(&vertexBufferGpuAddress);
    pTestCommandBuffer->SetVertexBuffer(0, vertexBufferGpuAddress, sizeof(DefaultVertex), g_RectangleVertexBufferDataSize);

    for (int textureIndex = 0; textureIndex < m_TextureCount; ++textureIndex)
    {
        TextureData* pTextureData = &m_TextureDataArray[textureIndex];
        pTestCommandBuffer->SetTextureAndSampler(
            textureIndex, nn::gfx::ShaderStage_Pixel,
            pTextureData->descriptorSlot, m_TextureSamplerDescriptorSlot);
    }
}

void GpuBenchmarkTextureFetch::DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount)
{
    NN_ASSERT(runCount > 0);

    for (int repeatIndex = 0; repeatIndex < runCount; ++repeatIndex)
    {
        pTestCommandBuffer->Draw(nn::gfx::PrimitiveTopology_TriangleStrip, 4, 0);
    }
}

void GpuBenchmarkTextureFetch::RenderDebug(nn::gfx::CommandBuffer* pTestCommandBuffer)
{
    NN_UNUSED(pTestCommandBuffer);

#if defined(NN_GFXUTIL_GPUBENCHMARK_TEXTURE_FETCH_DEBUG)
    pTestCommandBuffer->SetTextureStateTransition(
        &m_RenderTexture, nullptr,
        nn::gfx::TextureState_ColorTarget, nn::gfx::PipelineStageBit_RenderTarget,
        nn::gfx::TextureState_ShaderRead, nn::gfx::PipelineStageBit_PixelShader);

    int shaderVariationIndex = ComputeRenderQuadShaderVariationIndex(1, 1); // instance + blit texture
    nn::gfx::Shader* pShader = m_RenderQuadResShader.pResShaderContainer->GetResShaderVariation(shaderVariationIndex)->GetResShaderProgram(m_RenderQuadResShader.codeType)->GetShader();
    pTestCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);

    pTestCommandBuffer->SetTextureAndSampler(
        0, nn::gfx::ShaderStage_Pixel,
        m_RenderTextureViewDescriptorSlot,
        m_RenderTextureSamplerDescriptorSlot);

    int scale = 3;
    int offsetX = 2;
    int offsetY = 2;

    pTestCommandBuffer->Draw(
        nn::gfx::PrimitiveTopology_TriangleStrip, 4,
        (scale << 8) | (offsetX << 16) | (offsetY << 24));
#endif
}


void GpuBenchmarkTextureFetch::PrintResults(nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter)
{
    NN_ASSERT(runCount > 0);

    uint64_t gpuTimeElapsedValueInNs = static_cast<uint64_t>(gpuTimeElapsed.GetNanoSeconds());
    uint64_t cpuTimeElapsedValueInNs = static_cast<uint64_t>(cpuTimeElapsed.GetNanoSeconds());

    uint64_t gpuTimeElapsedAvgValueInNs = gpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);
    uint64_t cpuTimeElapsedAvgValueInNs = cpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);

    uint64_t pixelCount = static_cast<uint64_t>(runCount) * static_cast<uint64_t>(m_RenderSize * m_RenderSize);

    uint64_t texelPerSecond = 0;
    if (gpuTimeElapsedValueInNs > 0)
    {
        texelPerSecond = (pixelCount * static_cast<uint64_t>(m_TextureCount) * 1000000000LL) / gpuTimeElapsedValueInNs;
    }

    uint64_t pixelPerSecond = 0;
    if (gpuTimeElapsedValueInNs > 0)
    {
        pixelPerSecond = (pixelCount * 1000000000LL) / gpuTimeElapsedValueInNs;
    }

    pDebugFontTextWriter->Print("%12lu texels/sec\n", texelPerSecond);
    pDebugFontTextWriter->Print("%12lu pixels/sec\n", pixelPerSecond);
    pDebugFontTextWriter->Print("gpu time (ns): %8lu\n", gpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("cpu time (ns): %8lu\n", cpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("total gpu time (ns): %12lu\n", gpuTimeElapsedValueInNs);
    pDebugFontTextWriter->Print("total cpu time (ns): %12lu\n", cpuTimeElapsedValueInNs);
}

void GpuBenchmarkTextureFetch::CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer)
{
    int renderSize = m_RenderSize;

    nn::gfx::BufferTextureCopyRegion bufferTextureCopyRegion;
    bufferTextureCopyRegion.SetDefault();
    bufferTextureCopyRegion.SetBufferImageHeight(renderSize);
    bufferTextureCopyRegion.SetBufferImageWidth(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().SetDefault();
    bufferTextureCopyRegion.EditTextureCopyRegion().SetWidth(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().SetHeight(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().EditSubresource().SetDefault();

    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture);
    pCommandBuffer->CopyImageToBuffer(&m_OutputCopyBuffer, &m_RenderTexture, bufferTextureCopyRegion);
    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Write);
}

void GpuBenchmarkTextureFetch::MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize)
{
    *pOutBuffer = m_OutputCopyBuffer.Map();
    *pOutBufferSize = m_OutputCopyBufferSize;
}

void GpuBenchmarkTextureFetch::UnmapResultBuffer()
{
    m_OutputCopyBuffer.Unmap();
}

const char* GpuBenchmarkTextureFetch::GetName() const
{
    return ClassName;
}

BenchmarkType GpuBenchmarkTextureFetch::GetType() const
{
    return BenchmarkType_TextureFetch;
}

int GpuBenchmarkTextureFetch::GetPropertyCount() const
{
    return m_PropertyArray.GetCount();
}

int GpuBenchmarkTextureFetch::FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

int GpuBenchmarkTextureFetch::FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize)
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkTextureFetch::FindPropertyByName(const char* propertyName)
{
    return m_PropertyArray.FindPropertyByName(propertyName);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkTextureFetch::GetPropertyByIndex(int index)
{
    return m_PropertyArray.Get(index);
}


const char* GpuBenchmarkTextureFetch::GetFormatString(nn::gfx::ImageFormat format)
{
    switch (format)
    {
    case nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm:
        return "unorm_8_8_8_8";

    case nn::gfx::ImageFormat_Bc1_Unorm:
        return "unorm_bc1";

    case nn::gfx::ImageFormat_Bc2_Unorm:
        return "unorm_bc2";

    default:
        NN_UNEXPECTED_DEFAULT;
    }

}

void GpuBenchmarkTextureFetch::GetInputTextureName(
    char* pBuffer, int bufferSize,
    TextureSize size, nn::gfx::ImageFormat format)
{
    snprintf(pBuffer, bufferSize, "texturefetch_%s_%d", GetFormatString(format), GetDimensionInPixelFromSize(size));
}

} } } // namespace nnt { namespace gfx { namespace util {
