﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include "gfxUtilGpuBenchmark_GpuBenchmarkMrt.h"

#include "gfxUtilGpuBenchmark_ResHelpers.h"
#include "gfxUtilGpuBenchmark_PropertyMacros.h"
#include "gfxUtilGpuBenchmark_ComputeMrtShaderVariationIndex.h"

#include "gfxUtilGpuBenchmark_BuiltinMrtShader.h"

namespace nnt { namespace gfx { namespace util {


namespace {

struct Vertex
{
    float x;
    float y;
    float z;

    float u;
    float v;
};

const Vertex rectangleVertexBufferData[] =
{
    { -1.0f, -1.0f, 0.0f, 0.0f, 0.0f,},
    { -1.0f,  1.0f, 0.0f, 0.0f, 1.0f,},
    { 1.0f, -1.0f, 0.0f, 1.0f, 0.0f, },
    { 1.0f,  1.0f, 0.0f, 1.0f, 1.0f, },
};

} // anonymous namespace
const char* GpuBenchmarkMrt::ClassName = "Mrt";

GpuBenchmarkMrt::GpuBenchmarkMrt()
: m_RenderSize(1024)
, m_RenderFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm)
, m_MrtCount(1)
, m_RenderTexture()
, m_RenderTextureColorTargetView()
, m_ViewportScissorState()
, m_ResShader()
, m_PropertyArray()
{
}

GpuBenchmarkMrt::~GpuBenchmarkMrt()
{
}

void GpuBenchmarkMrt::Initialize(ResourceAllocator* pResourceAllocator)
{
    NN_UNUSED(pResourceAllocator);

    // RenderSize
    GpuBenchmarkPropertyHolder* pPropertyRenderSize = m_PropertyArray.Get(Property_RenderSize);
    BENCHMARK_PROPERTY_INTEGER_RANGE_DEFINITION(
        pPropertyRenderSize, "RenderSize",
        GpuBenchmarkMrt::GetRenderSize,
        GpuBenchmarkMrt::SetRenderSize,
        1024, 2048, 1024);

    // RenderFormat
    GpuBenchmarkPropertyHolder* pPropertyRenderFormat = m_PropertyArray.Get(Property_RenderFormat);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyRenderFormat, "RenderFormat",
        nn::gfx::ImageFormat, pResourceAllocator,
        GpuBenchmarkMrt::GetRenderFormat,
        GpuBenchmarkMrt::SetRenderFormat,
        "ImageFormat_R8_G8_B8_A8_Unorm", nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm,
        "ImageFormat_R16_G16_B16_A16_Float", nn::gfx::ImageFormat_R16_G16_B16_A16_Float);

    // MrtCount
    GpuBenchmarkPropertyHolder* pPropertyMrtCount = m_PropertyArray.Get(Property_MrtCount);
    BENCHMARK_PROPERTY_INTEGER_RANGE_DEFINITION(
        pPropertyMrtCount, "MrtCount",
        GpuBenchmarkMrt::GetMrtCount,
        GpuBenchmarkMrt::SetMrtCount,
        1, 4, 1);
}

void GpuBenchmarkMrt::Finalize(ResourceAllocator* pResourceAllocator)
{
    NN_UNUSED(pResourceAllocator);

    for (int i = 0; i < m_PropertyArray.GetCount(); ++i)
    {
        m_PropertyArray.Get(i)->Finalize();
    }
}


void GpuBenchmarkMrt::InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    GpuBenchmark::InitializeGfxObjects(pResourceAllocator, pDevice);

    NN_ASSERT(m_MrtCount <= m_MrtCountMax);

    nn::gfx::TileMode tileMode = nn::gfx::TileMode_Optimal;

    size_t renderTargetTotalSize = 0;

    for (int mrtIndex = 0; mrtIndex < m_MrtCount; ++mrtIndex)
    {
        m_OutputCopyBufferRenderTargetSize[mrtIndex] = InitializeColorRenderTarget(
            &m_RenderTexture[mrtIndex], nullptr,
            &m_RenderTextureColorTargetView[mrtIndex],
            (mrtIndex == 0) ?&m_ViewportScissorState : nullptr,
            m_RenderSize, m_RenderSize, m_RenderFormat, tileMode,
            pResourceAllocator, pDevice);

        renderTargetTotalSize += m_OutputCopyBufferRenderTargetSize[mrtIndex];
    }

    m_OutputCopyBufferTotalSize = renderTargetTotalSize;

    nn::gfx::Buffer::InfoType outputCopyBufferInfo;
    outputCopyBufferInfo.SetDefault();
    outputCopyBufferInfo.SetGpuAccessFlags(
        nn::gfx::GpuAccess_Image | nn::gfx::GpuAccess_Write);
    outputCopyBufferInfo.SetSize(renderTargetTotalSize);
    InitializeBuffer(
        &m_OutputCopyBuffer, outputCopyBufferInfo,
        pResourceAllocator, MemoryPoolType_Data, pDevice);
    ClearBufferContent(&m_OutputCopyBuffer, renderTargetTotalSize);

    nn::gfx::Buffer::InfoType rectangleVertexBufferInfo;
    {
        rectangleVertexBufferInfo.SetDefault();
        rectangleVertexBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);
        rectangleVertexBufferInfo.SetSize(sizeof(rectangleVertexBufferData));
    }

    InitializeBuffer(
        &m_RectangleVertexBuffer, rectangleVertexBufferInfo,
        pResourceAllocator, MemoryPoolType_Data, pDevice);
    CopyContentToBuffer(&m_RectangleVertexBuffer, rectangleVertexBufferData, sizeof(rectangleVertexBufferData));

    InitializeResShader(&m_ResShader, g_MrtShaderData, sizeof(g_MrtShaderData), pResourceAllocator, pDevice);
}

void GpuBenchmarkMrt::FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    FinalizeResShader(&m_ResShader, pResourceAllocator, pDevice);

    FinalizeBuffer(&m_RectangleVertexBuffer, pResourceAllocator, pDevice);

    FinalizeBuffer(&m_OutputCopyBuffer, pResourceAllocator, pDevice);

    for (int mrtIndex = 0; mrtIndex < m_MrtCount; ++mrtIndex)
    {
        FinalizeColorRenderTarget(
            &m_RenderTexture[mrtIndex], nullptr,
            &m_RenderTextureColorTargetView[mrtIndex],
            (mrtIndex == 0) ? &m_ViewportScissorState : nullptr,
            pResourceAllocator, pDevice);
    }

    GpuBenchmark::FinalizeGfxObjects(pResourceAllocator, pDevice);
}

void GpuBenchmarkMrt::PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer)
{
    nn::gfx::ColorTargetView* pTestTarget[m_MrtCountMax] = { nullptr, nullptr, nullptr, nullptr };

    for (int mrtIndex = 0; mrtIndex < m_MrtCount; ++mrtIndex)
    {
        nn::gfx::ColorTargetView* pColorTargetView = &m_RenderTextureColorTargetView[mrtIndex];
        pTestCommandBuffer->ClearColor(pColorTargetView, .05f, .05f, .05f, 1.0f, nullptr);
        pTestTarget[mrtIndex] = pColorTargetView;
    }

    pTestCommandBuffer->SetRenderTargets(m_MrtCount, pTestTarget, nullptr);
    pTestCommandBuffer->SetViewportScissorState(&m_ViewportScissorState);

    int shaderVariationIndex = ComputeMrtShaderVariationIndex(m_MrtCount - 1);
    NN_ASSERT(shaderVariationIndex < m_ResShader.pResShaderContainer->GetShaderVariationCount());
    nn::gfx::Shader* pShader = m_ResShader.pResShaderContainer->GetResShaderVariation(shaderVariationIndex)->GetResShaderProgram(m_ResShader.codeType)->GetShader();
    pTestCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);

    nn::gfx::GpuAddress vertexBufferGpuAddress;
    m_RectangleVertexBuffer.GetGpuAddress(&vertexBufferGpuAddress);
    pTestCommandBuffer->SetVertexBuffer(0, vertexBufferGpuAddress, sizeof(Vertex), sizeof(rectangleVertexBufferData));

}

void GpuBenchmarkMrt::DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount)
{
    NN_ASSERT(runCount > 0);

    for (int i = 0; i < runCount; ++i)
    {
        pTestCommandBuffer->Draw(nn::gfx::PrimitiveTopology_TriangleStrip, 4, 0);
    }
}


void GpuBenchmarkMrt::PrintResults(nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter)
{
    NN_ASSERT(runCount > 0);

    uint64_t gpuTimeElapsedValueInNs = static_cast<uint64_t>(gpuTimeElapsed.GetNanoSeconds());
    uint64_t cpuTimeElapsedValueInNs = static_cast<uint64_t>(cpuTimeElapsed.GetNanoSeconds());

    uint64_t gpuTimeElapsedAvgValueInNs = gpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);
    uint64_t cpuTimeElapsedAvgValueInNs = cpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);

    uint64_t pixelCount = static_cast<uint64_t>(runCount) * static_cast<uint64_t>(m_RenderSize * m_RenderSize);

    uint64_t gpuFillrateAvgMPixPerSec = 0;
    if (gpuTimeElapsedValueInNs > 0)
    {
        gpuFillrateAvgMPixPerSec = pixelCount / (gpuTimeElapsedValueInNs / 1000);
    }

    pDebugFontTextWriter->Print("fillRateAvgKPixPerSec:%12lu\n", gpuFillrateAvgMPixPerSec);
    pDebugFontTextWriter->Print("gpu time (ns): %8lu\n", gpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("cpu time (ns): %8lu\n", cpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("total gpu time (ns): %12lu\n", gpuTimeElapsedValueInNs);
    pDebugFontTextWriter->Print("total cpu time (ns): %12lu\n", cpuTimeElapsedValueInNs);
}

void GpuBenchmarkMrt::CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer)
{
    int renderSize = GetRenderSize();
    int bufferOffset = 0;

    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture);
    for (int mrtIndex = 0; mrtIndex < m_MrtCount; ++mrtIndex)
    {
        nn::gfx::BufferTextureCopyRegion bufferTextureCopyRegion;
        bufferTextureCopyRegion.SetDefault();
        bufferTextureCopyRegion.SetBufferImageHeight(renderSize);
        bufferTextureCopyRegion.SetBufferImageWidth(renderSize);
        bufferTextureCopyRegion.SetBufferOffset(bufferOffset);
        bufferTextureCopyRegion.EditTextureCopyRegion().SetDefault();
        bufferTextureCopyRegion.EditTextureCopyRegion().SetWidth(renderSize);
        bufferTextureCopyRegion.EditTextureCopyRegion().SetHeight(renderSize);
        bufferTextureCopyRegion.EditTextureCopyRegion().EditSubresource().SetDefault();

        pCommandBuffer->CopyImageToBuffer(&m_OutputCopyBuffer, &m_RenderTexture[mrtIndex], bufferTextureCopyRegion);

        bufferOffset += static_cast<int>(m_OutputCopyBufferRenderTargetSize[mrtIndex]);
    }
    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Write);
}

void GpuBenchmarkMrt::MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize)
{
    *pOutBuffer = m_OutputCopyBuffer.Map();
    *pOutBufferSize = m_OutputCopyBufferTotalSize;
}

void GpuBenchmarkMrt::UnmapResultBuffer()
{
    m_OutputCopyBuffer.Unmap();
}

const char* GpuBenchmarkMrt::GetName() const
{
    return ClassName;
}

BenchmarkType GpuBenchmarkMrt::GetType() const
{
    return BenchmarkType_Mrt;
}

int GpuBenchmarkMrt::GetPropertyCount() const
{
    return m_PropertyArray.GetCount();
}

int GpuBenchmarkMrt::FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

int GpuBenchmarkMrt::FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize)
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkMrt::FindPropertyByName(const char* propertyName)
{
    return m_PropertyArray.FindPropertyByName(propertyName);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkMrt::GetPropertyByIndex(int index)
{
    return m_PropertyArray.Get(index);
}

} } } // namespace nnt { namespace gfx { namespace util {
