﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include "gfxUtilGpuBenchmark_GpuBenchmarkCompositeTest.h"

#include "gfxUtilGpuBenchmark_ResHelpers.h"
#include "gfxUtilGpuBenchmark_PropertyMacros.h"
#include "gfxUtilGpuBenchmark_ComputeCompositeTestShaderVariationIndex.h"
#include "gfxUtilGpuBenchmark_ResourceAllocator.h"

#include "gfxUtilGpuBenchmark_BuiltinCompositeTestShader.h"

namespace nnt { namespace gfx { namespace util {

namespace {

struct ConstantBuffer
{
    int vertexGridSize;
    float colorFactor;
    float colorOffset;
};

#if defined(NN_GFXUTIL_GPUBENCHMARK_COMPOSITE_TEST_DEBUG)

void PrintResShaderReflectionDic(const char* name, nn::util::ResDic* pSamplerDic)
{
    if (pSamplerDic == nullptr)
        return;

    NN_LOG("%s:\n", name);
    for (int i = 0; i < pSamplerDic->GetCount(); ++i)
    {
        NN_LOG("%d: %s\n", i, pSamplerDic->GetKey(i).data());
    }
}

void PrintResShaderReflectionStageData(nn::gfx::ResShaderReflectionStageData* pResShaderReflectionStageData)
{
    PrintResShaderReflectionDic("pShaderInputDic", pResShaderReflectionStageData->pShaderInputDic.Get());
    PrintResShaderReflectionDic("pShaderOutputDic", pResShaderReflectionStageData->pShaderOutputDic.Get());
    PrintResShaderReflectionDic("pSamplerDic", pResShaderReflectionStageData->pSamplerDic.Get());
    PrintResShaderReflectionDic("pConstantBufferDic", pResShaderReflectionStageData->pConstantBufferDic.Get());
    PrintResShaderReflectionDic("pUnorderedAccessBufferDic", pResShaderReflectionStageData->pUnorderedAccessBufferDic.Get());
    PrintResShaderReflectionDic("pImageDic", pResShaderReflectionStageData->pImageDic.Get());

    nn::gfx::ResShaderReflectionStageData2* pResShaderReflectionStageData2 = pResShaderReflectionStageData->pReflectionStageData2.Get();
    if (pResShaderReflectionStageData2 != nullptr)
    {
        PrintResShaderReflectionDic("pSeparateTextureDic", pResShaderReflectionStageData2->pSeparateTextureDic.Get());
        PrintResShaderReflectionDic("pSeparateSamplerDic", pResShaderReflectionStageData2->pSeparateSamplerDic.Get());
    }
}
#endif

}

const char* GpuBenchmarkCompositeTest::ClassName = "CompositeTest";

GpuBenchmarkCompositeTest::GpuBenchmarkCompositeTest()
{
}

GpuBenchmarkCompositeTest::~GpuBenchmarkCompositeTest()
{
}

void GpuBenchmarkCompositeTest::Initialize(ResourceAllocator* pResourceAllocator)
{
    NN_UNUSED(pResourceAllocator);

    // RenderSize
    GpuBenchmarkPropertyHolder* pPropertyRenderSize  = m_PropertyArray.Get(Property_RenderSize);
    BENCHMARK_PROPERTY_INTEGER_RANGE_DEFINITION(
        pPropertyRenderSize, "RenderSize",
        GpuBenchmarkCompositeTest::GetRenderSize,
        GpuBenchmarkCompositeTest::SetRenderSize,
        2048, 4096, 1024);

    // RenderTargetMode
    GpuBenchmarkPropertyHolder* pPropertyRenderTargetMode = m_PropertyArray.Get(Property_RenderTargetMode);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyRenderTargetMode, "RenderTargetMode",
        RenderTargetMode, pResourceAllocator,
        GpuBenchmarkCompositeTest::GetRenderTargetMode,
        GpuBenchmarkCompositeTest::SetRenderTargetMode,
        "8_Int", RenderTargetMode_8I,
        "16_Float", RenderTargetMode_16F);

    // DepthMode
    GpuBenchmarkPropertyHolder* pPropertyDepthMode = m_PropertyArray.Get(Property_DepthMode);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyDepthMode, "DepthMode",
        DepthMode, pResourceAllocator,
        GpuBenchmarkCompositeTest::GetDepthMode,
        GpuBenchmarkCompositeTest::SetDepthMode,
        "Off", DepthMode_Off,
        "On", DepthMode_On);

    // ShaderLoadMode
    GpuBenchmarkPropertyHolder* pPropertyShaderLoadMode = m_PropertyArray.Get(Property_ShaderLoadMode);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyShaderLoadMode, "ShaderLoadMode",
        ShaderLoadMode, pResourceAllocator,
        GpuBenchmarkCompositeTest::GetShaderLoadMode,
        GpuBenchmarkCompositeTest::SetShaderLoadMode,
        "None", ShaderLoadMode_None,
        "1xMatrixMul", ShaderLoadMode_1xMatrixMul,
        "4xMatrixMul", ShaderLoadMode_2xMatrixMul);

    // TextureMode
    GpuBenchmarkPropertyHolder* pPropertyTextureMode = m_PropertyArray.Get(Property_TextureMode);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyTextureMode, "TextureMode",
        TextureMode, pResourceAllocator,
        GpuBenchmarkCompositeTest::GetTextureMode,
        GpuBenchmarkCompositeTest::SetTextureMode,
        "None", TextureMode_None,
        "1xBC1" ,TextureMode_1xBC1,
        "2xBC1", TextureMode_2xBC1);

    // BlendingMode
    GpuBenchmarkPropertyHolder* pPropertyBlendingMode = m_PropertyArray.Get(Property_BlendingMode);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyBlendingMode, "BlendingMode",
        BlendingMode, pResourceAllocator,
        GpuBenchmarkCompositeTest::GetBlendingMode,
        GpuBenchmarkCompositeTest::SetBlendingMode,
        "Off", BlendingMode_Off,
        "AlphaAdd", BlendingMode_AlphaAdd);

    // VertexAttributeMode
    GpuBenchmarkPropertyHolder* pPropertyVertexAttributeMode = m_PropertyArray.Get(Property_VertexAttributeMode);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyVertexAttributeMode, "VertexAttributeMode",
        VertexAttributeMode, pResourceAllocator,
        GpuBenchmarkCompositeTest::GetVertexAttributeMode,
        GpuBenchmarkCompositeTest::SetVertexAttributeMode,
        "None", VertexAttributeMode_None,
        "1", VertexAttributeMode_1,
        "2", VertexAttributeMode_2,
        "8", VertexAttributeMode_8);

    // VertexGridSize
    GpuBenchmarkPropertyHolder* pPropertyVertexGridSize = m_PropertyArray.Get(Property_VertexGridSize);
    BENCHMARK_PROPERTY_INTEGER_RANGE_DEFINITION(
        pPropertyVertexGridSize, "VertexGridSize",
        GpuBenchmarkCompositeTest::GetVertexGridSize,
        GpuBenchmarkCompositeTest::SetVertexGridSize,
        256, 512, 128);
}

void GpuBenchmarkCompositeTest::Finalize(ResourceAllocator* pResourceAllocator)
{
    NN_UNUSED(pResourceAllocator);

    for (int i = 0; i < m_PropertyArray.GetCount(); ++i)
    {
        m_PropertyArray.Get(i)->Finalize();
    }
}


void GpuBenchmarkCompositeTest::InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    GpuBenchmark::InitializeGfxObjects(pResourceAllocator, pDevice);

    InitializeGfxObjectsRenderTarget(pResourceAllocator, pDevice);

    InitializeResShader(&m_ResShader, g_CompositeTestShaderData, sizeof(g_CompositeTestShaderData), pResourceAllocator, pDevice);

#if defined(NN_GFXUTIL_GPUBENCHMARK_COMPOSITE_TEST_DEBUG)
    for (int shaderVariationIndex = 0; shaderVariationIndex < m_ResShader.pResShaderContainer->GetShaderVariationCount(); ++shaderVariationIndex)
    {
        NN_LOG("Variation %d\n", shaderVariationIndex);

        nn::gfx::ResShaderProgram* pResShaderProgram = m_ResShader.pResShaderContainer->GetResShaderVariation(shaderVariationIndex)->GetResShaderProgram(m_ResShader.codeType);
        nn::gfx::ResShaderReflectionData* pResShaderReflectionData = pResShaderProgram->ToData().pShaderReflection.Get();

        NN_LOG("Vertex shader\n");

        nn::util::BinTPtr<nn::gfx::ResShaderReflectionStageData> pPtrResShaderReflectionVertexStageData = pResShaderReflectionData->pVertexReflection;
        PrintResShaderReflectionStageData(pPtrResShaderReflectionVertexStageData.Get());
    }
#endif

    InitializeGfxObjectsTexture(pResourceAllocator, pDevice);
    InitializeGfxObjectsVertexBuffer(pResourceAllocator, pDevice);
    InitializeGfxObjectsTranform(pResourceAllocator, pDevice);
    InitializeGfxObjectsBlendState(pResourceAllocator, pDevice);
}

void GpuBenchmarkCompositeTest::FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    FinalizeGfxObjectsBlendState(pResourceAllocator, pDevice);
    FinalizeGfxObjectsTranform(pResourceAllocator, pDevice);
    FinalizeGfxObjectsVertexBuffer(pResourceAllocator, pDevice);
    FinalizeGfxObjectsTexture(pResourceAllocator, pDevice);
    FinalizeResShader(&m_ResShader, pResourceAllocator, pDevice);
    FinalizeGfxObjectsRenderTarget(pResourceAllocator, pDevice);

    GpuBenchmark::FinalizeGfxObjects(pResourceAllocator, pDevice);
}


void GpuBenchmarkCompositeTest::PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer)
{
    nn::gfx::ColorTargetView* pTestTarget = &m_RenderTextureColorTargetView;
    nn::gfx::DepthStencilView* pTestDepthStencil = (m_DepthMode == DepthMode_On) ? &m_DepthStencilView : nullptr;

    pTestCommandBuffer->ClearColor(pTestTarget, 0.0f, 0.0f, 0.0f, 1.0f, nullptr);
    if (pTestDepthStencil != nullptr)
        pTestCommandBuffer->ClearDepthStencil(pTestDepthStencil, 0.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

    pTestCommandBuffer->SetRenderTargets(1, &pTestTarget, pTestDepthStencil);
    pTestCommandBuffer->SetViewportScissorState(&m_ViewportScissorState);

    if (m_VertexAttributeMode != VertexAttributeMode_None)
        pTestCommandBuffer->SetVertexState(&m_VertexState);

    pTestCommandBuffer->SetBlendState(&m_BlendState);
    pTestCommandBuffer->SetDepthStencilState(&m_DepthStencilState);

    const int shaderVariationIndex = ComputeCompositeTestShaderVariationIndex(m_ShaderLoadMode, m_VertexAttributeMode, m_TextureMode);
    nn::gfx::ResShaderProgram* pResShaderProgram =
        m_ResShader.pResShaderContainer->GetResShaderVariation(shaderVariationIndex)->GetResShaderProgram(m_ResShader.codeType);

#if defined(NN_SDK_BUILD_DEBUG)
    nn::gfx::ResShaderReflectionData* pResShaderReflectionData = pResShaderProgram->ToData().pShaderReflection.Get();
    if (pResShaderReflectionData != nullptr)
    {
        nn::gfx::ResShaderReflectionStageData* pResShaderReflectionPixelStageData = pResShaderReflectionData->pPixelReflection.Get();
        nn::util::ResDic* pPixelSamplerDic = pResShaderReflectionPixelStageData->pSamplerDic.Get();
        NN_ASSERT(
            ((pPixelSamplerDic == nullptr) && (GetTextureCount() == 0)) ||
            ((pPixelSamplerDic != nullptr) && (pPixelSamplerDic->GetCount() == GetTextureCount())));

        nn::gfx::ResShaderReflectionStageData* pResShaderReflectionVertexStageData = pResShaderReflectionData->pVertexReflection.Get();
        nn::util::ResDic* pVertexInputDic = pResShaderReflectionVertexStageData->pShaderInputDic.Get();
        NN_ASSERT((pVertexInputDic != nullptr) && (pVertexInputDic->GetCount() == (GetVertexAttributeCount() + 1)));
    }
#endif

    nn::gfx::Shader* pShader = pResShaderProgram->GetShader();
    pTestCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);

    pTestCommandBuffer->SetConstantBuffer(0, nn::gfx::ShaderStage_Vertex, m_ConstantBufferDescriptorSlot);
    pTestCommandBuffer->SetConstantBuffer(0, nn::gfx::ShaderStage_Pixel, m_ConstantBufferDescriptorSlot);

    if (m_ShaderLoadMode != ShaderLoadMode_None)
    {
        pTestCommandBuffer->SetConstantBuffer(1, nn::gfx::ShaderStage_Vertex, m_TransformConstantBufferDescriptorSlot);
        pTestCommandBuffer->SetConstantBuffer(1, nn::gfx::ShaderStage_Pixel, m_TransformConstantBufferDescriptorSlot);
    }

    if (m_VertexAttributeMode != VertexAttributeMode_None)
    {
        nn::gfx::GpuAddress vertexBufferGpuAddress;
        m_VertexBuffer.GetGpuAddress(&vertexBufferGpuAddress);
        pTestCommandBuffer->SetVertexBuffer(0, vertexBufferGpuAddress, sizeof(DefaultVertex), g_RectangleVertexBufferDataSize);
    }

    int textureCount = GetTextureCount();
    for (int textureIndex = 0; textureIndex < textureCount; ++textureIndex)
    {
        TextureData* pTextureData = &m_TextureDataArray[textureIndex];
        pTestCommandBuffer->SetTextureAndSampler(
            textureIndex, nn::gfx::ShaderStage_Pixel,
            pTextureData->descriptorSlot, m_TextureSamplerDescriptorSlot);
    }
}

void GpuBenchmarkCompositeTest::DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount)
{
    NN_ASSERT(runCount > 0);

    int vertexCount = GetVertexCount();
    bool blendEnabled = GetBlendEnabled();

    nn::gfx::ColorTargetView* pTestTarget = &m_RenderTextureColorTargetView;
    nn::gfx::DepthStencilView* pTestDepthStencil = (m_DepthMode == DepthMode_On) ? &m_DepthStencilView : nullptr;

    for (int repeatIndex = 0; repeatIndex < runCount; ++repeatIndex)
    {
        if (blendEnabled)
        {
            pTestCommandBuffer->ClearColor(pTestTarget, 0.5f, 0.5f, 0.5f, 0.5f, nullptr);
            if (pTestDepthStencil != nullptr)
                pTestCommandBuffer->ClearDepthStencil(pTestDepthStencil, 0.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);
            pTestCommandBuffer->SetRenderTargets(1, &pTestTarget, pTestDepthStencil);
            pTestCommandBuffer->SetViewportScissorState(&m_ViewportScissorState);
        }

        pTestCommandBuffer->Draw(nn::gfx::PrimitiveTopology_TriangleList, vertexCount, 0);
    }
}


void GpuBenchmarkCompositeTest::PrintResults(nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter)
{
    NN_ASSERT(runCount > 0);

    uint64_t gpuTimeElapsedValueInNs = static_cast<uint64_t>(gpuTimeElapsed.GetNanoSeconds());
    uint64_t cpuTimeElapsedValueInNs = static_cast<uint64_t>(cpuTimeElapsed.GetNanoSeconds());

    uint64_t gpuTimeElapsedAvgValueInNs = gpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);
    uint64_t cpuTimeElapsedAvgValueInNs = cpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);

    pDebugFontTextWriter->Print("gpu time (ns): %8lu\n", gpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("cpu time (ns): %8lu\n", cpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("total gpu time (ns): %12lu\n", gpuTimeElapsedValueInNs);
    pDebugFontTextWriter->Print("total cpu time (ns): %12lu\n", cpuTimeElapsedValueInNs);
}

void GpuBenchmarkCompositeTest::CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer)
{
    int renderSize = GetRenderSize();

    nn::gfx::BufferTextureCopyRegion bufferTextureCopyRegion;
    bufferTextureCopyRegion.SetDefault();
    bufferTextureCopyRegion.SetBufferImageHeight(renderSize);
    bufferTextureCopyRegion.SetBufferImageWidth(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().SetDefault();
    bufferTextureCopyRegion.EditTextureCopyRegion().SetWidth(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().SetHeight(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().EditSubresource().SetDefault();

    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture);
    pCommandBuffer->CopyImageToBuffer(&m_OutputCopyBuffer, &m_RenderTexture, bufferTextureCopyRegion);
    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Write);
}

void GpuBenchmarkCompositeTest::MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize)
{
    *pOutBuffer = m_OutputCopyBuffer.Map();
    *pOutBufferSize = m_OutputCopyBufferSize;
}

void GpuBenchmarkCompositeTest::UnmapResultBuffer()
{
    m_OutputCopyBuffer.Unmap();
}


const char* GpuBenchmarkCompositeTest::GetName() const
{
    return ClassName;
}

BenchmarkType GpuBenchmarkCompositeTest::GetType() const
{
    return BenchmarkType_CompositeTest;
}

int GpuBenchmarkCompositeTest::GetPropertyCount() const
{
    return m_PropertyArray.GetCount();
}

int GpuBenchmarkCompositeTest::FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

int GpuBenchmarkCompositeTest::FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize)
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkCompositeTest::FindPropertyByName(const char* propertyName)
{
    return m_PropertyArray.FindPropertyByName(propertyName);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkCompositeTest::GetPropertyByIndex(int index)
{
    return m_PropertyArray.Get(index);
}

nn::gfx::ImageFormat GpuBenchmarkCompositeTest::GetRenderTargetFormat() const
{
    switch (m_RenderTargetMode)
    {
    case RenderTargetMode_8I:       return nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
    case RenderTargetMode_16F:      return nn::gfx::ImageFormat_R16_G16_B16_A16_Float;
    default:                        NN_UNEXPECTED_DEFAULT;
    }
}

nn::gfx::ImageFormat GpuBenchmarkCompositeTest::GetDepthStencilFormat() const
{
    if (m_DepthMode == DepthMode_Off)
        return nn::gfx::ImageFormat_Undefined;

    switch (m_RenderTargetMode)
    {
    case RenderTargetMode_8I:       return nn::gfx::ImageFormat_D24_Unorm_S8_Uint;
    case RenderTargetMode_16F:      return nn::gfx::ImageFormat_D32_Float;
    default:                        NN_UNEXPECTED_DEFAULT;
    }
}


int GpuBenchmarkCompositeTest::GetTextureCount() const
{
    switch (m_TextureMode)
    {
    case TextureMode_None:      return 0;
    case TextureMode_1xBC1:     return 1;
    case TextureMode_2xBC1:     return 2;
    case TextureMode_4xBC1:     return 4;
    default:                    NN_UNEXPECTED_DEFAULT;
    }
}
int GpuBenchmarkCompositeTest::GetVertexAttributeCount() const
{
    switch (m_VertexAttributeMode)
    {
    case VertexAttributeMode_None:      return 0;
    case VertexAttributeMode_1:         return 1;
    case VertexAttributeMode_2:         return 2;
    case VertexAttributeMode_4:         return 4;
    case VertexAttributeMode_8:         return 8;
    default:                            NN_UNEXPECTED_DEFAULT;
    }
}

int GpuBenchmarkCompositeTest::GetVertexCount() const
{
    return m_VertexGridSize * m_VertexGridSize * 6;
}

bool GpuBenchmarkCompositeTest::GetBlendEnabled() const
{
    switch (m_BlendingMode)
    {
    case BlendingMode_Off:          return false;
    case BlendingMode_AlphaAdd:     return true;
    case BlendingMode_AlphaSub:     return true;
    default:                        NN_UNEXPECTED_DEFAULT;
    }
}

nn::gfx::BlendFunction GpuBenchmarkCompositeTest::GetBlendFunction() const
{
    switch (m_BlendingMode)
    {
    case BlendingMode_Off:          return nn::gfx::BlendFunction_Add;
    case BlendingMode_AlphaAdd:     return nn::gfx::BlendFunction_Add;
    case BlendingMode_AlphaSub:     return nn::gfx::BlendFunction_Subtract;
    default:                        NN_UNEXPECTED_DEFAULT;
    }
}

void GpuBenchmarkCompositeTest::InitializeGfxObjectsRenderTarget(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    nn::gfx::ImageFormat renderTargetFormat = GetRenderTargetFormat();
    nn::gfx::ImageFormat depthStencilFormat = GetDepthStencilFormat();

    m_OutputCopyBufferSize = InitializeColorRenderTarget(
        &m_RenderTexture, &m_OutputCopyBuffer,
        &m_RenderTextureColorTargetView, &m_ViewportScissorState,
        m_RenderSize, m_RenderSize, renderTargetFormat, m_TileMode,
        pResourceAllocator, pDevice);

    bool depthEnabled = (depthStencilFormat != nn::gfx::ImageFormat_Undefined);
    if (depthEnabled)
    {
        nn::gfx::Texture::InfoType depthTextureInfo;
        {
            depthTextureInfo.SetDefault();
            depthTextureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_DepthStencil);
            depthTextureInfo.SetWidth(m_RenderSize);
            depthTextureInfo.SetHeight(m_RenderSize);
            depthTextureInfo.SetDepth(1);
            depthTextureInfo.SetImageFormat(depthStencilFormat);
            depthTextureInfo.SetMipCount(1);
            depthTextureInfo.SetArrayLength(1);
            depthTextureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
            depthTextureInfo.SetTileMode(m_TileMode);
        }
        InitializeTexture(
            &m_DepthStencilTexture, depthTextureInfo,
            pResourceAllocator, MemoryPoolType_RenderTarget, pDevice);

        nn::gfx::DepthStencilView::InfoType depthStencilViewInfo;
        {
            depthStencilViewInfo.SetDefault();
            depthStencilViewInfo.SetTexturePtr(&m_DepthStencilTexture);
            depthStencilViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        }

        m_DepthStencilView.Initialize(pDevice, depthStencilViewInfo);
    }

    nn::gfx::DepthStencilState::InfoType depthStencilInfo;
    {
        depthStencilInfo.SetDefault();
        depthStencilInfo.SetDepthTestEnabled(depthEnabled);
        depthStencilInfo.SetDepthComparisonFunction(nn::gfx::ComparisonFunction_Always);
        depthStencilInfo.SetDepthWriteEnabled(depthEnabled);
    }
    m_DepthStencilState.Initialize(pDevice, depthStencilInfo);
}

void GpuBenchmarkCompositeTest::FinalizeGfxObjectsRenderTarget(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    nn::gfx::ImageFormat depthStencilFormat = GetDepthStencilFormat();
    bool depthEnabled = (depthStencilFormat != nn::gfx::ImageFormat_Undefined);

    m_DepthStencilState.Finalize(pDevice);

    if (depthEnabled)
    {
        m_DepthStencilView.Finalize(pDevice);
        FinalizeTexture(&m_DepthStencilTexture, pResourceAllocator, pDevice);
    }

    FinalizeColorRenderTarget(
        &m_RenderTexture, &m_OutputCopyBuffer, &m_RenderTextureColorTargetView, &m_ViewportScissorState,
        pResourceAllocator, pDevice);
    m_OutputCopyBufferSize = 0;
}

void GpuBenchmarkCompositeTest::InitializeGfxObjectsTexture(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    const int textureCount = GetTextureCount();
    const nn::gfx::ImageFormat textureFormat = nn::gfx::ImageFormat_Bc1_Unorm;
    const int textureDimension = 512;
    const int mipCount = 5;
    const nn::gfx::FilterMode textureFilterMode = nn::gfx::FilterMode_MinLinear_MagLinear_MipLinear;

    for (int textureIndex = 0; textureIndex < textureCount; ++textureIndex)
    {
        TextureData* pTextureData = &m_TextureDataArray[textureIndex];

        nn::gfx::Texture::InfoType textureInfo;
        {
            textureInfo.SetDefault();
            textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
            textureInfo.SetHeight(textureDimension);
            textureInfo.SetWidth(textureDimension);
            textureInfo.SetImageFormat(textureFormat);
            textureInfo.SetTileMode(nn::gfx::TileMode_Optimal);
            textureInfo.SetMipCount(mipCount);
        }


        InitializeTexture(&pTextureData->texture, textureInfo, pResourceAllocator, MemoryPoolType_Data, pDevice);

        nn::gfx::TextureView::InfoType textureViewInfo;
        {
            textureViewInfo.SetDefault();
            textureViewInfo.SetTexturePtr(&pTextureData->texture);
            textureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
            textureViewInfo.SetImageFormat(textureFormat);
            textureViewInfo.EditSubresourceRange().EditMipRange().SetMipCount(mipCount);
        }

        pTextureData->textureView.Initialize(pDevice, textureViewInfo);

        pTextureData->slotIndex = pResourceAllocator->AllocateAndSetTextureViewToDescriptorPool(
            &pTextureData->textureView, &pTextureData->descriptorSlot);
    }

    nn::gfx::Sampler::InfoType textureSamplerInfo;
    {
        textureSamplerInfo.SetDefault();
        textureSamplerInfo.SetFilterMode(textureFilterMode);
        textureSamplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToEdge);
        textureSamplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToEdge);
        textureSamplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
    }
    m_TextureSampler.Initialize(pDevice, textureSamplerInfo);
    m_TextureSamplerSlotIndex = pResourceAllocator->AllocateAndSetSamplerToDescriptorPool(
        &m_TextureSampler, &m_TextureSamplerDescriptorSlot);
}

void GpuBenchmarkCompositeTest::FinalizeGfxObjectsTexture(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    const int textureCount = GetTextureCount();

    pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_Sampler, m_TextureSamplerSlotIndex);
    m_TextureSamplerSlotIndex = -1;
    m_TextureSampler.Finalize(pDevice);

    for (int textureIndex = 0; textureIndex < textureCount; ++textureIndex)
    {
        TextureData* pTextureData = &m_TextureDataArray[textureIndex];
        pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_TextureView, pTextureData->slotIndex);

        pTextureData->textureView.Finalize(pDevice);
        FinalizeTexture(&pTextureData->texture, pResourceAllocator, pDevice);

        pTextureData->slotIndex = -1;
    }
}

void GpuBenchmarkCompositeTest::InitializeGfxObjectsVertexBuffer(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    if (m_VertexAttributeMode != VertexAttributeMode_None)
    {
        const nn::gfx::AttributeFormat attributeFormat = nn::gfx::AttributeFormat_32_32_32_32_Float;
        int attributeSize = 4 * 4; // AttributeFormat_32_32_32_32_Float
        int attributeCount = GetVertexAttributeCount();

        m_VertexStride = attributeSize * attributeCount;

        size_t vertexBufferSizeInBytes = static_cast<size_t>(m_VertexStride) * GetVertexCount();

        nn::gfx::Buffer::InfoType vertexBufferInfo;
        {
            vertexBufferInfo.SetDefault();
            vertexBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);
            vertexBufferInfo.SetSize(vertexBufferSizeInBytes);
        }
        InitializeBuffer(&m_VertexBuffer, vertexBufferInfo, pResourceAllocator, MemoryPoolType_Data, pDevice);

        void* pVertexBuffer = m_VertexBuffer.Map();
        memset(pVertexBuffer, 0, vertexBufferInfo.GetSize());
        m_VertexBuffer.Unmap();
        m_VertexBuffer.FlushMappedRange(0, vertexBufferInfo.GetSize());

        nn::gfx::VertexState::InfoType vertexStateInfo;
        {
            nn::gfx::VertexAttributeStateInfo attributeStateInfo[m_MaxAttributeCount];

            ptrdiff_t vertexOffset = 0;
            for (int attributeIndex = 0; attributeIndex < attributeCount; ++attributeIndex)
            {
                nn::gfx::VertexAttributeStateInfo* pVertexAttributeStateInfo = &attributeStateInfo[attributeIndex];
                pVertexAttributeStateInfo->SetBufferIndex(0);
                pVertexAttributeStateInfo->SetFormat(attributeFormat);
                pVertexAttributeStateInfo->SetOffset(vertexOffset);
                pVertexAttributeStateInfo->SetShaderSlot(attributeIndex);

                vertexOffset += attributeSize;
            }
            NN_ASSERT(vertexOffset == m_VertexStride);

            nn::gfx::VertexBufferStateInfo vertexBufferStateInfo;
            vertexBufferStateInfo.SetDefault();
            vertexBufferStateInfo.SetStride(m_VertexStride);

            vertexStateInfo.SetDefault();
            vertexStateInfo.SetVertexAttributeStateInfoArray(attributeStateInfo, attributeCount);
            vertexStateInfo.SetVertexBufferStateInfoArray(&vertexBufferStateInfo, 1);
        }
        InitializeVertexState(&m_VertexState, vertexStateInfo, pResourceAllocator, pDevice);
    }

    nn::gfx::Buffer::InfoType constantBufferInfo;
    {
        constantBufferInfo.SetDefault();
        constantBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        constantBufferInfo.SetSize(sizeof(ConstantBuffer));
    }
    InitializeBuffer(&m_ConstantBuffer, constantBufferInfo, pResourceAllocator, MemoryPoolType_ConstantBuffer, pDevice);

    ConstantBuffer* pConstantBuffer = nn::util::BytePtr(m_ConstantBuffer.Map()).Get<ConstantBuffer>();
    pConstantBuffer->vertexGridSize = m_VertexGridSize;
    pConstantBuffer->colorFactor = 1.0f;
    pConstantBuffer->colorOffset = 0.0f;
    m_ConstantBuffer.Unmap();
    m_ConstantBuffer.FlushMappedRange(0, constantBufferInfo.GetSize());

    m_ConstantBufferDescriptorSlotIndex = pResourceAllocator->AllocateAndSetBufferViewToDescriptorPool(
        &m_ConstantBuffer, constantBufferInfo.GetSize(), &m_ConstantBufferDescriptorSlot);
}

void GpuBenchmarkCompositeTest::FinalizeGfxObjectsVertexBuffer(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    FinalizeBuffer(&m_ConstantBuffer, pResourceAllocator, pDevice);
    pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_BufferView, m_ConstantBufferDescriptorSlotIndex);
    m_ConstantBufferDescriptorSlotIndex = 1;

    if (m_VertexAttributeMode != VertexAttributeMode_None)
    {
        FinalizeVertexState(&m_VertexState, pResourceAllocator, pDevice);
        FinalizeBuffer(&m_VertexBuffer, pResourceAllocator, pDevice);
    }
}

void GpuBenchmarkCompositeTest::InitializeGfxObjectsTranform(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    nn::gfx::Buffer::InfoType constantBufferInfo;
    {
        constantBufferInfo.SetDefault();
        constantBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        constantBufferInfo.SetSize(m_TransformConstantBufferSize);

    }

    InitializeBuffer(&m_TransformConstantBuffer, constantBufferInfo, pResourceAllocator, MemoryPoolType_ConstantBuffer, pDevice);
    {
        float* pConstantBufferMemory = nn::util::BytePtr(m_TransformConstantBuffer.Map()).Get<float>();
        memset(pConstantBufferMemory, 0, m_TransformConstantBufferSize);

        for (int i = 0; i < m_ConstantBufferMatrixCount; ++i)
        {
            pConstantBufferMemory[0] = 1.0f;
            pConstantBufferMemory[5] = 1.0f;
            pConstantBufferMemory[10] = 1.0f;
            pConstantBufferMemory[15] = 1.0f;
            pConstantBufferMemory += 16;
        }

        m_TransformConstantBuffer.Unmap();
        m_TransformConstantBuffer.FlushMappedRange(0, m_TransformConstantBufferSize);
    }

    m_TransformConstantBufferDescriptorSlotIndex = pResourceAllocator->AllocateAndSetBufferViewToDescriptorPool(
        &m_TransformConstantBuffer, constantBufferInfo.GetSize(), &m_TransformConstantBufferDescriptorSlot);
}

void GpuBenchmarkCompositeTest::FinalizeGfxObjectsTranform(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    FinalizeBuffer(&m_TransformConstantBuffer, pResourceAllocator, pDevice);
    pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_BufferView, m_TransformConstantBufferDescriptorSlotIndex);
    m_TransformConstantBufferDescriptorSlotIndex = -1;
}

void GpuBenchmarkCompositeTest::InitializeGfxObjectsBlendState(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    bool                    blendEnabled            = GetBlendEnabled();
    nn::gfx::BlendFactor    blendSourceFactor       = nn::gfx::BlendFactor_SourceAlpha;
    nn::gfx::BlendFactor    blendDestinationFactor  = nn::gfx::BlendFactor_OneMinusSourceAlpha;
    nn::gfx::BlendFunction  blendFunction           = GetBlendFunction();

    nn::gfx::BlendTargetStateInfo targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red | nn::gfx::ChannelMask_Green | nn::gfx::ChannelMask_Blue | nn::gfx::ChannelMask_Alpha);
    targetInfo.SetBlendEnabled(blendEnabled);
    targetInfo.SetColorBlendFunction(blendFunction);
    targetInfo.SetDestinationColorBlendFactor(blendDestinationFactor);
    targetInfo.SetSourceColorBlendFactor(blendSourceFactor);

    nn::gfx::BlendState::InfoType blendInfo;
    blendInfo.SetDefault();
    blendInfo.SetBlendTargetStateInfoArray(&targetInfo, 1);
    blendInfo.SetBlendConstant(0.5f, 0.5f, 0.5f, 0.5f);
    InitializeBlendState(&m_BlendState, blendInfo, pResourceAllocator, pDevice);
}

void GpuBenchmarkCompositeTest::FinalizeGfxObjectsBlendState(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    FinalizeBlendState(&m_BlendState, pResourceAllocator, pDevice);
}



} } } // namespace nnt { namespace gfx { namespace util {
