﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include "gfxUtilGpuBenchmark_GpuBenchmark.h"

#include "gfxUtilGpuBenchmark_Property.h"
#include "gfxUtilGpuBenchmark_PropertyArray.h"
#include "gfxUtilGpuBenchmark_ResShader.h"

#include <nn/gfx/gfx_Texture.h>
#include <nn/gfx/gfx_State.h>
#include <nn/gfx/gfx_DescriptorSlot.h>

//#define NN_GFXUTIL_GPUBENCHMARK_BLITTEXTURE_DEBUG

namespace nnt { namespace gfx { namespace util {

class GpuBenchmarkBlitTexture : public GpuBenchmark
{
public:
    static const char* ClassName;

    enum RenderSizeConfiguration
    {
        RenderSizeConfiguration_1280x720,
        RenderSizeConfiguration_1920x1080,
    };

    enum RenderNvnCompression
    {
        RenderNvnCompression_Enabled,
        RenderNvnCompression_ForceDisable,
    };

    enum BlendingMode
    {
        BlendingMode_Off,
        BlendingMode_AlphaAdd,
    };

    enum TextureSize
    {
        TextureSize_1280x720,
        TextureSize_1920x1080,
        TextureSize_256x256,
        TextureSize_512x512,
        TextureSize_1024x1024,
        TextureSize_2048x2048,
    };

    enum TextureFillMode
    {
        TextureFillMode_SingleColor,
        TextureFillMode_ColorPattern,
    };

private:
    nn::gfx::Texture                    m_RenderTexture;
    nn::gfx::ColorTargetView            m_RenderTextureColorTargetView;
    nn::gfx::ViewportScissorState       m_ViewportScissorState;

    nn::gfx::Texture                    m_Texture;
    nn::gfx::TextureView                m_TextureView;
    int                                 m_TextureSlotIndex;
    nn::gfx::DescriptorSlot             m_TextureDescriptorSlot;

    nn::gfx::Sampler                    m_TextureSampler;
    int                                 m_TextureSamplerSlotIndex;
    nn::gfx::DescriptorSlot             m_TextureSamplerDescriptorSlot;

    nn::gfx::BlendState                 m_BlendState;

    ResShader                           m_ResShader;

    nn::gfx::Buffer                     m_OutputCopyBuffer;
    size_t                              m_OutputCopyBufferSize;


    RenderSizeConfiguration             m_RenderSizeConfiguration;
    nn::gfx::ImageFormat                m_RenderFormat;
#if NN_GFX_IS_TARGET_NVN
    RenderNvnCompression                m_RenderNvnCompression;
#endif
    BlendingMode                        m_BlendingMode;
    TextureSize                         m_TextureSize;
    nn::gfx::FilterMode                 m_TextureFilterMode;
    nn::gfx::ImageFormat                m_TextureFormat;
    TextureFillMode                     m_TextureFillMode;

    enum Property
    {
        Property_RenderSizeConfiguration = 0,
        Property_RenderFormat,
#if NN_GFX_IS_TARGET_NVN
        Property_RenderNvnCompression,
#endif
        Property_BlendingMode,
        Property_TextureSize,
        Property_TextureFilterMode,
        Property_TextureFormat,
        Property_TextureFillMode,
        Property_Count,
    };

    PropertyArray<Property_Count>       m_PropertyArray;

public:
                                                GpuBenchmarkBlitTexture();
    virtual                                     ~GpuBenchmarkBlitTexture();

    virtual void                                Initialize(ResourceAllocator* pResourceAllocator) NN_OVERRIDE;
    virtual void                                Finalize(ResourceAllocator* pResourceAllocator) NN_OVERRIDE;

    virtual void                                InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice) NN_OVERRIDE;
    virtual void                                FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice) NN_OVERRIDE;

    RenderSizeConfiguration                     GetRenderSizeConfiguration() const;
    void                                        SetRenderSizeConfiguration(RenderSizeConfiguration value);

    nn::gfx::ImageFormat                        GetRenderFormat() const;
    void                                        SetRenderFormat(nn::gfx::ImageFormat value);

#if NN_GFX_IS_TARGET_NVN
    RenderNvnCompression                        GetRenderNvnCompression() const;
    void                                        SetRenderNvnCompression(RenderNvnCompression value);
#endif

    BlendingMode                                GetBlendingMode() const;
    void                                        SetBlendingMode(BlendingMode value);

    TextureSize                                 GetTextureSize() const;
    void                                        SetTextureSize(TextureSize value);

    nn::gfx::FilterMode                         GetTextureFilterMode() const;
    void                                        SetTextureFilterMode(nn::gfx::FilterMode value);

    nn::gfx::ImageFormat                        GetTextureFormat() const;
    void                                        SetTextureFormat(nn::gfx::ImageFormat value);

    TextureFillMode                             GetTextureFillMode() const;
    void                                        SetTextureFillMode(TextureFillMode value);

    virtual void                                PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer) NN_OVERRIDE;
    virtual void                                DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount) NN_OVERRIDE;
    void                                        RenderDebug(nn::gfx::CommandBuffer* pTestCommandBuffer) NN_OVERRIDE;

    virtual void                                PrintResults(nn::TimeSpan gpuTimeElapsed, nn::TimeSpan cpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter) NN_OVERRIDE;

    virtual void                                CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer) NN_OVERRIDE;
    virtual void                                MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize) NN_OVERRIDE;
    virtual void                                UnmapResultBuffer() NN_OVERRIDE;

    virtual const char*                         GetName() const NN_OVERRIDE;
    virtual BenchmarkType                       GetType() const NN_OVERRIDE;
    virtual int                                 GetPropertyCount() const NN_OVERRIDE;
    virtual int                                 FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const NN_OVERRIDE;
    virtual int                                 FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) NN_OVERRIDE;
    virtual GpuBenchmarkPropertyHolder*         FindPropertyByName(const char* propertyName) NN_OVERRIDE;
    virtual GpuBenchmarkPropertyHolder*         GetPropertyByIndex(int index) NN_OVERRIDE;

private:
    int                                         GetRenderWidth() const;
    int                                         GetRenderHeight() const;

    int                                         GetTextureWidth() const;
    int                                         GetTextureHeight() const;

    void                                        GetBlendConfiguration(
                                                    bool* pOutBlendEnabled,
                                                    nn::gfx::BlendFunction* pOutBlendFunction,
                                                    nn::gfx::BlendFactor* pOutBlendDestinationFactor,
                                                    nn::gfx::BlendFactor* pOutBlendSourceFactor);

    void                                        InitializeRenderTexture(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);
    void                                        FinalizeRenderTexture(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);
};

inline GpuBenchmarkBlitTexture::RenderSizeConfiguration GpuBenchmarkBlitTexture::GetRenderSizeConfiguration() const
{
    return m_RenderSizeConfiguration;
}

inline void GpuBenchmarkBlitTexture::SetRenderSizeConfiguration(RenderSizeConfiguration value)
{
    m_RenderSizeConfiguration = value;
}

inline nn::gfx::ImageFormat GpuBenchmarkBlitTexture::GetRenderFormat() const
{
    return m_RenderFormat;
}

inline void GpuBenchmarkBlitTexture::SetRenderFormat(nn::gfx::ImageFormat value)
{
    m_RenderFormat = value;
}

#if NN_GFX_IS_TARGET_NVN
inline GpuBenchmarkBlitTexture::RenderNvnCompression GpuBenchmarkBlitTexture::GetRenderNvnCompression() const
{
    return m_RenderNvnCompression;
}

inline void GpuBenchmarkBlitTexture::SetRenderNvnCompression(RenderNvnCompression value)
{
    m_RenderNvnCompression = value;
}
#endif

inline GpuBenchmarkBlitTexture::BlendingMode GpuBenchmarkBlitTexture::GetBlendingMode() const
{
    return m_BlendingMode;
}

inline void GpuBenchmarkBlitTexture::SetBlendingMode(BlendingMode value)
{
    m_BlendingMode = value;
}

inline GpuBenchmarkBlitTexture::TextureSize GpuBenchmarkBlitTexture::GetTextureSize() const
{
    return m_TextureSize;
}

inline void GpuBenchmarkBlitTexture::SetTextureSize(TextureSize value)
{
    m_TextureSize = value;
}

inline nn::gfx::FilterMode GpuBenchmarkBlitTexture::GetTextureFilterMode() const
{
    return m_TextureFilterMode;
}

inline void GpuBenchmarkBlitTexture::SetTextureFilterMode(nn::gfx::FilterMode value)
{
    m_TextureFilterMode = value;
}

inline nn::gfx::ImageFormat GpuBenchmarkBlitTexture::GetTextureFormat() const
{
    return m_TextureFormat;
}

inline void GpuBenchmarkBlitTexture::SetTextureFormat(nn::gfx::ImageFormat value)
{
    m_TextureFormat = value;
}

inline GpuBenchmarkBlitTexture::TextureFillMode GpuBenchmarkBlitTexture::GetTextureFillMode() const
{
    return m_TextureFillMode;
}

inline void GpuBenchmarkBlitTexture::SetTextureFillMode(TextureFillMode value)
{
    m_TextureFillMode = value;
}


} } } // namespace nnt { namespace gfx { namespace util {
