﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "gfxUtilAgingTest_LibraryState.h"

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/crypto/crypto_Sha1Generator.h>

#include "gfxUtilAgingTest_BmpReader.h"

#include <gfxUtilGpuBenchmark_JsonStreamer.h>
#include <gfxUtilGpuBenchmark_Property.h>
#include <gfxUtilGpuBenchmark_Factory.h>
#include <gfxUtilGpuBenchmark_ResHelpers.h>
#include <gfxUtilGpuBenchmark_PlatformId.h>


#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif


#if defined(NN_SDK_BUILD_DEBUG)
#   define ASSERT_SAME_THREAD() do { NN_ASSERT(m_OwnerThreadId == nn::os::GetThreadId(nn::os::GetCurrentThread())); } while (false)
#else
#   define ASSERT_SAME_THREAD() do { } while (false)
#endif

namespace nnt { namespace gfx { namespace util { namespace agingtest {

namespace
{

nn::gfx::ImageFormat GetImageFormat(PixelFormat pixelFormat)
{
    switch (pixelFormat)
    {
        case PixelFormat_RGB:       return nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
        case PixelFormat_RGBA:      return nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
        default:                    NN_UNEXPECTED_DEFAULT;
    }
}

int GetDestinationPixelSizeInBytes(PixelFormat pixelFormat)
{
    switch (pixelFormat)
    {
    case PixelFormat_RGB:       return 4;
    case PixelFormat_RGBA:      return 4;
    default:                    NN_UNEXPECTED_DEFAULT;
    }

}

bool MustConvertPixelFormat(PixelFormat pixelFormat)
{
    switch (pixelFormat)
    {
    case PixelFormat_RGB:       return true;
    case PixelFormat_RGBA:      return false;
    default:                    NN_UNEXPECTED_DEFAULT;
    }

}

void ConvertPixelLine(
    uint8_t* pDestinationBuffer,
    const uint8_t* pSourceBuffer,
    int width, PixelFormat sourcePixelFormat)
{

    if (sourcePixelFormat == PixelFormat_RGBA)
    {
        memcpy(pDestinationBuffer, pSourceBuffer, width * 4);
        return;
    }

    uint8_t* pWriteBuffer = pDestinationBuffer;
    const uint8_t* pReadBuffer = pSourceBuffer;

    for (int pixel = 0; pixel < width; ++pixel)
    {
        if (sourcePixelFormat == PixelFormat_RGB)
        {
            pWriteBuffer[0] = pReadBuffer[0];
            pWriteBuffer[1] = pReadBuffer[1];
            pWriteBuffer[2] = pReadBuffer[2];
            pWriteBuffer[3] = 0xFF;

            pWriteBuffer += 4;
            pReadBuffer += 3;
        }
        else
        {
            NN_ABORT();
        }
    }
}
#if NN_GFX_IS_TARGET_GL || NN_GFX_IS_TARGET_VK
void SetTextureConvtentForGlVk(
    const void* pPixelBuffer, size_t pixelBufferSizeInBytes,
    int width, int height, ptrdiff_t pitch, PixelFormat pixelFormat,
    nns::gfx::GraphicsFramework* pGfw, nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType,
    ptrdiff_t memoryPoolBufferOffset, size_t memoryPoolBufferSize)
{
    NN_UNUSED(pixelBufferSizeInBytes);
    NN_UNUSED(memoryPoolBufferSize);

    const uint8_t* pSourceCopyBuffer = reinterpret_cast<const uint8_t*>(pPixelBuffer);
    uint8_t* pMemoryPoolBuffer = pGfw->GetMemoryPool(memoryPoolType)->Map<uint8_t>() + memoryPoolBufferOffset;
    int destinationPixelSizeInBytes = GetDestinationPixelSizeInBytes(pixelFormat);
    ptrdiff_t destinationLineDataSizeInBytes = destinationPixelSizeInBytes * width;
    for (int line = 0; line < height; ++line)
    {
        NN_ASSERT(static_cast<size_t>((line + 1) * pitch) <= pixelBufferSizeInBytes);
        NN_ASSERT(static_cast<size_t>((line + 1) * destinationLineDataSizeInBytes) <= memoryPoolBufferSize);

        ConvertPixelLine(
            pMemoryPoolBuffer + (line * destinationLineDataSizeInBytes),
            pSourceCopyBuffer + (line * pitch),
            width,
            pixelFormat);
    }
}
#endif


#if NN_GFX_IS_TARGET_NVN
void SetTextureConvtentForNvn(
    NVNtexture* pNvnTexture,
    nns::gfx::GraphicsFramework* pGfw,
    const void* pPixelBuffer, size_t pixelBufferSizeInBytes,
    int width, int height, ptrdiff_t pitch, PixelFormat pixelFormat)
{
    NVNcopyRegion region;
    region.xoffset = 0;
    region.yoffset = 0;
    region.zoffset = 0;
    region.width = width;
    region.height = height;
    region.depth = 1;

    if (MustConvertPixelFormat(pixelFormat))
    {
        int destinationPixelSize = GetDestinationPixelSizeInBytes(pixelFormat);
        ptrdiff_t destinationConvertBufferPitch = destinationPixelSize * width;
        size_t destinationConvertBufferSizeInBytes = height * destinationConvertBufferPitch;
        uint8_t* pDestinationConvertBuffer = reinterpret_cast<uint8_t*>(pGfw->AllocateMemory(destinationConvertBufferSizeInBytes, 1));
        const uint8_t* pSourceConvertBuffer = reinterpret_cast<const uint8_t*>(pPixelBuffer);

        for (int line = 0; line < height; ++line)
        {
            ConvertPixelLine(
                pDestinationConvertBuffer + (line * destinationConvertBufferPitch),
                pSourceConvertBuffer + (line * pitch),
                width, pixelFormat);
        }

        nvnTextureWriteTexelsStrided(
            pNvnTexture, nullptr, &region,
            pDestinationConvertBuffer, destinationConvertBufferPitch, destinationConvertBufferSizeInBytes);

        pGfw->FreeMemory(pDestinationConvertBuffer);
    }
    else
    {
        nvnTextureWriteTexelsStrided(
            pNvnTexture, nullptr, &region,
            pPixelBuffer, pitch, pixelBufferSizeInBytes);
    }

    nvnTextureFlushTexels(pNvnTexture, nullptr, &region);
}
#endif



const int g_SuiteIndexShiftCount = 3;

void PrintBenchmarkInformation(const GpuBenchmark* pBenchmark, int selectedPropertyIndex, nn::gfx::util::DebugFontTextWriter* pDebugFontWriter)
{
    const GpuBenchmarkPropertyHolder** pDestinationArray = nullptr;
    int propertyCount = pBenchmark->GetPropertyCount();
    pDestinationArray = static_cast<const GpuBenchmarkPropertyHolder**>(alloca(propertyCount * sizeof(GpuBenchmarkPropertyHolder*)));
    NN_ASSERT(pDestinationArray != nullptr);
    pBenchmark->FillPropertyList(pDestinationArray, propertyCount);

    const size_t bufferSize = 256;
    char sBuffer[bufferSize];
    int bufferIndex = 0;

    snprintf(sBuffer, bufferSize, "Benchmark %s:\n", pBenchmark->GetName());
    pDebugFontWriter->Print(sBuffer);

    for (int i = 0; i < static_cast<int>(propertyCount); ++i)
    {
        const GpuBenchmarkPropertyHolder* pPropertyHolder = pDestinationArray[i];

        const char* propertyName = pPropertyHolder->GetName();

        PropertyType type = pPropertyHolder->GetType();
        NN_ASSERT(type < PropertyType_Max);
        const char* typeName = GetPropertyTypeName(type);

        bufferIndex = 0;
        if (selectedPropertyIndex >= 0)
            bufferIndex += snprintf(sBuffer + bufferIndex, bufferSize - bufferIndex, "[%c] ", (i == selectedPropertyIndex) ? 'X' : ' ');
        bufferIndex += snprintf(sBuffer + bufferIndex, bufferSize - bufferIndex, "%s (%s): ", propertyName, typeName);

        switch (type)
        {
        case PropertyType_IntegerRange:
            {
                bufferIndex += snprintf(sBuffer + bufferIndex, bufferSize - bufferIndex, "%d", pPropertyHolder->Get());
            }
            break;

        case PropertyType_Enumeration:
            {
                int value = pPropertyHolder->Get();
                const char* elementName = pPropertyHolder->ToEnum()->GetElementNameAt(value);
                bufferIndex += snprintf(sBuffer + bufferIndex, bufferSize - bufferIndex, "%s", elementName);
            }
            break;

        case PropertyType_Invalid:
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        NN_ASSERT(bufferIndex < (bufferSize - 1));
        sBuffer[bufferIndex] = '\n';
        sBuffer[bufferIndex + 1] = '\0';

        pDebugFontWriter->Print(sBuffer);
    }
}

void* PrimitiveRendererAllocateFunction(size_t size, size_t alignment, void* pUserData)
{
    NN_ASSERT_NOT_NULL(pUserData);
    nns::gfx::GraphicsFramework* pGfw = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
    return pGfw->AllocateMemory(size, alignment);
}

void PrimitiveRendererFreeFunction(void* ptr, void* pUserData)
{
    NN_ASSERT_NOT_NULL(pUserData);
    nns::gfx::GraphicsFramework* pGfw = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
    pGfw->FreeMemory(ptr);
}

}

LibraryState::LibraryState()
: m_Initialized(false)
, m_FrameStarted(false)
, m_Gfw()
, m_TestSuiteArray()
, m_DebugFontWriterData()
, m_pPrimitiveRenderer(nullptr)
, m_RuntimeGfxObjects()
, m_FrameRateTracker()
, m_ResultHistoryIndex(0)
, m_HashComparisonFailed(false)
#if defined(NN_SDK_BUILD_DEBUG)
, m_OwnerThreadId()
#endif
{
    NN_STATIC_ASSERT((1 << g_SuiteIndexShiftCount) >= LibraryState::m_ActiveTestSuiteMaxCount);
}

LibraryState::~LibraryState()
{
}



void LibraryState::Initialize()
{
    NN_ASSERT(!m_Initialized);

    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
#if defined(NN_SDK_BUILD_DEBUG)
    m_OwnerThreadId = nn::os::GetThreadId(nn::os::GetCurrentThread());
#endif


    size_t memoryPoolRequiredSize[nns::gfx::GraphicsFramework::MemoryPoolType_End] = { 0 };

    const size_t appCommandBufferPoolRequiredMemorySize     = 16777216;
    const size_t appRenderTargetPoolRequiredMemorySize      = (FrameBufferWidth * FrameBufferHeight * FrameBufferBpp * FrameBufferCount) * 2;
    const size_t appConstantBufferPoolRequiredMemorySize    = 128 * 1024;
    const size_t appDataPoolRequiredMemorySize              = 64 * 1024 * 1024;

    memoryPoolRequiredSize[nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer]   += appCommandBufferPoolRequiredMemorySize;
    memoryPoolRequiredSize[nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget]    += appRenderTargetPoolRequiredMemorySize;
    memoryPoolRequiredSize[nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer]  += appConstantBufferPoolRequiredMemorySize;
    memoryPoolRequiredSize[nns::gfx::GraphicsFramework::MemoryPoolType_Data]            += appDataPoolRequiredMemorySize;

    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);

    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(FrameBufferWidth);
    fwInfo.SetDisplayHeight(FrameBufferHeight);
    fwInfo.SetBufferCount(FrameBufferCount);
    fwInfo.SetSwapChainBufferCount(FrameBufferCount);
    for (int memoryPoolIndex = 0; memoryPoolIndex < nns::gfx::GraphicsFramework::MemoryPoolType_End; ++memoryPoolIndex)
    {
        nns::gfx::GraphicsFramework::MemoryPoolType poolType = static_cast<nns::gfx::GraphicsFramework::MemoryPoolType>(memoryPoolIndex);
        fwInfo.SetMemoryPoolSize(poolType, memoryPoolRequiredSize[memoryPoolIndex]);
    }
    fwInfo.SetRootCommandBufferCommandMemorySize(8 * 1024 * 1024);

    fwInfo.SetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType_BufferView, 16);
    fwInfo.SetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType_TextureView, 1024);
    fwInfo.SetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType_Sampler, 1024);

#if defined(NN_SDK_BUILD_DEBUG)
    fwInfo.SetDebugMode(nn::gfx::DebugMode_Full);
#else
    fwInfo.SetDebugMode(nn::gfx::DebugMode_Disable);
#endif

    m_Gfw.Initialize(fwInfo);

    nnt::gfx::util::ResourceAllocator::InfoType ResourceAllocatorInfo;
    ResourceAllocatorInfo.SetDefault();
    ResourceAllocatorInfo.SetMemoryAllocator(
        m_Gfw.GetAllocateFunction(),
        m_Gfw.GetFreeFunction(),
        m_Gfw.GetAllocateFunctionUserData());
    m_ResourceAllocator.Initialize(m_Gfw.GetDevice(), ResourceAllocatorInfo);

    nn::gfx::Queue::InfoType gpuBenchmarkQueueInfo;
    gpuBenchmarkQueueInfo.SetDefault();
    gpuBenchmarkQueueInfo.SetCapability(nn::gfx::QueueCapability_Graphics
        | nn::gfx::QueueCapability_Compute | nn::gfx::QueueCapability_Copy);
    m_GpuBenchmarkQueue.Initialize(m_Gfw.GetDevice(), gpuBenchmarkQueueInfo);

    nn::gfx::Semaphore::InfoType gpuBenchmarkQueueSemaphoreInfo;
    gpuBenchmarkQueueSemaphoreInfo.SetDefault();
    m_GpuBenchmarkQueueSemaphore.Initialize(m_Gfw.GetDevice(), gpuBenchmarkQueueSemaphoreInfo);

    InitializeTestGfxObjects(FrameBufferWidth, FrameBufferHeight, FrameBufferCount);

    m_FrameRateTracker.Initialize();

    m_TestSuiteArray.Initialize();

    m_TextureResourceDataArray.Initialize();
    m_SamplerDescriptorSlotIndex = m_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);
    m_Gfw.SetSamplerToDescriptorPool(
        m_SamplerDescriptorSlotIndex,
        m_Gfw.GetSampler(nns::gfx::GraphicsFramework::SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Clamp));
    m_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler)->GetDescriptorSlot(&m_SamplerDescriptorSlot, m_SamplerDescriptorSlotIndex);

    nn::gfx::BlendTargetStateInfo textureBlendTargetInfo;
    {
        textureBlendTargetInfo.SetDefault();
        textureBlendTargetInfo.SetBlendEnabled(true);
        textureBlendTargetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
        textureBlendTargetInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
        textureBlendTargetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
        textureBlendTargetInfo.SetSourceAlphaBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
        textureBlendTargetInfo.SetDestinationAlphaBlendFactor(nn::gfx::BlendFactor_One);
        textureBlendTargetInfo.SetAlphaBlendFunction(nn::gfx::BlendFunction_Add);
    }

    nn::gfx::BlendState::InfoType textureBlendStateInfo;
    textureBlendStateInfo.SetDefault();
    textureBlendStateInfo.SetBlendTargetStateInfoArray(&textureBlendTargetInfo, 1);

    nnt::gfx::util::InitializeBlendState(
        &m_Draw2dRectTextureBlendState, textureBlendStateInfo,
        &m_ResourceAllocator, m_Gfw.GetDevice());


    m_HashComparisonFailed = false;

    m_ResultHistoryIndex = 0;
    memset(m_ResultHistory, 0, sizeof(m_ResultHistory));
    ResetResultHistory();

    m_Initialized = true;
}

void LibraryState::Finalize()
{
    nnt::gfx::util::FinalizeBlendState(
        &m_Draw2dRectTextureBlendState, &m_ResourceAllocator, m_Gfw.GetDevice());

    m_TextureResourceDataArray.Finalize();

    m_TestSuiteArray.Finalize();

    m_FrameRateTracker.Finalize();

    FinalizeTestGfxObjects();

    m_GpuBenchmarkQueue.Finalize(m_Gfw.GetDevice());
    m_GpuBenchmarkQueueSemaphore.Finalize(m_Gfw.GetDevice());
    m_ResourceAllocator.Finalize(m_Gfw.GetDevice());

    m_Gfw.Finalize();
    m_Initialized = false;
}


void LibraryState::GetTestPlatformId(char* buffer, int bufferSize)
{
    nnt::gfx::util::GetPlatformId(buffer, bufferSize);
}

nn::Result LibraryState::LoadTestSuiteFromFile(TestSuiteHandle* pOutHandle, const char* szFilePath)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);

    json::Document* pDocument = nullptr;

    json::Create(&pDocument, nullptr);

    nn::Result result = json::LoadFromFile(pDocument, szFilePath);
    if (result.IsFailure())
    {
        json::Finalize(pDocument);
        return result;
    }

    TestSuiteHandle handle;
    if (!AllocateTestSuite(&handle))
    {
        json::Finalize(pDocument);
        return nn::ResultSuccess();
    }

    TestSuite* pTestSuite = m_TestSuiteArray.Get(handle);

    pTestSuite->pJsonDocument = pDocument;
    pTestSuite->pTestCaseIterator = nullptr;

    memset(pTestSuite->filterPattern, 0, sizeof(pTestSuite->filterPattern));
    pTestSuite->startIndex = 0;
    pTestSuite->maxCount = MaxCountAllTests;
    pTestSuite->iterationIndex = 0;

    pTestSuite->pGpuBenchmark = nullptr;

    pTestSuite->previousCpuDuration = nn::TimeSpan(0);
    pTestSuite->previousGpuDuration = nn::TimeSpan(0);

    pTestSuite->cpuResult.Clear();
    pTestSuite->gpuResult.Clear();

    pTestSuite->testCount = 0;



    *pOutHandle = handle;
    return nn::ResultSuccess();
}

void LibraryState::UnloadTestSuite(TestSuiteHandle handle)
{
    TestSuite* pTestSuite = m_TestSuiteArray.Get(handle);

    NN_ASSERT(pTestSuite->pJsonDocument != nullptr);
    NN_ASSERT(pTestSuite->pTestCaseIterator == nullptr);
    NN_ASSERT(pTestSuite->pGpuBenchmark == nullptr);

    json::Finalize(pTestSuite->pJsonDocument);
    pTestSuite->pJsonDocument = nullptr;

    FreeTestSuite(handle);
}

int LibraryState::InitializeTestSuite(TestSuiteHandle handle, const char* filterPattern, int startIndex, int maxCount)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    NN_ASSERT((filterPattern == nullptr) || (strlen(filterPattern) < (TestSuite::FilterPatternMaxLength - 1)));

    TestSuite* pTestSuite = m_TestSuiteArray.Get(handle);
    NN_ASSERT(pTestSuite->pJsonDocument != nullptr);
    NN_ASSERT(pTestSuite->pTestCaseIterator == nullptr);
    NN_ASSERT(pTestSuite->pGpuBenchmark == nullptr);

    if (filterPattern != nullptr)
    {
        strncpy(pTestSuite->filterPattern, filterPattern, TestSuite::FilterPatternMaxLength - 1);
        pTestSuite->filterPattern[TestSuite::FilterPatternMaxLength - 1] = '\0';
    }
    else
    {
        memset(pTestSuite->filterPattern, 0, TestSuite::FilterPatternMaxLength);
    }

    pTestSuite->startIndex      = startIndex;
    pTestSuite->maxCount        = maxCount;
    pTestSuite->iterationIndex  = 0;

    int testCount = 0;
    json::TestCaseIterator* pTestCaseIterator = NULL;

    json::InitializeTestIterator(&pTestCaseIterator, pTestSuite->pJsonDocument);
    int firstIndex = 0;
    while (firstIndex < startIndex)
    {
        if (!json::MoveToNextTestCase(filterPattern, pTestCaseIterator))
        {
            break;
        }

        firstIndex++;
    }

    json::CopyTestIterator(&pTestSuite->pTestCaseIterator, pTestCaseIterator);

    if (firstIndex == startIndex)
    {
        while ((maxCount == MaxCountAllTests) || (testCount < maxCount))
        {
            if (!json::MoveToNextTestCase(filterPattern, pTestCaseIterator))
            {
                break;
            }

            NN_LOG("Found test case %s\n", nnt::gfx::util::json::GetTestCaseId(pTestCaseIterator));
            testCount++;
        }
    }

    json::FinalizeTestIterator(pTestCaseIterator);

    return testCount;
}

void LibraryState::FinalizeTestSuite(TestSuiteHandle handle)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);

    TestSuite* pTestSuite = m_TestSuiteArray.Get(handle);

    if (pTestSuite->pTestCaseIterator != nullptr)
    {
        json::FinalizeTestIterator(pTestSuite->pTestCaseIterator);
        pTestSuite->pTestCaseIterator = nullptr;
    }

    if (pTestSuite->pGpuBenchmark != nullptr)
    {
        LogTestResults(pTestSuite);
        FinalizeTestCaseData(pTestSuite);
    }
}


bool LibraryState::HasMoreTests(TestSuiteHandle handle) const
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);

    const TestSuite* pTestSuite = m_TestSuiteArray.Get(handle);
    if (pTestSuite->pTestCaseIterator == nullptr)
    {
        return false;
    }

    return json::HasMoreTestCase(pTestSuite->filterPattern, pTestSuite->pTestCaseIterator);
}

bool LibraryState::BeginNextTest(TestSuiteHandle handle)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);

    TestSuite* pTestSuite = m_TestSuiteArray.Get(handle);

    if (pTestSuite->pGpuBenchmark != nullptr)
    {
        LogTestResults(pTestSuite);
        FinalizeTestCaseData(pTestSuite);
    }

    bool hasMoreTestCases = false;

    if ((pTestSuite->maxCount == MaxCountAllTests) || (pTestSuite->iterationIndex < pTestSuite->maxCount))
    {
        pTestSuite->iterationIndex++;

        hasMoreTestCases = json::MoveToNextTestCase(pTestSuite->filterPattern, pTestSuite->pTestCaseIterator);
        if (hasMoreTestCases)
        {
            InitializeTestCaseData(pTestSuite);
        }
    }

    return hasMoreTestCases;
}

const char* LibraryState::GetTestName(TestSuiteHandle handle) const
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);

    const TestSuite* pTestSuite = m_TestSuiteArray.Get(handle);
    NN_ASSERT(pTestSuite->pTestCaseIterator != nullptr);
    const char* testName = json::GetTestCaseName(pTestSuite->pTestCaseIterator);
    return testName;
}

const char* LibraryState::GetTestCaseId(TestSuiteHandle handle) const
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);

    const TestSuite* pTestSuite = m_TestSuiteArray.Get(handle);
    NN_ASSERT(pTestSuite->pTestCaseIterator != nullptr);
    const char* testCaseId = json::GetTestCaseId(pTestSuite->pTestCaseIterator);
    return testCaseId;
}

bool LibraryState::RunTest(int* pOutTestResultFlags, HashComparisonMode hashComparisonMode, TestSuiteHandle handle)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);

    TestSuite* pTestSuite = m_TestSuiteArray.Get(handle);
    NN_ASSERT(pTestSuite->pTestCaseIterator != nullptr);

    nnt::gfx::util::RunGpuBenchmarkCommandList(
        &pTestSuite->previousCpuDuration, &pTestSuite->previousGpuDuration,
        &m_GpuBenchmarkQueue, &m_RuntimeGfxObjects);

    return ValidateAndUpdateTestResult(pOutTestResultFlags, hashComparisonMode, pTestSuite);
}

nn::Result LibraryState::CreateTextureFromResourceFile(TextureHandle* pOutHandle, const char* szFilePath)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);

    nn::Result result;
    nn::fs::FileHandle hFile;
    int64_t fileSize = 0;
    size_t bytesRead = 0;
    nn::util::BinaryFileHeader fileHeader;

    result = nn::fs::OpenFile(&hFile, szFilePath, nn::fs::OpenMode_Read);
    if (result.IsFailure())
        return result;

    result = nn::fs::GetFileSize(&fileSize, hFile);
    if (result.IsFailure())
    {
        nn::fs::CloseFile(hFile);
        return result;
    }

    result = nn::fs::ReadFile(&bytesRead, hFile, 0, &fileHeader, sizeof(fileHeader), nn::fs::ReadOption::MakeValue(0));
    if (bytesRead != sizeof(fileHeader))
    {
        nn::fs::CloseFile(hFile);
        return result;
    }

    size_t alignment = fileHeader.GetAlignment();

    void* pReadBuffer = m_Gfw.AllocateMemory(static_cast<size_t>(fileSize), alignment);
    result = nn::fs::ReadFile(&bytesRead, hFile, 0, pReadBuffer, static_cast<size_t>(fileSize));
    nn::fs::CloseFile(hFile);

    if (bytesRead != static_cast<size_t>(fileSize))
    {
        return result;
    }

    FreeListElementHandle handle = m_TextureResourceDataArray.Allocate();
    TextureResourceData* pTextureResourceData = m_TextureResourceDataArray.Get(handle);

    pTextureResourceData->pBuffer = pReadBuffer;

    nn::gfx::ResTextureFile* pResTextureFile = nn::gfx::ResTextureFile::ResCast(pTextureResourceData->pBuffer);
    pResTextureFile->Initialize(m_Gfw.GetDevice());

    int textureCount = pResTextureFile->GetTextureDic()->GetCount();
    NN_ASSERT(textureCount >= 1);

    int baseSlotIndex = m_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, textureCount);
    pTextureResourceData->textureBaseSlotIndex = baseSlotIndex;

    nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture(0);
    pResTexture->Initialize(m_Gfw.GetDevice());

    nn::gfx::TextureView* pTextureView = pResTexture->GetTextureView();
    m_Gfw.SetTextureViewToDescriptorPool(baseSlotIndex, pTextureView);

    m_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->GetDescriptorSlot(
        &pTextureResourceData->textureDescriptorSlot, baseSlotIndex);

    pResTexture->SetUserDescriptorSlot(pTextureResourceData->textureDescriptorSlot);

    pTextureResourceData->pResTextureFile = pResTextureFile;
    pTextureResourceData->pBmpData = nullptr;

    *pOutHandle = handle;

    return result;
}

nn::Result LibraryState::CreateTextureFromBmpBuffer(TextureHandle* pOutHandle, const void* bmpBuffer, size_t bmpBufferSizeInBytes)
{
    BmpInputStream is;
    is.bufferSize = bmpBufferSizeInBytes;
    is.pBuffer = reinterpret_cast<const uint8_t*>(bmpBuffer);
    is.readOffset = 0;

    BmpFileInfo bmpFileInfo;
    if (!ReadBmpFileInfoFromStream(&is, &bmpFileInfo))
    {
        return nn::fs::ResultHostFileCorrupted();
    }

    NN_ASSERT((bmpFileInfo.bpp % CHAR_BIT) == 0);
    size_t bmpDecodeBufferSizeInBytes = (bmpFileInfo.width * bmpFileInfo.height * bmpFileInfo.bpp) / CHAR_BIT;
    void* pBmpDecodeBuffer = m_Gfw.AllocateMemory(bmpDecodeBufferSizeInBytes, sizeof(uintptr_t));

    if (!ReadBmpFromStream(&is, &bmpFileInfo, pBmpDecodeBuffer, bmpDecodeBufferSizeInBytes))
    {
        m_Gfw.FreeMemory(pBmpDecodeBuffer);
        return nn::fs::ResultHostFileCorrupted();
    }

    nn::Result result = CreateTextureFromPixelBuffer(
        pOutHandle,
        pBmpDecodeBuffer, bmpDecodeBufferSizeInBytes,
        bmpFileInfo.width, bmpFileInfo.height, bmpFileInfo.stride,
        bmpFileInfo.bpp == 24 ? PixelFormat_RGB : PixelFormat_RGBA);


    m_Gfw.FreeMemory(pBmpDecodeBuffer);
    pBmpDecodeBuffer = nullptr;

    return result;
}

nn::Result LibraryState::CreateTextureFromBmpFile(TextureHandle* pOutHandle, const char* szFilePath)
{
    nn::Result result;
    nn::fs::FileHandle hFile;
    int64_t fileSize = 0;
    size_t bytesRead = 0;

    result = nn::fs::OpenFile(&hFile, szFilePath, nn::fs::OpenMode_Read);
    if (result.IsFailure())
        return result;

    result = nn::fs::GetFileSize(&fileSize, hFile);
    if (result.IsFailure())
    {
        nn::fs::CloseFile(hFile);
        return result;
    }

    size_t readBufferSize = static_cast<size_t>(fileSize);
    void* pReadBuffer = m_Gfw.AllocateMemory(readBufferSize, sizeof(uintptr_t));

    result = nn::fs::ReadFile(
        &bytesRead, hFile, 0,
        pReadBuffer, readBufferSize, nn::fs::ReadOption::MakeValue(0));
    if (bytesRead != readBufferSize)
    {
        m_Gfw.FreeMemory(pReadBuffer);
        nn::fs::CloseFile(hFile);
        return result;
    }

    result = CreateTextureFromBmpBuffer(pOutHandle, pReadBuffer, bytesRead);

    m_Gfw.FreeMemory(pReadBuffer);
    nn::fs::CloseFile(hFile);

    return result;
}

nn::Result LibraryState::CreateTextureFromPixelBuffer(
    TextureHandle* pOutHandle, const void* pPixelBuffer, size_t pixelBufferSizeInBytes,
    int width, int height, ptrdiff_t pitch, PixelFormat pixelFormat)
{
    NN_UNUSED(pixelBufferSizeInBytes);
    NN_ASSERT(pixelBufferSizeInBytes >= static_cast<size_t>(height * pitch));

    void* pBuffer = m_Gfw.AllocateMemory(sizeof(BmpData), NN_ALIGNOF(BmpData));
    BmpData* pBmpData = new (pBuffer) BmpData();

    nn::gfx::ImageFormat imageFormat = GetImageFormat(pixelFormat);

    pBmpData->textureInfo.SetDefault();
    pBmpData->textureInfo.SetWidth(width);
    pBmpData->textureInfo.SetHeight(height);
    pBmpData->textureInfo.SetArrayLength(1);
    pBmpData->textureInfo.SetMipCount(1);
    pBmpData->textureInfo.SetImageFormat(imageFormat);
    pBmpData->textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    pBmpData->textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
    pBmpData->textureInfo.SetTileMode(nn::gfx::TileMode_Optimal);

    size_t textureBufferAlignment = nn::gfx::Texture::CalculateMipDataAlignment(m_Gfw.GetDevice(), pBmpData->textureInfo);
    size_t textureBufferSize = nn::gfx::Texture::CalculateMipDataSize(m_Gfw.GetDevice(), pBmpData->textureInfo);

    nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType =
        nns::gfx::GraphicsFramework::MemoryPoolType_Data;

    ptrdiff_t textureBufferMemoryPoolOffset =
        m_Gfw.AllocatePoolMemory(memoryPoolType, textureBufferSize, textureBufferAlignment);
    NN_ASSERT(textureBufferMemoryPoolOffset >= 0);

#if NN_GFX_IS_TARGET_GL || NN_GFX_IS_TARGET_VK
    SetTextureConvtentForGlVk(
        pPixelBuffer, pixelBufferSizeInBytes, width, height, pitch, pixelFormat,
        &m_Gfw, memoryPoolType, textureBufferMemoryPoolOffset, textureBufferSize);
#endif

    pBmpData->texture.Initialize(
        m_Gfw.GetDevice(), pBmpData->textureInfo,
        m_Gfw.GetMemoryPool(memoryPoolType), textureBufferMemoryPoolOffset, textureBufferSize);

#if NN_GFX_IS_TARGET_NVN
    SetTextureConvtentForNvn(
        pBmpData->texture.ToData()->pNvnTexture, &m_Gfw,
        pPixelBuffer, pixelBufferSizeInBytes, width, height, pitch, pixelFormat);
#endif

    nn::gfx::TextureView::InfoType textureViewInfo;
    textureViewInfo.SetDefault();
    textureViewInfo.SetTexturePtr(&pBmpData->texture);
    textureViewInfo.SetImageFormat(pBmpData->textureInfo.GetImageFormat());
    textureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    pBmpData->textureView.Initialize(
        m_Gfw.GetDevice(), textureViewInfo);


    FreeListElementHandle handle = m_TextureResourceDataArray.Allocate();
    TextureResourceData* pTextureResourceData = m_TextureResourceDataArray.Get(handle);

    pTextureResourceData->pBuffer = pBuffer;
    pTextureResourceData->pBmpData = pBmpData;
    pTextureResourceData->pResTextureFile = nullptr;

    int baseSlotIndex = m_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    pTextureResourceData->textureBaseSlotIndex = baseSlotIndex;

    m_Gfw.SetTextureViewToDescriptorPool(baseSlotIndex, &pBmpData->textureView);
    m_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->GetDescriptorSlot(
        &pTextureResourceData->textureDescriptorSlot, baseSlotIndex);


    *pOutHandle = handle;
    return nn::ResultSuccess();
}

void LibraryState::FreeTextureResource(TextureHandle handle)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);

    TextureResourceData* pTextureResourceData = m_TextureResourceDataArray.Get(handle);

    if (pTextureResourceData->pResTextureFile != nullptr)
    {
        int textureCount = pTextureResourceData->pResTextureFile->GetTextureDic()->GetCount();
        NN_ASSERT(textureCount >= 1);

        nn::gfx::ResTexture* pResTexture = pTextureResourceData->pResTextureFile->GetResTexture(0);
        pResTexture->Finalize(m_Gfw.GetDevice());

        pTextureResourceData->pResTextureFile->Finalize(m_Gfw.GetDevice());
        pTextureResourceData->pResTextureFile = nullptr;
    }

    if (pTextureResourceData->pBmpData != nullptr)
    {
        pTextureResourceData->pBmpData->texture.Finalize(m_Gfw.GetDevice());
        pTextureResourceData->pBmpData->textureView.Finalize(m_Gfw.GetDevice());
        pTextureResourceData->pBmpData->~BmpData();
        pTextureResourceData->pBmpData = nullptr;
    }

    if (pTextureResourceData->pBuffer != nullptr)
    {
        m_Gfw.FreeMemory(pTextureResourceData->pBuffer);
        pTextureResourceData->pBuffer = nullptr;
    }

    if (pTextureResourceData->textureBaseSlotIndex != -1)
    {
        m_Gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, pTextureResourceData->textureBaseSlotIndex);
        pTextureResourceData->textureBaseSlotIndex = -1;
    }

    m_TextureResourceDataArray.Free(handle);
}

int LibraryState::GetTextureWidth(TextureHandle handle)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);

    TextureResourceData* pTextureResourceData = m_TextureResourceDataArray.Get(handle);
    if (pTextureResourceData->pResTextureFile != nullptr)
    {
        return pTextureResourceData->pResTextureFile->GetResTexture(0)->GetTextureInfo()->GetWidth();
    }
    else if (pTextureResourceData->pBmpData != nullptr)
    {
        return pTextureResourceData->pBmpData->textureInfo.GetWidth();
    }
    else
    {
        NN_ABORT();
    }


    return -1;
}

int LibraryState::GetTextureHeight(TextureHandle handle)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);

    TextureResourceData* pTextureResourceData = m_TextureResourceDataArray.Get(handle);
    if (pTextureResourceData->pResTextureFile != nullptr)
    {
        return pTextureResourceData->pResTextureFile->GetResTexture(0)->GetTextureInfo()->GetHeight();
    }
    else if (pTextureResourceData->pBmpData != nullptr)
    {
        return pTextureResourceData->pBmpData->textureInfo.GetHeight();
    }
    else
    {
        NN_ABORT();
    }

    return -1;
}

void LibraryState::StartFrame()
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    NN_ASSERT(!m_FrameStarted);

    m_Gfw.AcquireTexture(0);
    m_Gfw.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));
    nn::gfx::SyncResult frameSyncResult = m_Gfw.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));
    NN_ASSERT(frameSyncResult == nn::gfx::SyncResult_Success);

    m_Gfw.BeginFrame(0);

    nn::gfx::CommandBuffer* pRootCommandBuffer = m_Gfw.GetRootCommandBuffer(0);
    nn::gfx::ColorTargetView* pScreenTarget = m_Gfw.GetColorTargetView();
    pRootCommandBuffer->ClearColor(pScreenTarget, 0.0f, 0.0f, 0.0f, 0.0f, nullptr);
    pRootCommandBuffer->SetRenderTargets(1, &pScreenTarget, nullptr);
    pRootCommandBuffer->SetViewportScissorState(m_Gfw.GetViewportScissorState());

    m_pPrimitiveRenderer->Update(0);
    m_pPrimitiveRenderer->SetDefaultParameters();

    m_FrameStarted = true;
}

void LibraryState::EndFrame()
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    NN_ASSERT(m_FrameStarted);

    m_DebugFontWriterData.debugFontWriter.Draw(GetCommandBuffer());

    m_Gfw.EndFrame(0);
    m_Gfw.ExecuteCommand(0);
    m_Gfw.QueuePresentTexture(1);

    nn::gfx::SyncResult frameSyncResult = m_Gfw.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));
    NN_ASSERT(frameSyncResult == nn::gfx::SyncResult_Success);

    m_FrameRateTracker.UpdateFps();

    m_FrameStarted = false;
}

void LibraryState::DrawTestResults(TestSuiteHandle handle, int positionX, int positionY, float scale, const Color& color)
{
    m_DebugFontWriterData.debugFontWriter.SetFixedWidthEnabled(true);
    m_DebugFontWriterData.debugFontWriter.SetFixedWidth(10 * scale);
    m_DebugFontWriterData.debugFontWriter.SetScale(scale, scale);

    float textPositionX = static_cast<float>(positionX);
    float textPositionY = static_cast<float>(positionY);
    m_DebugFontWriterData.debugFontWriter.SetCursor(textPositionX, textPositionY);

    m_DebugFontWriterData.debugFontWriter.SetTextColor(color);

    TestSuite* pTestSuite = m_TestSuiteArray.Get(handle);
    NN_ASSERT(pTestSuite->pTestCaseIterator != nullptr);

    const char* testId = json::GetTestCaseId(pTestSuite->pTestCaseIterator);
    int warmUpCount = json::GetTestCaseWarmUpCount(pTestSuite->pTestCaseIterator);
    int testRepeatCount = json::GetTestCaseRepeatCount(pTestSuite->pTestCaseIterator);

    m_DebugFontWriterData.debugFontWriter.Print(
        "Test results: %s (warmUp:%d repeat:%d)\n",
        testId, warmUpCount, testRepeatCount);

    PrintBenchmarkInformation(pTestSuite->pGpuBenchmark, -1, &m_DebugFontWriterData.debugFontWriter);

    int runCount = json::GetTestCaseRepeatCount(pTestSuite->pTestCaseIterator);

    pTestSuite->pGpuBenchmark->PrintResults(
        pTestSuite->previousCpuDuration,
        pTestSuite->previousGpuDuration,
        runCount,
        &m_DebugFontWriterData.debugFontWriter);
}

void LibraryState::DrawFont(const char* text, int positionX, int positionY, float scale, const Color& color)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    NN_ASSERT(m_FrameStarted);

    m_DebugFontWriterData.debugFontWriter.SetFixedWidthEnabled(false);
    m_DebugFontWriterData.debugFontWriter.SetScale(scale, scale);

    float textPositionX = static_cast<float>(positionX);
    float textPositionY = static_cast<float>(positionY);
    m_DebugFontWriterData.debugFontWriter.SetCursor(textPositionX, textPositionY);

    m_DebugFontWriterData.debugFontWriter.SetTextColor(color);

    m_DebugFontWriterData.debugFontWriter.Print(text);
}

void LibraryState::Draw2DFrame(int positionX, int positionY, int width, int height, const Color& color)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    NN_ASSERT(m_FrameStarted);

    m_pPrimitiveRenderer->SetColor(color);
    m_pPrimitiveRenderer->SetDepthStencilState(GetCommandBuffer(), nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);
    m_pPrimitiveRenderer->SetBlendState(GetCommandBuffer(), nns::gfx::PrimitiveRenderer::BlendType::BlendType_Opacity);
    GetCommandBuffer()->SetRasterizerState(
        m_pPrimitiveRenderer->GetRasterizerState(
            nn::gfx::PrimitiveTopologyType_Triangle,
            nn::gfx::CullMode_None, nn::gfx::FillMode_Solid));

    m_pPrimitiveRenderer->Draw2DFrame(
        GetCommandBuffer(),
        static_cast<float>(positionX), static_cast<float>(positionY),
        static_cast<float>(width), static_cast<float>(height));
}

void LibraryState::Draw2DRect(int positionX, int positionY, int width, int height, const Color& color)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    NN_ASSERT(m_FrameStarted);

    m_pPrimitiveRenderer->SetColor(color);
    m_pPrimitiveRenderer->SetDepthStencilState(GetCommandBuffer(), nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);
    m_pPrimitiveRenderer->SetBlendState(GetCommandBuffer(), nns::gfx::PrimitiveRenderer::BlendType::BlendType_Opacity);
    GetCommandBuffer()->SetRasterizerState(
        m_pPrimitiveRenderer->GetRasterizerState(
            nn::gfx::PrimitiveTopologyType_Triangle,
            nn::gfx::CullMode_None, nn::gfx::FillMode_Solid));

    m_pPrimitiveRenderer->Draw2DRect(
        GetCommandBuffer(),
        static_cast<float>(positionX), static_cast<float>(positionY),
        static_cast<float>(width), static_cast<float>(height));
}

void LibraryState::Draw2DRectTexture(int positionX, int positionY, int width, int height, TextureHandle handle)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    NN_ASSERT(m_FrameStarted);

    TextureResourceData* pTextureResourceData = m_TextureResourceDataArray.Get(handle);

    m_pPrimitiveRenderer->SetDepthStencilState(GetCommandBuffer(), nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);
    GetCommandBuffer()->SetBlendState(&m_Draw2dRectTextureBlendState);
    GetCommandBuffer()->SetRasterizerState(
        m_pPrimitiveRenderer->GetRasterizerState(
            nn::gfx::PrimitiveTopologyType_Triangle,
            nn::gfx::CullMode_None, nn::gfx::FillMode_Solid));

    m_pPrimitiveRenderer->Draw2DRect(
        GetCommandBuffer(),
        static_cast<float>(positionX), static_cast<float>(positionY),
        static_cast<float>(width), static_cast<float>(height),
        pTextureResourceData->textureDescriptorSlot,
        m_SamplerDescriptorSlot);
}

void LibraryState::DrawTestDebugInformation(TestSuiteHandle handle)
{
    const TestSuite* pTestSuite = m_TestSuiteArray.Get(handle);
    if (pTestSuite->pGpuBenchmark != nullptr)
    {
        nn::gfx::CommandBuffer* pCommandBuffer = GetCommandBuffer();
        RenderGpuBenchmarkDebug(pTestSuite->pGpuBenchmark, &m_RuntimeGfxObjects, pCommandBuffer);

        pCommandBuffer->SetDescriptorPool(m_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_BufferView));
        pCommandBuffer->SetDescriptorPool(m_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView));
        pCommandBuffer->SetDescriptorPool(m_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler));
    }
}

nn::gfx::Device* LibraryState::GetGfxDevice()
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    return m_Gfw.GetDevice();
}

nn::gfx::CommandBuffer* LibraryState::GetGfxCommandBuffer()
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    return m_Gfw.GetRootCommandBuffer(0);
}

nn::gfx::util::DebugFontTextWriter* LibraryState::GetDebugFontTextWriter()
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    NN_ASSERT(m_FrameStarted);
    return &m_DebugFontWriterData.debugFontWriter;
}

nns::gfx::PrimitiveRenderer::Renderer* LibraryState::GetPrimitiveRenderer()
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    NN_ASSERT(m_FrameStarted);
    return m_pPrimitiveRenderer;
}

void LibraryState::GetFps(float* pOutMinFps, float* pOutMaxFps, float* pOutAverageFps)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    NN_ASSERT(m_FrameStarted);

    m_FrameRateTracker.GetFps(pOutMinFps, pOutMaxFps, pOutAverageFps);
}


bool LibraryState::AllocateTestSuite(TestSuiteHandle* pOutTestSuiteHandle)
{
    ASSERT_SAME_THREAD();
    NN_ASSERT(m_Initialized);
    NN_ASSERT(pOutTestSuiteHandle != nullptr);

    FreeListElementHandle testSuiteHandle = m_TestSuiteArray.Allocate();
    if (testSuiteHandle == FreeListElementInvalidHandle)
        return false;

    TestSuite* pTestSuite = m_TestSuiteArray.Get(testSuiteHandle);
    pTestSuite->active = true;

    *pOutTestSuiteHandle = testSuiteHandle;
    return true;
}

void LibraryState::FreeTestSuite(TestSuiteHandle handle)
{
    TestSuite* pTestSuite = m_TestSuiteArray.Get(handle);

    NN_ASSERT(pTestSuite->pJsonDocument == nullptr);

    pTestSuite->active = false;
    m_TestSuiteArray.Free(handle);
}

void LibraryState::InitializeDebugFontTextWriter(int bufferCount)
{
    const size_t TextWriterCharCountMax = 32 * 1024;

    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(TextWriterCharCountMax);
    info.SetBufferCount(bufferCount);
    info.SetUserMemoryPoolEnabled(false);

    m_DebugFontWriterData.debugFontWriterHeapSize =
        nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(m_Gfw.GetDevice(), info);

    m_DebugFontWriterData.pDebugFontWriterHeap = m_Gfw.AllocateMemory(m_DebugFontWriterData.debugFontWriterHeapSize, sizeof(int));

    m_DebugFontWriterData.debugFontWriter.Initialize(
        m_Gfw.GetDevice(), info,
        m_DebugFontWriterData.pDebugFontWriterHeap, m_DebugFontWriterData.debugFontWriterHeapSize,
        nullptr, 0, 0);

    m_DebugFontWriterData.debugFontWriterTextureSlotIndex = m_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    m_DebugFontWriterData.debugFontWriterSamplerSlotIndex = m_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);

    m_DebugFontWriterData.debugFontWriter.SetDisplayWidth(m_Gfw.GetDisplayWidth());
    m_DebugFontWriterData.debugFontWriter.SetDisplayHeight(m_Gfw.GetDisplayHeight());

    m_DebugFontWriterData.debugFontWriter.SetTextureDescriptor(m_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView), m_DebugFontWriterData.debugFontWriterTextureSlotIndex);
    m_DebugFontWriterData.debugFontWriter.SetSamplerDescriptor(m_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler), m_DebugFontWriterData.debugFontWriterSamplerSlotIndex);
}


void LibraryState::FinalizeDebugFontTextWriter()
{
    m_DebugFontWriterData.debugFontWriter.Finalize();
    m_Gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, m_DebugFontWriterData.debugFontWriterTextureSlotIndex);
    m_Gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, m_DebugFontWriterData.debugFontWriterSamplerSlotIndex);
    m_Gfw.FreeMemory(m_DebugFontWriterData.pDebugFontWriterHeap);
}

void LibraryState::InitializePrimitiveRenderer(int screenWidth, int screenHeight, int bufferCount)
{
    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(PrimitiveRendererAllocateFunction, &m_Gfw);
    info.SetAdditionalBufferSize(1024 * 4);
    info.SetMultiBufferQuantity(bufferCount);

    m_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(m_Gfw.GetDevice(), info);
    m_pPrimitiveRenderer->SetScreenWidth(screenWidth);
    m_pPrimitiveRenderer->SetScreenHeight(screenHeight);
}

void LibraryState::FinalizePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::DestroyRenderer(m_pPrimitiveRenderer, m_Gfw.GetDevice(), PrimitiveRendererFreeFunction, &m_Gfw);
}

void LibraryState::InitializeTestGfxObjects(int screenWidth, int screenHeight, int bufferCount)
{
    InitializePrimitiveRenderer(screenWidth, screenHeight, bufferCount);
    InitializeDebugFontTextWriter(bufferCount);

    InitializeRuntimeGfxObjects(
        &m_RuntimeGfxObjects, m_Gfw.GetDevice(), &m_ResourceAllocator);
}

void LibraryState::FinalizeTestGfxObjects()
{
    FinalizeRuntimeGfxObjects(
        &m_RuntimeGfxObjects, m_Gfw.GetDevice(), &m_ResourceAllocator);

    FinalizeDebugFontTextWriter();
    FinalizePrimitiveRenderer();
}

GpuBenchmark* LibraryState::InitializeBenchmark(const char* testName, json::TestCaseIterator* pTestCaseIterator)
{
    GpuBenchmark* pGpuBenchmark = CreateBenchmarkFromName(testName, &m_ResourceAllocator);
    pGpuBenchmark->Initialize(&m_ResourceAllocator);

    if (pTestCaseIterator != nullptr)
        json::ApplyTestCaseConfiguration(pTestCaseIterator, pGpuBenchmark);

    pGpuBenchmark->InitializeGfxObjects(&m_ResourceAllocator, m_Gfw.GetDevice());

    return pGpuBenchmark;
}

void LibraryState::FinalizeBenchmark(GpuBenchmark* pGpuBenchmark)
{
    pGpuBenchmark->FinalizeGfxObjects(&m_ResourceAllocator, m_Gfw.GetDevice());
    pGpuBenchmark->Finalize(&m_ResourceAllocator);
    DestroyBenchmark(pGpuBenchmark, &m_ResourceAllocator);
}

bool LibraryState::ValidateAndUpdateTestResult(int* pOutTestResultFlags, HashComparisonMode hashComparisonMode, TestSuite* pTestSuite)
{
    bool needNoreSamples = true;
    int resultFlags = 0;

    const int testCountWarmUp = 16;
    const int testCountMinForValidation = m_ResultHistoryLength + testCountWarmUp;
    const int validationFactor = 3;
    const float failureThreshold = 0.1f;

    if (pTestSuite->testCount >= testCountWarmUp)
    {
        json::BenchmarkTestResult expectedResult;
        GetTestCaseAdjustedExpectedResult(pTestSuite->pTestCaseIterator, &expectedResult);

        ValidationResult cpuResult = nnt::gfx::util::ValidateResultStandardDeviation(
            pTestSuite->previousCpuDuration, expectedResult.cpuTimeAverage, expectedResult.cpuTimeStandardDeviation,
            validationFactor);
        pTestSuite->cpuResult.Update(cpuResult);

        ValidationResult gpuResult = nnt::gfx::util::ValidateResultStandardDeviation(
            pTestSuite->previousGpuDuration, expectedResult.gpuTimeAverage, expectedResult.gpuTimeStandardDeviation,
            validationFactor);
        pTestSuite->gpuResult.Update(gpuResult);
    }

    if (HashComparisonRequested(hashComparisonMode, pTestSuite->testCount, testCountMinForValidation))
    {
        typedef nn::crypto::Sha1Generator HashGenerator;
        uint8_t testHashContent[HashGenerator::HashSize] = {};
        uint8_t referenceHashContent[HashGenerator::HashSize] = {};

        pTestSuite->pGpuBenchmark->HashResultBuffer<HashGenerator>(
            testHashContent, HashGenerator::HashSize);

        int referenceHashSize = nnt::gfx::util::json::GetTestCaseOutputBufferHash(
            pTestSuite->pTestCaseIterator, referenceHashContent, HashGenerator::HashSize);

        bool hashCompareResult =
            (referenceHashSize == HashGenerator::HashSize)
            && (memcmp(testHashContent, referenceHashContent, HashGenerator::HashSize) == 0);

        if (!hashCompareResult)
        {
            m_HashComparisonFailed = true;
        }
    }

    if (pTestSuite->testCount >= testCountMinForValidation)
    {
        if (pTestSuite->cpuResult.UpdateIsFailed(failureThreshold))
        {
            resultFlags = resultFlags | RunTestResult_FlagFailCpu;
        }

        if (pTestSuite->gpuResult.UpdateIsFailed(failureThreshold))
        {
            resultFlags = resultFlags | RunTestResult_FlagFailGpu;
        }

        if (m_HashComparisonFailed)
        {
            resultFlags = resultFlags | RunTestResult_FlagFailFrameBufferContent;
        }

        needNoreSamples = false;
    }


    AppendToResultHistory(pTestSuite->previousCpuDuration, pTestSuite->previousGpuDuration, resultFlags);

    pTestSuite->testCount++;

    *pOutTestResultFlags = resultFlags;
    return needNoreSamples;
}


void LibraryState::ResetResultHistory()
{
    m_ResultHistoryIndex = 0;

    for (int i = 0; i < m_ResultHistoryLength; ++i)
    {
        m_ResultHistory[i].testResultFlags = -1;
    }

    m_HashComparisonFailed = false;}


void LibraryState::AppendToResultHistory(nn::TimeSpan cpuDuration, nn::TimeSpan gpuDuration, int testResultFlags)
{
    int index = m_ResultHistoryIndex;
    ResultHistoryElement* pResultHistoryElement = &m_ResultHistory[index];

    pResultHistoryElement->cpuDuration = cpuDuration;
    pResultHistoryElement->gpuDuration = gpuDuration;
    pResultHistoryElement->testResultFlags = testResultFlags;

    m_ResultHistoryIndex = (index + 1) % m_ResultHistoryLength;

}
template<bool IsCpu>
nn::TimeSpan LibraryState::GetDuration(const ResultHistoryElement* pResultHistoryElement) const
{
    if (IsCpu)
        return pResultHistoryElement->cpuDuration;
    else
        return pResultHistoryElement->gpuDuration;
}


template<bool IsCpu>
inline uint64_t LibraryState::GetDurationInNs(const ResultHistoryElement* pResultHistoryElement) const
{
    return GetDuration<IsCpu>(pResultHistoryElement).GetNanoSeconds();
}

template<bool IsCpu>
void LibraryState::ComputeStandardDeviationFromHistory(
    uint64_t* pOutMean, uint64_t* pOutStandardDeviation) const
{
    uint64_t mean = 0;
    int elementCount = 0;

    for (int historyIndex = 0; historyIndex < m_ResultHistoryLength; ++historyIndex)
    {
        const ResultHistoryElement* pResultHistoryElement = &m_ResultHistory[historyIndex];
        if (pResultHistoryElement->testResultFlags >= 0)
        {
            uint64_t newValue = mean + GetDurationInNs<IsCpu>(pResultHistoryElement);
            NN_ASSERT(newValue >= mean);
            mean = newValue;
            elementCount++;
        }
    }

    if (elementCount <= 1)
    {
        *pOutMean = mean;
        *pOutStandardDeviation = 0;
        return;
    }

    mean /= static_cast<uint64_t>(elementCount);

    uint64_t squareDiffSum = 0;

    for (int historyIndex = 0; historyIndex < m_ResultHistoryLength; ++historyIndex)
    {
        const ResultHistoryElement* pResultHistoryElement = &m_ResultHistory[historyIndex];
        if (pResultHistoryElement->testResultFlags >= 0)
        {
            int64_t diff = mean - GetDurationInNs<IsCpu>(pResultHistoryElement);
            uint64_t squaredDiff = diff * diff;
            NN_ASSERT(squaredDiff >= static_cast<uint64_t>(diff < 0 ? -diff : diff));
            uint64_t newValue = squareDiffSum + squaredDiff;
            NN_ASSERT(newValue >= squareDiffSum);
            squareDiffSum = newValue;
        }
    }

    uint64_t variance = squareDiffSum / static_cast<uint64_t>(elementCount - 1);
    uint64_t standardDeviation = static_cast<uint64_t>(sqrt(static_cast<double>(variance)));

    *pOutMean = mean;
    *pOutStandardDeviation = standardDeviation;
}

template<bool IsCpu>
int LibraryState::FindFailedCountFromHistory(
    uint64_t mean, uint64_t standardDeviation, int factor) const
{
    int failCount = 0;

    for (int historyIndex = 0; historyIndex < m_ResultHistoryLength; ++historyIndex)
    {
        const ResultHistoryElement* pResultHistoryElement = &m_ResultHistory[historyIndex];
        if (pResultHistoryElement->testResultFlags >= 0)
        {
            nn::TimeSpan value = GetDuration<IsCpu>(pResultHistoryElement);

            if (nnt::gfx::util::ValidateResultStandardDeviation(
                value, mean, standardDeviation, factor))
            {
                failCount++;
            }
        }
    }

    return failCount;
}

void LibraryState::ComputeStandardDeviationFromHistory(
    uint64_t* pOutCpuMean, uint64_t* pOutCpuStandardDeviation,
    uint64_t* pOutGpuMean, uint64_t* pOutGpuStandardDeviation) const
{
    ComputeStandardDeviationFromHistory<true>(pOutCpuMean, pOutCpuStandardDeviation);
    ComputeStandardDeviationFromHistory<false>(pOutGpuMean, pOutGpuStandardDeviation);
}


void LibraryState::LogTestResults(const TestSuite* pTestSuite) const
{
    if (pTestSuite->testCount <= 0)
        return;

    const ValidationResultData* pCpuResult = &pTestSuite->cpuResult;
    const ValidationResultData* pGpuResult = &pTestSuite->gpuResult;
    double cpuFailureRateRate = pCpuResult->ComputeFailureRate();
    double gpuFailureRateRate = pGpuResult->ComputeFailureRate();
    NN_LOG("Test failure rate cpu:%3.2f%% gpu:%3.2f%%\n",
        cpuFailureRateRate * 100,
        gpuFailureRateRate * 100);

    //if (pCpuResult->GetIsFailed() || pGpuResult->GetIsFailed()
    //    || (cpuFailureRateRate > 0.01) || (gpuFailureRateRate > 0.01))
    {
        double cpuFailureRateOverPercent = pCpuResult->ComputeFailureRateOver() * 100;
        double cpuFailureRateBelowPercent = pCpuResult->ComputeFailureRateBelow() * 100;
        NN_LOG("CPU failure rate over:%3.2f%% below:%3.2f%%\n", cpuFailureRateOverPercent, cpuFailureRateBelowPercent);

        double gpuFailureRateOverPercent = pGpuResult->ComputeFailureRateOver() * 100;
        double gpuFailureRateBelowPercent = pGpuResult->ComputeFailureRateBelow() * 100;
        NN_LOG("GPU failure rate over:%3.2f%% below:%3.2f%%\n", gpuFailureRateOverPercent, gpuFailureRateBelowPercent);

        uint64_t cpuMean = 0;
        uint64_t cpuStandardDeviation = 0;
        uint64_t gpuMean = 0;
        uint64_t gpuStandardDeviation = 0;

        ComputeStandardDeviationFromHistory(
            &cpuMean, &cpuStandardDeviation,
            &gpuMean, &gpuStandardDeviation);

        json::BenchmarkTestResult expectedResult;
        GetTestCaseAdjustedExpectedResult(pTestSuite->pTestCaseIterator, &expectedResult);

        int64_t diff;
        double percentage;
        double percentageInSd;
        int cpuFailCount;
        int gpuFailCount;
        int factor = 3;
        NN_LOG("Result details:\n");

        cpuFailCount = FindFailedCountFromHistory<true>(cpuMean, cpuStandardDeviation, factor);
        gpuFailCount = FindFailedCountFromHistory<false>(gpuMean, gpuStandardDeviation, factor);
        NN_LOG("[own data] cpuFailCount %d gpuFailCount %d\n", cpuFailCount, gpuFailCount);

        cpuFailCount = FindFailedCountFromHistory<true>(
            expectedResult.cpuTimeAverage, expectedResult.cpuTimeStandardDeviation, factor);
        gpuFailCount = FindFailedCountFromHistory<false>(
            expectedResult.gpuTimeAverage, expectedResult.gpuTimeStandardDeviation, factor);
        NN_LOG("[expected data] cpuFailCount %d gpuFailCount %d\n", cpuFailCount, gpuFailCount);

        diff = cpuMean - expectedResult.cpuTimeAverage;
        percentage = static_cast<double>(diff * 100) / static_cast<double>(expectedResult.cpuTimeAverage);
        percentageInSd = static_cast<double>(diff) / static_cast<double>(expectedResult.cpuTimeStandardDeviation);
        NN_LOG("cpu mean:               expected:%8lu result:%8lu diff:%8ld (%2.3f%%, %2.3fsd)\n",
            expectedResult.cpuTimeAverage, cpuMean, diff, percentage, percentageInSd);

        diff = cpuStandardDeviation - expectedResult.cpuTimeStandardDeviation;
        percentage = static_cast<double>(diff * 100) / static_cast<double>(expectedResult.cpuTimeStandardDeviation);
        NN_LOG("cpu standard deviation: expected:%8lu result:%8lu diff:%8ld (%2.3f%%)\n",
            expectedResult.cpuTimeStandardDeviation, cpuStandardDeviation, diff, percentage);

        diff = gpuMean - expectedResult.gpuTimeAverage;
        percentage = static_cast<double>(diff * 100) / static_cast<double>(expectedResult.gpuTimeAverage);
        percentageInSd = static_cast<double>(diff) / static_cast<double>(expectedResult.gpuTimeStandardDeviation);
        NN_LOG("gpu mean:               expected:%8lu result:%8lu diff:%8ld (%2.3f%%, %2.3fsd)\n",
            expectedResult.gpuTimeAverage, gpuMean, diff, percentage, percentageInSd);

        diff = gpuStandardDeviation - expectedResult.gpuTimeStandardDeviation;
        percentage = static_cast<double>(diff * 100) / static_cast<double>(expectedResult.gpuTimeStandardDeviation);
        NN_LOG("gpu standard deviation: expected:%8lu result:%8lu diff:%8ld (%2.3f%%)\n",
            expectedResult.gpuTimeStandardDeviation, gpuStandardDeviation, diff, percentage);
    }
}

void LibraryState::InitializeTestCaseData(TestSuite* pTestSuite)
{
    NN_ASSERT(pTestSuite->pGpuBenchmark == nullptr);

    m_GpuBenchmarkQueue.Flush();
    m_GpuBenchmarkQueue.Sync();

    const char* testCaseName = json::GetTestCaseName(pTestSuite->pTestCaseIterator);
    pTestSuite->pGpuBenchmark = InitializeBenchmark(testCaseName, pTestSuite->pTestCaseIterator);
    pTestSuite->testCount = 0;

    int warmUpCount = json::GetTestCaseWarmUpCount(pTestSuite->pTestCaseIterator);
    int runCount = json::GetTestCaseRepeatCount(pTestSuite->pTestCaseIterator);

    nnt::gfx::util::RecordGpuBenchmarkCommandList(
        pTestSuite->pGpuBenchmark, &m_RuntimeGfxObjects,
        warmUpCount, runCount);

    pTestSuite->cpuResult.Clear();
    pTestSuite->gpuResult.Clear();

    ResetResultHistory();
}

void LibraryState::FinalizeTestCaseData(TestSuite* pTestSuite)
{
    NN_ASSERT(pTestSuite->pGpuBenchmark != nullptr);

    m_GpuBenchmarkQueue.Flush();
    m_GpuBenchmarkQueue.Sync();

    FinalizeBenchmark(pTestSuite->pGpuBenchmark);
    pTestSuite->pGpuBenchmark = nullptr;

}

void LibraryState::GetTestCaseAdjustedExpectedResult(
    nnt::gfx::util::json::TestCaseIterator* pTestCaseIterator,
    nnt::gfx::util::json::BenchmarkTestResult* pAdjustedExpectedResult) const
{
    json::GetTestCaseExpectedResult(pTestCaseIterator, pAdjustedExpectedResult);

    // 1.33%
    const uint64_t cpuThresholdDivider = 75;
    // 1.25%
    const uint64_t gpuThresholdDivider = 80;
    {
        uint64_t cpuStandardDeviationThreshold =
            (pAdjustedExpectedResult->cpuTimeAverage + (cpuThresholdDivider / 2)) / cpuThresholdDivider;
        if (pAdjustedExpectedResult->cpuTimeStandardDeviation < cpuStandardDeviationThreshold)
            pAdjustedExpectedResult->cpuTimeStandardDeviation = cpuStandardDeviationThreshold;
    }
    {
        uint64_t gpuStandardDeviationThreshold =
            (pAdjustedExpectedResult->gpuTimeAverage + (gpuThresholdDivider / 2)) / gpuThresholdDivider;
        if (pAdjustedExpectedResult->gpuTimeStandardDeviation < gpuStandardDeviationThreshold)
            pAdjustedExpectedResult->gpuTimeStandardDeviation = gpuStandardDeviationThreshold;
    }
}

bool LibraryState::HashComparisonRequested(
    HashComparisonMode hashComparisonMode, int testCount, int testCountMinForValidation)
{
    switch (hashComparisonMode)
    {
    case HashComparisonMode_Disable:
        return false;
    case HashComparisonMode_Enable:
        return true;
    case HashComparisonMode_FirstRunOnly:
        return (testCount == 0);
    case HashComparisonMode_LastRunOnly:
        return (testCount == testCountMinForValidation);
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}


} } } } // namespace nnt { namespace gfx { namespace util { namespace agingtest {
