﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>

namespace nnt { namespace gfx { namespace util { namespace agingtest {


typedef int FreeListElementHandle;

static const FreeListElementHandle FreeListElementInvalidHandle = -1;

template<typename T, int Capacity>
class FreeList
{
private:
    static const int m_InvalidAllocationCounter = -1;

    union ListElement
    {
        ListElement*    pNext;
        uint8_t         content[sizeof(T)];
    };

    int             m_AllocationCounter;
    int             m_IndexBitCount;
    int             m_AllocationCounterBitCount;

    ListElement*    m_pListHead;

    int             m_ContentAllocationCounter[Capacity];
    ListElement     m_ContentArray[Capacity];

public:
                            FreeList();
                            ~FreeList();

    void                    Initialize();
    void                    Finalize();

    FreeListElementHandle   Allocate();
    void                    Free(FreeListElementHandle handle);

    T*                      Get(FreeListElementHandle handle);
    const T*                Get(FreeListElementHandle handle) const;

private:

    ListElement*            GetListElementFromHandle(FreeListElementHandle handle);
    const ListElement*      GetListElementFromHandle(FreeListElementHandle handle) const;

    int                     GetElementIndex(const ListElement* element) const;

    FreeListElementHandle   MakeHandle(int elementIndex, int allocationCount) const;
    int                     GetHandleElementIndex(FreeListElementHandle handle) const;
    int                     GetHandleAllocationCounter(FreeListElementHandle handle) const;

    bool                    IsFreeElement(const ListElement* element) const;
    int                     CountFreeElements() const;
};

template<typename T, int Capacity>
FreeList<T, Capacity>::FreeList()
: m_AllocationCounter(0)
, m_IndexBitCount(0)
, m_AllocationCounterBitCount(0)
, m_pListHead(nullptr)
{
}

template<typename T, int Capacity>
FreeList<T, Capacity>::~FreeList()
{
}

template<typename T, int Capacity>
void FreeList<T, Capacity>::Initialize()
{
    m_AllocationCounter = 0;
    m_IndexBitCount = 0;
    m_AllocationCounterBitCount = 0;

    m_IndexBitCount = ((sizeof(int) * CHAR_BIT) - 1) - nn::util::cntl0(Capacity);
    m_AllocationCounterBitCount = (sizeof(int) * CHAR_BIT) - m_IndexBitCount - 1;
    NN_ASSERT(m_IndexBitCount > 0);
    NN_ASSERT(m_AllocationCounterBitCount > 0);
    NN_ASSERT((m_IndexBitCount  + m_AllocationCounterBitCount) == ((sizeof(int) * CHAR_BIT) - 1));


    memset(m_ContentAllocationCounter, 0, sizeof(m_ContentAllocationCounter));
    for (int elementIndex = 0; elementIndex < Capacity; ++elementIndex)
    {
        m_ContentAllocationCounter[elementIndex] = m_InvalidAllocationCounter;
    }

    memset(m_ContentArray, 0, sizeof(m_ContentArray));
    for (int elementIndex = 0; elementIndex < (Capacity - 1); ++elementIndex)
    {
        m_ContentArray[elementIndex].pNext = &m_ContentArray[elementIndex + 1];
    }
    m_ContentArray[Capacity - 1].pNext = nullptr;
    m_pListHead = &m_ContentArray[0];
}

template<typename T, int Capacity>
void FreeList<T, Capacity>::Finalize()
{
    NN_ASSERT(CountFreeElements() == Capacity);
}

template<typename T, int Capacity>
FreeListElementHandle FreeList<T, Capacity>::Allocate()
{
    ListElement* pListElement = m_pListHead;
    if (pListElement == nullptr)
    {
        return FreeListElementInvalidHandle;
    }
    ListElement* pNextListElement = pListElement->pNext;
    m_pListHead = pNextListElement;

    new (&pListElement->content) T();

    int index = GetElementIndex(pListElement);
    int allocationCounter = m_AllocationCounter;
    FreeListElementHandle handle = MakeHandle(index, allocationCounter);
    NN_ASSERT(m_ContentAllocationCounter[index] == m_InvalidAllocationCounter);
    m_ContentAllocationCounter[index] = allocationCounter;

    m_AllocationCounter = (m_AllocationCounter + 1) & ((1 << m_AllocationCounterBitCount) - 1);

    return handle;
}

template<typename T, int Capacity>
void FreeList<T, Capacity>::Free(FreeListElementHandle handle)
{
    ListElement* pListElement = GetListElementFromHandle(handle);
    reinterpret_cast<T*>(&pListElement->content)->~T();

    int index = GetElementIndex(pListElement);
    NN_ASSERT(m_ContentAllocationCounter[index] != m_InvalidAllocationCounter);
    m_ContentAllocationCounter[index] = m_InvalidAllocationCounter;

    pListElement->pNext = m_pListHead;
    m_pListHead = pListElement;
}

template<typename T, int Capacity>
inline T* FreeList<T, Capacity>::Get(FreeListElementHandle handle)
{
    ListElement* pListElement = GetListElementFromHandle(handle);
    return reinterpret_cast<T*>(&pListElement->content);
}

template<typename T, int Capacity>
inline const T* FreeList<T, Capacity>::Get(FreeListElementHandle handle) const
{
    const ListElement* pListElement = GetListElementFromHandle(handle);
    return reinterpret_cast<const T*>(&pListElement->content);
}

template<typename T, int Capacity>
const typename FreeList<T, Capacity>::ListElement* FreeList<T, Capacity>::GetListElementFromHandle(FreeListElementHandle handle) const
{
    int index = GetHandleElementIndex(handle);
    int allocationCounter = GetHandleAllocationCounter(handle);
    NN_ASSERT(allocationCounter == m_ContentAllocationCounter[index]);
    const ListElement* pListElement = &m_ContentArray[index];
    return pListElement;
}

template<typename T, int Capacity>
typename FreeList<T, Capacity>::ListElement* FreeList<T, Capacity>::GetListElementFromHandle(FreeListElementHandle handle)
{
    int index = GetHandleElementIndex(handle);
    int allocationCounter = GetHandleAllocationCounter(handle);
    NN_ASSERT(allocationCounter == m_ContentAllocationCounter[index]);
    ListElement* pListElement = &m_ContentArray[index];
    return pListElement;
}

template<typename T, int Capacity>
int FreeList<T, Capacity>::GetElementIndex(const ListElement* element) const
{
    ptrdiff_t offset = element - m_ContentArray;
    NN_ASSERT(offset >= 0);
    NN_ASSERT(offset < Capacity);
    return static_cast<int>(offset);
}

template<typename T, int Capacity>
FreeListElementHandle FreeList<T, Capacity>::MakeHandle(int elementIndex, int allocationCounter) const
{
    NN_ASSERT(allocationCounter < (1 << m_AllocationCounterBitCount));
    NN_ASSERT(elementIndex < (1 << m_IndexBitCount));
    return (allocationCounter << m_IndexBitCount) | elementIndex;
}

template<typename T, int Capacity>
int FreeList<T, Capacity>::GetHandleElementIndex(FreeListElementHandle handle) const
{
    return handle & ((1 << m_IndexBitCount) - 1);
}

template<typename T, int Capacity>
int FreeList<T, Capacity>::GetHandleAllocationCounter(FreeListElementHandle handle) const
{
    return (handle >> m_IndexBitCount) & ((1 << m_AllocationCounterBitCount) - 1);
}

template<typename T, int Capacity>
bool FreeList<T, Capacity>::IsFreeElement(const ListElement* element) const
{
    const ListElement* pListElement = m_pListHead;
    while (pListElement != nullptr)
    {
        if (pListElement == element)
            return true;

        pListElement = pListElement->pNext;
    }

    return false;
}

template<typename T, int Capacity>
int FreeList<T, Capacity>::CountFreeElements() const
{
    int elementCount = 0;
    ListElement* pListElement = m_pListHead;
    while (pListElement != nullptr)
    {
        int index = GetElementIndex(pListElement);
        NN_ASSERT(m_ContentAllocationCounter[index] == m_InvalidAllocationCounter);

        elementCount++;
        pListElement = pListElement->pNext;
    }

    return elementCount;
}


} } } } // namespace nnt { namespace gfx { namespace util { namespace agingtest {
