﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "gfxUtilAgingTest_BmpReader.h"

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <memory.h>

namespace nnt { namespace gfx { namespace util { namespace agingtest {

namespace {

struct BmpHeader
{
    uint8_t  magic[2];    /* Magic bytes 'B' and 'M'. */
    uint32_t fileSize;   /* Size of whole file. */
    uint32_t unused;      /* Should be 0. */
    uint32_t dataOffset; /* Offset from beginning of file to bitmap data. */

};


struct BmpInfoHeader
{
    uint32_t    infoSize;   /* Size of info struct (> sizeof(bmp_info)). */
    int32_t     width;       /* Width of image. */
    int32_t     height;      /* Height (< 0 means right-side up). */
    uint16_t    planes;      /* Planes (should be 1). */
    uint16_t    bits;        /* Number of bits (1, 4, 8, 16, 24, or 32). */
    uint32_t    compression; /* 0 = none, 1 = 8-bit RLE, 2 = 4-bit RLE, etc. */
    int32_t     bV4SizeImage;
    uint32_t    bV4XPelsPerMeter;
    uint32_t    bV4YPelsPerMeter;
    int32_t     bV4ClrUsed;
    int32_t     bV4ClrImportant;
};

struct BmpInfoHeaderV4
{
    int32_t     bV4RedMask;
    int32_t     bV4GreenMask;
    int32_t     bV4BlueMask;
    int32_t     bV4AlphaMask;
    int32_t     bV4CSType;
    int32_t     bV4EndpointRedX;
    int32_t     bV4EndpointRedY;
    int32_t     bV4EndpointRedZ;
    int32_t     bV4EndpointGreenX;
    int32_t     bV4EndpointGreenY;
    int32_t     bV4EndpointGreenZ;
    int32_t     bV4EndpointBlueX;
    int32_t     bV4EndpointBlueY;
    int32_t     bV4EndpointBlueZ;
    int32_t     bV4GammaRed;
    int32_t     bV4GammaGreen;
    int32_t     bV4GammaBlue;
};
struct BmpInfoHeaderV5
{
    int32_t     bV5Intent;
    int32_t     bV5ProfileData;
    int32_t     bV5ProfileSize;
    int32_t     bV5Reserved;
};

int IsEof(BmpInputStream* pIs)
{
    return pIs->readOffset == pIs->bufferSize;
}

int ReadByte(BmpInputStream* pIs)
{
    int value = pIs->pBuffer[pIs->readOffset];
    pIs->readOffset = pIs->readOffset + 1;
    return value;
}


bool ReadLittleEndian(uint32_t* dest, int bytes, BmpInputStream* pIs)
{
    uint32_t shift = 0;

    *dest = 0;

    while(bytes--)
    {
        if (IsEof(pIs))
            return false;

        int byte = ReadByte(pIs);

        *dest += (uint32_t)byte << shift;
        shift += 8;
    }

    return true;
}

template<typename T>
bool ReadLittleEndian(T* dest, BmpInputStream* pIs)
{
    uint32_t tmp = 0;
    if (ReadLittleEndian(&tmp, sizeof(T), pIs))
    {
        *dest = static_cast<T>(tmp);
        return true;
    }

    return false;

}

template<typename T>
bool ReadArrayLittleEndian(T* dest, size_t count, BmpInputStream* pIs)
{
    for (size_t i = 0; i < count; ++i)
    {
        if (!ReadLittleEndian(&dest[i], pIs))
            return false;
    }

    return true;
}


void DecodeBmpLine(
    uint8_t* pOutputBuffer, const uint8_t* pInputBuffer, size_t pixelCount, int bpp,
    int maskRed, int maskGreen, int maskBlue, int maskAlpha)
{
    NN_ASSERT((bpp == 24) || (bpp == 32));

    int destinationPixelSizeInBytes = (bpp == 24) ? 3 : 4;
    int sourcePixelSizeInBytes = (bpp == 24) ? 3 : 4;

    int shiftRed = nn::util::cntt0(maskRed);
    int shiftGreen = nn::util::cntt0(maskGreen);
    int shiftBlue = nn::util::cntt0(maskBlue);
    int shiftAlpha = nn::util::cntt0(maskAlpha);

    for (size_t i = 0; i < pixelCount; ++i)
    {
        //int shift = 24;
        int value = 0;
        int componentCount = bpp / 8;
        for (int c = 0; c < componentCount; ++c)
        {
            value |= pInputBuffer[c] << (c * 8);
        }
        value <<= ((4 - componentCount) * 8);

        int red = (value & maskRed) >> shiftRed;
        int green = (value & maskGreen) >> shiftGreen;
        int blue = (value & maskBlue) >> shiftBlue;
        int alpha = (value & maskAlpha) >> shiftAlpha;

        *(pOutputBuffer + 0) = static_cast<uint8_t>(red);
        *(pOutputBuffer + 1) = static_cast<uint8_t>(green);
        *(pOutputBuffer + 2) = static_cast<uint8_t>(blue);

        if (bpp == 32)
        {
            *(pOutputBuffer + 3) = static_cast<uint8_t>(alpha);
        }

        pOutputBuffer += destinationPixelSizeInBytes;
        pInputBuffer += sourcePixelSizeInBytes;
    }
}


bool DecodeBmp(
    const BmpFileInfo* pBmpFileInfo, const uint8_t* pInputBuffer, uint8_t* pOutputBuffer, size_t outputBufferSizeInBytes,
    int maskRed, int maskGreen, int maskBlue, int maskAlpha)
{
    size_t destinationPixelStride = pBmpFileInfo->stride;
    size_t sourcePixelStride = nn::util::align_up(pBmpFileInfo->stride, 4);

    size_t ouputDataSizeInBytes = pBmpFileInfo->height * destinationPixelStride;
    if (ouputDataSizeInBytes > outputBufferSizeInBytes)
        return false;

    if ((pBmpFileInfo->bpp != 24) && (pBmpFileInfo->bpp != 32))
        return false;

    // will reverse lines
    uint8_t*        pCurrentOutputLineBuffer = pOutputBuffer + ouputDataSizeInBytes - destinationPixelStride;
    const uint8_t*  pCurrentInputLineBuffer = pInputBuffer;
    const uint8_t*  pCurrentInputLineBufferEnd = pInputBuffer + (sourcePixelStride * pBmpFileInfo->height);

    while (pCurrentInputLineBuffer < pCurrentInputLineBufferEnd)
    {
        DecodeBmpLine(
            pCurrentOutputLineBuffer, pCurrentInputLineBuffer, pBmpFileInfo->width, pBmpFileInfo->bpp,
            maskRed, maskGreen, maskBlue, maskAlpha);
        pCurrentInputLineBuffer += sourcePixelStride;
        pCurrentOutputLineBuffer -= destinationPixelStride;
    }

    return true;
}

bool ReadBmpHeader(BmpHeader* pOutHeader, BmpInputStream* pIs)
{
    size_t availableSizeInStream = pIs->bufferSize - pIs->readOffset;

    pOutHeader->magic[0] = static_cast<uint8_t>(ReadByte(pIs));
    pOutHeader->magic[1] = static_cast<uint8_t>(ReadByte(pIs));
    ReadLittleEndian(&pOutHeader->fileSize, pIs);
    ReadLittleEndian(&pOutHeader->unused, pIs);
    ReadLittleEndian(&pOutHeader->dataOffset, pIs);

    if ((pOutHeader->magic[0] != 'B') || (pOutHeader->magic[1] != 'M'))
    {
        return false;
    }
    if (pOutHeader->fileSize > availableSizeInStream)
    {
        return false;
    }

    return true;
}

bool ReadBmpInfoHeader(BmpInfoHeader* pOutInfo, BmpInputStream* pIs)
{
    ReadLittleEndian(&pOutInfo->infoSize, pIs);

    if (pOutInfo->infoSize < sizeof(BmpInfoHeader))
    {
        return false;
    }

    ReadLittleEndian(&pOutInfo->width, pIs);
    ReadLittleEndian(&pOutInfo->height, pIs);
    ReadLittleEndian(&pOutInfo->planes, pIs);
    ReadLittleEndian(&pOutInfo->bits, pIs);
    ReadLittleEndian(&pOutInfo->compression, pIs);
    ReadLittleEndian(&pOutInfo->bV4SizeImage, pIs);
    ReadLittleEndian(&pOutInfo->bV4XPelsPerMeter, pIs);
    ReadLittleEndian(&pOutInfo->bV4YPelsPerMeter, pIs);
    ReadLittleEndian(&pOutInfo->bV4ClrUsed, pIs);
    ReadLittleEndian(&pOutInfo->bV4ClrImportant, pIs);

    if ((pOutInfo->infoSize != sizeof(BmpInfoHeader))
        && (pOutInfo->infoSize != (sizeof(BmpInfoHeader) + sizeof(BmpInfoHeaderV4)))
        && (pOutInfo->infoSize != (sizeof(BmpInfoHeader) + sizeof(BmpInfoHeaderV4) + sizeof(BmpInfoHeaderV5))))
    {
        return false;
    }

    if ((pOutInfo->width == 0) || (pOutInfo->height == 0))
    {
        return false;
    }
    if (pOutInfo->planes != 1)
    {
        return false;
    }
    if ((pOutInfo->bits != 24) && (pOutInfo->bits != 32))
    {
        return false;
    }
    if ((pOutInfo->bits == 24) && (pOutInfo->compression != 0))
    {
        return false;
    }
    if ((pOutInfo->bits == 32) && ((pOutInfo->compression != 0) && (pOutInfo->compression != 3)))
    {
        return false;
    }

    return true;

}

bool ReadBmpInfoHeaderV4(BmpInfoHeaderV4* pOutInfoV4, BmpInputStream* pIs)
{
    ReadLittleEndian(&pOutInfoV4->bV4RedMask, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4GreenMask, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4BlueMask, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4AlphaMask, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4CSType, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4EndpointRedX, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4EndpointRedY, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4EndpointRedZ, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4EndpointGreenX, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4EndpointGreenY, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4EndpointGreenZ, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4EndpointBlueX, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4EndpointBlueY, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4EndpointBlueZ, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4GammaRed, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4GammaGreen, pIs);
    ReadLittleEndian(&pOutInfoV4->bV4GammaBlue, pIs);

    return true;
}

bool ReadBmpInfoHeaderV5(BmpInfoHeaderV5* pOutInfoV5, BmpInputStream* pIs)
{
    ReadLittleEndian(&pOutInfoV5->bV5Intent, pIs);
    ReadLittleEndian(&pOutInfoV5->bV5ProfileData, pIs);
    ReadLittleEndian(&pOutInfoV5->bV5ProfileSize, pIs);
    ReadLittleEndian(&pOutInfoV5->bV5Reserved, pIs);

    return true;
}

} // anonymous namespace

bool ReadBmpFileInfoFromStream(const BmpInputStream* pIs, BmpFileInfo* pBmpFileInfo)
{
    BmpHeader header;
    BmpInfoHeader infoHeader;
    BmpInputStream is = *pIs;

    if (!ReadBmpHeader(&header, &is))
    {
        return false;
    }
    if (!ReadBmpInfoHeader(&infoHeader, &is))
    {
        return false;
    }

    pBmpFileInfo->width = infoHeader.width;
    pBmpFileInfo->height = infoHeader.height;
    pBmpFileInfo->bpp = infoHeader.bits;
    pBmpFileInfo->stride = (infoHeader.width * infoHeader.bits) / CHAR_BIT;

    size_t dataSize = (pBmpFileInfo->height * pBmpFileInfo->width * infoHeader.bits) / CHAR_BIT;
    if (is.bufferSize < (dataSize + header.dataOffset))
        return false;

    return true;
}

bool ReadBmpFromStream(BmpInputStream* pIs, const BmpFileInfo* pBmpFileInfo, void* pPixelOutBuffer, size_t pixelOutBufferSizeInBytes)
{
    BmpHeader       header;
    BmpInfoHeader   bmpInfoHeader;

    if (!ReadBmpHeader(&header, pIs))
    {
        return false;
    }

    int maskRed = 0xFF << 24;
    int maskGreen = 0xFF << 16;
    int maskBlue = 0xFF << 8;
    int maskAlpha = 0xFF << 0;


    if (!ReadBmpInfoHeader(&bmpInfoHeader, pIs))
    {
        return false;
    }

    if (bmpInfoHeader.infoSize >= (sizeof(BmpInfoHeader) + sizeof(BmpInfoHeaderV4)))
    {
        BmpInfoHeaderV4 bmpInfoHeaderV4;

        if (!ReadBmpInfoHeaderV4(&bmpInfoHeaderV4, pIs))
        {
            return false;
        }

        if (bmpInfoHeader.compression == 3)
        {
            maskRed = bmpInfoHeaderV4.bV4RedMask;
            maskGreen = bmpInfoHeaderV4.bV4GreenMask;
            maskBlue = bmpInfoHeaderV4.bV4BlueMask;
            maskAlpha = bmpInfoHeaderV4.bV4AlphaMask;
        }

        if (bmpInfoHeader.infoSize >= (sizeof(BmpInfoHeader) + sizeof(BmpInfoHeaderV4) + sizeof(BmpInfoHeaderV5)))
        {
            BmpInfoHeaderV5 bmpInfoHeaderV5;

            if (!ReadBmpInfoHeaderV5(&bmpInfoHeaderV5, pIs))
            {
                return false;
            }
        }
    }
    else
    {
        if (bmpInfoHeader.compression != 0)
        {
            return false;
        }
    }

    DecodeBmp(
        pBmpFileInfo, pIs->pBuffer + header.dataOffset,
        static_cast<uint8_t*>(pPixelOutBuffer), pixelOutBufferSizeInBytes,
        maskRed, maskGreen, maskBlue, maskAlpha);

    return true;
};

} } } } // namespace nnt { namespace gfx { namespace util { namespace agingtest {

