﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx/gfx_Types.h>
#include <nn/util/util_Color.h>

namespace nns { namespace gfx { namespace PrimitiveRenderer {
class Renderer;
} } } // namespace nns { namespace gfx { namespace PrimitiveRenderer {


namespace nnt { namespace gfx { namespace util { namespace agingtest {


//---------------------------------------------------------------------------
//! @brief InitializeTestSuiteの4つ目の索引で使ったら、全部のテストケースを戻します。
//---------------------------------------------------------------------------
const int MaxCountAllTests = -1;

//---------------------------------------------------------------------------
//! @brief 表示のためのカラー型
//---------------------------------------------------------------------------
typedef nn::util::Color4u8 Color;

//---------------------------------------------------------------------------
//! @brief よく使われているカラーの定義
//---------------------------------------------------------------------------
struct DefaultColor
{
    static const Color White;
    static const Color Gray;
    static const Color Black;
    static const Color Red;
    static const Color Green;
    static const Color Blue;
};

//---------------------------------------------------------------------------
//! @brief テストのハンドル
//---------------------------------------------------------------------------
typedef int TestSuiteHandle;

//---------------------------------------------------------------------------
//! @brief テクスチャーのハンドル
//---------------------------------------------------------------------------
typedef int TextureHandle;


//---------------------------------------------------------------------------
//! @brief テクスチャーのハンドル
//! @deprecated
//---------------------------------------------------------------------------
NN_DEPRECATED typedef TextureHandle TextureResourceHandle;

//---------------------------------------------------------------------------
//! @brief ピクセルのフォーマット
//---------------------------------------------------------------------------
enum PixelFormat
{
    PixelFormat_RGB,
    PixelFormat_RGBA,
};

//---------------------------------------------------------------------------
//! @brief RunTestでハッシュ計算のタイミングのモード
//---------------------------------------------------------------------------
enum HashComparisonMode
{
    HashComparisonMode_Disable,         // 比較しない
    HashComparisonMode_Enable,          // 比較する
    HashComparisonMode_FirstRunOnly,    // 最初のRunTestの呼び出しのみ
    HashComparisonMode_LastRunOnly,     // RunTestの戻り値がfalseになる時
};

//---------------------------------------------------------------------------
//! @brief RunTest 関数の結果
//---------------------------------------------------------------------------
enum RunTestResultFlags
{
    RunTestResult_FlagFailCpu = 0x01,
    RunTestResult_FlagFailGpu = 0x02,
    RunTestResult_FlagFailFrameBufferContent = 0x04,
};

//---------------------------------------------------------------------------
//! @brief ラブラリーの初期化
//---------------------------------------------------------------------------
void Initialize() NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief ラブラリーの終了処理
//---------------------------------------------------------------------------
void Finalize() NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief nn::gfx::Ｄevice のインスタンスの取得
//!
//! @return nn::gfx::Ｄevice のインスタンスのポインタを返します
//---------------------------------------------------------------------------
nn::gfx::Device* GetGfxDevice() NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief nn::gfx::CommandBuffer のインスタンスの取得
//!
//! @return nn::gfx::CommandBuffer のインスタンスのポインタを返します
//---------------------------------------------------------------------------
nn::gfx::CommandBuffer* GetCommandGfxBuffer() NN_NOEXCEPT;


//---------------------------------------------------------------------------
//! @brief フレームバッファの幅の取得
//!
//! @return フレームバッファの幅を返します。
//---------------------------------------------------------------------------
int GetFrameBufferWidth() NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief フレームバッファの縦の取得
//!
//! @return フレームバッファの縦を返します。
//---------------------------------------------------------------------------
int GetFrameBufferHeight() NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief プラットフォームのＩＤ
//
//! @param[out] buffer          目的のバッファ
//! @param[in] bufferSize       目的のバッファのサイズ
//---------------------------------------------------------------------------
void GetTestPlatformId(char* buffer, int bufferSize) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief ファイルからテストの設定を読みます。
//!
//! @param[out] pOutHandle      リソースのハンドル
//! @param[in] szFilePath       リソースファイルのパス
//! @return                     処理の結果が返ります。
//! @retval ResultSuccess       成功しました。
//! @retval ResultPathNotFound  対象ファイルが存在しません。
//---------------------------------------------------------------------------
nn::Result LoadTestSuiteFromFile(TestSuiteHandle* pOutHandle, const char* szFilePath) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief リソースの解放
//!
//! @param[in] handle           LoadTestSuiteFromFile() からもらったハンドル
//---------------------------------------------------------------------------
void UnloadTestSuite(TestSuiteHandle handle) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief パターンに一致するテスト
//!
//! @param[in] handle           LoadTestSuiteFromFile() からもらったハンドル
//! @param[in] filterPattern    検索のパタン
//! @param[in] startIndex       一致するテストのリストにはこの索引からテストを行う
//! @param[in] maxCount         最高のテストの数。MaxCountAllTestsなら全部のテストケースを戻します
//! @return                     テストの数
//---------------------------------------------------------------------------
int InitializeTestSuite(TestSuiteHandle handle, const char* filterPattern, int startIndex, int maxCount) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief リソースの解放
//!
//! @param[in] handle LoadTestSuiteFromFile() からもらったハンドル
//---------------------------------------------------------------------------
void FinalizeTestSuite(TestSuiteHandle handle) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief テストの有無
//!
//! @param[in] handle   LoadTestSuiteFromFile() からもらったハンドル
//! @return             テストが残っている場合は true を返します。そうでない場合は false を返します。
//---------------------------------------------------------------------------
bool HasMoreTests(TestSuiteHandle handle) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief 次のテストを用意します。
//!
//! @param[in] handle   LoadTestSuiteFromFile() からもらったハンドル
//! @return             テストが残っている場合は true を返します。そうでない場合は false を返します。
//---------------------------------------------------------------------------
bool BeginNextTest(TestSuiteHandle handle) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief テストの名前を戻します。
//!
//! @param[in] handle   LoadTestSuiteFromFile() からもらったハンドル
//! @return             選ばされたテストの名前を返します。
//---------------------------------------------------------------------------
const char* GetTestName(TestSuiteHandle handle) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief テストの識別名を戻します。
//!
//! @param[in] handle   LoadTestSuiteFromFile() からもらったハンドル
//! @return             選ばされたテストの識別名を返します。
//---------------------------------------------------------------------------
const char* GetTestCaseId(TestSuiteHandle handle) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief テストを行います
//!
//! @param[out]     pResultFlags 行ったテストの結果に応じてRunTestResultFlagsの組み合す
//! @param[in]      hashComparisonMode 出力バッファハッシュの比較モード
//! @param[in]      handle LoadTestSuiteFromFile() からもらったハンドル
//! @return         テスト結果がそろっていた場合、falseを戻します。trueの戻り値の場合、
//!                 判断するために結果がまだ足りていない状態です。
//---------------------------------------------------------------------------
bool RunTest(int* pOutTestResultFlags, HashComparisonMode hashComparisonMode, TestSuiteHandle handle) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief ファイルからテクスチャーリソースを読み込ます。
//!
//! @deprecated
//!
//! @param[in] szFilePath       リソースファイルのパス
//! @param[out] pOutHandle      リソースのハンドル
//! @return                     処理の結果が返ります。
//! @retval ResultSuccess       成功しました。
//! @retval ResultPathNotFound  対象ファイルが存在しません。
//---------------------------------------------------------------------------
NN_DEPRECATED nn::Result LoadTextureResourceFromFile(const char* szFilePath, TextureHandle* pOutHandle) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief ファイルからテクスチャーリソースを読み込ます。
//!
//! @param[out] pOutHandle      リソースのハンドル
//! @param[in] szFilePath       リソースファイルのパス
//! @return                     処理の結果が返ります。
//! @retval ResultSuccess       成功しました。
//! @retval ResultPathNotFound  対象ファイルが存在しません。
//---------------------------------------------------------------------------
nn::Result CreateTextureFromResourceFile(TextureHandle* pOutHandle, const char* szFilePath) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief バッファからテクスチャーリソースを読み込ます。
//!
//! @param[out] pOutHandle          リソースのハンドル
//! @param[in] bmpBuffer            BMP ファイルのバッファ
//! @param[in] bmpBufferSizeInBytes BMP ファイルのバッファのサイズ
//! @return                         処理の結果が返ります。
//! @retval ResultSuccess           成功しました。
//---------------------------------------------------------------------------
nn::Result CreateTextureFromBmpBuffer(TextureHandle* pOutHandle, const void* bmpBuffer, size_t bmpBufferSizeInBytes) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief ファイルからテクスチャーリソースを読み込ます。
//!
//! @param[out] pOutHandle          リソースのハンドル
//! @param[in] szFilePath       リソースファイルのパス
//! @return                         処理の結果が返ります。
//! @retval ResultSuccess           成功しました。
//! @retval ResultPathNotFound      対象ファイルが存在しません。
//---------------------------------------------------------------------------
nn::Result CreateTextureFromBmpFile(TextureHandle* pOutHandle, const char* szFilePath) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief ファイルからテクスチャーリソースを読み込ます。
//!
//! @param[out] pOutHandle          リソースのハンドル
//! @param[in] szFilePath       リソースファイルのパス
//! @return                         処理の結果が返ります。
//! @retval ResultSuccess           成功しました。
//! @retval ResultPathNotFound      対象ファイルが存在しません。
//---------------------------------------------------------------------------
nn::Result CreateTextureFromPixelBuffer(
    TextureHandle* pOutHandle, const void* pPixelBuffer, size_t pixelBufferSizeInBytes,
    int width, int height, int pitch, PixelFormat pixelFormat) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief リソースの解放
//! @param[in] handle      LoadTextureResourceFromFile() からもらったハンドル
//---------------------------------------------------------------------------
void FreeTextureResource(TextureHandle handle) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief テクスチャの幅の取得
//! @param[in] handle      LoadTextureResourceFromFile() からもらったハンドル
//! @return テクスチャの幅を返します。
//---------------------------------------------------------------------------
int GetTextureWidth(TextureHandle handle) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief テクスチャの縦の取得
//! @param[in] handle      LoadTextureResourceFromFile() からもらったハンドル
//! @return テクスチャの縦を返します。
//---------------------------------------------------------------------------
int GetTextureHeight(TextureHandle handle) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief フレームが始めます。この関数を呼び出して、表示のAPIが使えます。
//---------------------------------------------------------------------------
void StartFrame() NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief テストの結果を表示します
//!
//! @param[in] handle       LoadTestSuiteFromFile() からもらったハンドル
//! @param[in] positionX    画面のｘ位置
//! @param[in] positionY    画面のｙ位置
//! @param[in] scale        文字のサイズ
//! @param[in] color        フォントの色
//---------------------------------------------------------------------------
void DrawTestResults(TestSuiteHandle handle, int positionX, int positionY, float scale, const Color& color) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief スクリーンにフォントの表示
//!
//! @param[in] text         表示したい文字列
//! @param[in] positionX    画面のｘ位置
//! @param[in] positionY    画面のｙ位置
//! @param[in] scale        文字のサイズ
//! @param[in] color        フォントの色
//---------------------------------------------------------------------------
void DrawFont(const char* text, int positionX, int positionY, float scale, const Color& color) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief スクリーンに2D枠を描画します。
//!
//! @param[in] positionX    2D枠の左上頂点のx座標
//! @param[in] positionY    2D枠の左上頂点のy座標
//! @param[in] width        2D枠の幅
//! @param[in] height       2D枠の高さ
//! @param[in] color        2D枠の色
//---------------------------------------------------------------------------
void Draw2DFrame(int positionX, int positionY, int width, int height, const Color& color) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief スクリーンに四角形を描画します。
//!
//! @param[in] positionX    四角形の左上頂点のx座標
//! @param[in] positionY    四角形の左上頂点のy座標
//! @param[in] width　      四角形の幅
//! @param[in] height       四角形の高さ
//! @param[in] color        四角形の色
//---------------------------------------------------------------------------
void Draw2DRect(int positionX, int positionY, int width, int height, const Color& color) NN_NOEXCEPT;


//---------------------------------------------------------------------------
//! @brief スクリーンに四角形テクスチャーを描画します。
//!
//! @param[in] positionX    四角形の左上頂点のx座標
//! @param[in] positionY    四角形の左上頂点のy座標
//! @param[in] width　      四角形の幅
//! @param[in] height       四角形の高さ
//! @param[in] handle
//---------------------------------------------------------------------------
void Draw2DRectTexture(int positionX, int positionY, int width, int height, TextureHandle handle) NN_NOEXCEPT;


//---------------------------------------------------------------------------
//! @brief デバッグ用の情報（レンダーターゲット、テクスチャーなど）を画面に描画します。
//!
//! @param[in] handle       LoadTestSuiteFromFile() からもらったハンドル
//---------------------------------------------------------------------------
void DrawTestDebugInformation(TestSuiteHandle handle) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief 画面の nn::gfx::CommandBuffer のインスタンスにアクセサ
//!
//! @return nn::gfx::CommandBuffer のインスタンスのポインタを返します
//---------------------------------------------------------------------------
nn::gfx::CommandBuffer* GetCommandBuffer() NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief PrimitiveRenderer のインスタンスにアクセサ
//!
//! @return nns::gfx::PrimitiveRenderer::Renderer のインスタンスのポインタを返します
//---------------------------------------------------------------------------
nns::gfx::PrimitiveRenderer::Renderer* GetPrimitiveRenderer() NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief フレームの終わり。これから表示されたものを画面で見えます。
//---------------------------------------------------------------------------
void EndFrame() NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief アプリのFPS
//!
//! @param[out] pOutMinFps      最高の1秒あたりのフレーム数を返します
//! @param[out] pOutMaxFps      最低の1秒あたりのフレーム数を返します
//! @param[out] pOutAverageFps  平均の1秒あたりのフレーム数を返します
//---------------------------------------------------------------------------
void GetFps(float* pOutMinFps, float *pOutMaxFps, float* pOutAverageFps) NN_NOEXCEPT;


} } } } // namespace nnt { namespace gfx { namespace util { namespace agingtest {
