﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>
#include <algorithm>

#include <gfx/demo.h>

#include "textureHandle.h"
#include "simpleParticle.h"
#include "gsParticle.h"

#if NN_GFX_IS_TARGET_GL
#include <GL/glew.h>
#endif

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

#include <nnt.h>
#include <nnt/nnt_Argument.h>

/*
 * Surface size.
 */
#define SURFACE_WIDTH ( DEMOColorBufferInfo.GetWidth() )
#define SURFACE_HEIGHT ( DEMOColorBufferInfo.GetHeight() )

/*
 * Shaders.
 */
int g_ShaderFileIdx = 0;
// Debug shader.
static const char* const DEBUG_SHADER[] =
{
    "shaders/gpuParticle/debugSimple",
    "shaders/gpuParticle/debugSimpleHlslcc",
};

// Shader objects.
static DEMOGfxPipeline s_DebugPipeline;

// Uniform locations.
typedef struct DEBUG_MATRICES
{
    Mtx44 viewMtx;
    Mtx44 projMtx;
} TDebugMatrices;
static DEMOGfxBuffer s_DebugMatrices[ 2 ];
static int s_DebugMatrixBuffer = 0;
static int s_DebugMatricesLoc = -1;

// Attribute streams.
enum DEBUG_ATTR
{
    DEBUG_VPOS = 0,
    DEBUG_VCOLOR,
    DEBUG_MAX_ATTR,
};

// Debug vertices.
typedef struct DEBUG_VERTEX
{
    f32 pos[3];
    u32 color;
} TDebugVertex;
static DEMOGfxBuffer s_DebugVtxBuf;

/*
 * Global variables.
 */
// Matrices
static Mtx44 s_ViewMatrix44;
static Mtx44 s_ProjMatrix44;
static nn::gfx::ViewportScissorState g_DebugViewportScissorState;
static void* g_pDebugViewportScissorData;

static const int NUM_DEBUG_VERTICES = 44;

// Camera attributes.
static Vec s_UpVector;
static Vec s_CamTarget;
static Vec s_CamPosition;

// Timing & performance.
static u64 s_PrevOSTime = 0;
static f32 s_DeltaOSTime = 0.0f;
#if NN_GFX_IS_TARGET_GX
static GX2CounterInfo* s_pPerfInfo;
static u32 s_PerfTime = 0;
static f32 s_GpuTime = 0.0f;
static f32 s_ShaderBusyVs;
static f32 s_ShaderBusyGs;
static f32 s_ShaderBusyPs;
#endif

// Menu.
enum MENU_ITEM
{
    MODE = 0,
    USE_GS,
    PARTICLE_NUM,
    RESET_CAM,
    ANIMATION_SPEED,
    PAUSE,
    HELP_STATE,

    MAX_MENU_ITEM,
};
static s32 s_CurrentItem = 0;
static PARTICLE_MODE s_CurrentMode = SNOW;
static bool s_ShowHelp = true;

// Animation attributes.
static bool s_AnimationState = true;
static f32 s_AnimationSpeed = 1.0f;

// Keys.
static bool s_ButtonUpPressed = false;
static bool s_ButtonDownPressed = false;
static bool s_ButtonAPressed = false;
static bool s_ButtonBPressed = false;

// Use GS flag.
static bool s_UseGS = true;

static s32 s_frameCount=0;

/*
 * Function prototypes.
 */
static void InitCamera(Mtx44 outView44, Mtx44 outProj44);
static void UpdateCamera(Mtx44 outView44);
static void ResetCamera();
static void InitScene();
static void TermScene();
static void DrawScene();
static void UpdatePad(f32 frameTime);

static void DrawGridLines();

static void InitPerformanceCounter();
static void TermPerformanceCounter();
static void BeginPerformanceCounter();
static void EndPerformanceCounter();

static void DrawMenu();


/*
 * Main function.
 */
//extern "C" void nnMain()
TEST(GfxGpuParticle, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    // Set random seed.
    DEMOSRand(1);

    // Initialize scene & particle system.
    s_frameCount = 0;
    s_PrevOSTime = 0;
    s_DeltaOSTime = 0.0f;
#if NN_GFX_IS_TARGET_GX
    s_PerfTime = 0;
    s_GpuTime = 0.0f;
    s_ShaderBusyVs = 0.0f;
    s_ShaderBusyGs = 0.0f;
    s_ShaderBusyPs = 0.0f;
#endif
    s_AnimationState = true;
    s_AnimationSpeed = 1.0f;

    InitParticle();
    InitScene();
    InitPerformanceCounter();
    InitTexture();
    InitSimpleParticle();
    InitGsParticle();

    // Main loop.
    while (DEMOIsRunning())
    {
        f32 update = 1.0f;

        u64 currentTime = OSGetTime();
        s_DeltaOSTime   = (f32)OSTicksToMilliseconds(currentTime - s_PrevOSTime) / 1000.0f;
        s_PrevOSTime = currentTime;

        UpdatePad(update);
        if (s_AnimationState)
            UpdateParticle(update * s_AnimationSpeed);

        DEMOGfxBeforeRender();

        BeginPerformanceCounter();

        DrawScene();
        if(s_frameCount>0)
        {
            DrawMenu();
        }

        s_frameCount++;

        EndPerformanceCounter();

        DEMOGfxDoneRender();
    }

    // Terminate scene & particle system.
    TermGsParticle();
    TermSimpleParticle();
    TermTexture();
    TermPerformanceCounter();
    TermScene();

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}



// Initialize view-projection matrix.
static void InitCamera(Mtx44 outView44, Mtx44 outProj44)
{
    // Matrices are in row major.
    f32 pers = 60.0f;
    f32 aspect = static_cast< f32 >(SURFACE_WIDTH) / static_cast< f32 >(SURFACE_HEIGHT);
    f32 znear = 10.0f;
    f32 zfar = 1000.0f;

    // Compute perspective matrix
    MTXPerspective(outProj44, pers, aspect, znear, zfar);

    ResetCamera();
    UpdateCamera(outView44);
}

// Update camera.
static void UpdateCamera(Mtx44 outView44)
{
    Mtx lookAtMatrix;
    MTXLookAt(lookAtMatrix, &s_CamPosition, &s_UpVector, &s_CamTarget);
    MTX34To44(lookAtMatrix, outView44);
}

// Reset camera.
static void ResetCamera()
{
    const Vec up = { 0.0f, 1.0f, 0.0f };
    const Vec target = { 0.0f, 25.0f, 0.0f };
    const Vec pos = { 0.0f, 25.0f, 75.0f };

    s_UpVector = up;
    s_CamTarget = target;
    s_CamPosition = pos;
}

// Initialize scene.
static void InitScene()
{
    // Intialize the pipeline
    s_DebugPipeline.SetDefaults();

    DEMOGfxLoadShadersFromFile( &s_DebugPipeline.shaders, 0, DEBUG_SHADER[g_ShaderFileIdx] );
    // Get uniform locations.
    s_DebugMatrices[ 0 ].Initialize( sizeof( TDebugMatrices ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    s_DebugMatrices[ 1 ].Initialize( sizeof( TDebugMatrices ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    s_DebugMatricesLoc = s_DebugPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_matrices" );

    // Init vertex attribute.
    DEMOGfxInitShaderAttribute( &s_DebugPipeline.shaders, "a_position", 0, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute( &s_DebugPipeline.shaders, "a_color", 0, 3 * sizeof( float ), nn::gfx::AttributeFormat_8_8_8_8_Unorm );

    // Init vertex buffer
    DEMOGfxInitShaderVertexBuffer( &s_DebugPipeline.shaders, 0, 3 * sizeof( float ) + 4 * sizeof( uint8_t ), 0 );

    // Create debug vertex buffer.
    s_DebugVtxBuf.Initialize( sizeof( TDebugVertex ) * NUM_DEBUG_VERTICES, NULL, nn::gfx::GpuAccess_VertexBuffer, 0 );

    TDebugVertex* pvtx = s_DebugVtxBuf.Map< TDebugVertex >();

    // Set-up grid vertices.
    for(s32 ii = 0; ii <= 10; ii++)
    {
        f32 x_grid, z_grid;

        x_grid = -100.0f + static_cast< f32 >(ii) * 20.0f;
        z_grid = -100.0f + static_cast< f32 >(ii) * 20.0f;

#if NN_GFX_IS_TARGET_GX
        u32 color = 0xccccccff;
#else
        u32 color = 0xffcccccc;
#endif

        pvtx[0].pos[0] = x_grid;
        pvtx[0].pos[1] = 0.0f;
        pvtx[0].pos[2] = -100.0f;
        pvtx[0].color = color;

        pvtx[1].pos[0] = x_grid;
        pvtx[1].pos[1] = 0.0f;
        pvtx[1].pos[2] = 100.0f;
        pvtx[1].color = color;

        pvtx[2].pos[0] = 100.0f;
        pvtx[2].pos[1] = 0.0f;
        pvtx[2].pos[2] = z_grid;
        pvtx[2].color = color;

        pvtx[3].pos[0] = -100.0f;
        pvtx[3].pos[1] = 0.0f;
        pvtx[3].pos[2] = z_grid;
        pvtx[3].color = color;

        pvtx += 4;
    }
    s_DebugVtxBuf.Unmap();

    // Initialize view & projection matrix.
    InitCamera(s_ViewMatrix44, s_ProjMatrix44);

    // Set smaller font size.
    DEMOFontSetGridSize(90, 35);

    DEMOGfxSetViewportScissorState( &g_DebugViewportScissorState, &g_pDebugViewportScissorData,
        0.0f, 0.0f, static_cast< float >( SURFACE_WIDTH ), static_cast< float >( SURFACE_HEIGHT ),
        0.0f, 1.0f, static_cast< float >( SURFACE_HEIGHT ), false );

    s_DebugPipeline.depthStencilStateInfo.SetDepthTestEnabled( false );
    s_DebugPipeline.depthStencilStateInfo.SetDepthWriteEnabled( false );

    s_DebugPipeline.blendTargetStateCount = 1;
    s_DebugPipeline.colorTargetStateCount = 1;
    s_DebugPipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    s_DebugPipeline.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );
    s_DebugPipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    s_DebugPipeline.blendTargetStateInfoArray[ 0 ].SetBlendEnabled( false );

    s_DebugPipeline.Initialize( &DEMODevice );
}

// Terminate scene.
static void TermScene()
{
    // Free vertex & pixel shader.
    s_DebugPipeline.Finalize( &DEMODevice );

    // Free buffers.
    s_DebugMatrices[ 0 ].Finalize();
    s_DebugMatrices[ 1 ].Finalize();

    // Special care needed due to MEM1 alloc
    s_DebugVtxBuf.buffer.Finalize( &DEMODevice );
    s_DebugVtxBuf.memPool->Finalize();
    DEMOGfxFreeMEM1(s_DebugVtxBuf.data);

    // Free Viewport
    g_DebugViewportScissorState.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( g_pDebugViewportScissorData );
}

// Render scene.
static void DrawScene()
{
    // Clear buffers.
    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    DEMOCommandBuffer.ClearColor( pCurrentScanBuffer, 0.2f, 0.2f, 0.2f, 0.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOGfxSetDefaultRenderTarget();

    // Set viewport & scissoring rectangle.
    DEMOCommandBuffer.SetViewportScissorState( &g_DebugViewportScissorState );

    // Draw grid lines.
    DrawGridLines();
    if (s_UseGS)
        DrawGsParticle(s_ViewMatrix44, s_ProjMatrix44);
    else
        DrawSimpleParticle(s_ViewMatrix44, s_ProjMatrix44);
}

// Process input pad.
static void UpdatePad(f32 frameTime)
{
    DEMOPadRead();
    u16 button = DEMOPadGetButton(0);
    u16 buttonPressed = DEMOPadGetButtonDown(0);

    s_ButtonUpPressed = buttonPressed & DEMO_PAD_BUTTON_UP ? true : false;
    s_ButtonDownPressed = buttonPressed & DEMO_PAD_BUTTON_DOWN ? true : false;
    s_ButtonAPressed = buttonPressed & DEMO_PAD_BUTTON_A ? true : false;
    s_ButtonBPressed = buttonPressed & DEMO_PAD_BUTTON_B ? true : false;

    // L-stick to move forward/backward/stride.
    {
        // Use L-trigger to move faster.
        f32 scale = (button & DEMO_PAD_TRIGGER_L ? 10.0f : 5.0f) * frameTime;
        f32 sx = static_cast< f32 >(DEMOPadGetStickX(0)) / 255.0f;
        f32 sy = static_cast< f32 >(DEMOPadGetStickY(0)) / 255.0f;

        Vec vec;
        Vec tempVec;
        tempVec.x = sx * scale;
        tempVec.y = 0.0f;
        tempVec.z = -sy * scale;

        Mtx44 inv;
        MTX44Inverse(s_ViewMatrix44, inv);
        MTX44MultVecSR(inv, &tempVec, &vec);
        Vec tempCamLoc = s_CamPosition;
        Vec tempCamTgt = s_CamTarget;
        VECAdd(&vec, &tempCamLoc, &s_CamPosition);
        VECAdd(&vec, &tempCamTgt, &s_CamTarget);

        UpdateCamera(s_ViewMatrix44);
    }

    // R-stick to rotate view.
    {
        // Use R-trigger to rotate faster.
        f32 scale = (button & DEMO_PAD_TRIGGER_R ? 4.0f : 2.0f) * frameTime;
        f32 sx = static_cast< f32 >(DEMOPadGetSubStickX(0)) / 255.0f;
        f32 sy = static_cast< f32 >(DEMOPadGetSubStickY(0)) / 255.0f;

        Vec eyev;
        Vec tempEyev;
        VECSubtract(&s_CamTarget, &s_CamPosition, &eyev);
        Vec wupv = { 0.0f, 1.0f, 0.0f };
        VECCrossProduct(&eyev, &s_UpVector, &tempEyev);
        VECNormalize(&tempEyev, &eyev);

        Mtx44 rot;
        Mtx44 rot0;
        Mtx44 rot1;
        MTX44RotAxisRad(rot0, &eyev, MTXDegToRad(sy * scale));
        MTX44RotAxisRad(rot1, &wupv, MTXDegToRad(-sx * scale));

        MTX44Concat(rot0, rot1, rot);

        Vec camv;
        Vec tempCamv;
        VECSubtract(&s_CamTarget, &s_CamPosition, &tempCamv);
        MTX44MultVecSR(rot, &tempCamv, &camv);

        VECAdd(&camv, &s_CamPosition, &s_CamTarget);
        Vec tempUp = s_UpVector;
        MTX44MultVecSR(rot, &tempUp, &s_UpVector);

        UpdateCamera(s_ViewMatrix44);
    }

    // Control pad up-down to select menu item.
    if (s_ButtonUpPressed)
        s_CurrentItem = s_CurrentItem == 0 ? (MAX_MENU_ITEM - 1) : (s_CurrentItem - 1);
    else if (s_ButtonDownPressed)
        s_CurrentItem = (s_CurrentItem + 1) % MAX_MENU_ITEM;

    // Handle selected menu item.
    switch (s_CurrentItem)
    {
    case MODE:
        // Particle mode.
        if (s_ButtonAPressed || s_ButtonBPressed)
        {
            switch (s_CurrentMode)
            {
            case SNOW:
                s_CurrentMode = SAKURA;
                break;
            case SAKURA:
                s_CurrentMode = MIST;
                break;
            case MIST:
                s_CurrentMode = SNOW;
                break;
            default:
                break;
            }

            if (s_UseGS)
                GsParticleMode(s_CurrentMode);
            else
                SimpleParticleMode(s_CurrentMode);
        }
        break;
    case USE_GS:
        // Use geometry shader.
        if (s_ButtonAPressed || s_ButtonBPressed)
        {
            s_UseGS = !s_UseGS;
            if (s_UseGS)
                GsParticleMode(s_CurrentMode);
            else
                SimpleParticleMode(s_CurrentMode);
        }
        break;
    case PARTICLE_NUM:
        // Increase/decrease number of particles.
        if (button & DEMO_PAD_BUTTON_A)
        {
            s32 add = 1;
            if (button & DEMO_PAD_TRIGGER_L)
                add = 100;
            else if (button & DEMO_PAD_TRIGGER_R)
                add = 1000;
            AddParticle(add);
        }
        else if (button & DEMO_PAD_BUTTON_B)
        {
            s32 add = -1;
            if (button & DEMO_PAD_TRIGGER_L)
                add = -100;
            else if (button & DEMO_PAD_TRIGGER_R)
                add = -1000;
            AddParticle(add);
        }
        break;
    case RESET_CAM:
        // Use geometry shader.
        if (s_ButtonAPressed || s_ButtonBPressed)
            ResetCamera();
        break;
    case ANIMATION_SPEED:
        // Animation speed.
        if (button & DEMO_PAD_BUTTON_A)
        {
            f32 add = 0.005f;
            if (button & DEMO_PAD_TRIGGER_L)
                add = 0.01f;
            else if (button & DEMO_PAD_TRIGGER_R)
                add = 0.02f;
            s_AnimationSpeed += add;
            if (s_AnimationSpeed > 1.0f)
                s_AnimationSpeed = 1.0f;
        }
        else if (button & DEMO_PAD_BUTTON_B)
        {
            f32 add = -0.005f;
            if (button & DEMO_PAD_TRIGGER_L)
                add = -0.01f;
            else if (button & DEMO_PAD_TRIGGER_R)
                add = -0.02f;
            s_AnimationSpeed += add;
            if (s_AnimationSpeed < 0.005f)
                s_AnimationSpeed = 0.005f;
        }
        break;
    case PAUSE:
        // Animation state (pause animation or resume animation).
        if (s_ButtonAPressed || s_ButtonBPressed)
        {
            s_AnimationState = !s_AnimationState;
        }
        break;
    case HELP_STATE:
        // Show/hide help message.
        if (s_ButtonAPressed || s_ButtonBPressed)
            s_ShowHelp = !s_ShowHelp;
        break;
    default:
        break;
    }
} // NOLINT(impl/function_size)

#if NN_GFX_IS_TARGET_GL
static void SetLineWidth( const void* )
{
    glLineWidth( 2.0f );
}
#endif

// Draw grid lines.
static void DrawGridLines()
{
    // Switch uniform buffers
    s_DebugMatrixBuffer ^= 1;

    // Set uniforms.
    TDebugMatrices* pMatrices = s_DebugMatrices[ s_DebugMatrixBuffer ].Map< TDebugMatrices >();
    memcpy( pMatrices->viewMtx, s_ViewMatrix44, sizeof( s_ViewMatrix44 ) );
    memcpy( pMatrices->projMtx, s_ProjMatrix44, sizeof( s_ProjMatrix44 ) );

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMatrices, sizeof( *pMatrices ) );
#endif
    s_DebugMatrices[ s_DebugMatrixBuffer ].Unmap();

    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ConstantBuffer );

    // Set uniforms
    DEMOCommandBuffer.SetConstantBuffer( s_DebugMatricesLoc,
        nn::gfx::ShaderStage_Vertex, s_DebugMatrices[ s_DebugMatrixBuffer ].gpuAddress, s_DebugMatrices[ s_DebugMatrixBuffer ].size );

    // Set vertex attributes.
    DEMOCommandBuffer.SetVertexBuffer( 0, s_DebugVtxBuf.gpuAddress, sizeof( TDebugVertex ), s_DebugVtxBuf.size );

    // Set render states.
    DEMOCommandBuffer.SetPipeline( &s_DebugPipeline.pipeline );

#if NN_GFX_IS_TARGET_GX
    GX2SetLineWidth(2);
#elif NN_GFX_IS_TARGET_GL
    DEMOCommandBuffer.Gl4SetUserCommand( SetLineWidth, NULL );
#elif NN_GFX_IS_TARGET_NVN
    nvnCommandBufferSetLineWidth( DEMOCommandBuffer.ToData()->pNvnCommandBuffer, 2.0f );
#endif

    // Draw grid lines.
    DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_LineList, NUM_DEBUG_VERTICES, 0 );
}

// Initialize performance counter.
static void InitPerformanceCounter()
{
#if NN_GFX_IS_TARGET_GX
    s_pPerfInfo = static_cast< GX2CounterInfo* >(DEMOGfxAllocMEM2(sizeof(GX2CounterInfo),
                                                                  GX2_DEFAULT_BUFFER_ALIGNMENT));
#endif
}

// Terminate performance counter.
static void TermPerformanceCounter()
{
#if NN_GFX_IS_TARGET_GX
    DEMOGfxFreeMEM2(s_pPerfInfo);
#endif
}

// Begin measuring graphic performance.
static void BeginPerformanceCounter()
{
#if NN_GFX_IS_TARGET_GX
    if (s_PerfTime == 0)
    {
        GX2Boolean counter = GX2_FALSE;

        GX2ResetCounterInfo(s_pPerfInfo);
        counter = GX2InitCounterInfo(s_pPerfInfo, GX2_COUNTER_GRBM_0, GX2_STAT_GRBM_GUI_ACTIVE);
        if (GX2_TRUE != counter)
        {
            ASSERT(false && "Cannot measure GX2_STAT_GRBM_GUI_ACTIVE on GX2_COUNTER_GRBM_0");
        }
        counter = GX2InitCounterInfo(s_pPerfInfo, GX2_COUNTER_SQ_0, _GX2StatId(GX2_STAT_SQ_THREAD_LEVEL_PER_TYPE | GX2_SQ_VS | GX2_SQ_ES));
        if (GX2_TRUE != counter)
        {
            ASSERT(false && "Cannot measure (GX2_STAT_SQ_THREAD_LEVEL_PER_TYPE | GX2_SQ_VS | GX2_SQ_ES) on GX2_COUNTER_SQ_0");
        }
        counter = GX2InitCounterInfo(s_pPerfInfo, GX2_COUNTER_SQ_1, _GX2StatId(GX2_STAT_SQ_THREAD_LEVEL_PER_TYPE | GX2_SQ_GS));
        if (GX2_TRUE != counter)
        {
            ASSERT(false && "Cannot measure (GX2_STAT_SQ_THREAD_LEVEL_PER_TYPE | GX2_SQ_GS) on GX2_COUNTER_SQ_1");
        }
        counter = GX2InitCounterInfo(s_pPerfInfo, GX2_COUNTER_SQ_2, _GX2StatId(GX2_STAT_SQ_THREAD_LEVEL_PER_TYPE | GX2_SQ_PS));
        if (GX2_TRUE != counter)
        {
            ASSERT(false && "Cannot measure (GX2_STAT_SQ_THREAD_LEVEL_PER_TYPE | GX2_SQ_PS) on GX2_COUNTER_SQ_2");
        }
        counter = GX2InitCounterInfo(s_pPerfInfo, GX2_COUNTER_SQ_3, _GX2StatId(GX2_STAT_SQ_THREAD_LEVEL_PER_TYPE | GX2_SQ_VS | GX2_SQ_ES | GX2_SQ_GS | GX2_SQ_PS));
        if (GX2_TRUE != counter)
        {
            ASSERT(false && "Cannot measure (GX2_STAT_SQ_THREAD_LEVEL_PER_TYPE | GX2_SQ_VS | GX2_SQ_ES | GX2_SQ_GS | GX2_SQ_PS | GX2_SQ_ES) on GX2_COUNTER_SQ_3");
        }

        GX2SetCounterInfo(s_pPerfInfo);
        GX2ResetCounters(s_pPerfInfo);
        GX2StartCounters(s_pPerfInfo);
    }
    s_PerfTime = (s_PerfTime + 1) % 10;
#endif
}

// End measuring graphic performance.
static void EndPerformanceCounter()
{
#if NN_GFX_IS_TARGET_GX
    if (s_PerfTime == 0)
    {
        u64 gpu_time = GX2_INVALID_COUNTER_VALUE_U64;
        u64 sq_es_vs = GX2_INVALID_COUNTER_VALUE_U64;
        u64 sq_gs = GX2_INVALID_COUNTER_VALUE_U64;
        u64 sq_ps = GX2_INVALID_COUNTER_VALUE_U64;
        u64 sq_es_vs_gs_ps = GX2_INVALID_COUNTER_VALUE_U64;

        GX2StopCounters(s_pPerfInfo);
        GX2SampleCounters(s_pPerfInfo);

        // Emulate DrawDone
        DEMOCommandBuffer.End();
        DEMOQueue.ExecuteCommand( &DEMOCommandBuffer, NULL );
        DEMOQueue.Sync();
        DEMOCommandBuffer.Begin();

        gpu_time = GX2GetCounterResult(s_pPerfInfo, GX2_STAT_GRBM_GUI_ACTIVE);
        sq_es_vs = GX2GetCounterResult(s_pPerfInfo, _GX2StatId(GX2_STAT_SQ_THREAD_LEVEL_PER_TYPE | GX2_SQ_VS | GX2_SQ_ES));
        sq_gs = GX2GetCounterResult(s_pPerfInfo, _GX2StatId(GX2_STAT_SQ_THREAD_LEVEL_PER_TYPE | GX2_SQ_GS));
        sq_ps = GX2GetCounterResult(s_pPerfInfo, _GX2StatId(GX2_STAT_SQ_THREAD_LEVEL_PER_TYPE | GX2_SQ_PS));
        sq_es_vs_gs_ps = GX2GetCounterResult(s_pPerfInfo, _GX2StatId(GX2_STAT_SQ_THREAD_LEVEL_PER_TYPE | GX2_SQ_VS | GX2_SQ_ES | GX2_SQ_GS | GX2_SQ_PS));

        s_GpuTime = static_cast< f32 >(gpu_time) / 550000.0f / 10; // average milliseconds over 10 frames

        if(sq_es_vs_gs_ps)
        {
            s_ShaderBusyVs = 100.0f * f32(sq_es_vs / f64(sq_es_vs_gs_ps));
            s_ShaderBusyGs = 100.0f * f32(sq_gs / f64(sq_es_vs_gs_ps));
            s_ShaderBusyPs = 100.0f * f32(sq_ps / f64(sq_es_vs_gs_ps));
        }
        else
        {
            s_ShaderBusyVs = 0.0f;
            s_ShaderBusyGs = 0.0f;
            s_ShaderBusyPs = 0.0f;
        }
    }
#endif
}

// Draw menu.
static void DrawMenu()
{
    u32 line = 4;

    const char* mode[] = {
        "Snow",
        "Sakura",
        "Mist"
        };

    DEMOFontPrintf(3, 2, "Frame rate: %.1f fps", 1.0f / std::max(1.0e-7f, s_DeltaOSTime));

    DEMOFontPrintf(3, static_cast< f32 >( line + s_CurrentItem ), ">");

    DEMOFontPrintf(4, static_cast< f32 >( line++ ), " Particle demo: %s", mode[s_CurrentMode]);
    DEMOFontPrintf(4, static_cast< f32 >( line++ ), " Use geometry shader: %s", s_UseGS ? "YES" : "NO");
    DEMOFontPrintf(4, static_cast< f32 >( line++ ), " Number of particles: %d", currentParticleNum);
    DEMOFontPrintf(4, static_cast< f32 >( line++ ), " Reset camera");
    DEMOFontPrintf(4, static_cast< f32 >( line++ ), " Animation speed: %.3f", s_AnimationSpeed);
    DEMOFontPrintf(4, static_cast< f32 >( line++ ), " %s", s_AnimationState ? "Pause" : "Resume");
    DEMOFontPrintf(4, static_cast< f32 >( line++ ), " %s", s_ShowHelp ? "Hide help" : "Show help");
    line++;

    if (s_ShowHelp)
    {
        DEMOFontPrintf(3, static_cast< f32 >( line++ ), "Controller:");
        DEMOFontPrintf(3, static_cast< f32 >( line++ ), "- Use L/R-stick to move/rotate camera.");
        DEMOFontPrintf(3, static_cast< f32 >( line++ ), "- Hold L/R-button to move/rotate faster.");
        DEMOFontPrintf(3, static_cast< f32 >( line++ ), "- Use control pad up/down to select an item.");
        DEMOFontPrintf(3, static_cast< f32 >( line++ ), "- Use a/b to change value.");
        DEMOFontPrintf(3, static_cast< f32 >( line++ ), "- L/R-button with a/b will change value gradually.");
    }

    line = 26;

    DEMOFontPrintf(3, static_cast< f32 >( line++ ), "Vertex buffer:");
    DEMOFontPrintf(3, static_cast< f32 >( line++ ), "   %d bytes",
                   s_UseGS ? GsParticleVtxBufSize() : SimpleParticleVtxBufSize());
#if NN_GFX_IS_TARGET_GX
    line++;
    DEMOFontPrintf(3, static_cast< f32 >( line++ ), "VS: %.3f%%", s_ShaderBusyVs);
    DEMOFontPrintf(3, static_cast< f32 >( line++ ), "GS: %.3f%%", s_ShaderBusyGs);
    DEMOFontPrintf(3, static_cast< f32 >( line++ ), "PS: %.3f%%", s_ShaderBusyPs);
    DEMOFontPrintf(3, static_cast< f32 >( line++ ), "GPU time: %.3fms", s_GpuTime);
#endif
}
