﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <cmath>

#include <algorithm>

#include <gfx/demo.h>
#include <nnt.h>
#include <nnt/nnt_Argument.h>

#if NN_GFX_IS_TARGET_GX
#include <cafe/gx2.h>
#endif

////////////////////////////////////////////////////
//
// Assets data, types and interface for demos
//
////////////////////////////////////////////////////
#define SURFACE_WIDTH  (DEMOColorBufferInfo.GetWidth())
#define SURFACE_HEIGHT  (DEMOColorBufferInfo.GetHeight())

#define REDUCTION_WIDTH     SURFACE_WIDTH / 2
#define REDUCTION_HEIGHT    SURFACE_HEIGHT / 2

static int g_ShaderFileIdx = 0;
static const char * const ShaderList[][4] =
{
    {
        "shaders/godray/background",
        "shaders/godray/mask",
        "shaders/godray/godray",
        "shaders/godray/copy",
    },
    {
        "shaders/godray/backgroundHlslcc",
        "shaders/godray/maskHlslcc",
        "shaders/godray/godrayHlslcc",
        "shaders/godray/copyHlslcc",
    },
};

#if NN_GFX_IS_TARGET_GX || NN_GFX_IS_TARGET_D3D

#define FLIP(y) (y)

#else

#ifndef DEMO_BUFFER_ALIGN
// this has to be a #define, not a static const, because __declspec(align) only accepts literal constants
#define DEMO_BUFFER_ALIGN 4096 // NOLINT(readability/define)
#endif

#define FLIPY(y) (-(y))

#endif

static const char * const TextureList[] =
{
    "textures/godray/cubemap"
};

#if NN_GFX_IS_TARGET_GX || NN_GFX_IS_TARGET_D3D
static f32 s_quadPT[] =
{
    -1.0f,  1.0f,
    0.0f,  0.0f,

    -1.0f, -1.0f,
    0.0f,  1.0f,

    1.0f,  1.0f,
    1.0f,  0.0f,

    1.0f, -1.0f,
    1.0f,  1.0f,
};
#else
static f32 s_quadPT[] =
{
    -1.0f,  FLIPY(1.0f),
    0.0f,  0.0f,

    -1.0f, FLIPY(-1.0f),
    0.0f,  1.0f,

    1.0f,  FLIPY(1.0f),
    1.0f,  0.0f,

    1.0f, FLIPY(-1.0f),
    1.0f,  1.0f,
};
#endif

#if NN_GFX_IS_TARGET_GX || NN_GFX_IS_TARGET_D3D
static f32 s_quadRPT[] =
{
    -1.0f,  1.0f,
    1.0f,  0.0f,
    -1.0f, -1.0f,
    1.0f,  1.0f,
    1.0f,  1.0f,
    0.0f,  0.0f,
    1.0f, -1.0f,
    0.0f,  1.0f,
};
#else
static f32 s_quadRPT[] =
{
    -1.0f, FLIPY(1.0f),
    1.0f,  0.0f,    // v_position.x = a_texcoord.x * 2 - 1 = 1  // v_postion.y = 1.0 - 2*a_texcoord.y = 1.0

    -1.0f, FLIPY(-1.0f),
    1.0f,  1.0f,

    1.0f,  FLIPY(1.0f),
    0.0f,  0.0f,

    1.0f,  FLIPY(-1.0f),
    0.0f,  1.0f,
};
#endif

#if NN_GFX_IS_TARGET_GX || NN_GFX_IS_TARGET_D3D
static f32 s_quadP[] =
{
    -1.0f,  1.0f,
    -1.0f, -1.0f,
    1.0f,  1.0f,
    1.0f, -1.0f,
};
#else
static f32 s_quadP[] =
{
    -1.0f,  1.0f,
    -1.0f, -1.0f,
    1.0f,  1.0f,
    1.0f, -1.0f,
};
#endif

static bool s_firstFrame;

//static const u32 QUAD_P_SIZE  = sizeof(s_quadP);
//static const u32 QUAD_PT_SIZE = sizeof(s_quadPT);
static const u32 POS_OFFSET = 0;
static const u32 TEXCOORD_OFFSET = sizeof(f32) * 2;

static nn::gfx::ViewportScissorState frameViewport;
static nn::gfx::ViewportScissorState reductionViewport;

static nn::gfx::Texture reductionDepthStencilTexture;
static nn::gfx::DepthStencilView reductionDepthStencilView;
static DEMOGfxMemPool* reductionDepthStencilMemPool;

static DEMOGfxBuffer     attrPositionBuffer;
static DEMOGfxBuffer     attrPositionTextureBuffer;
static DEMOGfxBuffer     attrResolvedPositionTextureBuffer;

static DEMOGfxShader   g_BackGroundShader;
static u32             g_BackGroundCubemapLocation;
static DEMOGfxTexture      backGroundTexture;
static nn::gfx::Sampler      backGroundSampler;
static nn::gfx::DescriptorSlot      backGroundSamplerSlot;

static nn::gfx::Texture      resolvedTexture;
static nn::gfx::TextureInfo  resolvedTextureInfo;
static nn::gfx::TextureView  resolvedTextureView;
static nn::gfx::DescriptorSlot  resolvedTextureSlot;
static nn::gfx::ColorTargetView resolvedColorTargetView;
static DEMOGfxMemPool*          resolvedTexturePool = NULL;

static nn::gfx::Sampler      linearSampler;
static nn::gfx::Sampler      pointSampler;
static nn::gfx::DescriptorSlot      linearSamplerSlot;
static nn::gfx::DescriptorSlot      pointSamplerSlot;

static DEMOGfxShader   g_MaskShader;
static u32             g_MaskSourceTextureLocation;

static nn::gfx::ColorTargetView  maskColorTargetView;
static nn::gfx::Texture      maskTextureBuffer;
static nn::gfx::TextureInfo  maskTextureBufferInfo;
static nn::gfx::TextureView  maskTextureBufferView;
static nn::gfx::DescriptorSlot  maskTextureBufferSlot;
static DEMOGfxMemPool*       maskTextureBufferPool = NULL;

static nn::gfx::ColorTargetView  feedBackColorBuffer[2];
static nn::gfx::Texture          feedBackTexture[2];
static nn::gfx::TextureInfo      feedBackTextureInfo[2];
static nn::gfx::TextureView      feedBackTextureView[2];
static nn::gfx::DescriptorSlot      feedBackTextureSlot[2];
static DEMOGfxMemPool*           feedBackTexturePool[ 2 ] = { 0 };

static DEMOGfxShader   g_GodrayShader;
static u32             g_GodraySourceTextureLocation;

static DEMOGfxShader   g_CopyShader;
static u32             g_CopySourceTextureLocation;

static Vec g_objPt = { -0.975f, 0.200f, -0.094f };
static Vec g_camUp = { 0.0f, 1.0f, 0.0f };
static Vec g_camLoc = { 0.0f, 0.0f, 0.0f };
static Vec g_sunDirection = { -0.903f, 0.347f, -0.254f };
static Mtx44 g_vpMtx;
static Mtx44 g_projMtx;
static Mtx44 g_viewMtx;

static f32             g_godrayDecay = 0.985f;
static u32             g_godrayLoop  = 3;
static const u32       MAX_GODRAY_LOOP = 8;

struct VpiDir
{
    Mtx44   u_viewProjI;
    float            u_direction[3];
};
static DEMOGfxBuffer ub_vpiBuffer;

static int             ub_backgroundSlot;  // slot for background pixel shader constant buffer
static int             ub_maskSlot;  // slot for mask pixel shader constant buffer

struct UB_Godray
{
    f32 u_godrayParams[16];
    f32 u_godrayCenter[4];
};
static int             ub_godraySlot;  // slot for godray pixel shader constant buffer
static DEMOGfxBuffer   ub_godrayBuffer[MAX_GODRAY_LOOP];

static nn::gfx::Pipeline backgroundPipeline;
static nn::gfx::Pipeline maskPipeline;
static nn::gfx::Pipeline godrayPipeline;
static nn::gfx::Pipeline copyPipeline;
static nn::gfx::Pipeline copyBlendPipeline;

static nn::gfx::VertexBufferStateInfo backgroundVertexBufferStateInfo;
static nn::gfx::VertexBufferStateInfo maskVertexBufferStateInfo;
static nn::gfx::VertexBufferStateInfo godrayVertexBufferStateInfo;
static nn::gfx::VertexBufferStateInfo copyVertexBufferStateInfo;

////////////////////////////////////////////////////
//
// Prototypes
//
////////////////////////////////////////////////////

static void InitScene(void);
static void InitBackgroundShader(void);
static void InitMaskShader(void);
static void InitGodrayShader(void);
static void InitCopyShader(void);

static void UpdateScene(void);
static void DrawScene(f32 dt);
static void FreeScene(void);

static void CameraInit(Mtx44* resultProjMtx44, Mtx44* resultViewMtx44);

////////////////////////////////////////////////////
//
// Functions
//
////////////////////////////////////////////////////

// Initialize a constant buffer object
static void InitConstBuffer( DEMOGfxBuffer& constantBuffer, size_t size )
{
    constantBuffer.Initialize(size, NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);
}

static nn::gfx::BlendTargetStateInfo defaultBlendTargetInfo[1];

// Initialize a pipeline object
static void InitPipeline( nn::gfx::Pipeline& pipeline, DEMOGfxShader *shader, nn::gfx::VertexBufferStateInfo* bufferStateInfo, bool blendEnable )
{
    nn::gfx::Pipeline::InfoType info;
    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;

    bool depthEnable = false;

    info.SetDefault();
    info.SetShaderPtr(shader->GetShader());

    rasterizerStateInfo.SetDefault();
    rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_None );
    rasterizerStateInfo.SetScissorEnabled(true);

    depthStencilStateInfo.SetDefault();
    depthStencilStateInfo.SetDepthTestEnabled( depthEnable );
    depthStencilStateInfo.SetDepthWriteEnabled( depthEnable );

    vertexStateInfo.SetDefault();
    vertexStateInfo.SetVertexAttributeStateInfoArray( shader->vertexAttributes, shader->attribCount );
    vertexStateInfo.SetVertexBufferStateInfoArray( bufferStateInfo, 1 );

    blendStateInfo.SetDefault();

    {
        defaultBlendTargetInfo[0].SetDefault();
        defaultBlendTargetInfo[0].SetBlendEnabled( blendEnable );
        defaultBlendTargetInfo[0].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
        defaultBlendTargetInfo[0].SetSourceColorBlendFactor( nn::gfx::BlendFactor_One );
        defaultBlendTargetInfo[0].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );

        defaultBlendTargetInfo[0].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );
        defaultBlendTargetInfo[0].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_One );
        defaultBlendTargetInfo[0].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    }
    blendStateInfo.SetBlendTargetStateInfoArray( defaultBlendTargetInfo, 1 );

    renderTargetStateInfo.SetDefault();

    info.SetBlendStateInfo(&blendStateInfo);
    info.SetDepthStencilStateInfo(&depthStencilStateInfo);
    info.SetRasterizerStateInfo(&rasterizerStateInfo);
    info.SetVertexStateInfo(&vertexStateInfo);
    info.SetRenderTargetStateInfo(&renderTargetStateInfo);

    size_t dataSize = nn::gfx::Pipeline::GetRequiredMemorySize( info );
    void* data = DEMOGfxAllocMEM2( dataSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    pipeline.SetMemory( data, dataSize );
    pipeline.Initialize( &DEMODevice, info );
}

static void InitViewport( nn::gfx::ViewportScissorState& viewport, int width, int height )
{
    void *ptr = NULL;
    DEMOGfxSetViewportScissorState( &viewport, &ptr,
        0.0f, 0.0f, static_cast< float >( width ), static_cast< float >( height ),
        0.0f, 1.0f, static_cast< float >( height ), false );
}

static void InitTextureBuffer( nn::gfx::Texture& buffer, nn::gfx::TextureInfo& bufferInfo, nn::gfx::TextureView& bufferView,
    nn::gfx::DescriptorSlot* pSlot, DEMOGfxMemPool** ppPool, int width, int height, nn::gfx::ImageFormat fmt, nn::gfx::TileMode tileMode )
{
    NN_UNUSED( bufferInfo );
    NN_UNUSED( tileMode );
    DEMOGfxSetupTextureBuffer( &buffer, &bufferView, pSlot, NULL, NULL, ppPool, width, height, 1, 1, nn::gfx::ImageDimension_2d, fmt, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );
}

static void InitSamplers()
{
    DEMOGfxInitSampler( &backGroundSampler, &backGroundSamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge, nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint, nn::gfx::ComparisonFunction_Always );
    DEMOGfxInitSampler( &linearSampler, &linearSamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge, nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint, nn::gfx::ComparisonFunction_Always );
    DEMOGfxInitSampler( &pointSampler, &pointSamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge, nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );
}

static void InitVertexBuffers()
{
    attrPositionBuffer.Initialize(sizeof(s_quadP), s_quadP, nn::gfx::GpuAccess_VertexBuffer, 0);
    attrPositionTextureBuffer.Initialize(sizeof(s_quadPT), s_quadPT, nn::gfx::GpuAccess_VertexBuffer, 0);
    attrResolvedPositionTextureBuffer.Initialize(sizeof(s_quadRPT), s_quadRPT, nn::gfx::GpuAccess_VertexBuffer, 0);
}

// Initialization
static void InitScene()
{
    nn::gfx::ImageFormat imgFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb;
    InitViewport( frameViewport, SURFACE_WIDTH, SURFACE_HEIGHT );
    InitViewport( reductionViewport, REDUCTION_WIDTH, REDUCTION_HEIGHT );

    //=============================================================================================
    //Vertex
    InitVertexBuffers();

    //=============================================================================================
    //Texture
    if ( !backGroundTexture.Initialize( TextureList[ 0 ] ) )
    {
        ASSERT(!"Error reading texture file");
    }

    // set up texture buffer
    InitTextureBuffer( resolvedTexture, resolvedTextureInfo, resolvedTextureView, &resolvedTextureSlot, &resolvedTexturePool, SURFACE_WIDTH, SURFACE_HEIGHT, imgFormat, nn::gfx::TileMode_Optimal );
    DEMOGfxSetupColorView( &resolvedTexture, &resolvedColorTargetView, nn::gfx::ImageDimension_2d, imgFormat );

    // set up mask texture
    InitTextureBuffer( maskTextureBuffer, maskTextureBufferInfo, maskTextureBufferView, &maskTextureBufferSlot, &maskTextureBufferPool, REDUCTION_WIDTH, REDUCTION_HEIGHT, nn::gfx::ImageFormat_R16_G16_B16_A16_Float, nn::gfx::TileMode_Optimal );
    DEMOGfxSetupColorView( &maskTextureBuffer, &maskColorTargetView, nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R16_G16_B16_A16_Float );

    // set up dummy depth buffer
    {
        DEMOGfxSetupTextureBuffer( &reductionDepthStencilTexture, NULL, NULL, NULL, &reductionDepthStencilView, &reductionDepthStencilMemPool, REDUCTION_WIDTH, REDUCTION_HEIGHT, 1, 1,
            nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_D16_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );
    }

    for (u32 i = 0;i < 2;++i)
    {
        InitTextureBuffer( feedBackTexture[i], feedBackTextureInfo[i], feedBackTextureView[i], &feedBackTextureSlot[i], &feedBackTexturePool[i], REDUCTION_WIDTH, REDUCTION_HEIGHT, nn::gfx::ImageFormat_R16_G16_B16_A16_Float, nn::gfx::TileMode_Optimal );
        DEMOGfxSetupColorView( &feedBackTexture[ i ], &feedBackColorBuffer[ i ], nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R16_G16_B16_A16_Float );
    }

    //=============================================================================================
    //Sampler
    InitSamplers();

    // Init shaders
    InitBackgroundShader();
    InitMaskShader();
    InitGodrayShader();
    InitCopyShader();
}

    //=============================================================================================
    //BackGround
void InitBackgroundShader()
{
    DEMOGfxLoadShadersFromFile(&g_BackGroundShader, 0, ShaderList[g_ShaderFileIdx][0]);
    DEMOGfxInitShaderAttribute(&g_BackGroundShader,
                               "a_position",
                               0,
                               POS_OFFSET,
                               nn::gfx::AttributeFormat_32_32_Float);
    g_BackGroundCubemapLocation = (u32)g_BackGroundShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );

    ub_backgroundSlot = g_BackGroundShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_background" );
    InitConstBuffer( ub_vpiBuffer, sizeof(VpiDir) );
    backgroundVertexBufferStateInfo.SetDefault();
    backgroundVertexBufferStateInfo.SetStride( sizeof(float) * 2 );
    InitPipeline( backgroundPipeline, &g_BackGroundShader, &backgroundVertexBufferStateInfo, false );
}
    //=============================================================================================
    //Mask
static void InitMaskShader()
{
    DEMOGfxLoadShadersFromFile(&g_MaskShader, 0, ShaderList[g_ShaderFileIdx][1]);
    DEMOGfxInitShaderAttribute(&g_MaskShader,
                               "a_position",
                               0,
                               POS_OFFSET,
                               nn::gfx::AttributeFormat_32_32_Float);

    DEMOGfxInitShaderAttribute(&g_MaskShader,
                               "a_texcoord",
                               0,
                               TEXCOORD_OFFSET,
                               nn::gfx::AttributeFormat_32_32_Float);

    g_MaskSourceTextureLocation   = (u32)g_MaskShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );

    ub_maskSlot = g_MaskShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_mask" );
    maskVertexBufferStateInfo.SetDefault();
    maskVertexBufferStateInfo.SetStride( sizeof(float) * 4 );
    InitPipeline( maskPipeline, &g_MaskShader, &maskVertexBufferStateInfo, false );
}
    //=============================================================================================
    //Godray

static void InitGodrayShader()
{
    DEMOGfxLoadShadersFromFile(&g_GodrayShader, 0, ShaderList[g_ShaderFileIdx][2]);

    DEMOGfxInitShaderAttribute(&g_GodrayShader,
                               "a_position",
                               0,
                               POS_OFFSET,
                               nn::gfx::AttributeFormat_32_32_Float);

    DEMOGfxInitShaderAttribute(&g_GodrayShader,
                               "a_texcoord",
                               0,
                               TEXCOORD_OFFSET,
                               nn::gfx::AttributeFormat_32_32_Float);

    g_GodraySourceTextureLocation = (u32)g_GodrayShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );
    ub_godraySlot = g_GodrayShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_godray" );
    for (int i = 0; i < MAX_GODRAY_LOOP; i++)
    {
        InitConstBuffer( ub_godrayBuffer[i], sizeof(UB_Godray) );
    }
    godrayVertexBufferStateInfo.SetDefault();
    godrayVertexBufferStateInfo.SetStride( sizeof(float) * 4 );
    InitPipeline( godrayPipeline, &g_GodrayShader, &godrayVertexBufferStateInfo, false );
}

    //=============================================================================================
    //Copy
static void InitCopyShader()
{
    DEMOGfxLoadShadersFromFile(&g_CopyShader, 0, ShaderList[g_ShaderFileIdx][3]);

    DEMOGfxInitShaderAttribute(&g_CopyShader,
                               "a_position",
                               0,
                               POS_OFFSET,
                               nn::gfx::AttributeFormat_32_32_Float);

    DEMOGfxInitShaderAttribute(&g_CopyShader,
                               "a_texcoord",
                               0,
                               TEXCOORD_OFFSET,
                               nn::gfx::AttributeFormat_32_32_Float);

    g_CopySourceTextureLocation = g_CopyShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );

    copyVertexBufferStateInfo.SetDefault();
    copyVertexBufferStateInfo.SetStride(sizeof(float) * 4);

    InitPipeline( copyPipeline, &g_CopyShader, &copyVertexBufferStateInfo, false );
    InitPipeline( copyBlendPipeline, &g_CopyShader, &copyVertexBufferStateInfo, true );
}

// Update camera
static void CameraInit(Mtx44* resultProjMtx44, Mtx44* resultViewMtx44)
{
    Mtx   lookAtMtx34;

    f32   pers = 50.0f;
    f32 aspect = ( f32 ) SURFACE_WIDTH / ( f32 ) SURFACE_HEIGHT;
    f32  znear = 0.1f;
    f32   zfar = 10000.0f;

    MTXPerspective( *resultProjMtx44, pers, aspect, znear, zfar );

    MTXLookAt( lookAtMtx34, &g_camLoc, &g_camUp, &g_objPt );
    MTX34To44( lookAtMtx34, *resultViewMtx44 );

}

static void HandleRightStick()
{
    // R Stick rotates the camera
    f32 sx = (f32)DEMOPadGetSubStickX(0) / 255.0f;
    f32 sy = (f32)DEMOPadGetSubStickY(0) / 255.0f;

    Vec eyev, tempEyev;
    VECSubtract( &g_objPt, &g_camLoc, &eyev );

    Vec wupv =  { 0.0f, 1.0f, 0.0f };
    VECCrossProduct( &eyev, &g_camUp, &tempEyev );
    VECNormalize( &tempEyev, &eyev );

    Mtx44 rot, rot0, rot1;
    MTX44RotAxisRad( rot0, &eyev, MTXDegToRad( sy * 5.0f ) );
    MTX44RotAxisRad( rot1, &wupv, MTXDegToRad( sx * 5.0f ) );

    MTX44Concat( rot0, rot1, rot );

    Vec camv, tempCamv;
    VECSubtract( &g_objPt, &g_camLoc, &tempCamv );
    MTX44MultVecSR(rot, &tempCamv, &camv);

    VECAdd( &camv, &g_camLoc, &g_objPt );
    Vec tempUp = g_camUp;
    MTX44MultVecSR(rot, &tempUp, &g_camUp);

    CameraInit( &g_projMtx, &g_viewMtx);
}

static void HandleLeftStick()
{
    // L Stick rotates the sun
    f32 sx = (f32)DEMOPadGetStickX(0) / 255.0f;
    f32 sy = (f32)DEMOPadGetStickY(0) / 255.0f;

    Vec eyev, tempEyev;
    VECSubtract( &g_objPt, &g_camLoc, &eyev );
    VECCrossProduct( &eyev, &g_camUp, &tempEyev );
    VECNormalize( &tempEyev, &eyev );

    Mtx44 rot, rot0, rot1;

    MTX44RotAxisRad( rot0, &g_camUp, MTXDegToRad( sx * 5.0f ) );
    MTX44RotAxisRad( rot1, &eyev, MTXDegToRad( sy * 5.0f ) );

    MTX44Concat( rot0, rot1, rot );

    Vec tmp;
    MTX44MultVecSR(rot, &g_sunDirection, &tmp);

    memcpy( &g_sunDirection, &tmp, sizeof(tmp));
}

// Update processing
static void UpdateScene()
{
    DEMOPadRead();
    u16 button = DEMOPadGetButton(0);
    u16 buttonDown = DEMOPadGetButtonDown(0);

    if (DEMO_PAD_BUTTON_X & buttonDown)
    {
        g_godrayLoop = g_godrayLoop < MAX_GODRAY_LOOP ? g_godrayLoop + 1 : g_godrayLoop;
    }
    if (DEMO_PAD_BUTTON_Y & buttonDown)
    {
        g_godrayLoop = g_godrayLoop > 1 ? g_godrayLoop - 1 : g_godrayLoop;
    }

    if (DEMO_PAD_BUTTON_A & button)
    {
        g_godrayDecay += 0.025f;
        g_godrayDecay = std::min( 1.f, g_godrayDecay);
    }
    if (DEMO_PAD_BUTTON_B & button)
    {
        g_godrayDecay -= 0.025f;
        g_godrayDecay = std::max( 0.5f, g_godrayDecay);
    }

    if (DEMO_PAD_TRIGGER_R & buttonDown)
    {
        VECNormalize( &g_objPt, &g_sunDirection );
    }

    HandleRightStick();
    HandleLeftStick();


    MTX44Concat( g_projMtx, g_viewMtx, g_vpMtx );
}

static void DrawBackground( )
{
    const nn::gfx::ColorTargetView* renderTargets[1];
    renderTargets[0] = DEMOGetColorBufferView();
#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOCommandBuffer.SetRenderTargets( 1, renderTargets, &DEMODepthBufferView );
    DEMOCommandBuffer.SetViewportScissorState( &frameViewport );

    DEMOCommandBuffer.SetPipeline( &backgroundPipeline );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer );

    if ( (int)g_BackGroundCubemapLocation >= 0 )
    {
        DEMOCommandBuffer.SetTextureAndSampler( g_BackGroundCubemapLocation, nn::gfx::ShaderStage_Pixel, backGroundTexture.GetDescriptorSlot( 0 ), backGroundSamplerSlot );
    }

    DEMOCommandBuffer.SetVertexBuffer( 0, attrPositionBuffer.gpuAddress, 2 * sizeof(float), attrPositionBuffer.size );
    DEMOCommandBuffer.SetConstantBuffer( ub_backgroundSlot, nn::gfx::ShaderStage_Pixel, ub_vpiBuffer.gpuAddress, ub_vpiBuffer.size );
    DEMOCommandBuffer.Draw(nn::gfx::PrimitiveTopology_TriangleStrip, 4, 0);
    DEMOCommandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
}

static void CreateMask()
{
    const nn::gfx::ColorTargetView* renderTargets[1];
    renderTargets[0] = &maskColorTargetView;
    DEMOCommandBuffer.SetRenderTargets( 1, renderTargets, &reductionDepthStencilView );
    DEMOCommandBuffer.SetViewportScissorState( &reductionViewport );

    DEMOCommandBuffer.SetPipeline(&maskPipeline);
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_ColorBuffer );

    DEMOCommandBuffer.SetVertexBuffer( 0, attrResolvedPositionTextureBuffer.gpuAddress, 4 * sizeof(float), attrResolvedPositionTextureBuffer.size );
    DEMOCommandBuffer.SetConstantBuffer( ub_maskSlot, nn::gfx::ShaderStage_Pixel, ub_vpiBuffer.gpuAddress, ub_vpiBuffer.size );
    if ( (int)g_MaskSourceTextureLocation >= 0 )
    {
        DEMOCommandBuffer.SetTextureAndSampler( g_MaskSourceTextureLocation, nn::gfx::ShaderStage_Pixel, resolvedTextureSlot, linearSamplerSlot );
    }
    DEMOCommandBuffer.Draw(nn::gfx::PrimitiveTopology_TriangleStrip, 4, 0);
    DEMOCommandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
}

static void CreateGodRays()
{
    Vec out;

    MTX44MultVec(g_vpMtx, &g_sunDirection, &out);

    f32 uv[4]={ -out.x * 0.5f + 0.5f, -out.y * 0.5f + 0.5f, 0.f, 0.f};

#if !NN_GFX_IS_TARGET_GX && !NN_GFX_IS_TARGET_D3D
            // For GL it is necessary to flip texture coordinates
            uv[1] = 1.0f - uv[1];
#endif
    //Stretch image with ping-pong
    for (s32 i = (s32)g_godrayLoop - 1; i >= 0; i--)
    {
        DEMOCommandBuffer.SetPipeline(&godrayPipeline);
        const nn::gfx::ColorTargetView* renderTargets[1];
        renderTargets[0] = &feedBackColorBuffer[ i & 1 ];
        DEMOCommandBuffer.SetRenderTargets( 1, renderTargets, &reductionDepthStencilView );
        DEMOCommandBuffer.SetViewportScissorState( &reductionViewport );
        DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_ColorBuffer );
        DEMOCommandBuffer.SetVertexBuffer( 0, attrPositionTextureBuffer.gpuAddress, 4 * sizeof(float), attrPositionTextureBuffer.size );

        f32 param[8][2];
        f32 sample_decay = 1.f / 8.f;
        for (u32 j = 0; j < 8; ++j)
        {
            param[j][0] = (f32)j * powf(0.5,f32(i)) * 1.0f;
            param[j][1] = sample_decay;
            sample_decay *= g_godrayDecay;
        }
        // initialize uniform block data
        {
#if NN_GFX_IS_TARGET_GX
            size_t dataSize = 20 * sizeof(float);
#endif
            f32* pdata = ub_godrayBuffer[i].Map< float >();
            memcpy( &pdata[0], &param, 16 * sizeof( float ) );
            memcpy( &pdata[16], &uv, 4 * sizeof(float) );
#if NN_GFX_IS_TARGET_GX
            GX2EndianSwap( pdata, dataSize );
#endif
            ub_godrayBuffer[i].Unmap();
        }
        DEMOCommandBuffer.SetConstantBuffer( ub_godraySlot, nn::gfx::ShaderStage_Pixel, ub_godrayBuffer[i].gpuAddress, ub_godrayBuffer[i].size );

        if ((u32)i == g_godrayLoop - 1)
        {
            DEMOCommandBuffer.SetTextureAndSampler( g_GodraySourceTextureLocation, nn::gfx::ShaderStage_Pixel, maskTextureBufferSlot, linearSamplerSlot );
        }
        else
        {
            DEMOCommandBuffer.SetTextureAndSampler( g_GodraySourceTextureLocation, nn::gfx::ShaderStage_Pixel, feedBackTextureSlot[ (i + 1) & 1 ], linearSamplerSlot );
        }
        DEMOCommandBuffer.Draw(nn::gfx::PrimitiveTopology_TriangleStrip, 4, 0);
        DEMOCommandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
    }
}

// do final composition
static void DoFinalRender()
{
    DEMOCommandBuffer.SetPipeline(&copyPipeline);
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer );
    DEMOCommandBuffer.SetViewportScissorState( &frameViewport );

    // render resolved image
    DEMOCommandBuffer.SetVertexBuffer( 0, attrResolvedPositionTextureBuffer.gpuAddress, 4 * sizeof(float), attrResolvedPositionTextureBuffer.size );
    DEMOCommandBuffer.SetTextureAndSampler( g_CopySourceTextureLocation, nn::gfx::ShaderStage_Pixel, resolvedTextureSlot, pointSamplerSlot );
    DEMOCommandBuffer.Draw(nn::gfx::PrimitiveTopology_TriangleStrip, 4, 0);

    // additive blend of god-ray filter
    DEMOCommandBuffer.SetPipeline(&copyBlendPipeline);
    DEMOCommandBuffer.SetVertexBuffer( 0, attrPositionTextureBuffer.gpuAddress, 4 * sizeof(float), attrPositionTextureBuffer.size );
    DEMOCommandBuffer.SetTextureAndSampler( g_CopySourceTextureLocation, nn::gfx::ShaderStage_Pixel, feedBackTextureSlot[0], linearSamplerSlot );
    DEMOCommandBuffer.Draw(nn::gfx::PrimitiveTopology_TriangleStrip, 4, 0);
}

// Rendering
static void DrawScene(f32 dt)
{
    Mtx44 VPI;

    MTX44Inverse( g_vpMtx, VPI);

    // initialize uniform block data
    {
#if NN_GFX_IS_TARGET_GX
        size_t dataSize = 19 * sizeof(float);
#endif
        f32* pdata = ub_vpiBuffer.Map< float >( );
        memcpy( &pdata[0], &VPI, 16 * sizeof( float ) );
        memcpy( &pdata[16], &g_sunDirection, 3 * sizeof(float) );
#if NN_GFX_IS_TARGET_GX
        GX2EndianSwap( pdata, dataSize );
#endif
        ub_vpiBuffer.Unmap();
    }

    DEMOGfxBeforeRender();

    //-------------------------------------------------------------------------
    //Render the background
    DrawBackground();

    // copy background to the resolvedTextureBuffer
    nn::gfx::TextureSubresource dstSubresource;
    nn::gfx::TextureCopyRegion srcCopyRegion;
    dstSubresource.SetDefault();
    srcCopyRegion.SetDefault();
    srcCopyRegion.SetWidth( DEMOColorBufferInfo.GetWidth() );
    srcCopyRegion.SetHeight( DEMOColorBufferInfo.GetHeight() );

    DEMOCommandBuffer.CopyImage( &resolvedTexture, dstSubresource, 0, 0, 0, DEMOGetColorBuffer(), srcCopyRegion );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    //-------------------------------------------------------------------------
    //Use threshold to create mask
    CreateMask();

    // Create god-rays
    CreateGodRays();

    DEMOGfxSetDefaultRenderTarget();

    // Render
    DoFinalRender();

    // Set Demo Font state
    if (s_firstFrame == false)
    {
        DEMOFontPrintf(5,3, "PostProcess Godray : %.1f fps", 1.0f / std::max(1.0e-7f, dt));
        DEMOFontPrintf(5,4, "Godray Decay (a/b) : %.02f",g_godrayDecay);
        DEMOFontPrintf(5,5, "Feedback (x/y) : %d",g_godrayLoop);
        DEMOFontPrintf(5,6, "Move Camera (R Stick)");
        DEMOFontPrintf(5,7, "Move Light Source (L Stick)");
        DEMOFontPrintf(5,8, "Reset Light Source (R Trigger)");
    }
    DEMOGfxDoneRender();
}

// Freeing
static void FreeScene()
{
    DEMOGfxFreeShaders( &g_BackGroundShader);
    DEMOGfxFreeShaders( &g_MaskShader);
    DEMOGfxFreeShaders( &g_GodrayShader);
    DEMOGfxFreeShaders( &g_CopyShader);

    attrPositionBuffer.Finalize( );

    attrPositionTextureBuffer.Finalize( );
    attrResolvedPositionTextureBuffer.Finalize( );

    for (int i = 0; i < 8; i++)
    {
        ub_godrayBuffer[i].Finalize();
    }
    ub_vpiBuffer.Finalize();

    resolvedTexture.Finalize(&DEMODevice);
    resolvedTextureView.Finalize(&DEMODevice);
    resolvedColorTargetView.Finalize(&DEMODevice);
    resolvedTexturePool->Finalize();
    delete resolvedTexturePool;

    reductionDepthStencilView.Finalize(&DEMODevice);
    reductionDepthStencilTexture.Finalize(&DEMODevice);
    reductionDepthStencilMemPool->Finalize();
    delete reductionDepthStencilMemPool;

    DEMOGfxFreeMEM2( reductionViewport.GetMemory() );
    reductionViewport.Finalize(&DEMODevice);

    DEMOGfxFreeMEM2( frameViewport.GetMemory() );
    frameViewport.Finalize(&DEMODevice);

    maskTextureBuffer.Finalize(&DEMODevice);
    maskTextureBufferView.Finalize(&DEMODevice);
    maskColorTargetView.Finalize(&DEMODevice);
    maskTextureBufferPool->Finalize();
    for (int i = 0; i < 2; i++)
    {
        feedBackColorBuffer[i].Finalize(&DEMODevice);
        feedBackTexture[i].Finalize(&DEMODevice);
        feedBackTextureView[i].Finalize(&DEMODevice);
        feedBackTexturePool[i]->Finalize();
        delete feedBackTexturePool[ i ];
    }

    backgroundPipeline.Finalize(&DEMODevice);
    copyBlendPipeline.Finalize(&DEMODevice);
    copyPipeline.Finalize(&DEMODevice);
    maskPipeline.Finalize(&DEMODevice);
    godrayPipeline.Finalize(&DEMODevice);

    pointSampler.Finalize(&DEMODevice);
    linearSampler.Finalize(&DEMODevice);
    backGroundSampler.Finalize(&DEMODevice);
}

//extern "C" void nnMain()
TEST(GfxGodray, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;
    InitScene();
    s_firstFrame = true;

    OSTime  lastTime = OSGetTime();
    while (DEMOIsRunning())
    {
        f32 dtime = OSTicksToMilliseconds(OSGetTime() - lastTime) / 1000.0f;
        lastTime = OSGetTime();

        UpdateScene();
        DrawScene(dtime);

        s_firstFrame = false;
    }

    FreeScene();
    backGroundTexture.Finalize();

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
