﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

////===========================================================================
///  demoTest.c
///
///     This contains test-related functions for the demos
///
////===========================================================================

#include <gfx/demo.h>
#if NN_GFX_IS_TARGET_GX
#include <nn/save.h>
#include <nn/act.h>
#include <cafe/env.h>
#else
#include <nnt.h>
#endif

#include <cstdio>
#include <cstring>

#include <nnt/nnt_Argument.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_LocationName.h>
#include <nnt/graphics/testGraphics_PerformanceProfileData.h>

//----------------------------------------------------------------------

// Global Data
static const size_t DEMO_TEST_MAX_PATH = 1024;

typedef struct _DEMOTestDataStore {
    u32 count;
    u32 stopSelect;
    u32 testSelect;
    u32 captureSelect;
    u32 dumpSelect;
    u32 pm4CaptureSelect;
    u32 pm4CaptureNumFrames;
    bool result;
    bool bPerfProfileDataCaptureEnable;
    bool useHlslccGlsl;
    char pm4CaptureFilename[ DEMO_TEST_MAX_PATH ];
    char compareFileName[DEMO_TEST_MAX_PATH];
    char captureFile[DEMO_TEST_MAX_PATH];
    char argv0[DEMO_TEST_MAX_PATH];
    char perfProfileDataFile[DEMO_TEST_MAX_PATH];
    nnt::graphics::PerformanceProfileData* pPerfProfileData;
    s64 initStart;
    s64 initEnd;
} DEMOTestDataStore;

DEMOTestDataStore DEMOTestData;

namespace {
char s_BuildTargetName[ MAX_ASSET_DIR_LEN ] = { 0 };
const char* GetBuildTargetName()
{
#if NN_GFX_IS_TARGET_GX
    return "Cafe";
#else
    static bool s_FirstTime = true;
    if ( s_FirstTime )
    {
        char** argv = nnt::GetHostArgv();

        NN_SDK_ASSERT( NULL != argv );
        NN_SDK_ASSERT( NULL != argv[ 0 ] );

        const char* basePath = "Tests\\Outputs\\";
        char* buildTargetName = std::strstr( argv[ 0 ], basePath );
        NN_SDK_ASSERT( NULL != buildTargetName );
        buildTargetName += strlen( basePath );

        char* removeSlash = std::strstr( buildTargetName, "\\" );
        NN_SDK_ASSERT( NULL != removeSlash );

        // Copy over only the test name
        strncat( s_BuildTargetName, buildTargetName,
            nn::util::BytePtr( buildTargetName ).Distance( removeSlash ) );

        s_FirstTime = false;
    }

    return s_BuildTargetName;
#endif
}
//----------------------------------------------------------------------

static void* AllocateProfilingMemory( size_t size )
{
    return std::malloc( size );
}

static void DeallocateProfilingMemory( void* ptr, size_t size )
{
    NN_UNUSED( size );

    std::free( ptr );
}

}

//----------------------------------------------------------------------

// Utility hash functions

u16 DEMOTestHashCRC16(const void *datap, u32 size)
{
    static const u16 crc16_table[16] = {
        0x0000, 0xCC01, 0xD801, 0x1400,
        0xF001, 0x3C00, 0x2800, 0xE401,
        0xA001, 0x6C00, 0x7800, 0xB401,
        0x5000, 0x9C01, 0x8801, 0x4400,
    };

    u32     r = 0;
    const u8 *p = (const u8 *)datap;

    while (size--)
    {
        u32     data = *p;
        p++;

        r = (r >> 4) ^ crc16_table[(r ^ data) & 0xf];
        data >>= 4;
        r = (r >> 4) ^ crc16_table[(r ^ data) & 0xf];
    }

    return (u16)r;
}

u32 DEMOTestHashCRC32(const void *datap, u32 size)
{
    static const u32 crc32_table[16] = {
        0x00000000U, 0x1DB71064U, 0x3B6E20C8U, 0x26D930ACU,
        0x76DC4190U, 0x6B6B51F4U, 0x4DB26158U, 0x5005713CU,
        0xEDB88320U, 0xF00F9344U, 0xD6D6A3E8U, 0xCB61B38CU,
        0x9B64C2B0U, 0x86D3D2D4U, 0xA00AE278U, 0xBDBDF21CU,
    };

    u32     r = 0xffffffffU;
    const u8 *p = (const u8 *)datap;

    while (size--)
    {
        u32     data = *p;
        p++;

        r = (r >> 4) ^ crc32_table[(r ^ data) & 0xf];
        data >>= 4;
        r = (r >> 4) ^ crc32_table[(r ^ data) & 0xf];
    }

    return ~r;
}

//----------------------------------------------------------------------


void DEMOTestInit(int argc, char *argv[])
{
    int i;
    char *p;

    DEMOTestData.count = 0;
    DEMOTestData.stopSelect = 0xFFFFFFFF;
    DEMOTestData.testSelect = 0xFFFFFFFF;
    DEMOTestData.captureSelect = 0xFFFFFFFF;
    DEMOTestData.dumpSelect = 0xFFFFFFFF;
    DEMOTestData.pm4CaptureSelect = 0xFFFFFFFF;
    DEMOTestData.pm4CaptureNumFrames = 1;
    DEMOTestData.pm4CaptureFilename[0] = '\0';
    DEMOTestData.result = true;
    DEMOTestData.bPerfProfileDataCaptureEnable = false;
    DEMOTestData.useHlslccGlsl = false;
    DEMOTestData.compareFileName[0] = '\0';
    DEMOTestData.captureFile[0] = '\0';
#if NN_GFX_IS_TARGET_GX
    strncpy( DEMOTestData.captureFile, "/vol/save/common/GX2Capture.tga", sizeof( DEMOTestData.captureFile ) );
#elif NN_GFX_IS_TARGET_GL
    strncpy( DEMOTestData.captureFile, "assets:/GLCapture.tga", sizeof( DEMOTestData.captureFile ) );
#elif NN_GFX_IS_TARGET_NVN
    strncpy( DEMOTestData.captureFile, "assets:/NVNCapture.tga", sizeof( DEMOTestData.captureFile ) );
#elif NN_GFX_IS_TARGET_D3D
    strncpy( DEMOTestData.captureFile, "assets:/D3DCapture.tga", sizeof( DEMOTestData.captureFile ) );
#else
#error Need to define the test capture output file name!
#endif

    if (argc && (argv != NULL))
    {
        strcpy(DEMOTestData.argv0, argv[0]); //note this might be different to OSGetArgcArgv(), e.g. under Meta.
    } else {
        strcpy(DEMOTestData.argv0, "unknown");
    }

#define SKIP_NON_DIGIT(c) ((c)!=0&&((c)<'0'||(c)>'9'))
    for (i = 0; i < argc; ++i)
    {
        // Simple parameter reading for now
        p = strstr(argv[i], "STOP_SELECT=");
        if (p)
        {
            DEMOTestData.stopSelect = (u32) atoi(p + strlen("STOP_SELECT="));
        }

        p = strstr(argv[i], "TEST_SELECT=");
        if (p)
        {
            DEMOTestData.testSelect = (u32) atoi(p + strlen("TEST_SELECT="));
#if NN_GFX_IS_TARGET_GX
            FSInit(); // Need to initialize FS before SAVEInit().
            SAVEInit();
            SAVEStatus status = SAVEInitSaveDir(ACT_SLOT_NO_COMMON);
            if (status != SAVE_STATUS_OK)
            {
                OSHalt("Failed to make common save directory for tests.\n");
            }
#endif
        }

        p = strstr(argv[i], "CAPTURE_SELECT=");
        if (p)
        {
            DEMOTestData.captureSelect = (u32) atoi(p + strlen("CAPTURE_SELECT="));
        }

        p = strstr(argv[i], "GX2R_DUMP_SELECT=");
        if (p)
        {
            DEMOTestData.dumpSelect = (u32) atoi(p + strlen("GX2R_DUMP_SELECT="));
        }

        p = strstr(argv[i], "PM4_SELECT=");
        if (p)
        {
            u32 endFrame=0;
            u32 numParams = sscanf(p + strlen("PM4_SELECT="), "%d-%d", &DEMOTestData.pm4CaptureSelect, &endFrame);
            DEMOTestData.pm4CaptureNumFrames = (numParams == 2) ? endFrame - DEMOTestData.pm4CaptureSelect + 1 : 1;
        }

        p = strstr(argv[i], "PM4_FILE=");
        if (p)
        {
            strcpy(DEMOTestData.pm4CaptureFilename, p + strlen("PM4_FILE="));
            // replace EOL or ',' with terminating 0
            p = DEMOTestData.pm4CaptureFilename;
            for( ; *p != '\n' && *p != '\r' && *p != ',' && *p != 0; p++ ) {}
            *p = 0;
        }

        p = strstr(argv[i], "CAPTURE_DIR=");
        if (p)
        {
            strcpy(DEMOTestData.captureFile, p + strlen("CAPTURE_DIR="));
            // replace EOL or ',' with terminating 0
            p = DEMOTestData.captureFile;
            for( ; *p != '\n' && *p != '\r' && *p != ',' && *p != 0; p++ ) {}
            *p = 0;

            size_t length = strnlen( DEMOTestData.captureFile, DEMO_TEST_MAX_PATH );
            const char ext[] = { '.', 'b', 'm', 'p' };
            if( memcmp( &DEMOTestData.captureFile[ length - 4 ], ext, sizeof( ext ) ) )
            {
                // Create the directory in case it does not exist
                DEMOFSCreateDirectory( DEMOTestData.captureFile );

#if NN_GFX_IS_TARGET_GX
                strncat( DEMOTestData.captureFile, "/GX2Capture.tga", sizeof( DEMOTestData.captureFile ) );
#elif NN_GFX_IS_TARGET_GL
                strncat( DEMOTestData.captureFile, "/GLCapture.tga", sizeof( DEMOTestData.captureFile ) );
#elif NN_GFX_IS_TARGET_NVN
                strncat( DEMOTestData.captureFile, "/NVNCapture.tga", sizeof( DEMOTestData.captureFile ) );
#elif NN_GFX_IS_TARGET_D3D
                strncat( DEMOTestData.captureFile, "/D3DCapture.tga", sizeof( DEMOTestData.captureFile ) );
#else
#error Need to define the test capture output file name!
#endif
            }
            else // if specified fine name.
            {
                char path[DEMO_TEST_MAX_PATH] = {};
                size_t strLength = strnlen( DEMOTestData.captureFile, DEMO_TEST_MAX_PATH );
                memcpy( path, DEMOTestData.captureFile, strLength );
                for( char* ptr =  &path[strLength]; ( *ptr == '\\' ) ? false : ( *ptr == '/' ) ? false : true; --ptr )
                {
                    *ptr = '\0';
                }
                DEMOFSCreateDirectory( path );
            }
        }

        p = strstr(argv[i], "NNT_UI2D_PERF_PROFILE_DATA_PATH=");
        if (p)
        {
            nn::time::PosixTime posixTime;
            nn::time::CalendarTime  calendarTime;
            nn::time::StandardUserSystemClock::GetCurrentTime( &posixTime );
            nn::time::ToCalendarTime( &calendarTime, NULL, posixTime );

            strncpy( DEMOTestData.perfProfileDataFile, p + strlen("NNT_UI2D_PERF_PROFILE_DATA_PATH="), DEMO_TEST_MAX_PATH );

            char* comma = NULL;
            if ( ( comma = strchr( DEMOTestData.perfProfileDataFile, ',' ) ) != NULL )
            {
                *comma = '\0';
            }

            // Make sure the output path exists
            DEMOFSCreateDirectory( DEMOTestData.perfProfileDataFile );

            char fileDateString[ DEMO_TEST_MAX_PATH ];
            nn::util::SNPrintf( fileDateString, DEMO_TEST_MAX_PATH,
                "%04d%02d%02d_%02d%02d%02d",
                calendarTime.year, calendarTime.month, calendarTime.day,
                calendarTime.hour, calendarTime.minute, calendarTime.second );
            strncat( DEMOTestData.perfProfileDataFile, DEMOTestGetTestName(), DEMO_TEST_MAX_PATH );
            strncat( DEMOTestData.perfProfileDataFile, "_", DEMO_TEST_MAX_PATH );
            strncat( DEMOTestData.perfProfileDataFile, fileDateString, DEMO_TEST_MAX_PATH );
            strncat( DEMOTestData.perfProfileDataFile, ".json", DEMO_TEST_MAX_PATH );

            DEMOTestData.pPerfProfileData = new nnt::graphics::PerformanceProfileData;
            DEMOAssert( NULL != DEMOTestData.pPerfProfileData );

            nnt::graphics::PerformanceProfileData* pPerfProfileData = DEMOTestData.pPerfProfileData;
            pPerfProfileData->Initialize( 1024, AllocateProfilingMemory, DeallocateProfilingMemory );
            pPerfProfileData->SetName( DEMOTestGetTestName() );

            char timeString[ 32 ];
            nn::util::SNPrintf( timeString, 32,
                "%04d-%02d-%02d %02d:%02d:%02d",
                calendarTime.year, calendarTime.month, calendarTime.day,
                calendarTime.hour, calendarTime.minute, calendarTime.second );
            pPerfProfileData->SetDate( timeString );
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#if defined(NN_BUILD_CONFIG_ADDRESS_64)
            const char* platformString = "NXFP2-a64";
#else
            const char* platformString = "NXFP2-a32";
#endif
#elif NN_GFX_IS_TARGET_GX
            const char* platformString = "Cafe";
#else
            const char* platformString = "PC";
#endif
            pPerfProfileData->SetEnvironmentText( platformString );

            DEMOTestData.initStart = OSGetTime();
            DEMOTestData.bPerfProfileDataCaptureEnable = true;
        }

        p = strstr(argv[i], "COMPARE_FILE=");
        if (p)
        {
#if NN_GFX_IS_TARGET_GX
            char nxSdkRoot[ MAX_ASSET_DIR_LEN ];
            strcpy( DEMOTestData.compareFileName, "/vol/hfio01/" );
            int retVal = ENVGetEnvironmentVariable( "NX_SDK_ROOT", nxSdkRoot, sizeof( nxSdkRoot ) );
            DEMOAssert( !retVal );

            char* pSdkRoot = strstr( nxSdkRoot, "\\" );
            while (pSdkRoot)
            {
                *pSdkRoot = '/';
                pSdkRoot = strstr( pSdkRoot, "\\" );
            }

            pSdkRoot = strstr( nxSdkRoot, ":/" );

            // Append the drive letter
            char driveLetter = tolower( pSdkRoot[ -1 ] );
            strncat( DEMOTestData.compareFileName, &driveLetter, 1 );

            // Append the path
            strcat( DEMOTestData.compareFileName, pSdkRoot + 1 );

            if ( DEMOTestData.compareFileName[ strlen(DEMOTestData.compareFileName) - 1 ] != '/' )
            {
                strcat( DEMOTestData.compareFileName, "/" );
            }
#else
            // Append the image reference path
            strcat( DEMOTestData.compareFileName, "assets:/" );
#endif
            strcat( DEMOTestData.compareFileName, "Externals/TestBinaries/Gfx/ReferenceImages/" );
            strcat( DEMOTestData.compareFileName, DEMOTestGetTestName() );
            strcat( DEMOTestData.compareFileName, "/" );

#if NN_GFX_IS_TARGET_NVN
            strcat( DEMOTestData.compareFileName, "NX/" );
#elif NN_GFX_IS_TARGET_GL
            strcat( DEMOTestData.compareFileName, "Generic/" );
#elif NN_GFX_IS_TARGET_GX
            strcat( DEMOTestData.compareFileName, "Cafe/" );
#elif NN_GFX_IS_TARGET_D3D
            strcat( DEMOTestData.compareFileName, "GenericD3d/" );
#endif
            strcat(DEMOTestData.compareFileName, p + strlen("COMPARE_FILE="));
            // replace EOL or ',' with terminating 0
            p = DEMOTestData.compareFileName;
            for( ; *p != '\n' && *p != '\r' && *p != ',' && *p != 0; p++ ) {}
            *p = 0;
            strcat( DEMOTestData.compareFileName, ".tga" );
        }

        p = strstr(argv[i], "PERF_WORKS_FRAME=");
        if (p)
        {
            DEMOTestData.captureSelect = (u32) atoi(p + strlen("CAPTURE_SELECT="));
        }

        p = strstr(argv[i], "PERF_WORKS_FRAME=");
        if (p)
        {
            DEMOTestData.captureSelect = (u32) atoi(p + strlen("CAPTURE_SELECT="));
        }

        p = strstr(argv[i], "USE_HLSLCC_GLSL");
        if (p)
        {
            DEMOTestData.useHlslccGlsl = true;
        }
    }

#if NN_GFX_IS_TARGET_GX
    if (DEMOTestData.pm4CaptureSelect != 0xFFFFFFFF &&
        DEMOTestData.pm4CaptureFilename[0]=='\0')
    {
        // if we asked for a PM4 capture but didn't specify a filename, default to argv[0].4mp
        char baseName[256]="";
        strcpy(baseName, argv[0]);
        char* dot=strrchr(baseName, '.');
        if (NULL != dot)
        {
            *dot = 0;
        }
        strcat(baseName, ".4mp");
        strcpy(DEMOTestData.pm4CaptureFilename, baseName);
    }
#endif
} // NOLINT(impl/function_size)

void DEMOTestShutdown()
{
#if !NN_GFX_IS_TARGET_GX
    EXPECT_TRUE( DEMOTestResult() ) << "DEMO: Test Result: FAIL";
#endif
}

void DEMOTestCheck()
{
#if NN_GFX_IS_TARGET_GX
    // Do this even on first frame. Treat 1 as the first frame number to appear consistent with other checks,
    // which are done at the top of the following frame
    if (DEMOTestData.count + 1 == DEMOTestData.pm4CaptureSelect)
    {
        char str[DEMO_TEST_MAX_PATH]="";

        OSCalendarTime td;
        OSTicksToCalendarTime(OSGetTime(), &td);

        sprintf(str, "DEMO: start PM4 capture of %s frame %d-%d to file \"%s\" at %4d-%02d-%02d %02d:%02d:%02d",
                    DEMOTestData.argv0, DEMOTestData.pm4CaptureSelect, DEMOTestData.pm4CaptureSelect + DEMOTestData.pm4CaptureNumFrames - 1,
                    DEMOTestData.pm4CaptureFilename,
                    td.year, td.mon + 1, td.mday, td.hour, td.min, td.sec);
        // Print it to the console
        OSReport(str); OSReport("\n");

        // Start the capture
        GX2DebugCaptureStart(DEMOTestData.pm4CaptureFilename, GX2_DEBUG_CAPTURE_DEFAULT);

        // Add the info string to the capture itself as a nop tag
        DEMOGfxDebugTagComment(str);
    }
#endif

    if (DEMOTestData.count > 0)
    {
#if NN_GFX_IS_TARGET_GX
        // Do this first so we don't get the screen capture code in the PM4 capture
        if (DEMOTestData.count == DEMOTestData.pm4CaptureSelect + DEMOTestData.pm4CaptureNumFrames - 1)
        {
            char str[DEMO_TEST_MAX_PATH]="";
            sprintf(str, "DEMO: end PM4 capture at frame %d to file \"%s\"", DEMOTestData.count, DEMOTestData.pm4CaptureFilename);

            DEMOGfxDebugTagComment(str);
            OSReport(str); OSReport("\n");

            GX2DebugCaptureEnd(GX2_DEBUG_CAPTURE_DEFAULT);
        }
#endif
        if (DEMOTestData.count == DEMOTestData.testSelect)
        {
            DEMOTestCompare();
            DEMOStopRunning();

            if ( !DEMOTestResult() )
            {
#if NN_GFX_IS_TARGET_GX
                DEMOPrintf("Test result: FAIL\n");
#else
                EXPECT_TRUE( DEMOTestResult() ) << "DEMO: Image comparison failure";
#endif
            }
        }
        if (DEMOTestData.count == DEMOTestData.captureSelect)
        {
            DEMOTestCapture();
            DEMOStopRunning();
        }
        if (DEMOTestData.count == DEMOTestData.stopSelect)
        {
            DEMOStopRunning();
        }
#if NN_GFX_IS_TARGET_GX
        if (DEMOTestData.count == DEMOTestData.dumpSelect)
        {
            GX2TempDumpResources();
        }
#endif
    }
    DEMOTestData.count++;
}

void _DEMODumpSerial(const u8 *, const u32);

extern bool _DEMOIsScanBufferSwap();

void DEMOTestCompare()
{
    if ( !DEMOTestData.compareFileName[ 0 ] )
    {
        // Capture comparison is not required.
        return;
    }

    u32 cmpLen;
    u8 *cmpBuf;
    u32 i;
    bool testPassed;

    nn::gfx::ImageFormat format = DEMOColorBufferInfo.GetImageFormat();
    DEMOCaptureInit(DEMOColorBufferInfo.GetWidth(), DEMOColorBufferInfo.GetHeight(),
        ( format == nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm ) ? RGBA8 : SRGB8 );

    {
        const int MaxScanBufferCount = 6;
        nn::gfx::Texture* pScanBufferArray[ MaxScanBufferCount ];
        int scanBufferCount = DEMOSwapChain.GetScanBuffers( NULL, 0 );
        NN_SDK_ASSERT( MaxScanBufferCount >= scanBufferCount );
        DEMOSwapChain.GetScanBuffers( pScanBufferArray, MaxScanBufferCount );
        int idxCopyScanBuffer = ( DEMOSwapChain.AcquireNextScanBufferIndex()
            + scanBufferCount - 1 ) % scanBufferCount;
        DEMOCaptureCopy( pScanBufferArray[ idxCopyScanBuffer ], NULL );
    }

    DEMOPrintf("Capture CRC=%08x\n", DEMOTestHashCRC32(DEMOCaptureData.TGAData, DEMOCaptureData.TGALength));

    cmpBuf = (u8 *) DEMOFSSimpleRead(DEMOTestData.compareFileName, &cmpLen);
    DEMOPrintf("Compare CRC=%08x\n", DEMOTestHashCRC32(cmpBuf, cmpLen));

    // A test is considered to have passed if the image captured from the
    // color buffer matches the "golden" image. In the event that the TGA
    // compressed image lengths differ, or bytes within the image differ,
    // consider this test to have failed.
    testPassed = ( cmpLen == DEMOCaptureData.TGALength );

    if (testPassed)
    {

        for(i=0; i<cmpLen; i++)
        {
            if (cmpBuf[i] != DEMOCaptureData.TGAData[i])
            {

                testPassed = false;
                break;
            }
        }
    }

    if (false == testPassed)
    {
        // Before printing a failure message, output the TGA file.
        DEMOCaptureCopy(NULL, DEMOTestData.captureFile);
        DEMOTestSetResult( testPassed );
    } else {
        DEMOPrintf("Test result: PASS\n");
    }

    DEMOFree(cmpBuf);
    DEMOCaptureShutdown();
}

void DEMOTestCapture()
{
    DEMOCaptureInit(DEMOColorBufferInfo.GetWidth(), DEMOColorBufferInfo.GetHeight(),
                    (DEMOColorBufferInfo.GetImageFormat() == nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm) ? RGBA8 : SRGB8);

    const int MaxScanBufferCount = 6;
    nn::gfx::Texture* pScanBufferArray[ MaxScanBufferCount ];
    int scanBufferCount = DEMOSwapChain.GetScanBuffers( NULL, 0 );
    NN_SDK_ASSERT( MaxScanBufferCount >= scanBufferCount );
    DEMOSwapChain.GetScanBuffers( pScanBufferArray, MaxScanBufferCount );
    int idxCopyScanBuffer = ( DEMOSwapChain.AcquireNextScanBufferIndex()
        + scanBufferCount - 1 ) % scanBufferCount;
    DEMOCaptureCopy( pScanBufferArray[ idxCopyScanBuffer ], DEMOTestData.captureFile );
    DEMOPrintf("DEMO_CAPTURE_FINISH\n");
    DEMOPrintf("Capture CRC=%08x\n", DEMOTestHashCRC32(DEMOCaptureData.TGAData, DEMOCaptureData.TGALength));
    DEMOCaptureShutdown();
}

void DEMOTestSetResult( bool result )
{
    // Once the test has failed make sure it remains that way
    if ( !result )
    {
        DEMOTestData.result = result;
    }
}

bool DEMOTestResult()
{
    return DEMOTestData.result;
}

u32 DEMOTestFrame()
{
    return DEMOTestData.count;
}

void DEMOTestCheckPerfBegin()
{
    if (DEMOTestData.testSelect == 0xFFFFFFFF)
    {
        return;
    }

    if ( DEMOTestData.count == 1 )
    {
        DEMOTestData.initEnd = OSGetTime();
    }

    if (DEMOTestData.count == DEMOTestData.testSelect)
    {
    }
}

void DEMOTestCheckPerfEnd()
{
    if (DEMOTestData.testSelect == 0xFFFFFFFF)
    {
        return;
    }

#ifdef NN_PERF_PROFILE_ENABLED
    if ( DEMOTestData.bPerfProfileDataCaptureEnable )
    {
        nn::perf::CpuMeter* pCpuMeter = NN_PERF_GET_USER_METER( DEMOTestPerfIndices_FrameCpu );
        nn::perf::GpuMeter* pGpuMeter = NN_PERF_GET_GPU_METER();
        int64_t cpuTime = 0ll;
        int64_t gpuTime = 0ll;

        if ( pCpuMeter != NULL && pGpuMeter != NULL )
        {
            //FIXME SIGLONTD-8465: cpuTime = pCpuMeter->GetLastTotalSpan().GetMicroSeconds();
            gpuTime = pGpuMeter->GetLastTotalSpan().GetMicroSeconds();
            DEMOTestData.pPerfProfileData->SetCpuValue( DEMOTestData.count, cpuTime );
            DEMOTestData.pPerfProfileData->SetGpuValue( DEMOTestData.count, gpuTime );
        }

        DEMOTestData.pPerfProfileData->SetMemoryUsageValue( DEMOTestData.count, DEMOGetUsedMemory() );

        if ( DEMOTestData.count == 1 )
        {
            DEMOTestData.pPerfProfileData->SetInitializationLoadValue( 0 );
            //FIXME SIGLONTD-8465:     OSTicksToMicroseconds( DEMOTestData.initEnd - DEMOTestData.initStart ) );
        }

        if ( DEMOTestData.count == DEMOTestData.testSelect )
        {
            const char* TestName = DEMOTestGetTestName();
            const char* BuildTarget = GetBuildTargetName();
            DEMOPrintf( "##teamcity[buildStatisticValue key='%s %s %s' value='%lld']\n", TestName, BuildTarget, "CPU Init Time", OSTicksToMicroseconds( DEMOTestData.initEnd - DEMOTestData.initStart ) );
            DEMOPrintf( "##teamcity[buildStatisticValue key='%s %s %s' value='%lld']\n", TestName, BuildTarget, "CPU Frame Time",
                DEMOTestData.pPerfProfileData->GetAverageValue( nnt::graphics::PerformanceProfileData::FrameValueType::FrameValueType_CpuLoad ) );
            DEMOPrintf( "##teamcity[buildStatisticValue key='%s %s %s' value='%lld']\n", TestName, BuildTarget, "GPU Frame Time",
                DEMOTestData.pPerfProfileData->GetAverageValue( nnt::graphics::PerformanceProfileData::FrameValueType::FrameValueType_GpuLoad ) );
            DEMOTestData.pPerfProfileData->Write( DEMOTestData.perfProfileDataFile );
            DEMOTestData.pPerfProfileData->Finalize();
            DEMOStopRunning();
        }
    }
#endif
}

static char s_TestName[ MAX_ASSET_DIR_LEN ] = { 0 };
const char* DEMOTestGetTestName()
{
    static bool s_FirstTime = true;
    if ( s_FirstTime )
    {
        char** argv = nnt::GetHostArgv();

        NN_SDK_ASSERT( NULL != argv );
        NN_SDK_ASSERT( NULL != argv[ 0 ] );

#if NN_GFX_IS_TARGET_GX
        strcpy( s_TestName, argv[ 0 ] );
        char* p = strstr( s_TestName, ".rpx" );
        NN_SDK_ASSERT( NULL != p );
        *p = '\0';
#else
        char* testNameStart = std::strstr( argv[ 0 ], "testGfx_" );
        NN_SDK_ASSERT( NULL != testNameStart );

        char* removeSlash = std::strstr( testNameStart, "\\" );
        NN_SDK_ASSERT( NULL != removeSlash );

#if NN_GFX_IS_TARGET_D3D
        removeSlash -= 3; // skip D3d suffix
        bool isD3dSuffix = strncmp( removeSlash, "D3d", 3 ) == 0;
        NN_ASSERT( isD3dSuffix, "Program name suffix should be D3d.\n" );
#endif

        // Copy over only the test name
        strncat( s_TestName, testNameStart,
            nn::util::BytePtr( testNameStart ).Distance( removeSlash ) );
#endif

        s_FirstTime = false;
    }

    return s_TestName;
}

bool DEMOTestIsUseHlslccGlsl()
{
    return DEMOTestData.useHlslccGlsl;
}
