﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

////===========================================================================
///  demoPad.c
///
///     This is a pad code for the demo library.
///
////===========================================================================

#include <gfx/demo.h>
#include <nn/nn_Macro.h>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_FROM_WINDOWS_SDK_HEADERS
#ifndef NOMINMAX
#define NOMINMAX
#endif
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#include <nn/nn_Windows.h>
NN_PRAGMA_POP_WARNINGS

#include <algorithm>

#ifdef _DEBUG
#include <cstdio>
#endif

// -------------------------------------------------------------------------
//  Global Values
// -------------------------------------------------------------------------
// Demo Pad Status
DemoPadStatus  g_DemoPad[ DEMO_PAD_MAX_CONTROLLERS ];


// ------------------------------------------------------------------------
//  Macro definitions
// ------------------------------------------------------------------------
#define padButtonDown(buttonLast, button)   \
    ((u16) (((buttonLast) ^ (button)) & (button)))

#define padButtonUp(buttonLast, button)     \
    ((u16) (((buttonLast) ^ (button)) & (buttonLast)))

// -------------------------------------------------------------------------
//  Internal Fuctions
// -------------------------------------------------------------------------
// Attaches informations such as down/up, stick direction.

static const unsigned int BUTTON_IS_DOWN = 0x8000;

static u16 SetButtonDown ( u16 button, int mask, u16 keyState )
{

    if ( (keyState & BUTTON_IS_DOWN) != 0)
    {
        button |= mask;
    }
    else
    {
        button &= ~mask;
    }
    return button;
}

// -------------------------------------------------------------------------
//  Initialize
// -------------------------------------------------------------------------
void DEMOPadInit()
{
    u32 i;

    // Reset exported pad status
    for (i = 0 ; i < DEMO_PAD_MAX_CONTROLLERS ; i++)
    {
        g_DemoPad[i].button = 0;
        g_DemoPad[i].stickX = 0;
        g_DemoPad[i].stickY = 0;
        g_DemoPad[i].substickX = 0;
        g_DemoPad[i].substickY = 0;
        g_DemoPad[i].triggerLeft = 0;
        g_DemoPad[i].triggerRight = 0;
        g_DemoPad[i].analogA = 0;
        g_DemoPad[i].analogB = 0;
        g_DemoPad[i].err = 0;
        g_DemoPad[i].lastButton = 0;
        g_DemoPad[i].buttonDown = 0;
        g_DemoPad[i].buttonUp = 0;
        g_DemoPad[i].dirs = 0;
        g_DemoPad[i].dirsNew = 0;
        g_DemoPad[i].dirsReleased = 0;
        g_DemoPad[i].stickDeltaX = 0;
        g_DemoPad[i].stickDeltaY = 0;
        g_DemoPad[i].substickDeltaX = 0;
        g_DemoPad[i].substickDeltaY = 0;
    }
}

// -------------------------------------------------------------------------
//  ShutDown
// -------------------------------------------------------------------------
void DEMOPadShutdown()
{
    u32 i;
    // Reset exported pad status
    for (i = 0 ; i < DEMO_PAD_MAX_CONTROLLERS ; i++)
    {
        g_DemoPad[i].button = 0;
        g_DemoPad[i].stickX = 0;
        g_DemoPad[i].stickY = 0;
        g_DemoPad[i].substickX = 0;
        g_DemoPad[i].substickY = 0;
        g_DemoPad[i].triggerLeft = 0;
        g_DemoPad[i].triggerRight = 0;
        g_DemoPad[i].analogA = 0;
        g_DemoPad[i].analogB = 0;
        g_DemoPad[i].err = 0;
        g_DemoPad[i].lastButton = 0;
        g_DemoPad[i].buttonDown = 0;
        g_DemoPad[i].buttonUp = 0;
        g_DemoPad[i].dirs = 0;
        g_DemoPad[i].dirsNew = 0;
        g_DemoPad[i].dirsReleased = 0;
        g_DemoPad[i].stickDeltaX = 0;
        g_DemoPad[i].stickDeltaY = 0;
        g_DemoPad[i].substickDeltaX = 0;
        g_DemoPad[i].substickDeltaY = 0;
    }
}

// -------------------------------------------------------------------------
//  Read
// -------------------------------------------------------------------------

u32 DEMOPadRead ()
{
    u16 buttons = g_DemoPad[0].button;
    u16 lastButtons = buttons;
    POINT mousePos;
    static POINT lastPos;
    static bool lastPosIsLeft = false;
    static bool lastPosIsRight = false;
    static const int MOUSE_SCALE = -2;

    g_DemoPad[0].lastButton = lastButtons;
    if (DEMO_HasFocus())
    {
        buttons = SetButtonDown(buttons, DEMO_PAD_BUTTON_A, GetKeyState('A'));
        buttons = SetButtonDown(buttons, DEMO_PAD_BUTTON_B, GetKeyState('B'));
        buttons = SetButtonDown(buttons, DEMO_PAD_BUTTON_X, GetKeyState('X'));
        buttons = SetButtonDown(buttons, DEMO_PAD_BUTTON_Y, GetKeyState('Y'));
        buttons = SetButtonDown(buttons, DEMO_PAD_TRIGGER_Z, GetKeyState('Z'));
        buttons = SetButtonDown(buttons, DEMO_PAD_TRIGGER_L, GetKeyState('L'));
        buttons = SetButtonDown(buttons, DEMO_PAD_TRIGGER_R, GetKeyState('R'));
        buttons = SetButtonDown(buttons, DEMO_PAD_BUTTON_DOWN, GetKeyState(VK_DOWN));
        buttons = SetButtonDown(buttons, DEMO_PAD_BUTTON_UP, GetKeyState(VK_UP));
        buttons = SetButtonDown(buttons, DEMO_PAD_BUTTON_LEFT, GetKeyState(VK_LEFT));
        buttons = SetButtonDown(buttons, DEMO_PAD_BUTTON_RIGHT, GetKeyState(VK_RIGHT));
    }
    else
    {
        buttons = 0;
    }
    g_DemoPad[0].button = buttons;
    g_DemoPad[0].buttonDown = padButtonDown(lastButtons, buttons);
    g_DemoPad[0].buttonUp = padButtonUp(lastButtons, buttons);

    g_DemoPad[0].stickDeltaX = 0;
    g_DemoPad[0].stickDeltaY = 0;
    g_DemoPad[0].substickDeltaX = 0;
    g_DemoPad[0].substickDeltaY = 0;
    if (DEMO_HasFocus() && GetCursorPos(&mousePos) != 0)
    {
        // check for stick emulation
        if (0 != (GetKeyState(VK_LBUTTON) & BUTTON_IS_DOWN))
        {
            if (lastPosIsLeft)
            {
                // Clamp to Cafe Pad boundaries (-128, 127)
                s8 mouseChangeX = static_cast< s8 >( std::max( -128L, std::min( 127L, ( mousePos.x - lastPos.x ) ) ) );
                s8 mouseChangeY = static_cast< s8 >( std::max( -128L, std::min( 127L, ( mousePos.y - lastPos.y ) ) ) );
                g_DemoPad[ 0 ].stickX = MOUSE_SCALE * mouseChangeX;
                g_DemoPad[ 0 ].stickY = MOUSE_SCALE * mouseChangeY;
                //printf("stick = %d %d\n", g_DemoPad[0].stickX, g_DemoPad[0].stickY);
            }
            lastPosIsLeft = true;
        }
        else
        {
            lastPosIsLeft = false;
        }

        if (0 != (GetKeyState(VK_RBUTTON) & BUTTON_IS_DOWN))
        {
            if (lastPosIsRight)
            {
                s8 mouseChangeX = static_cast< s8 >( std::max< LONG >( -128, std::min< LONG >( 127, ( mousePos.x - lastPos.x ) ) ) );
                s8 mouseChangeY = static_cast< s8 >( std::max< LONG >( -128, std::min< LONG >( 127, ( mousePos.y - lastPos.y ) ) ) );
                g_DemoPad[ 0 ].substickX = MOUSE_SCALE * mouseChangeX;
                g_DemoPad[ 0 ].substickY = MOUSE_SCALE * mouseChangeY;
            }
            //printf("substick= %d %d\n", g_DemoPad[0].substickX, g_DemoPad[0].substickY);
            lastPosIsRight = true;
        }
        else
        {
            lastPosIsRight = false;
        }
        lastPos = mousePos;
    }
    return TRUE;
}

