﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// -----------------------------------------------------------------------------
//  demoSystem.h
//
// -----------------------------------------------------------------------------

#pragma once

#ifndef ASSERT
#include <cassert>
#define ASSERT assert
#endif

#ifdef CAFE
#include <cafe/os.h>
#endif

#define DEMO_INLINE static inline

#define DEMOAssert(expr) NN_SDK_ASSERT(expr)

#define DEMORoundUp256B(x) (((u32)(x) + 256 - 1) & ~(256 - 1))
#define DEMORoundUp32B(x) (((u32)(x) + 32 - 1) & ~(32 - 1))
#define DEMORoundUp4B(x) (((u32)(x) + 4 - 1) & ~(4 - 1))

#define DEMORoundUpForIO(x) (((u32)(x) + PPC_IO_BUFFER_ALIGN - 1) & ~(PPC_IO_BUFFER_ALIGN - 1))

#define DEMO_RAND_MAX ((u32)0xFFFF)

/// @addtogroup demoSystem
/// @{

/// \brief Initialize System
///
/// The DEMO library provides a common application framework
/// that is used in many of the example demos distributed with SDK.
/// The source code for the DEMO library is also distributed with the SDK.
///
/// This function initializes various system components including:
/// - OS: Base operation system
/// - Memory: All of the main memory is then allocated into a
///   heap that can be managed with \ref DEMOAlloc
void DEMOInit(void);

/// \brief Have \ref DEMOIsRunning() return false the next time it is called
///
/// This allows a controlled shutdown by letting the current loop finish
void DEMOStopRunning(void);

/// \brief Shutdown System
///
/// Shuts down all system components initialized by \ref DEMOInit
void DEMOShutdown(void);

/// \brief Set the main core
///
/// Sets the main core for DEMOIsRunning
void DEMOSetMainCore(int core);

/// \brief Set the random seed
///
/// Seeds the random number generator.
void DEMOSRand( u32 seed );

/// \brief Random float number generator
///
/// Returns a random number: 0 <= n <= 1.
/// \retval The next random number.
f32 DEMOFRand( void );

/// \brief Random integer number generator
///
/// Returns a random number: 0 <= n <= DEMO_RAND_MAX.
/// \retval The next random number.
u32 DEMORand( void );

/// \brief Type used for default DEMO memory allocator
typedef void* (*DEMODefaultAllocateFunc)(size_t byteCount, size_t alignment);

/// \brief Type used for default DEMO memory free function
typedef void (*DEMODefaultFreeFunc)(void* pMem);

/// \brief Set default functions to use for memory allocation/freeing.
///
/// These will be the functions called by DEMOAlloc/DEMOAllocEx/DEMOFree.
/// Those entry points are used by the DEMO libs when they need to allocate memory.
/// (Except for when non-regular-MEM2 arenas are needed.)
///
/// If not set by the user, these will just call MEMAllocFromDefaultHeap/MEMFreeToDefaultHeap.
///
/// \param pfnAlloc pointer to allocator function
/// \param pfnFree  pointer to free function
void DEMOSetDefaultAllocator(DEMODefaultAllocateFunc pfnAlloc, DEMODefaultFreeFunc pfnFree);

/// \brief Get default functions to use for memory allocation/freeing.
/// \param ppfnAlloc pointer to get pointer to allocator function
/// \param ppfnFree  pointer to get pointer to free function
void DEMOGetDefaultAllocator(DEMODefaultAllocateFunc *ppfnAlloc, DEMODefaultFreeFunc *ppfnFree);

/// \brief Allocate memory
///
/// \param size Size to allocate
/// \retval Pointer to the allocated buffer if allocation succeeded
void* DEMOAlloc( size_t size );

/// \brief Allocate memory with specific alignment
///
/// \param size Size to allocate
/// \param align Alignment to use for allocation
/// \retval Pointer to the allocated buffer if allocation succeeded
void* DEMOAllocEx( size_t size, size_t align );

/// \brief Free memory
///
/// \param ptr Pointer to the buffer to be deallocated
void DEMOFree(void* ptr);

/// \brief Return amount of used memory
///
/// \return Returns the amount of used memory if the default allocator is in use.
/// Otherwise it returns 0.
size_t DEMOGetUsedMemory();

/// \brief Get demo running state
///
/// \note This function also calls the DEMO Test functions, and therefore
///       it is expected that this function is only called once prior to each
///       main loop iteration.
///
/// \retval TRUE if \ref DEMOInit() has been called and DEMOStopRunning() has not been called; false otherwise.
BOOL DEMOIsRunning(void);

typedef void (*DEMOReleaseCallbackFunc)(void);

/// \brief Sets the callback for when a release occurs
///
/// \param func Function to call upon release message
void DEMOSetReleaseCallback(DEMOReleaseCallbackFunc func);

/// \brief Get current time
///
/// \retval The value of time in seconds since 1970-01-01 00:00:00
DEMO_INLINE
s64 DEMOGetTime()
{
#ifdef CAFE
    s64 diffSinceEpoch = 946627200;
    return OSTicksToSeconds(OSGetSystemTime()) + diffSinceEpoch;
#else
    return (s64)time(NULL);
#endif
}

/// \brief Prints formatted message to debug output
///
/// \param msg Pointer to a null-terminated string including format specification
/// (equivalent to C's standard output function).
/// \param ... Optional argument
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
#define DEMOPrintf NN_LOG
#else
void DEMOPrintf (const char* msg, ...);
#endif

/// \brief Perform a "fast" CPU copy
///
/// This copy makes no assumptions about source & dest alignment.
/// Both source & dest buffers are flushed out of the CPU cache.
/// The buffers must not overlap.
///
/// \note This function does not use locked-cache DMA or any non-CPU hardware.
/// It is therefore not the fastest way to copy, but it is usually faster than
/// just calling memcpy.  Better copy functions may become available in the future.
///
void DEMOFastCopy(void *dst, void *src, u32 size);

/// @}
