﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// -----------------------------------------------------------------------------
//  demoPad.h
//
// -----------------------------------------------------------------------------

#pragma once

#include <gfx/demo.h>

/// @addtogroup demoPad Demo Pad
/// @{

static const int DEMO_PAD_MAX_CONTROLLERS = 4;

static const int DEMO_PAD_BUTTON_LEFT =   0x0001; ///< Button Left (Win: Left Arrow.)
static const int DEMO_PAD_BUTTON_RIGHT =  0x0002; ///< Button Right (Win: Right Arrow.)
static const int DEMO_PAD_BUTTON_DOWN =   0x0004; ///< Button Down (Win: Down Arrow.)
static const int DEMO_PAD_BUTTON_UP =     0x0008; ///< Button Up (Win: Up Arrow.)
static const int DEMO_PAD_TRIGGER_Z =     0x0010; ///< Trigger Z (Win: Z key.)
static const int DEMO_PAD_TRIGGER_R =     0x0020; ///< Trigger R (Win: R key.)
static const int DEMO_PAD_TRIGGER_L =     0x0040; ///< Trigger L (Win: L key.)
static const int DEMO_PAD_BUTTON_A =      0x0100; ///< Button A (Win: A key.)
static const int DEMO_PAD_BUTTON_B =      0x0200; ///< Button B (Win: B key.)
static const int DEMO_PAD_BUTTON_X =      0x0400; ///< Button X (Win: X key.)
static const int DEMO_PAD_BUTTON_Y =      0x0800; ///< Button Y (Win: Y key.)
static const int DEMO_PAD_BUTTON_START =  0x1000; ///< Button Start (Win: S key.)

static const int DEMO_STICK_THRESHOLD =   0x0030; ///< Stick threshold
static const int DEMO_STICK_UP =          0x1000; ///< Stick up
static const int DEMO_STICK_DOWN =        0x2000; ///< Stick down
static const int DEMO_STICK_LEFT =        0x4000; ///< Stick left
static const int DEMO_STICK_RIGHT =       0x8000; ///< Stick right
static const int DEMO_SUBSTICK_UP =       0x0100; ///< Substick up
static const int DEMO_SUBSTICK_DOWN =     0x0200; ///< Substick down
static const int DEMO_SUBSTICK_LEFT =     0x0400; ///< Substick left
static const int DEMO_SUBSTICK_RIGHT =    0x0800; ///< Substick right

/// extended pad status structure
typedef struct
{
    u16         button;          ///< Or-ed DEMO_PAD_BUTTON_* and DEMO_PAD_TRIGGER_* bits
    s8          stickX;          ///< -128 <= stickX       <= 127 (Win: Emulated from mouse cursor with mouse L button.)
    s8          stickY;          ///< -128 <= stickY       <= 127 (Win: Emulated from mouse cursor with mouse L button.)
    s8          substickX;       ///< -128 <= substickX    <= 127 (Win: Emulated from mouse cursor with mouse R button.)
    s8          substickY;       ///< -128 <= substickY    <= 127 (Win: Emulated from mouse cursor with mouse R button.)
    u8          triggerLeft;     ///< 0 <= triggerLeft  <= 255 (Win: Mouse L button. Not analog value. 0 or 255)
    u8          triggerRight;    ///< 0 <= triggerRight <= 255 (Win: Mouse R button. Not analog value. 0 or 255)
    u8          analogA;         ///< 0 <= analogA      <= 255 (Win: Enter button. Not analog value. 0 or 255)
    u8          analogB;         ///< 0 <= analogB      <= 255 (Win: BackSpace button. Not analog value. 0 or 255)
    s8          err;             ///< one of PAD_ERR_* number
    u16         lastButton;      ///< Last button;

    u16         buttonDown;       ///< Button down
    u16         buttonUp;         ///< Button up
    u16         dirs;             ///< Directions
    u16         dirsNew;          ///< New directions
    u16         dirsReleased;     ///< Released directions
    s16         stickDeltaX;      ///< Stick delta x
    s16         stickDeltaY;      ///< Stick delta y
    s16         substickDeltaX;   ///< Substick delta x
    s16         substickDeltaY;   ///< Substick delta y
} DemoPadStatus;

extern DemoPadStatus  g_DemoPad[ DEMO_PAD_MAX_CONTROLLERS ]; ///< the entity which keeps current pad status
extern u32              g_DemoNumValidPads;             ///< the entity which keeps current pad status


/// \brief Initializes the controllers.
///
/// This function initializes the controllers.
/// The PADInit function enables the controller sampling performed by hardware.
///
void DEMOPadInit (void);


/// \brief Shutdown the controllers.
///
/// This function shuts down the controllers.
void DEMOPadShutdown (void);


/// \brief Gets the status of all controllers at once.
///
/// This function is used to sample the current complete set of Controller inputs.
/// Once a sample has been read, the individual values can be examined.
///
/// \return Return logical OR of all the controllers
u32 DEMOPadRead (void);

/// \brief Return bit field that indicates the button.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return Bit field that indicates the button.
DEMO_INLINE u16 DEMOPadGetButton (u32 i)
{
    return g_DemoPad[i].button;
}

/// \brief Return bit field that indicates the button up state.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return Bit field that indicates the button state.
DEMO_INLINE u16 DEMOPadGetButtonUp (u32 i)
{
    return g_DemoPad[i].buttonUp;
}

/// \brief Return bit field that indicates the button down state.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return Bit field that indicates the button state.
DEMO_INLINE u16 DEMOPadGetButtonDown (u32 i)
{
    return g_DemoPad[i].buttonDown;
}

/// \brief Return bit field that specifies the directional state of the stick.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return Bit field that specifies the directional state of the stick.
DEMO_INLINE u16 DEMOPadGetDirs (u32 i)
{
    return g_DemoPad[i].dirs;
}

/// \brief Return bit field that specifies the directional state of the stick.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return Bit field that specifies the directional state of the stick.
DEMO_INLINE u16 DEMOPadGetDirsNew (u32 i)
{
    return g_DemoPad[i].dirsNew;
}

/// \brief Return bit field that specifies the directional state of the stick.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return Bit field that specifies the directional state of the stick.
DEMO_INLINE u16 DEMOPadGetDirsReleased (u32 i)
{
    return g_DemoPad[i].dirsReleased;
}

/// \brief Return the input value of the stick X.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return The input value of the stick X.
DEMO_INLINE s8 DEMOPadGetStickX (u32 i)
{
    return g_DemoPad[i].stickX;
}

/// \brief Return the input value of the stick Y.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return The input value of the stick Y.
DEMO_INLINE s8  DEMOPadGetStickY(u32 i)
{
    return g_DemoPad[i].stickY;
}

/// \brief Return the input value of the sub stick X.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return The input value of the sub stick X.
DEMO_INLINE s8  DEMOPadGetSubStickX(u32 i)
{
    return g_DemoPad[i].substickX;
}

/// \brief Return the input value of the sub stick Y.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return The input value of the sub stick Y.
DEMO_INLINE s8  DEMOPadGetSubStickY(u32 i)
{
    return g_DemoPad[i].substickY;
}

/// \brief Return analog value of L that indicates how long the trigger or analog button has been pressed.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return Analog value that indicates how long the trigger or analog button has been pressed.
DEMO_INLINE u8  DEMOPadGetTriggerL(u32 i)
{
    return g_DemoPad[i].triggerLeft;
}

/// \brief Return analog value of R that indicates how long the trigger or analog button has been pressed.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return Analog value that indicates how long the trigger or analog button has been pressed.
DEMO_INLINE u8  DEMOPadGetTriggerR(u32 i)
{
    return g_DemoPad[i].triggerRight;
}

/// \brief Return analog value of A that indicates how long the trigger or analog button has been pressed.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return Analog value that indicates how long the trigger or analog button has been pressed.
DEMO_INLINE u8  DEMOPadGetAnalogA(u32 i)
{
    return g_DemoPad[i].analogA;
}

/// \brief Return analog value of B that indicates how long the trigger or analog button has been pressed.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return Analog value that indicates how long the trigger or analog button has been pressed.
DEMO_INLINE u8  DEMOPadGetAnalogB(u32 i)
{
    return g_DemoPad[i].analogB;
}

/// \brief Return the state of the various Controller inputs.
///
/// \param i Specifies Controller to read. The first Controller is 0, the second Controller is 1, etc.
/// \return A PAD_ERR* value.
DEMO_INLINE s8  DEMOPadGetErr(u32 i)
{
    return g_DemoPad[i].err;
}

/// @} // demoPad
