﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// -----------------------------------------------------------------------------
//  demoFWB.h
//
// -----------------------------------------------------------------------------

#pragma once

#include <gfx/demo.h>

/// @addtogroup demoFileWriteBuffer
/// @{

/// \brief Data container for Demo file-write-buffer (FWB) object.
typedef struct _DEMOFWBFileInfo
{
    /// The DEMOFSFileInfo data.
    DEMOFSFileInfo fileInfo;

    /// \brief A pointer to the large write buffer that holds all of the writes until they are flushed to the file.
    /// MUST be \c PPC_IO_BUFFER_ALIGN byte aligned.
    void* writeBuffer;

    /// The size of the writeBuffer buffer in bytes.
    u32 writeBufferSize;

    /// Current write location within the writeBuffer.  This is the next empty index/location to write data.
    u32 writeBufferWriteLocation;

} DEMOFWBFileInfo;


/// \brief Open a file and initialize the file-write-buffer (FWB) object.
///
/// Opens a file for writing.
///
/// If the path name begins with '/', it is used as an absolute path name.
/// If it does not, then "/vol/content/" is pre-prended to the path name.
///
/// \note If /vol/content has been remapped to use DVDFS, then you may not
///       open files there in "w" (write) mode.  Use /vol/save instead.
///
/// \param path Pointer to file name to open
/// \param fwbFileInfo Pointer to a DEMOFWBFileInfo structure that will be filled in if the function succeeds
/// \param mode Pointer to file mode to use. i.e. "w" for write.
/// \param writeBufferSize The size of the buffer, that will be internally allocated and used to buffer up prints
///        before writing them to the file. A larger buffer size will result in greater buffering and higher performance.
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_PERMISSION if it is not permitted to open current file.
/// \retval DEMO_FS_RESULT_CORRUPTION if a file is corrupted.
/// \retval DEMO_FS_FATAL_ERROR if a fatal error occurs during a read, the function returns.
s32 DEMOFWBOpenFile(const char* path, DEMOFWBFileInfo* fwbFileInfo, const char* mode, u32 writeBufferSize);


/// \brief Closes the file.
///
/// Closes the file tied to the file-write-buffer (FWB) object.  This will ensure all writes are flushed to the file.
///
/// \param fwbFileInfo File information for the file to close.
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_CORRUPTION if a file is corrupted.
/// \retval DEMO_FS_FATAL_ERROR if a fatal error occurs.
s32 DEMOFWBCloseFile(DEMOFWBFileInfo* fwbFileInfo);

/// \brief Flushes previous writes to the file-write-buffer (FWB) object to the file.
///        It is not required to ever call this function.  DEMOFWBCloseFile() will automatically
///        call this function and ensure all writes are flushed.
s32 DEMOFWBFlush(DEMOFWBFileInfo* fwbFileInfo);


/// \brief Writes data to the file-write-buffer (FWB) object.
///
/// \param fwbFileInfo Pointer to a DEMOFWBFileInfo structure.
/// \param bufferAddress Buffer address.
/// \param length Number of bytes to write.
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_FATAL_ERROR if there is an error.
s32 DEMOFWBWrite(DEMOFWBFileInfo* fwbFileInfo, void* bufferAddress, s32 length);


/// \brief Prints characters to the file-write-buffer (FWB) object.
///        No single write should be larger than writeBufferSize specified in DEMOFWBInit().
///        If a single write is too large, DEMO_FS_RESULT_AREA_FULL will be returned.
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_FATAL_ERROR if there is an error.
/// \retval DEMO_FS_RESULT_AREA_FULL if the single write is larger than writeBufferSize specified in DEMOFWBInit()
s32 DEMOFWBfprintf(DEMOFWBFileInfo* fwbFileInfo, const char * format, ...);

/// @}
