﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// -----------------------------------------------------------------------------
//  demoFs.h
//
// -----------------------------------------------------------------------------

#pragma once

#include <gfx/demo.h>

#if !NN_GFX_IS_TARGET_GX
#include <nn/fs.h>
#endif

/// @addtogroup demoFileSystem
/// @{

#if NN_GFX_IS_TARGET_GX
typedef struct _DEMOFSFileInfo
{
    void* pClient;
    s32 fileHandle;
} DEMOFSFileInfo;
#else
typedef nn::fs::FileHandle DEMOFSFileInfo;
#endif
typedef s32 DEMOFSDir;
typedef s32 DEMOFSDirEntry;

// At the moment, only 3 of these result values are ever returned:
// (This may change in the future.)

typedef enum
{
    DEMO_FS_RESULT_OK              =  0,  // YES!
    DEMO_FS_RESULT_ACCESS          = -1,  // no
    DEMO_FS_RESULT_PERMISSION      = -2,  // no
    DEMO_FS_RESULT_CORRUPTION      = -3,  // no
    DEMO_FS_RESULT_AREA_FULL       = -4,  // no
    DEMO_FS_RESULT_MAX_HANDLES     = -5,  // no
    DEMO_FS_RESULT_ALREADY_EXISTS  = -6,  // no
    DEMO_FS_RESULT_NOT_FOUND       = -7,  // YES!
    DEMO_FS_RESULT_NOT_EMPTY       = -8,  // no
    DEMO_FS_RESULT_FILE_OPEN       = -9,  // no
    DEMO_FS_RESULT_FATAL_ERROR     = -128 // YES!
} DEMOFSResultCode;


/// \brief Open a file handle for reading.
///
/// Opens a file for reading.
///
/// If the pathname begins with '/', it is used as an absolute path name.
/// If it does not, then "/vol/content/" is prepended to the path name.
/// The demoFSOpenFile function returns one of error codes.
///
/// \param path Pointer to file name to open
/// \param info Pointer to file information to use
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_PERMISSION if it is not permitted to open current file.
/// \retval DEMO_FS_RESULT_CORRUPTION if a file is corrupted.
/// \retval DEMO_FS_FATAL_ERROR if a fatal error occurs during a read, the function returns.
s32 DEMOFSOpenFile(const char* path, DEMOFSFileInfo* info);

/// \brief Open a file handle with a given mode.
///
/// Opens a file for reading/writing/etc.
///
/// If the pathname begins with '/', it is used as an absolute path name.
/// If it does not, then "/vol/content/" is prepended to the path name.
/// The DEMOFSOpenFileMode function returns one of error codes.
///
/// \note If /vol/content has been remapped to use DVDFS, then you may not
///       open files there in "w" (write) mode.  Use /vol/save instead.
///
/// \param path Pointer to file name to open
/// \param info Pointer to file information to use
/// \param mode Pointer to file mode to use
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_PERMISSION if it is not permitted to open current file.
/// \retval DEMO_FS_RESULT_CORRUPTION if a file is corrupted.
/// \retval DEMO_FS_FATAL_ERROR if a fatal error occurs during a read, the function returns.
s32 DEMOFSOpenFileMode(const char* path, DEMOFSFileInfo* info, const char* mode);

/// \brief Gets the size of the opened file.
///
/// Gets in advance the size of a file opened with the demoFSGetLength function.
///
/// This function is actually a macro defined in dvd.h.
///
/// \param fileInfo File information of the file
/// \param length The file size. As a return value, this size may not always be a multiple of 32. For this reason, when preparing buffers for reading files, this value must be rounded up to a multiple of 32
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_FATAL_ERROR if a fatal error occurs during a read, the function returns.
s32 DEMOFSGetLength(const DEMOFSFileInfo* fileInfo, u32* length);

/// \brief Synchronously reads from the opened file.
///
/// Synchronously reads data from a file. First opens the file with the DEMOOpenFile function.
///
/// \param fileInfo File information of the file.
/// \param addr Buffer address. Must be \c PPC_IO_BUFFER_ALIGN byte aligned.
/// \param length Number of bytes to read. Must be a multiple of 32.
/// \param offset File position to start reading from. Must be a multiple of 4.
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_ACCESS if it is no right to access file.
/// \retval DEMO_FS_RESULT_CORRUPTION if a file is corrupted.
/// \retval DEMO_FS_FATAL_ERROR if a fatal error occurs during a read, the function returns.
s32 DEMOFSRead(DEMOFSFileInfo* fileInfo, void* addr, size_t length, size_t offset);

/// \brief Writes data to a file.
///
/// This function may put the current thread to sleep.
///
/// \param fileInfo Pointer to a FSFileInfo structure.
/// \param addr Buffer address. Must be \c PPC_IO_BUFFER_ALIGN byte aligned.
/// \param length Number of bytes to write.
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_ACCESS if it is no right to access file.
/// \retval DEMO_FS_RESULT_CORRUPTION if a file is corrupted.
/// \retval DEMO_FS_RESULT_AREA_FULL if all regions given by application are used up. no more empty space.
s32 DEMOFSWrite(DEMOFSFileInfo* fileInfo, const void* addr, size_t length);

/// \brief Closes the file.
///
/// Closes the specified file.
/// if the FS library is not expecting a call (for example, during transfer) to the demoFSCloseFile function, the call to the function takes time.
/// Ensure that this situation doesn't occur.
/// (Namely, when calling demoFSCloseFile, make sure the transfer is complete.)
///
/// \param fileInfo File information for the file to close.
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_CORRUPTION if a file is corrupted.
/// \retval DEMO_FS_FATAL_ERROR if a fatal error occurs during a read, the function returns.
s32 DEMOFSCloseFile(DEMOFSFileInfo* fileInfo);

/// \brief Reads a file into an automatically created and allocated buffer.
///
/// This buffer is not aligned in any special way.
/// File is automatically closed after it's read from.
/// Call \ref DEMOFree() to free this buffer.
///
/// \param filename File string name to read
/// \param len Length of file returned here
///
void * DEMOFSSimpleRead(const char * filename, u32 * len);

/// \brief Reads a file into an automatically created and allocated buffer with alignment.
///
/// This buffer is aligned with alignSize.
/// File is automatically closed after it's read from.
/// Call \ref DEMOFree() to free this buffer.
///
/// \param filename File string name to read
/// \param len Length of file returned here
/// \param alignSize Alignment size for buffer
///
void * DEMOFSSimpleReadAlign(const char * filename, u32 * len, u32 alignSize);

/// \brief Scans directory for all the files in it.
///
/// This returns a list of the names of all the files in a directory.  Subdirectories
/// are automatically excluded.  Filenames, prefixed with pPrefixPath,
/// are returned, suitable for passing directly to DEMOFSOpenFile() or DEMOFSSimpleRead().
/// If the directory pathname begins with '/', it is used as an absolute path name.
/// If it does not, then "/vol/content/" is prepended to the path name.
///
/// \param pSearchPath Directory to search from
/// \param pPrefixPath Directory to prefix filenames with
/// \param pFileCount  Number of files in the directory
/// \param MaxFiles    Maximum number of file names to return from this function
/// \param ppFileNames Returned array of names of files.  The base array must be specified
///                    by the caller. (e.g char *ppFileNames[MaxFiles]; )
///                    Each non null string is allocated using DEMOAlloc() in this routine.
///                    Call DEMOFree() in caller to free them.
/// \return DEMO_FS_RESULT_OK on success, or error value.
s32 DEMOFSScanDir(const char *pSearchPath, const char *pPrefixPath, u32 *pFileCount, u32 MaxFiles, char** ppFileNames );

/// \brief Scans directory for all the files in it.
///
/// This returns a list of the names of all the files in a directory.  Subdirectories
/// are automatically excluded.  Full pathnames, beginning with the given directory pathname,
/// are returned, suitable for passing directly to DEMOFSOpenFile() or DEMOFSSimpleRead().
/// If the directory pathname begins with '/', it is used as an absolute path name.
/// If it does not, then "/vol/content/" is prepended to the path name.
///
/// \param pPath       Directory name
/// \param pFileCount  Number of files in the directory
/// \param MaxFiles    Maximum number of file names to return from this function
/// \param ppFileNames Returned array of names of files.  The base array must be specified
///                    by the caller. (e.g char *ppFileNames[MaxFiles]; )
///                    Each non null string is allocated using DEMOAlloc() in this routine.
///                    Call DEMOFree() in caller to free them.
/// \return DEMO_FS_RESULT_OK on success, or error value.
s32 DEMOFSSimpleScanDir(const char *pPath, u32 *pFileCount, u32 MaxFiles, char** ppFileNames );

/// \brief Rename a file or directory.
///
/// For each pathname, if it begins with '/', it is used as an absolute path name.
/// If it does not, then "/vol/content/" is prepended to the path name.
///
/// \note If /vol/content has been remapped to use DVDFS, then you may not
///       rename anything there.
///
/// \param oldpath Pointer to existing name to change
/// \param newpath Pointer to new name to use
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_FATAL_ERROR if a fatal error occurs during the operation.
s32 DEMOFSRename(const char* oldpath, const char *newpath);

/// \brief Remove a file or directory.
///
/// For each pathname, if it begins with '/', it is used as an absolute path name.
/// If it does not, then "/vol/content/" is prepended to the path name.
///
/// \note If /vol/content has been remapped to use DVDFS, then you may not
///       remove anything there.
///
/// \param path Pointer to existing name to remove
/// \param ignoreError If TRUE, ignore errors from underlying FSA remove call.
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_FATAL_ERROR if a fatal error occurs during the operation.
s32 DEMOFSRemove(const char* path, BOOL ignoreError);

/// \brief Create a directory recursively
///
/// For each pathname, if it begins with '/', it is used as an absolute path name.
/// If it does not, then "/vol/content/" is prepended to the path name.
///
/// \note If /vol/content has been remapped to use DVDFS, then you may not
///       rename anything there.
///
/// \param oldpath Pointer to existing name to change
/// \param newpath Pointer to new name to use
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_FATAL_ERROR if a fatal error occurs during the operation.
s32 DEMOFSCreateDirectory( const char* path );

/// @}
