﻿/*--------------------------------------------------------------------------------*
Copyright (C)Nintendo All rights reserved.

These coded instructions, statements, and computer programs contain proprietary
information of Nintendo and/or its licensed developers and are protected by
national and international copyright laws. They may not be disclosed to third
parties or copied or duplicated in any form, in whole or in part, without the
prior written consent of Nintendo.

The content herein is highly confidential and should be handled accordingly.
*--------------------------------------------------------------------------------*/
/**
* @examplesource{Parallel.cpp,PageSampleG3dParallel}
*
* @brief
* 並列処理を使用するサンプルプログラム
*/

/**
* @page PageSampleG3dParallel G3dDemo Parallel
* @tableofcontents
*
* @image html Applications\G3dDemo\parallel.png
*
* @brief
* サンプルプログラム Parallel の解説です。並列処理を使用したモデル更新をします。
*
* @section PageSampleG3dParallel_SectionBrief 概要
* 並列処理を使用したモデル更新を行うサンプルです。
*
* @section PageSampleG3dParallel_SectionFileStructure ファイル構成
* 本サンプルプログラムは @link ../../../Samples/Sources/Applications/G3dDemo
* Samples/Sources/Applications/G3dDemo @endlink 以下にあります。
*
* @section PageSampleG3dParallel_SectionNecessaryEnvironment 必要な環境
* 特になし。
*
* @section PageSampleG3dParallel_SectionHowToOperate 操作方法
* Joy-Con、DebugPad、PCの場合キーボード、マウスによる入力を用いて、以下の操作が可能です。
* <p>
* <table>
* <tr><th> 入力 </th><th> 動作 </th></tr>
* <tr><td> Startボタン、スペースキーまたはタッチスクリーン長押し、Joy-Con の+ボタン </td><td> メニューに戻ります。 </td></tr>
* </table>
* </p>
*
* @section PageSampleG3dParallel_SectionPrecaution 注意事項
* 特にありません。
*
* @section PageSampleG3dParallel_SectionHowToExecute 実行手順
* サンプルプログラムをビルドし、メニューから Parallel を選択し、実行してください。
*
* @section PageSampleG3dParallel_SectionDetail 解説
* このサンプルプログラムは、シェーダーアーカイブ、モデルリソースを読み込み、アニメーションを適用し、
* 並列処理での更新を行ったモデルを表示します。
*
* サンプルプログラムの処理の流れは以下の通りです。
*
* - シェーダーアーカイブ読み込み、初期化
* - モデルリソース読み込み、初期化およびテクスチャー初期化
* - サンプラーをディスクリプタプールに登録
* - nns::g3d を使用したモデルのセットアップ
* - 並列処理用のスレッドを初期化
* - ループ開始
* - 前フレームで作成したコマンドリスト実行
* - 各行列更新
* - 並列処理でのモデル更新
* - コマンドリスト生成
* - ループ開始に戻る
*
*/

#include "SampleSources/g3ddemo_GfxUtility.h"
#include "SampleSources/g3ddemo_ModelUtility.h"
#include "SampleSources/g3ddemo_ViewerUtility.h"

#include <nn/os.h>
#include <nn/g3d/g3d_Viewer.h>
#include <nn/g3d.h>
#include <nn/time.h>
#include <nnt/graphics/testGraphics_PerformanceProfileData.h>

namespace g3ddemo = nn::g3d::demo;

namespace {

#define RESOURCE_PATH "Resource:/"
const char* g_DemoBfshaPath = RESOURCE_PATH "shader/demo.bfsha";
const char* g_EnvBfresPath = RESOURCE_PATH "env.bfres";
const char* g_BoyBfresPath = RESOURCE_PATH "boy.bfres";

nn::util::Vector3fType g_CameraPosition;
nn::util::Vector3fType g_CameraUp;
nn::util::Vector3fType g_CameraTarget;

g3ddemo::ResourceHolder g_Holder;
g3ddemo::ScreenInfo g_ScreenInfo;
g3ddemo::LoadMeter g_LoadMeter;
nns::g3d::RenderView g_RenderView;

const size_t g_ThreadCommandBufferCount = 2; // コマンドバッファーをダブルバッファーで作成する
const size_t g_ThreadCount = 3; // 並列処理に使用するスレッド数
const size_t g_ThreadStackSize = 0x10000; // スレッド操作スレッドのスタックサイズ
const size_t g_WorkMemoryCount = 64; // ワークメモリーを確保できる個数

NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[g_ThreadCount][g_ThreadStackSize]; //スレッドのスタック
nn::os::ThreadType g_Thread[g_ThreadCount];
nn::os::EventType g_EventStart[g_ThreadCount];
nn::os::EventType g_EventEnd[g_ThreadCount];
nns::gfx::GraphicsFramework::CommandBuffer g_ThreadCommandBuffer[g_ThreadCount][g_ThreadCommandBufferCount];
const float g_GridDistance = 200.0f;
const int g_InstanceModelCount = 1000;
const int g_ModelLodCount = 2;
const float g_LodThresholds[g_ModelLodCount] = { 1.0f, 0.2f };
const float g_DoubleZnear = 1.0f / nn::util::TanEst(nn::util::DegreeToRadian(45.0f) * 0.5f);
const nn::Bit64 g_CoreMask[] = { 1, 2, 4 };

typedef void (*TaskFunc)(int, int); // 並列処理される関数ポインター型
TaskFunc g_TaskFunc;
int g_BufferIndex;
bool g_IsExit;

const int g_BufferingCount = 2;

class TaskQueue
{
public:
    TaskQueue() {}

    struct Task
    {
        int modelIndex;
    };

    void Initialize(int threadId, int queueCount)
    {
        m_ThreadId = threadId;
        m_QueueCount = queueCount;
        size_t size = sizeof(Task) * m_QueueCount;
        void* pBuffer = g3ddemo::AllocateMemory(size);
        m_Tasks.SetBuffer(pBuffer, size);
    }

    void Cleanup()
    {
        m_Tasks.Clear();
        if (void* pBuffer = m_Tasks.GetBuffer())
        {
            g3ddemo::FreeMemory(pBuffer);
        }
    }

    void PushBack(int modelIndex)
    {
        if (m_Tasks.GetCount() < m_QueueCount)
        {
            Task task;
            task.modelIndex = modelIndex;
            m_Tasks.PushBack(task);
        }
    }

    int GetTaskCount() const
    {
        return m_Tasks.GetCount();
    }

    Task* GetTask(int index)
    {
        return (index >= 0 && index < m_Tasks.GetCount()) ? &m_Tasks[index] : nullptr;
    }

    int GetThreadId() const
    {
        return m_ThreadId;
    }

private:
    g3ddemo::Vector<Task> m_Tasks;
    int m_ThreadId;
    int m_QueueCount;

} g_TaskQueue[g_ThreadCount];

nns::gfx::GraphicsFramework::CommandBuffer* GetThreadCommandBuffer(int threadIndex)
{
    return &g_ThreadCommandBuffer[threadIndex][g_BufferIndex];
}

void RunTaskQueue(TaskFunc taskFunc, int bufferIndex)
{
    // タスク処理に使用される関数を設定
    g_TaskFunc = taskFunc;
    g_BufferIndex = bufferIndex;

    // 並列処理の開始
    for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
    {
        nn::os::SignalEvent(&g_EventStart[threadIndex]);
    }

    // すべてのスレッドが終了するまで待つ
    for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
    {
        nn::os::WaitEvent(&g_EventEnd[threadIndex]);
    }
}

float CalcProjectionRatio(const nn::g3d::Sphere* sphere)
{
    float centerX = nn::util::VectorGetX(sphere->center);
    float centerY = nn::util::VectorGetY(sphere->center);
    float centerZ = nn::util::VectorGetZ(sphere->center);

    const nn::util::Matrix4x3fType& view = g_RenderView.GetViewMtx(0);
    nn::util::Vector3fType vec[4];
    nn::util::MatrixGetAxisX(&vec[0], view);
    nn::util::MatrixGetAxisY(&vec[1], view);
    nn::util::MatrixGetAxisZ(&vec[2], view);
    nn::util::MatrixGetAxisW(&vec[3], view);

    float view02 = nn::util::VectorGetZ(vec[0]);
    float view12 = nn::util::VectorGetZ(vec[1]);
    float view22 = nn::util::VectorGetZ(vec[2]);
    float view32 = nn::util::VectorGetZ(vec[3]);
    float viewZ = view02 * centerX + view12 * centerY + view22 * centerZ + view32;

    // バウンディングスフィアをカメラ投影したときの割合を計算
    return (viewZ != 0.0f) ? std::abs(sphere->radius * g_DoubleZnear / viewZ) : 1.0f;
}

void UpdateAnimModelObj(int modelIndex, int threadIndex)
{
    NN_UNUSED(threadIndex);

    nns::g3d::RenderModelObj* pRenderModelObj = g_Holder.renderModelObjs[modelIndex];
    float ratio = -1.0f;
    const nn::g3d::Sphere* sphere = pRenderModelObj->GetModelObj()->GetBounding();
    pRenderModelObj->SetDrawLodEnabled(0);

    if (sphere)
    {
        if (ratio < 0.0f)
        {
            ratio = CalcProjectionRatio(sphere);
        }

        for (int lodIndex = 1; lodIndex < g_ModelLodCount; ++lodIndex)
        {
            if (ratio < g_LodThresholds[lodIndex])
            {
                pRenderModelObj->SetDrawLodEnabled(lodIndex);
            }
        }
    }

    nns::g3d::ModelAnimObj* pModelAnimObj = g_Holder.modelAnimObjs[modelIndex];
    pModelAnimObj->CalculateView(0, g_BufferIndex, g_RenderView.GetViewMtx(0));
    pModelAnimObj->Calculate();
    pModelAnimObj->CalculateUniformBlock(g_BufferIndex);

    // 描画コマンドを追加
    nns::gfx::GraphicsFramework::CommandBuffer* pCommandBuffer = GetThreadCommandBuffer(threadIndex);
    pRenderModelObj->Draw(&pCommandBuffer->object, 0, g_BufferIndex);
}

void ExecuteTask(TaskQueue* pTaskQueue)
{
    int threadIndex = pTaskQueue->GetThreadId();
    nns::gfx::GraphicsFramework::CommandBuffer* pCommandBuffer = GetThreadCommandBuffer(threadIndex);
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();

    // コマンドバッファーをリセット
    pGfxFramework->ResetCommandBuffer(pCommandBuffer);

    // コマンドバッファーへの追加を開始
    pCommandBuffer->object.Begin();

    for (int taskIndex = 0, taskCount = pTaskQueue->GetTaskCount(); taskIndex < taskCount; ++taskIndex)
    {
        TaskQueue::Task* pTask = pTaskQueue->GetTask(taskIndex);
        g_TaskFunc(pTask->modelIndex, threadIndex);
    }

    // コマンドバッファーへの追加を終了
    pCommandBuffer->object.End();
}

void ThreadMain(void* arg)
{
    TaskQueue* pTaskQueue = reinterpret_cast<TaskQueue*>(arg);
    int threadId = pTaskQueue->GetThreadId();

    while (!g_IsExit)
    {
        nn::os::WaitEvent(&g_EventStart[threadId]);

        //NN_PERF_SET_COLOR(nn::util::Color4u8::Red());
        //NN_PERF_BEGIN_MEASURE();

        if (!g_IsExit) // 終了要求時には更新せず抜ける
        {
            ExecuteTask(pTaskQueue);
        }

        //NN_PERF_END_MEASURE();

        nn::os::SignalEvent(&g_EventEnd[threadId]);
    }
}

void Calculate(nns::gfx::GraphicsFramework* pGfxFramework, void* pUserData)
{
    NN_UNUSED(pGfxFramework);

    int uniformBufferIndex = *reinterpret_cast<int*>(pUserData);

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_BEGIN_MEASURE();

    // 計算
    {
        // カメラ設定
        VectorSet(&g_CameraPosition, 0.0f, 100.0f, 3600.0f);
        VectorSet(&g_CameraUp, 0.0f, 1.0f, 0.0f);
        VectorSet(&g_CameraTarget, 0.0f, 0.0f, 0.0f);
        {
            nn::util::Matrix4x3fType matrix;
            MatrixLookAtRightHanded(&matrix,
                g_CameraPosition,
                g_CameraTarget,
                g_CameraUp);
            g_RenderView.SetViewMtx(0, matrix);
        }

        // 投影行列
        {
            nn::util::Matrix4x4fType matrix;
            MatrixPerspectiveFieldOfViewRightHanded(&matrix,
                nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f, 10.0f, 40000.0f);
            g_RenderView.SetProjectionMtx(0, matrix);
        }

        // ビューの更新処理
        g_RenderView.Calculate(uniformBufferIndex);
    }

    NN_PERF_END_MEASURE();
}

void MakeCommand(nns::gfx::GraphicsFramework* pGfxFramework, int bufferIndex, void* pUserData)
{
    int uniformBufferIndex = *reinterpret_cast<int*>(pUserData);

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_BEGIN_MEASURE();

    // コマンド生成
    pGfxFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* pGfxCommandBuffer = pGfxFramework->GetRootCommandBuffer(bufferIndex);

        NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Blue());
        NN_PERF_BEGIN_MEASURE_GPU(pGfxCommandBuffer);

        // カラーターゲットをクリア
        nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
        pGfxCommandBuffer->ClearColor(pColor, 0.0f, 0.0f, 0.0f, 1.0f, nullptr);

        // 深度ステンシルをクリア
        nn::gfx::DepthStencilView* pDepth = pGfxFramework->GetDepthStencilView();
        pGfxCommandBuffer->ClearDepthStencil(pDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        // レンダーターゲットをセット
        pGfxCommandBuffer->SetRenderTargets(1, &pColor, pDepth);

        // ビューポートシザーステートをセット
        pGfxCommandBuffer->SetViewportScissorState(pGfxFramework->GetViewportScissorState());

        // モデルの描画コマンドの作成をマルチスレッドで並列実行
        RunTaskQueue(UpdateAnimModelObj, uniformBufferIndex);

        // 各スレッドで作成されたコマンドバッファーをメインのコマンドバッファーに追加
        for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
        {
            nns::gfx::GraphicsFramework::CommandBuffer* pCommandBuffer = GetThreadCommandBuffer(threadIndex);
            pGfxCommandBuffer->CallCommandBuffer(&pCommandBuffer->object);
        }

        // テキスト表示 コマンド生成
        {
            g_ScreenInfo.StartPrint();
            g_ScreenInfo.Print("[Parallel]\n");
            g_ScreenInfo.EndPrint();
            g_ScreenInfo.AdjustWindowSize();
            g_ScreenInfo.Draw(pGfxCommandBuffer);
        }

        // 計測メーター表示 コマンド生成
        {
            g_LoadMeter.Draw(pGfxCommandBuffer);
        }

        NN_PERF_END_MEASURE_GPU(pGfxCommandBuffer);
    }
    pGfxFramework->EndFrame(bufferIndex);

    NN_PERF_END_MEASURE();
}

// リソースの初期化
void InitializeResource(nn::gfx::Device* pDevice)
{
    g_Holder.Initialize();

    // シェーダーアーカイブの初期化
    nn::g3d::ResShaderArchive* pDemoShaderArchive;
    {
        nn::g3d::ResShaderFile* pFile = g3ddemo::LoadResource<nn::g3d::ResShaderFile>(g_DemoBfshaPath);
        pDemoShaderArchive = pFile->GetResShaderArchive();
        pDemoShaderArchive->Setup(pDevice);
        g_Holder.shaderFiles.PushBack(pFile);
    }

    // モデルリソースの初期化
    nn::g3d::ResFile* pEnvModelFile;
    {
        pEnvModelFile = g3ddemo::LoadResource<nn::g3d::ResFile>(g_EnvBfresPath);
        pEnvModelFile->Setup(pDevice);
        g_Holder.files.PushBack(pEnvModelFile);
    }
    nn::g3d::ResFile* pBoyModelFile;
    {
        pBoyModelFile = g3ddemo::LoadResource<nn::g3d::ResFile>(g_BoyBfresPath);
        pBoyModelFile->Setup(pDevice);
        g3ddemo::SetupTexture(pDevice, pBoyModelFile, g3ddemo::TextureBindCallback);
        g3ddemo::RegisterSamplerToDescriptorPool(pBoyModelFile);
        g_Holder.files.PushBack(pBoyModelFile);
    }

    // モデルの頂点ステート、デプスステンシルステート、ブレンドステート、ラスタライザーステート、シェーディングモデルを管理するRenderModelを作成
    nns::g3d::RenderModel* pDemoRenderModel;
    {
        nn::g3d::ResModel* pResModel = pBoyModelFile->GetModel(0);
        pDemoRenderModel = nns::g3d::CreateRenderModel(pDevice, pResModel);
        pDemoRenderModel->CreateDrawPass(pDevice, pDemoShaderArchive);
        g_Holder.renderModels.PushBack(pDemoRenderModel);
    }

    // ビューを初期化
    g_RenderView.Initialize(pDevice, 1, g_BufferingCount);

    // 描画時に必要となるg3dの管理外のユニフォームブロックを作成
    nn::g3d::ModelObj* pEnvModelObj;
    {
        nn::g3d::ResModel* pResModel = pEnvModelFile->GetModel(0);
        nn::g3d::ShaderUtility::BindShaderParam(pResModel->GetMaterial(0), pDemoShaderArchive->FindShadingModel("env"));
        nn::g3d::ModelObj::Builder builder(pResModel);
        builder.MaterialBufferingCount(g_BufferingCount);
        pEnvModelObj = nns::g3d::CreateModelObj(pDevice, builder);
        // このデモでは毎フレーム更新が必要でないため、ここで1度だけ計算を行う
        pEnvModelObj->CalculateMaterial(0);
        pEnvModelObj->CalculateMaterial(1);
        g_Holder.modelObjs.PushBack(pEnvModelObj);
    }

    nn::g3d::ResSkeletalAnim* pJumpAnim = pBoyModelFile->FindSkeletalAnim("boy_jump");

    // モデルおよびアニメーションの計算処理を簡略化するModelAnimObjを作成
    for (int instanceIndex = 0; instanceIndex < g_InstanceModelCount; ++instanceIndex)
    {
        nns::g3d::ModelAnimObj* pBoyModelAnimObj;
        {
            nn::g3d::ModelObj::Builder builder(pBoyModelFile->GetModel(0));
            // CPUとGPUで並列実行出来るようにバッファー数に2を指定
            builder.ShapeBufferingCount(g_BufferingCount);
            builder.SkeletonBufferingCount(g_BufferingCount);
            builder.MaterialBufferingCount(g_BufferingCount);
            builder.SetBoundingEnabled();
            pBoyModelAnimObj = nns::g3d::CreateModelAnimObj(pDevice, builder);
            g_Holder.modelAnimObjs.PushBack(pBoyModelAnimObj);

            // アニメーション追加
            int jumpId = pBoyModelAnimObj->CreateAnim(pJumpAnim);

            // 再生開始フレームをランダムに設定
            float frame = pJumpAnim->GetFrameCount() * (static_cast<float>(instanceIndex) / static_cast<float>(g_InstanceModelCount));
            pBoyModelAnimObj->GetAnimObj(jumpId)->GetFrameCtrl().SetFrame(frame);
        }

        // RenderModelとModelObjを組み合わせ、描画を行うRenderModelObjを作成
        nns::g3d::RenderModelObj* pBoyRenderModelObj;
        {
            pBoyRenderModelObj = nns::g3d::CreateRenderModelObj(pBoyModelAnimObj->GetModelObj(), pDemoRenderModel);
            g_Holder.renderModelObjs.PushBack(pBoyRenderModelObj);
            // このデモではシェーダーを動的に切り替えないため、ここでシェーダーを選択
            g3ddemo::WriteSkinningOption(pBoyRenderModelObj);
            pBoyRenderModelObj->UpdateShader(pDevice);
        }

        // EnvモデルのユニフォームブロックをRenderModelObjに設定
        {
            const nn::gfx::Buffer* buffers[g_BufferingCount];
            buffers[0] = pEnvModelObj->GetMaterial(0)->GetMaterialBlock(0);
            buffers[1] = pEnvModelObj->GetMaterial(0)->GetMaterialBlock(1);
            pBoyRenderModelObj->BindUniformBlock("env", buffers, g_BufferingCount);
        }

        {
            const nn::gfx::Buffer* buffers[g_BufferingCount];
            buffers[0] = g_RenderView.GetViewUniformBlock(0, 0);
            buffers[1] = g_RenderView.GetViewUniformBlock(0, 1);
            pBoyRenderModelObj->BindUniformBlock("view", buffers, g_BufferingCount);
        }
    }

    // シェーダーに必要なスクラッチメモリーを設定
    g3ddemo::SetShaderScratchMemory(g_Holder.shaderFiles);

    // レイアウト計算
    {
        int matrixSize = static_cast<int>(std::sqrt(static_cast<float>(g_Holder.modelAnimObjs.GetCount())));
        float center = (g_GridDistance * matrixSize) / 2;
        int xGrid = 0, zGrid = 0;

        int modelIndex = 0;
        for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
        {
            float randomX = (static_cast<float>(modelIndex) / static_cast<float>(matrixSize)) - 0.5f;
            float xStir = randomX * 0.5f * g_GridDistance;
            float randomZ = (static_cast<float>(modelIndex) / static_cast<float>(matrixSize)) - 0.5f;
            float zStir = randomZ * 0.5f * g_GridDistance;

            nn::util::Vector3fType position;
            nn::util::VectorSet(&position, xGrid * g_GridDistance - center + xStir, 0.0f, zGrid * g_GridDistance - center + zStir);
            pModelAnimObj->SetTranslate(position);

            ++xGrid;
            modelIndex++;

            if (xGrid == matrixSize)
            {
                xGrid = 0;
                ++zGrid;
            }
        }
    }
} //NOLINT(impl/function_size)

} // anonymous namespace

//--------------------------------------------------------------------------------------------------
int ParallelMain(bool isUseRankTurn)
{
    nn::os::Tick beginTick = nn::os::GetSystemTick();
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    pGfxFramework->SetFrameworkMode(nns::gfx::GraphicsFramework::FrameworkMode_DeferredSubmission);

    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // スクリーン情報表示を初期化
    {
        g_ScreenInfo.Initialize(pDevice);
        g_ScreenInfo.SetWindowPosition(16, 16);
        g_ScreenInfo.SetFontSize(20);
    }

    // 負荷計測クラスの初期化
    {
        nn::perf::LoadMeterCenterInfo info;
        info.SetDefault();
        info.SetCoreCount(3);
        info.SetCpuSectionCountMax(64);
        info.SetGpuSectionCountMax(64);
        info.SetCpuBufferCount(2);
        info.SetGpuBufferCount(3);
        g_LoadMeter.Initialize(pDevice, info);
    }

    // ワークメモリー管理の初期化
    g3ddemo::Vector<void*> workMemorys;
    {
        size_t size = sizeof(void*) * g_WorkMemoryCount;
        void* pBuffer = g3ddemo::AllocateMemory(size);
        workMemorys.SetBuffer(pBuffer, size);
    }

    // 現在のスレッド(メインスレッド)を 0 番コアに割り当て
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);

    // リソースの初期化
    InitializeResource(pDevice);

    // マルチタスク起動
    {
        g_IsExit = false;

        // 各スレッドで使用するコマンドバッファーを作成
        nn::gfx::CommandBuffer::InfoType info;
        info.SetDefault();
        info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
        info.SetCommandBufferType(nn::gfx::CommandBufferType_Nested);
        size_t commandMemorySize = 1024 * 1024 * 2;
        size_t controlMemorySize = 1024;
        for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
        {
            for (int bufferIndex = 0; bufferIndex < g_ThreadCommandBufferCount; ++bufferIndex)
            {
                pGfxFramework->InitializeCommandBuffer(&g_ThreadCommandBuffer[threadIndex][bufferIndex],
                    info, commandMemorySize, controlMemorySize);
            }
        }

        // スレッド上で処理するタスクを作成
        for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
        {
            TaskQueue* pTaskQueue = &g_TaskQueue[threadIndex];
            pTaskQueue->Initialize(threadIndex, g_InstanceModelCount);
        }

        // 各タスクに計算するモデルを振り分け
        int modelCount = g_Holder.renderModelObjs.GetCount();
        for (int modelIndex = 0; modelIndex < modelCount; ++modelIndex)
        {
            TaskQueue* pTaskQueue = &g_TaskQueue[modelIndex % g_ThreadCount];
            pTaskQueue->PushBack(modelIndex);
        }

        // 同期用イベントを作成
        for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
        {
            nn::os::InitializeEvent(&g_EventStart[threadIndex], false, nn::os::EventClearMode_AutoClear);
            nn::os::InitializeEvent(&g_EventEnd[threadIndex], false, nn::os::EventClearMode_AutoClear);
        }

        // コア番号を指定してスレッドを作成
        for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
        {
            nn::Result result;
            result = nn::os::CreateThread(&g_Thread[threadIndex], ThreadMain, &g_TaskQueue[threadIndex], g_ThreadStack[threadIndex], g_ThreadStackSize, nn::os::DefaultThreadPriority);
            NN_ASSERT(result.IsSuccess(), "Cannot create thread.");
            nn::os::SetThreadCoreMask(&g_Thread[threadIndex], threadIndex, g_CoreMask[threadIndex]);
        }

        // スレッドの実行を開始
        for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
        {
            nn::os::StartThread(&g_Thread[threadIndex]);
        }
    }

    int uniformBufferIndex = 0;
    pGfxFramework->SetCalculateCallback(Calculate, &uniformBufferIndex);
    pGfxFramework->SetMakeCommandCallback(MakeCommand, &uniformBufferIndex);
    pGfxFramework->ResetFrame();

    g3ddemo::Pad& pad = g3ddemo::GetPad();
    pad.Reset();

    int frameCount = 0;

    // 結果出力ファイルの初期化
    nnt::graphics::PerformanceProfileData  profile;
    profile.Initialize(1024, g3ddemo::AllocateMemory, g3ddemo::FreeMemory);
    profile.SetName("Parallel");
    nn::time::PosixTime posixTime;
    nn::time::CalendarTime  calendarTime;
    nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
    nn::time::ToCalendarTime(&calendarTime, NULL, posixTime);
    char timeString[32];
    nn::util::SNPrintf(timeString, 32,
        "%04d-%02d-%02d %02d:%02d:%02d",
        calendarTime.year, calendarTime.month, calendarTime.day,
        calendarTime.hour, calendarTime.minute, calendarTime.second);
    profile.SetDate(timeString);

    // 15 秒間計測する
    int profiledFrameCount = 0;
    int profileFrameCount = 60 * 15;
    if (isUseRankTurn)
    {
        profileFrameCount = 60 * 240;
    }

    nn::os::Tick endTick = nn::os::GetSystemTick();
    NN_LOG("InitialParallelCost:%f(ms)\n", (endTick.ToTimeSpan().GetMicroSeconds() - beginTick.ToTimeSpan().GetMicroSeconds()) / 1000.f);

    // メインループ
    while (NN_STATIC_CONDITION(1))
    {
        // タッチ長押しで終了
        if (g3ddemo::isExitDemo())
        {
            break;
        }

        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            break;
        }

        NN_PERF_BEGIN_FRAME();

        // フレーム処理実行
        pGfxFramework->ProcessFrame();
        uniformBufferIndex ^= 1;

        // システムコアが落ち着くまで開始後 5 秒間は計測しない
        if (frameCount > 300)
        {
            NN_PERF_END_FRAME_NO_CLEAR_PROFILE();
            profiledFrameCount++;
            // profileFrameCount 分計測したら計測を終了する
            if (profiledFrameCount == profileFrameCount)
            {
                break;
            }
        }
        else
        {
            NN_PERF_END_FRAME();
        }
        frameCount++;
    }
    pGfxFramework->QueueFinish();

    uint64_t cpu0ElapsedMicroSeconds = NN_PERF_GET_ELAPSED_TIME_CPU("CPU0", NULL, 0).GetMicroSeconds();
    uint64_t gpuElapsedMicroSeconds = NN_PERF_GET_ELAPSED_TIME_GPU(NULL, 0).GetMicroSeconds();

    float cpu0Average = cpu0ElapsedMicroSeconds / static_cast<float>(profiledFrameCount);
    float gpuAverage = gpuElapsedMicroSeconds / static_cast<float>(profiledFrameCount);

    NN_LOG("%f %f\n", cpu0Average, gpuAverage);
    // 結果をファイルに出力
    profile.SetCpuValue(0, static_cast<int64_t>(cpu0Average));
    profile.SetGpuValue(0, static_cast<int64_t>(gpuAverage));

    nn::util::SNPrintf(timeString, 32,
        "%04d%02d%02d_%02d%02d%02d",
        calendarTime.year, calendarTime.month, calendarTime.day,
        calendarTime.hour, calendarTime.minute, calendarTime.second);
    char filePath[1024];
    nn::util::SNPrintf(filePath, 1024, "Outputs:/Outputs/Parallel_%s.json", timeString);
    if (!isUseRankTurn)
    {
        profile.Write(filePath);
    }
    profile.Finalize();

    // マルチタスク終了
    {
        g_IsExit = true;

        // スレッドが終了するのを待つ
        for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
        {
            nn::os::SignalEvent(&g_EventStart[threadIndex]);
            nn::os::WaitThread(&g_Thread[threadIndex]);
        }

        // スレッドを破棄
        for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
        {
            nn::os::DestroyThread(&g_Thread[threadIndex]);
        }

        // 同期イベントを破棄
        for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
        {
            nn::os::FinalizeEvent(&g_EventStart[threadIndex]);
        }
    }

    // シェーダーの排他制御用のメモリーの破棄
    {
        int workMemoryCount = workMemorys.GetCount();
        for (int workMemoryIndex = 0; workMemoryIndex < workMemoryCount; ++workMemoryIndex)
        {
            void* pWorkMemory = workMemorys[workMemoryIndex];
            g3ddemo::FreeMemory(pWorkMemory);
        }

        if (void* pBuffer = workMemorys.GetBuffer())
        {
            g3ddemo::FreeMemory(pBuffer);
        }
    }

    // 各スレッドで使用したコマンドバッファーの破棄
    for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
    {
        for (int bufferIndex = 0; bufferIndex < g_ThreadCommandBufferCount; ++bufferIndex)
        {
            pGfxFramework->FinalizeCommandBuffer(&g_ThreadCommandBuffer[threadIndex][bufferIndex]);
        }
    }

    // タスクの破棄
    {
        for (int threadIndex = 0; threadIndex < g_ThreadCount; ++threadIndex)
        {
            TaskQueue* pTaskQueue = &g_TaskQueue[threadIndex];
            pTaskQueue->Cleanup();
        }
    }

    // 終了処理
    g3ddemo::DestroyAll(pDevice, &g_Holder);

    // ビュー用ユニフォームブロックの破棄
    g_RenderView.Finalize(pDevice);

    // 負荷計測メーターの破棄
    g_LoadMeter.Cleanup(pDevice);

    // スクリーン情報表示の破棄
    g_ScreenInfo.Cleanup(pDevice);

    pGfxFramework->SetCalculateCallback(nullptr, nullptr);
    pGfxFramework->SetMakeCommandCallback(nullptr, nullptr);

    return EXIT_SUCCESS;
} // NOLINT
