﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <vector>
#include <nn/mii.h>
#include <nn/nn_Assert.h>
#include <nn/g3d/g3d_ResModel.h>
#include <nn/g3d/g3d_ResMaterial.h>
#include <nn/g3d/g3d_Mii.h>
#include "testG3d_MiiG3dMiiCharModelBase.h"
#include "testG3d_MiiG3dMiiUtil.h"

//!< テストで使用するモデルに関する情報を保持しています。
//!< MaterialCallback, DrawParamCallback はモデルによって違うので G3dMiiCharModelBase を
//!< 継承した子クラスに実装します。
//!< 再利用する場合は必ず Finalize() 後に再利用して下さい。
class MiiG3dTestModelSimple : public G3dMiiCharModelBase
{
public:
    MiiG3dTestModelSimple()
        : m_MaterialCallbackParam()
        , m_DrawParamCallbackParam( s_MaterialNameDrawTypeTableArray,
                                    s_MaterialNameDrawTypeTableArraySize,
                                    s_BoneNameCreateModelTypeNoseTypeTableArray,
                                    s_BoneNameCreateModelTypeNoseTypeTableArraySize,
                                    this )
    {
        SetDrawParamCallback( this->DrawParamCallback );
        SetDrawParamCallbackParam( &this->m_DrawParamCallbackParam );
        SetMaterialCallback( this->MaterialCallback );
        SetMaterialCallbackParam( &this->m_MaterialCallbackParam );
    }

    virtual ~MiiG3dTestModelSimple()
    {
    }

    void Finalize()
    {
        m_DrawParamCallbackOutputArray.clear();
        G3dMiiCharModelBase::Finalize();
    }

    int GetVisibleBoneCount()
    {
        return static_cast<int>( s_VisibleBoneNameArraySize );
    }

    const char* GetVisibleBoneName( int idxVisibleBoneName )
    {
        NN_ASSERT( idxVisibleBoneName < static_cast<int>( s_VisibleBoneNameArraySize ) );
        return s_VisibleBoneNameArray[idxVisibleBoneName];
    }

    //!< shapeIndex に対応する DrawParamCallbackOutput を取得します。これがテストの正解値になります。
    const nn::g3d::mii::DrawParamCallbackOutput& GetGoldenDrawParamOutput( int shapeIndex )
    {
        NN_ASSERT( shapeIndex < static_cast<int>( m_DrawParamCallbackOutputArray.size() ) );
        return m_DrawParamCallbackOutputArray[shapeIndex];
    }

private:
    // render_info, shader_param のデフォルト値登録用コールバックパラメータ
    struct MaterialCallbackParam
    {
    public:
        MaterialCallbackParam() :
            renderInfoCullingName( "culling" ),
            shaderParamModulateTypeId( "mii_modulate_type" ),
            shaderParamConstColorId( "mii_const_color" )
        {
        }

        const char* renderInfoCullingName;
        const char* shaderParamModulateTypeId;
        const char* shaderParamConstColorId;
    };

    class DrawParamCallbackParam
    {
    public:
        explicit DrawParamCallbackParam( const MaterialNameDrawTypeTable* pMatTable, size_t matTableSize,
                                        const BoneNameCreateModelTypeNoseType* pBoneTable, size_t boneTableSize,
                                        MiiG3dTestModelSimple* pG3dMiiCharModel )
            : m_pMaterialNameDrawTypeTableArray( pMatTable )
            , m_MaterialNameDrawTypeTableArraySize( matTableSize )
            , pBoneNameCreateModelTypeNoseTypeArray( pBoneTable )
            , m_BoneNameCreateModelTypeNoseTypeArraySize( boneTableSize )
            , m_pG3dMiiCharModel( pG3dMiiCharModel )
        {
        }
        const MaterialNameDrawTypeTable*          m_pMaterialNameDrawTypeTableArray;
        size_t                                    m_MaterialNameDrawTypeTableArraySize;
        const BoneNameCreateModelTypeNoseType*    pBoneNameCreateModelTypeNoseTypeArray;
        size_t                                    m_BoneNameCreateModelTypeNoseTypeArraySize;

        //!< DrawParamCallbackParam を保持しているクラスへのポインタです。
        //!< テストの正解値を MiiG3dTestModelSimple にセットするために使用されます。
        MiiG3dTestModelSimple*                        m_pG3dMiiCharModel;
    };

    static const MaterialNameDrawTypeTable            s_MaterialNameDrawTypeTableArray[];
    static const BoneNameCreateModelTypeNoseType      s_BoneNameCreateModelTypeNoseTypeTableArray[];
    static const char*                                s_VisibleBoneNameArray[];
    static const size_t                               s_MaterialNameDrawTypeTableArraySize;
    static const size_t                               s_BoneNameCreateModelTypeNoseTypeTableArraySize;
    static const size_t                               s_VisibleBoneNameArraySize;

    // render_info, shader_param のデフォルト値登録用のコールバック関数
    static void MaterialCallback( nn::g3d::ResMaterial* pMaterial, const nn::mii::DrawParam* pDrawParam, void* pUserParam );
    MaterialCallbackParam                m_MaterialCallbackParam;

    // DrawParam 取得コールバック
    static void DrawParamCallback( nn::g3d::mii::DrawParamCallbackOutput* pDrawParamCallbackOutput, const nn::g3d::ResModel* pModel,
                                    const int shapeIndex, const nn::mii::CharModel* pCharModel, void* pCbParam );
    DrawParamCallbackParam                m_DrawParamCallbackParam;

    std::vector<int>                      m_VisibleBoneIndexArray;

    //!< テストの正解値を取得するために各 shape に対応する nn::g3d::mii::DrawParamCallbackOutput を保持します。
    //!< this::DrawParamCallback() で push_back() されます。
    std::vector<nn::g3d::mii::DrawParamCallbackOutput>    m_DrawParamCallbackOutputArray;

    //!< DrawParamCallback() 内で正解値をセットします。
    void SetDrawParamCallbackOutput( int index, const nn::g3d::mii::DrawParamCallbackOutput& drawParamCallbackOutput )
    {
        NN_ASSERT( index < static_cast<int>( m_DrawParamCallbackOutputArray.size() ) );
        m_DrawParamCallbackOutputArray[index] = drawParamCallbackOutput;
    }

    size_t GetDrawParamCallbackOutputArraySize()
    {
        return m_DrawParamCallbackOutputArray.size();
    }

    void ResizeDrawParamCallbackOutputArray( size_t arraySize )
    {
        m_DrawParamCallbackOutputArray.resize( arraySize );
    }

};
