﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testEft_CaptureParameter.h"
namespace nnt{
namespace eft{

//------------------------------------------------------------------------------
//  コンストラクタ
//------------------------------------------------------------------------------
TestCustomBehaviorParam::TestCustomBehaviorParam() : m_IsCameraZPosUpdate( false ),
                                     m_CameraZPosCurrent(0.0),
                                     m_CameraZPosPerFrame(0.0),
                                     m_IsProjectionUpdate(false),
                                     m_ProjectionPosPerFrame(0.0),
                                     m_CurrentH(0.0)
{
}

//------------------------------------------------------------------------------
//  デストラクタ
//------------------------------------------------------------------------------
TestCustomBehaviorParam::~TestCustomBehaviorParam()
{
}

//------------------------------------------------------------------------------
//  初期値に戻す
//------------------------------------------------------------------------------
void TestCustomBehaviorParam::Reset()
{
    m_IsCameraZPosUpdate = false;
    m_CameraZPosCurrent  = 0.0f;
    m_CameraZPosPerFrame = 0.0f;
    m_IsProjectionUpdate = false;
    m_ProjectionPosPerFrame = 0.0;
}

//------------------------------------------------------------------------------
//  カメラの設定
//------------------------------------------------------------------------------
void TestCustomBehaviorParam::SetCamera( nn::util::Matrix4x3fType*   pView,
                                 nn::util::Vector3fType&     cameraPosition,
                                 nn::util::Vector3fType&     target,
                                 nn::util::Vector3fType&     camUp )
{
    m_CameraPosition = cameraPosition;
    m_View           = pView;
    m_Target         = target;
    m_CamUp          = camUp;
}

//------------------------------------------------------------------------------
//  プロジェクションの設定
//------------------------------------------------------------------------------
void TestCustomBehaviorParam::SetProjection( nn::util::Matrix4x4fType* pProjctionMatrix,
                                     float                     fovy,
                                     float                     aspect,
                                     float                     zNear,
                                     float                     zFar )
{
    m_ProjctionMatrix = pProjctionMatrix;
    m_Fovy            = fovy;
    m_Aaspect         = aspect;
    m_ZNear           = zNear;
    m_ZFar            = zFar;
}

//------------------------------------------------------------------------------
//  テストカスタムパラメータの作成
//------------------------------------------------------------------------------
void TestCustomBehaviorParam::PreparCustomParam( nn::vfx::Handle* emitterSetHandle, float maxFrames )
{
    Reset();

    nn::util::MatrixIdentity( m_ProjctionMatrix );
    nn::util::MatrixPerspectiveFieldOfViewRightHanded( m_ProjctionMatrix, m_Fovy, m_Aaspect, m_ZNear, m_ZFar );

    uint16_t userDataBit = emitterSetHandle->GetEmitterSet()->GetUserDataBit();
    float userData1 = static_cast<float>( emitterSetHandle->GetEmitterSet()->GetUserData1() );
    float userData2 = static_cast<float>( emitterSetHandle->GetEmitterSet()->GetUserData2() );
    float userData3 = static_cast<float>( emitterSetHandle->GetEmitterSet()->GetUserData3() );
    float userData4 = static_cast<float>( emitterSetHandle->GetEmitterSet()->GetUserData4() );

    if ( ( userDataBit & TestCustomBehavior_UserDataBit_0 ) || ( userDataBit & TestCustomBehavior_UserDataBit_1 ) )
    {
        if ( userDataBit & TestCustomBehavior_UserDataBit_0 )    // Camera Y Z位置 注視点_Yの設定
        {
            nn::util::VectorSetY( &m_CameraPosition, userData1 );  // カメラの位置_Y
            nn::util::VectorSetZ( &m_CameraPosition, userData3 );  // カメラの位置_Z

            nn::util::VectorSetY( &m_Target, userData2 );          // 注視点_Y
        }

        if ( userDataBit & TestCustomBehavior_UserDataBit_1 )   // カメラ位置アニメ
        {
            m_IsCameraZPosUpdate = true;
            nn::util::VectorSetZ( &m_CameraPosition, userData3 );  // カメラの位置_Z
            m_CameraZPosCurrent = ( userData3 );
            m_CameraZPosPerFrame = ( ( userData4 - m_CameraZPosCurrent ) / maxFrames );

            float slope =  (nn::util::VectorGetY( m_CameraPosition ) -  nn::util::VectorGetY(m_Target) ) / nn::util::VectorGetZ( m_CameraPosition );
            m_CameraYPosPerFrame = ( ( slope * userData4 ) - ( nn::util::VectorGetY( m_CameraPosition ) -  nn::util::VectorGetY( m_Target )) ) / maxFrames;
        }
    }
    else if ( userDataBit & TestCustomBehavior_UserDataBit_2 )  // オルソ
    {
        m_IsProjectionUpdate = true;
        m_CurrentH = userData3 / 2.0f;
        float diff = ( userData4 - userData3 ) / 2.0f;
        m_ProjectionPosPerFrame = diff / (float)maxFrames;
        float w = m_CurrentH * m_Aaspect;
        nn::util::MatrixIdentity( m_ProjctionMatrix );
        nn::util::MatrixOrthographicOffCenterRightHanded( m_ProjctionMatrix, -w, w, -m_CurrentH, m_CurrentH, m_ZNear, m_ZFar );
    }

    nn::util::MatrixIdentity( m_View );
    nn::util::MatrixLookAtRightHanded( m_View, m_CameraPosition, m_Target, m_CamUp );
}

//------------------------------------------------------------------------------
//  カメラの更新
//------------------------------------------------------------------------------
void TestCustomBehaviorParam::Update( int frame )
{
    // 最初のフレーム時はスキップする。
    if ( frame == 0 ) return;

    if ( m_IsCameraZPosUpdate )
    {
        float yPos,zPos;
        zPos = nn::util::VectorGetZ( m_CameraPosition );
        zPos += m_CameraZPosPerFrame;

        yPos = nn::util::VectorGetY( m_CameraPosition );
        yPos += m_CameraYPosPerFrame;

        nn::util::VectorSetY( &m_CameraPosition, yPos );
        nn::util::VectorSetZ( &m_CameraPosition, zPos );
        nn::util::MatrixLookAtRightHanded( m_View, m_CameraPosition, m_Target, m_CamUp );
    }

    if ( m_IsProjectionUpdate )
    {
        m_CurrentH += m_ProjectionPosPerFrame;
        float w = m_CurrentH * m_Aaspect;
        nn::util::MatrixIdentity( m_ProjctionMatrix );
        nn::util::MatrixOrthographicOffCenterRightHanded( m_ProjctionMatrix, -w, w, -m_CurrentH, m_CurrentH, m_ZNear, m_ZFar );
    }
}

}
}
