﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/mem/mem_StandardAllocator.h>

namespace nnt{
namespace eft{

/**
 * @brief メモリプールアロケータ
 *
 */
class MemortPoolAllocator
{
public:
    MemortPoolAllocator()
    {
        m_PoolSize = 0;
        m_MemoryPtr = nullptr;
        m_Offset = 0;
    }

    void Initialize(nn::gfx::Device* pDevice, nn::mem::StandardAllocator* pAllocator, size_t poolSize, int property)
    {
        nn::gfx::MemoryPool::InfoType mpInfo;
        mpInfo.SetDefault();
        mpInfo.SetMemoryPoolProperty(property);

        m_PoolSize = nn::util::align_up(poolSize, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, mpInfo));
        m_MemoryPtr = pAllocator->Allocate(m_PoolSize, nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, mpInfo));
        NN_SDK_ASSERT_NOT_NULL(m_MemoryPtr);

        mpInfo.SetPoolMemory(m_MemoryPtr, m_PoolSize);
        m_MemortPool.Initialize(pDevice, mpInfo);
    }

    void Finalize(nn::gfx::Device* pDevice, nn::mem::StandardAllocator* pAllocator)
    {
        m_MemortPool.Finalize(pDevice);
        pAllocator->Free(m_MemoryPtr);
    }

    nn::gfx::MemoryPool* GetMemoryPool()
    {
        return &m_MemortPool;
    }
    ptrdiff_t GetCurrentPos() const
    {
        return m_Offset;
    }
    void Advance(ptrdiff_t add_size)
    {
        m_Offset += add_size;
        NN_SDK_ASSERT(GetCurrentPos() < static_cast<ptrdiff_t>(m_PoolSize));
    }
    void AlignUp(ptrdiff_t align_size)
    {
        m_Offset = nn::util::align_up( m_Offset, align_size );
    }
    void* Alloc(size_t size, size_t align)
    {
        AlignUp( align );
        void* ret = nn::util::BytePtr( m_MemoryPtr ).Advance( m_Offset ).Get();
        Advance( size );
        return ret;
    }
    void Reset()
    {
        m_Offset = 0;
    }

    ptrdiff_t AllocOffset( size_t size, size_t align )
    {
        AlignUp( align );
        ptrdiff_t ret = GetCurrentPos();
        Advance( size );
        return ret;
    }

private:
    size_t                      m_PoolSize;
    nn::gfx::MemoryPool         m_MemortPool;
    void*                       m_MemoryPtr;
    ptrdiff_t                   m_Offset;
};

}
}
