﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include <iostream>
#include "testGraphics_ImageIo.h"

namespace nnt{ namespace graphics{

    struct PNG_IHDR
    {
        uint32_t width;            //!< イメージの横幅
        uint32_t height;           //!<  イメージの縦幅
        uint8_t bitDepth;          //!<  ビット深度(1, 2, 4, 8, or 16 bits/channel)
        uint8_t colorType;         //!<  カラータイプ
        uint8_t compressionType;   //!<  圧縮手法
        uint8_t filterType;        //!<  フィルター手法
        uint8_t interlaceType;     //!<  インターレース手法
        uint8_t channels;          //!<  チャンネル数

        PNG_IHDR()
            : width(0)
            , height(0)
            , bitDepth(1)
            , colorType(0)
            , compressionType(0)
            , filterType(0)
            , interlaceType(0)
        {
        }

        /**
         * @brief ピクセル深度の取得
         *
         * @return     ピクセル深度を返します。
        */
        uint8_t GetPixelDepth() const { return bitDepth * channels; }

        /**
         * @brief 1行あたりのバイト数を取得
         *
         * @return     1行あたりのバイト数を返します。
        */
        uint32_t GetRowBytes() const { return ((width * static_cast<uint32_t>(GetPixelDepth()) + 7) >> 3); }
    };

    /**
    * @brief PNG入出力
    *
    * @details
    */
    class PngIO : ImageIo
    {
    public:

        /**
         * @brief カラーマスク
         *
         * @details カラータイプの組み合わせ用
         */
        enum PNG_COLOR_MASK
        {
            PNG_COLOR_MASK_PALETTE = 1, //!< パレット使用
            PNG_COLOR_MASK_COLOR   = 2, //!< カラー
            PNG_COLOR_MASK_ALPHA   = 4  //!< αチャンネル
        };

        /**
         * @brief カラータイプ
         *
         * @details 0 = グレイスケール
                    3 = パレット
                    2 = RGB
                    6 = RGB+アルファ
                    4 = グレイスケール+アルファ
         */
        enum PNG_COLOR_TYPE
        {
            PNG_COLOR_TYPE_GRAY         = 0,                                                //!< グレイスケール
            PNG_COLOR_TYPE_PALETTE      = (PNG_COLOR_MASK_COLOR | PNG_COLOR_MASK_PALETTE),  //!< パレット
            PNG_COLOR_TYPE_RGB          = (PNG_COLOR_MASK_COLOR),                           //!< RGB
            PNG_COLOR_TYPE_RGB_ALPHA    = (PNG_COLOR_MASK_COLOR | PNG_COLOR_MASK_ALPHA),    //!< RGB+アルファ
            PNG_COLOR_TYPE_GRAY_ALPHA   = PNG_COLOR_MASK_ALPHA                              //!< グレイスケール+アルファ
        };

        /**
         * @brief PNGファイル読み込み
         *
         * @param[out] pImage        イメージデータ
         * @param[in]  pPath         ファイルパス
         * @return     入力に成功したときtrueを返します。
        */
        static bool ReadPng( uint8_t** pImage, const char* pPath);

        /**
         * @brief PNGファイル読み込み
         *
         * @param[out] pImage        イメージデータ
         * @param[in]  pPath         ファイルパス
         * @param[out] ihdr          PNGのIHDRデータ
         * @return     入力に成功したときtrueを返します。
        */
        static bool ReadPng( uint8_t** pImage, const char* pPath, PNG_IHDR& ihdr );

        /**
         * @brief PNGファイル書き込み
         *
         * @param[in] pPath           ファイルパス
         * @param[in] pImage          イメージデータ
         * @param[in] width           横幅
         * @param[in] height          縦幅
         * @param[in] bitDepth        縦幅
         * @param[in] colorType       カラータイプ
         * @param[in] compressType    圧縮手法
         * @param[in] filterType      フィルター手法
         * @param[in] interlaceType   インターレース手法
         * @return     保存に成功したときtrueを返します。
        */
        static bool WritePng(
            const char* pPath,
            const uint8_t* pImage,
            int width,
            int height,
            uint8_t bitDepth,
            uint8_t colorType,
            uint8_t compressType,
            uint8_t filterType,
            uint8_t interlaceType);

        /**
         * @brief PNGファイル書き込み
         *
         * @param[in] pPath           ファイルパス
         * @param[in] pImage          イメージデータ
         * @param[in] ihdr            PNGのIHDRデータ
         * @return     保存に成功したときtrueを返します。
        */
        static bool WritePng( const char* pPath, const uint8_t* pImage, PNG_IHDR ihdr);

        static void FreePng(uint8_t** pImage);

    private:
        /**
         * @brief 圧縮率を上げるためピクセルデータにフィルターをかける
         *
         * @param[in] pInput  入力データ
         * @param[in] width   画像の横幅
         * @param[in] height  画像の縦幅
         * @param[in] pOutput 出力データ
        */
        static void CompressFilter( const uint8_t* pInput, int width, int height, uint8_t pixelBytes, uint8_t* pOutput );

        /**
         * @brief フィルタを適用したピクセルデータを復元
         *
         * @param[in] pInput        入力データ
         * @param[in] width         画像の横幅
         * @param[in] height        画像の縦幅
         * @param[in] pixelBytes    1ピクセルあたりのバイト数
         * @param[in] pOutput       出力データ
        */
        static void DecompressFilter( const uint8_t* pInput, int width, int height, uint8_t pixelBytes, uint8_t* pOutput );

    };


}}
