﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "testGraphics_Path.h"

#include <cstring>
#include <cctype>

#include <nn/nn_Assert.h>
#include <nn/util/util_StringUtil.h>

namespace nnt{ namespace graphics{
    //-------------------------------------------------------------------------------
    // Traits
    //-------------------------------------------------------------------------------
    char Path::GetDelimiter() NN_NOEXCEPT
    {
        return '/';
    }
    bool Path::IsDelimiter(char c) NN_NOEXCEPT
    {
        return c == '/' || c == '\\';
    }
    bool Path::IsMountNameDelimiter(char c) NN_NOEXCEPT
    {
        return c == ':';
    }

    //-------------------------------------------------------------------------------
    // Constructor
    //-------------------------------------------------------------------------------
    Path::Path() NN_NOEXCEPT
    {
        std::memset(m_Data, 0, Capacity);
    }
    Path::Path(const char* src) NN_NOEXCEPT
    {
        std::memset(m_Data, 0, Capacity);
        SetString(src);
    }

    bool Path::SetString(const char* src) NN_NOEXCEPT
    {
        if(src == nullptr)
        {
            m_Data[0] = '\0';
            return true;
        }
        int srcLength = nn::util::Strnlen(src, Capacity);
        if(srcLength == Capacity)
        {
            return false;
        }
        nn::util::Strlcpy(m_Data, src, Capacity);
        return true;
    }

    const char* Path::GetString() const NN_NOEXCEPT
    {
        return m_Data;
    }
    char* Path::GetString() NN_NOEXCEPT
    {
        return m_Data;
    }

    //-------------------------------------------------------------------------------
    // Length
    //-------------------------------------------------------------------------------
    int Path::GetLength() const NN_NOEXCEPT
    {
        int length = nn::util::Strnlen(m_Data, Capacity);
        NN_ASSERT(length < Capacity);
        return length;
    }
    bool Path::IsEmpty() const NN_NOEXCEPT
    {
        return m_Data[0] == '\0';
    }

    //-------------------------------------------------------------------------------
    // Compare
    //-------------------------------------------------------------------------------
    bool Path::IsEqual(const Path* path) const NN_NOEXCEPT
    {
        NN_ASSERT(path != nullptr);
        return nn::util::Strncmp(m_Data, path->m_Data, Capacity) == 0;
    }
    bool Path::IsEqual(const Path& path) const NN_NOEXCEPT
    {
        return IsEqual(&path);
    }

    //-------------------------------------------------------------------------------
    // MountName
    //-------------------------------------------------------------------------------
    int Path::GetMountNameLength() const NN_NOEXCEPT
    {
        int length = GetLength();
        for(int i = 0; i < length; i++)
        {
            char c = m_Data[i];
            if(IsMountNameDelimiter(c))
            {
                return i + 1;
            }
            else if(IsDelimiter(c))
            {
                return -1;
            }
        }
        return -1;
    }

    bool Path::HasMountName() const NN_NOEXCEPT
    {
        int mountNameLength = GetMountNameLength();
        return mountNameLength >= 0;
    }

    //-------------------------------------------------------------------------------
    // AbsolutePath
    //-------------------------------------------------------------------------------
    bool Path::IsAbsolute() const NN_NOEXCEPT
    {
        if(IsDelimiter(m_Data[0]))
        {
            return true;
        }
        if(HasMountName())
        {
            return true;
        }
        return false;
    }

    //-------------------------------------------------------------------------------
    // Root
    //-------------------------------------------------------------------------------
    int Path::GetRootLength() const NN_NOEXCEPT
    {
        int mountNameLength = GetMountNameLength();
        if(mountNameLength < 0)
        {
            mountNameLength = 0;
        }

        // "MountName:" につづく区切り文字の長さを計算
        int length = GetLength();
        int rootLength = mountNameLength;
        for(; rootLength < length; rootLength++)
        {
            if(!IsDelimiter(m_Data[rootLength]))
            {
                break;
            }
        }

        if(rootLength == 0)
        {
            return -1;
        }
        return rootLength;
    }

    bool Path::IsRoot() const NN_NOEXCEPT
    {
        int length = GetLength();
        int rootLength = GetRootLength();
        if(rootLength < 0)
        {
            return false;
        }
        return rootLength == length;
    }

    bool Path::MakeRoot() NN_NOEXCEPT
    {
        int rootLength = GetRootLength();
        if(rootLength < 0)
        {
            return false;
        }
        m_Data[rootLength] = '\0';
        return true;
    }

    bool Path::GetRoot(Path* pOutValue) const NN_NOEXCEPT
    {
        pOutValue->SetString(GetString());
        return pOutValue->MakeRoot();
    }
    Path Path::GetRoot() const NN_NOEXCEPT
    {
        Path path;
        if(!GetRoot(&path))
        {
            return Path();
        }
        return path;
    }
    //-------------------------------------------------------------------------------
    // Filename
    //-------------------------------------------------------------------------------
    int Path::GetFilenameStartPosition() const NN_NOEXCEPT
    {
        int length = GetLength();
        for(int pos = length - 1; pos >= 0; pos--)
        {
            char c = m_Data[pos];
            if(IsDelimiter(c) || IsMountNameDelimiter(c))
            {
                return pos + 1;
            }
        }
        return 0;
    }
    bool Path::GetFilename(Path* pOutValue) const NN_NOEXCEPT
    {
        int pos = GetFilenameStartPosition();
        pOutValue->SetString(m_Data + pos);
        return !pOutValue->IsEmpty();
    }
    Path Path::GetFilename() const NN_NOEXCEPT
    {
        Path result;
        GetFilename(&result);
        return result;
    }

    //-------------------------------------------------------------------------------
    // ParentDirectory
    //-------------------------------------------------------------------------------
    bool Path::MakeParent() NN_NOEXCEPT
    {
        if(IsEmpty() || IsRoot())
        {
            return false;
        }

        // 末尾が区切り文字だったら削除
        {
            int length = GetLength();
            if(IsDelimiter(m_Data[length - 1]))
            {
                m_Data[length - 1] = '\0';
            }
        }
        // 親ディレクトリを作る
        {
            int pos = GetFilenameStartPosition();
            m_Data[pos] = '\0';
            // 末尾の区切り文字を削除
            if(!IsEmpty() && !IsRoot())
            {
                m_Data[pos - 1] = '\0';
            }
        }
        return true;
    }
    bool Path::GetParent(Path* pOutValue) const NN_NOEXCEPT
    {
        NN_ASSERT(pOutValue != nullptr);
        *pOutValue = *this;
        return pOutValue->MakeParent();
    }
    Path Path::GetParent() const NN_NOEXCEPT
    {
        Path result;
        if(GetParent(&result))
        {
            return result;
        }
        else
        {
            return Path();
        }
    }

    //-------------------------------------------------------------------------------
    // Extension
    //-------------------------------------------------------------------------------
    int Path::GetExtensionStartPosition() const NN_NOEXCEPT
    {
        int length = GetLength();
        // 一番後ろの '.' の位置を探す
        for(int pos = length - 1; pos >= 0; pos--)
        {
            if(m_Data[pos] == '.')
            {
                return pos;
            }
            else if(IsDelimiter(m_Data[pos]))
            {
                return -1;
            }
        }
        return -1;
    }

    bool Path::IsEqualExtension(const char* extension, bool ignoreCase) const NN_NOEXCEPT
    {
        NN_ASSERT(extension != NULL);
        NN_ASSERT(extension[0] == '.');

        int length = GetLength();
        int extPos = GetExtensionStartPosition();
        if(extPos < 0)
        {
            return false;
        }

        int myExtLen = length - extPos;
        int argExtLen = nn::util::Strnlen(extension, myExtLen + 1);
        if(argExtLen != myExtLen)
        {
            return false;
        }

        for(int i = 0; i < myExtLen; i++)
        {
            int myChar = m_Data[extPos + i];
            int argChar = extension[i];
            if(ignoreCase)
            {
                myChar = tolower(myChar);
                argChar = tolower(argChar);
            }
            if(myChar != argChar)
            {
                return false;
            }
        }
        return true;
    }

    void Path::RemoveExtension() NN_NOEXCEPT
    {
        int pos = GetExtensionStartPosition();
        if(pos < 0)
        {
            return;
        }
        m_Data[pos] = '\0';
    }
    //-------------------------------------------------------------------------------
    // Normalize
    //-------------------------------------------------------------------------------
    void Path::NormalizeAssign() NN_NOEXCEPT
    {
        if(IsEmpty())
        {
            return;
        }
        int length = GetLength();
        int rootLength = GetRootLength();
        if(rootLength < 0)
        {
            rootLength = 0;
        }
        // ルート部分の区切り文字を統一
        for(int i = 0; i < rootLength; i++)
        {
            if(IsDelimiter(m_Data[i]))
            {
                m_Data[i] = GetDelimiter();
            }
        }
        // 区切り文字を統一＆連続する区切り文字をまとめる。
        {
            char* pDst = m_Data + rootLength;
            bool isPrevDelimiter = false;
            for(int i = rootLength; i < length; i++)
            {
                if(IsDelimiter(m_Data[i]))
                {
                    if(!isPrevDelimiter)
                    {
                        *pDst = GetDelimiter();
                        ++pDst;
                        isPrevDelimiter = true;
                    }
                }
                else
                {
                    *pDst = m_Data[i];
                    ++pDst;
                    isPrevDelimiter = false;
                }
            }
            // 末尾の区切り文字を削除
            if(IsDelimiter(*(pDst - 1)))
            {
                --pDst;
            }
            *pDst = '\0';
            length = static_cast<int>(pDst - m_Data);
        }
        // "." と ".." を処理
        {
            int dotCount = 0;
            // コピー先の開始位置。".." があったら戻る。
            int dstStart = rootLength;
            // コピー元の開始位置。戻ることはない。
            int srcStart = rootLength;
            // Root に処理済の ".." を加えた長さ。これより前には戻れない
            int upBase = rootLength;
            m_Data[length] = GetDelimiter();
            for(int srcPos = rootLength; srcPos <= length; srcPos++)
            {
                char c = m_Data[srcPos];
                if(IsDelimiter(c))
                {
                    if(dotCount == 1)
                    {
                        // "." は無視
                    }
                    else if(dotCount == 2)
                    {
                        // ".."
                        // 上がれるならば上がる
                        if(dstStart > upBase)
                        {
                            int pos;
                            for(pos = dstStart - 1; pos > upBase; pos--)
                            {
                                if(IsDelimiter(m_Data[pos]))
                                {
                                    break;
                                }
                            }
                            dstStart = pos;
                        }
                        // 上がれなければ ".." を連結
                        else
                        {
                            if(dstStart != rootLength)
                            {
                                m_Data[dstStart] = GetDelimiter();
                                dstStart++;
                                upBase++;
                            }
                            m_Data[dstStart + 0] = '.';
                            m_Data[dstStart + 1] = '.';
                            dstStart += 2;
                            upBase += 2;
                        }
                    }
                    else
                    {
                        if(dstStart != rootLength)
                        {
                            m_Data[dstStart] = GetDelimiter();
                            dstStart++;
                        }
                        while(srcStart < srcPos)
                        {
                            m_Data[dstStart] = m_Data[srcStart];
                            dstStart++;
                            srcStart++;
                        }
                    }
                    srcStart = srcPos + 1;
                    dotCount = 0;
                }
                else if(dotCount >= 0 && c == '.')
                {
                    dotCount++;
                }
                else
                {
                    dotCount = -1;
                }
            }
            m_Data[dstStart] = '\0';
        }
    }// NOLINT(impl/function_size)
    void Path::Normalize(Path* pOutValue) const NN_NOEXCEPT
    {
        pOutValue->SetString(GetString());
        pOutValue->NormalizeAssign();
    }
    Path Path::Normalize() const NN_NOEXCEPT
    {
        Path path;
        Normalize(&path);
        return path;
    }


    //-------------------------------------------------------------------------------
    // Combine
    //-------------------------------------------------------------------------------
    bool Path::CombineAssign(const Path* relativePath) NN_NOEXCEPT
    {
        NN_ASSERT(relativePath != nullptr);
        if(relativePath->IsEmpty())
        {
            return true;
        }
        int relativePathLength = relativePath->GetLength();
        if(relativePath->IsAbsolute())
        {
            return false;
        }
        int myPathLength = GetLength();
        int resultLength = myPathLength + relativePathLength;
        if(!IsDelimiter(m_Data[myPathLength - 1]))
        {
            resultLength++;
        }
        if(resultLength >= MaxLength)
        {
            return false;
        }
        if(!IsDelimiter(m_Data[myPathLength - 1]))
        {
            std::strncat(m_Data, "/", Capacity);
        }
        std::strncat(m_Data, relativePath->m_Data, Capacity);
        return true;
    }
    bool Path::CombineAssign(const Path& relativePath) NN_NOEXCEPT
    {
        return CombineAssign(&relativePath);
    }
    bool Path::Combine(Path* pOutValue, const Path* relativePath) const NN_NOEXCEPT
    {
        Path result(GetString());
        if(result.CombineAssign(relativePath))
        {
            pOutValue->SetString(result.GetString());
            return true;
        }
        return false;
    }
    Path Path::Combine(const Path& relativePath) const NN_NOEXCEPT
    {
        Path result(GetString());
        if(result.CombineAssign(relativePath))
        {
            return result;
        }
        return Path();
    }

}}
