﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/dd.h>

#include <nn/gpio/gpio.h>


namespace
{

// PINMUX レジスタの Field Name の列挙体
enum PinmuxFieldName
{
    PinmuxFieldName_PINMUX0  = 0,
    PinmuxFieldName_PINMUX1,
    PinmuxFieldName_PUPD2,
    PinmuxFieldName_PUPD3,
    PinmuxFieldName_TRISTATE,
    PinmuxFieldName_E_INPUT,
    PinmuxFieldName_E_OD,
    PinmuxFieldName_LOCK,
    PinmuxFieldName_IO_RESET,
    PinmuxFieldName_RCV_SEL,
};

// APB_MISC IO マッピング
const nn::dd::PhysicalAddress   ApbMiscPhysicalAddress   = 0x070000000ull;
const size_t                    ApbMiscAddressSize       = 0x4000;

}


namespace nnt { namespace gpio {


#if (defined(NN_GPIO_GEN) && (NN_GPIO_GEN == 2))
    #error Target platform is not supported with Gen 2 driver.
#else // gen1
    // テスト用の GPIO の Pad 番号
    enum PadName
    {
        PadName_GPIO_PU0 = 160,
        PadName_GPIO_PU1 = 161,
        PadName_GPIO_PU2 = 162,
        PadName_GPIO_PU3 = 163,
        PadName_GPIO_PU4 = 164,
        PadName_GPIO_PU5 = 165,
        PadName_GPIO_PU6 = 166,
    };

    // テスト対象の Pad の配列
    const PadName PadNameList[] =
    {
        PadName_GPIO_PU0,
        PadName_GPIO_PU1,
        PadName_GPIO_PU2,
        PadName_GPIO_PU3,
        PadName_GPIO_PU4,
        PadName_GPIO_PU5,
        PadName_GPIO_PU6,
    };
#endif

// テスト対象の Pad の数
const int NumberOfTestPad = sizeof(PadNameList) / sizeof(PadNameList[0]);

// テスト用 pinmux クラス
class Pinmux
{

public:
    Pinmux() NN_NOEXCEPT
    {
        m_ApbMiscVirtualAddress = nn::dd::QueryIoMappingAddress(ApbMiscPhysicalAddress, ApbMiscAddressSize);
        if (m_ApbMiscVirtualAddress == 0)
        {
            // physicalAddress が指す I/O アドレスがマッピングされていない
            NN_ABORT("I/O registers for 0x%llx are not mapped. Make sure the capability setting is properly set for this process.\n", ApbMiscPhysicalAddress);
        }
    }

    // テスト用　： pinmux で Pullup する関数(pinmux ライブラリ等が出来たら置き換える)
    void PullUp(nnt::gpio::PadName padName)
    {
        volatile nn::Bit32* address;


        address = reinterpret_cast<volatile nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3184 + (padName - nnt::gpio::PadName_GPIO_PU0) * 0x0004);

        *address |= (1 << PinmuxFieldName_PUPD3);
        *address &= ~(1 << PinmuxFieldName_PUPD2);
    }

    // テスト用　： pinmux で Pulldown する関数(pinmux ライブラリ等が出来たら置き換える)
    void PullDown(nnt::gpio::PadName padName)
    {
        volatile nn::Bit32* address;


        address = reinterpret_cast<volatile nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3184 + (padName - nnt::gpio::PadName_GPIO_PU0) * 0x0004);

        *address &= ~(1 << PinmuxFieldName_PUPD3);
        *address |= (1 << PinmuxFieldName_PUPD2);
    }

    // テスト用 ： OutPut が Input にループバックするように設定する(pinmux ライブラリ等が出来たら置き換える)
    void SetDirectionForLoopBack(nnt::gpio::PadName padName)
    {
        volatile nn::Bit32* address;

        address = reinterpret_cast<volatile nn::Bit32*>(m_ApbMiscVirtualAddress + 0x3184 + (padName - nnt::gpio::PadName_GPIO_PU0) * 0x0004);

        *address |= (1 << PinmuxFieldName_E_INPUT);
    }


private:
    uintptr_t m_ApbMiscVirtualAddress;


};




} } //nnt::gpio
