﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if ! defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/fs/fs_GameCardForInspection.h>
#include <nn/spl/spl_Api.h>
#endif

#include "GcCommon.h"


#include <cstring>
#include <cstdlib>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/sdmmc/sdmmc_Common.h>

#include <nn/gc/gc.h>
#include <nn/dd.h>

#include <nn/gc/detail/gc_Define.h>
#include <nn/gc/detail/gc_Util.h>
#include <nn/gc/detail/gc_Log.h>

#include <nn/gc/detail/gc_AsicHandler.h>
#include <nn/gc/detail/gc_DeviceDetector.h>
#include <nn/gc/detail/gc_DataIo.h>
#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/gc/detail/gc_GeneralIo.h>
#include <nn/gc/detail/gc_StateMachine.h>

#include "testGc_Integration_CardInsertionAndRemoval.h"


#define NN_TESTGC_EXPECT_SUCCESS(result) \
    NNT_EXPECT_RESULT_SUCCESS(result)

#define NN_TESTGC_EXPECT_INCLUDES(expect, result) \
    NNT_EXPECT_RESULT_FAILURE(expect, result)

// 定数
const size_t WorkBufferSize = 2 * 1024 * 1024;
const size_t MemoryHeapSize = 64 * 1024 * 1024;
const size_t DataBufferSize = MemoryHeapSize / 2;
const size_t MemoryHeapSizeForMalloc = MemoryHeapSize / 4;


char*     g_WorkBuffer;
uintptr_t g_WorkBufferAddress;
char*     g_DataBuffer;
uintptr_t g_DataBufferAddress;
uintptr_t g_MallocBufferAddress;
nn::os::EventType g_CardDetectionEvent;

nn::dd::DeviceAddressSpaceType g_Das;
nn::dd::DeviceVirtualAddress g_WorkBufferDeviceVirtualAddress;
nn::dd::DeviceVirtualAddress g_DataBufferDeviceVirtualAddress;

bool GcTest::m_isLogOutputEnable = true;

const char *g_AsicSessionStateName[] = {
    "Initial",
    "Establishing",
    "Established",
    "ReEstablishing",
};

const char *g_GameCardState[] = {
    "NotInserted",
    "Inserted",
    "Normal",
    "Secure",
};

const char *g_GcActionName[] = {
    "None",
    "Wait",
    "Wait2",
    "Wait3",
    "InsertCard",
    "RemoveCard",
    "InsertCardWait",
    "RemoveCardWait",
    "Activate",
    "Deactivate",
    "ChangeSecure",
    "PutToSleep",
    "Awaken",
    "Read",
    "ReadNormal",
    "ReadSecure",
};

using namespace nn::gc;
using namespace nn::gc::detail;

namespace
{
    uintptr_t g_GpioRegistersAddress = 0;

    void InitializeForDebug()
    {
        // PINMUX
        uintptr_t pinmuxRegistersAddress = nn::dd::QueryIoMappingAddress(0x70000000ull, 0x4000);
        // GPIO
        g_GpioRegistersAddress = nn::dd::QueryIoMappingAddress(0x6000d000ull, 0x1000);

        // PINMUX_AUX_GPIO_PZ3_0 出力設定
        uint32_t value = *reinterpret_cast<volatile uint32_t*>(pinmuxRegistersAddress + 0x3288);
        value &= (~(0x1 << 5));    // E_INPUT Disable
        value &= (~(0x1 << 4));    // TRISTATE Disable
        value = (value & (~(0x3 << 2))) | (0x1 << 2);   // PULL_DOWN
        *reinterpret_cast<volatile uint32_t*>(pinmuxRegistersAddress + 0x3288) = value;
        (void)(*reinterpret_cast<volatile uint32_t*>(pinmuxRegistersAddress + 0x3288));

        // GPIO_OUT PZ3 Low
        *reinterpret_cast<volatile uint16_t*>(g_GpioRegistersAddress + 0x6A4) = 0x0800;
        // GPIO_OE PZ3 Output Enable
        *reinterpret_cast<volatile uint16_t*>(g_GpioRegistersAddress + 0x694) = 0x0808;
        // GPIO_CNF PZ3 GPIO mode
        *reinterpret_cast<volatile uint16_t*>(g_GpioRegistersAddress + 0x684) = 0x0808;
        (void)(*reinterpret_cast<volatile uint16_t*>(g_GpioRegistersAddress + 0x684));

        // 波形確認前の維持
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    inline void SetGpioHighForDebug()
    {
        // GPIO_OUT PZ3 High
        *reinterpret_cast<volatile uint16_t*>(g_GpioRegistersAddress + 0x6A4) = 0x0808;
        (void)(*reinterpret_cast<volatile uint16_t*>(g_GpioRegistersAddress + 0x6A4));
    }

    inline void SetGpioLowForDebug()
    {
        // GPIO_OUT PZ3 Low
        *reinterpret_cast<volatile uint16_t*>(g_GpioRegistersAddress + 0x6A4) = 0x0800;
         (void)(*reinterpret_cast<volatile uint16_t*>(g_GpioRegistersAddress + 0x6A4));
    }
}

void GcTest::SetUp() NN_NOEXCEPT
{
#ifdef GC_DETAIL_LOG_TO_MEMORY_ENABLE
    nn::gc::detail::MemoryLogger::GetInstance().ClearLog();
#endif
    m_isLogOutputEnable = true;
}


void GcTest::TearDown() NN_NOEXCEPT
{
    if(m_isLogOutputEnable)
    {
#ifdef GC_DETAIL_LOG_TO_MEMORY_ENABLE
        nn::gc::detail::MemoryLogger::GetInstance().OutputLogToConsole();
#endif
    }
}

void PrintArray(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    for (size_t i = 0; i<bufferLength; i++)
    {
        if (i != 0 && i % 16 == 0)
        {
            NN_LOG("\n");
        }

        NN_LOG("%02X ", buffer[i]);
    }
    NN_LOG("\n");
}

extern "C" void nninitStartup()
{
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");

    // メモリヒープから WorkBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock(&g_WorkBufferAddress, WorkBufferSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock0.");

    // メモリヒープから DataBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock(&g_DataBufferAddress, DataBufferSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock1.");

    // メモリヒープから malloc で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock(&g_MallocBufferAddress, MemoryHeapSizeForMalloc);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock2.");

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator(reinterpret_cast<void*>(g_MallocBufferAddress), MemoryHeapSizeForMalloc);
}

// 挿抜があった際に呼ばれるコールバック関数：GC のスレッドをブロックしてしまうため、イベントのシグナルのみにとどめる必要がある
void SignalDetectionEvents(void *pParameter)
{
    NN_UNUSED(pParameter);
    nn::os::SignalEvent(&g_CardDetectionEvent);
}

void InitializeDriver()
{
    NN_LOG("Initialize gc driver.\n");

    // 自信で監視するイベントを用意する
    nn::os::InitializeEvent(&g_CardDetectionEvent, false, nn::os::EventClearMode_ManualClear);

    g_WorkBuffer = reinterpret_cast<char*>(g_WorkBufferAddress);
    g_DataBuffer = reinterpret_cast<char*>(g_DataBufferAddress);

#ifdef NN_BUILD_CONFIG_OS_WIN
    g_WorkBufferDeviceVirtualAddress = 0;
#else
    SetupDeviceAddressSpace(&g_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);
    g_WorkBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, g_WorkBufferAddress, WorkBufferSize, 0);
#endif
    nn::gc::Initialize(g_WorkBuffer, WorkBufferSize, g_WorkBufferDeviceVirtualAddress);

#ifdef NN_BUILD_CONFIG_OS_WIN
    g_DataBufferDeviceVirtualAddress = 0;
#else
    g_DataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, g_DataBufferAddress, DataBufferSize, g_WorkBufferDeviceVirtualAddress + WorkBufferSize);
#endif
    nn::gc::RegisterDeviceVirtualAddress(g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);
    nn::gc::RegisterDetectionEventCallback(SignalDetectionEvents, nullptr);

#ifndef NN_BUILD_CONFIG_OS_WIN
    InitializeForDebug();
#endif
}

void FinalizeDriver()
{
#if ! defined(NN_BUILD_CONFIG_OS_WIN)
    nn::gc::UnregisterDeviceVirtualAddress(g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);
    UnmapDeviceAddressSpaceAligned(&g_Das, g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);

    nn::gc::Finalize();
    UnmapDeviceAddressSpaceAligned(&g_Das, g_WorkBufferAddress, WorkBufferSize, g_WorkBufferDeviceVirtualAddress);

    CleanDeviceAddressSpace(&g_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);
#endif
}

nn::Result WaitFunction(bool (*func)(), bool targetFlag)
{
#ifdef NN_BUILD_CONFIG_OS_WIN
    const int timeoutMsec = 200;
#else
    const int timeoutMsec = 1000;
#endif
    const int waitMsec = 1;
    for(int i=0; i<INT32_MAX; i++)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(waitMsec));
        if((*func)() == targetFlag)
        {
            break;
        }
        if((timeoutMsec / waitMsec) < i)
        {
            return nn::fs::ResultGameCardUnknown();
        }
    }
    NN_RESULT_SUCCESS;
}

bool IsAsicSecure()
{
    return nn::gc::detail::StateMachine::GetInstance().IsAsicSecure();
}

void WaitForSessionEstablishment()
{
    NN_LOG("WaitForSessionEstablishment: \n");
    nn::Result result = WaitFunction(IsAsicSecure, true);
    NN_ASSERT(result.IsSuccess(), "session establishment timeout\n");
}

bool IsCardInsertedForTest()
{
    return DeviceDetector::GetInstance().GetGpioDetectPin();
}

bool IsCardInsertedGpioForTest()
{
    // Sleep 時は DeviceDetector が止まるので GeneralIo を直接見る
    return GeneralIo::GetInstance().GetGpioDetectPin();
}

nn::Result WaitForCardInserted(bool isInserted, bool isRawGpio)
{
    if(isRawGpio)
    {
        return WaitFunction(IsCardInsertedGpioForTest, isInserted);
    }
    return WaitFunction(IsCardInsertedForTest, isInserted);
}

nn::Result SetDetectPinStateForTest(bool isOn, bool isRawGpio)
{
    NN_LOG("Set GPIO detect pin to %d\n", isOn);
    GeneralIo::GetInstance().SetDetectPinStateForTest(isOn);
#ifndef NN_BUILD_CONFIG_OS_WIN
    if(isOn)
    {
        SetGpioHighForDebug();
    }
    else
    {
        SetGpioLowForDebug();
    }
#endif
    return WaitForCardInserted(isOn, isRawGpio);
}

nn::Result SetDetectPinStateForTest(bool isOn)
{
    return SetDetectPinStateForTest(isOn, false);
}

nn::Result TransitGcState(AsicSessionState asicState, GameCardState cardState)
{
    // DeviceDetector のスリープ復帰
    DeviceDetector::GetInstance().ResetPauseState();

    // sdmmc のスリープ復帰
    DataIo::GetInstance().AwakenSdmmc();

    // カードの初期状態をセット
    nn::Result result = SetDetectPinStateForTest(cardState != GameCardState_NotInserted);
    NN_ASSERT(result.IsSuccess(), "set detect pin timeout\n");

    // 強制リセット
    nn::gc::detail::StateMachine::GetInstanceForAsicHandler().SetForceResetFlag();
    nn::gc::Deactivate();

    // Check if initial state requested
    if(asicState <= AsicSessionState_Initial)
    {
        if(cardState > GameCardState_Inserted)
        {
            return nn::fs::ResultGameCardUnknown();
        }
        NN_RESULT_SUCCESS;
    }

    // Check if first establishing requested
    if(asicState == AsicSessionState_Establishing)
    {
        if(cardState > GameCardState_Inserted)
        {
            return nn::fs::ResultGameCardUnknown();
        }
        NN_RESULT_SUCCESS;
    }

    // Session establishment
    WaitForSessionEstablishment();
    NN_LOG("# ASIC session established\n");

    // Check if session established requested
    if(asicState == AsicSessionState_Established)
    {
        if(cardState <= GameCardState_Inserted)
        {
            NN_RESULT_SUCCESS;
        }
        if(cardState == GameCardState_Debug)
        {
            // TODO:
            NN_LOG("Warning: game card debug mode");
            NN_RESULT_SUCCESS;
        }
        if(cardState >= GameCardState_Normal)
        {
            NN_LOG("# activate card required\n");
            nn::gc::Activate();
        }
        if(cardState == GameCardState_Secure)
        {
            NN_LOG("# set to secure mode required\n");
            nn::gc::SetCardToSecureMode();
        }
        NN_RESULT_SUCCESS;
    }
    // Check if session reestablishing requested
    else if(asicState == AsicSessionState_ReEstablishing)
    {
        NN_LOG("# reestablishment required\n");
        // activate しておかないとセッション再構築が走らない
        nn::gc::Activate();
        // deactivate (== card removal であることは別途テストする：ここでは deactivate を使う)
        nn::gc::Deactivate();

        if(cardState > GameCardState_Inserted)
        {
            return nn::fs::ResultGameCardUnknown();
        }
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_SUCCESS;
}

bool CheckAsicSessionState(AsicSessionState asicState)
{
    bool result = false;
    nn::gc::detail::AsicState state = nn::gc::detail::StateMachine::GetInstance().GetAsicState();
    switch(asicState)
    {
    case AsicSessionState_Initial:
        result = (state == AsicState_Initial);
        break;
    case AsicSessionState_Establishing:
    case AsicSessionState_ReEstablishing:
        // TODO: コメント
        // ！注意
        result = (state == AsicState_Initial) || (state == AsicState_Secure);
        break;
    case AsicSessionState_Established:
        result = (state == AsicState_Secure);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if(result == false)
    {
        NN_LOG("Failure: asic state expected: %s, actual: %s\n", g_AsicSessionStateName[asicState], g_AsicSessionStateName[state]);
    }
    return result;
}

bool CheckGameCardState(GameCardState cardState)
{
    bool result = false;
    nn::gc::detail::GameCardMode state = nn::gc::detail::StateMachine::GetInstance().GetGameCardMode();
    switch(cardState)
    {
    case GameCardState_NotInserted:
        result = (state == GameCardMode_Initial) && (IsCardInsertedForTest() == false);
        break;
    case GameCardState_Inserted:
        result = (state == GameCardMode_Initial) && (IsCardInsertedForTest());
        break;
    case GameCardState_Normal:
        result = (state == GameCardMode_Normal);
        break;
    case GameCardState_Secure:
        result = (state == GameCardMode_Secure);
        break;
    case GameCardState_Debug:
        result = (state == GameCardMode_Debug);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if(result == false)
    {
        NN_LOG("Failure: game card state expected: %s, actual: %s\n", g_GameCardState[cardState], g_GameCardState[state]);
    }
    return result;
}

void CheckStateAndActionResult(AsicSessionState preAsicState, GameCardState preCardState, GcAction action, AsicSessionState expectedAsicState, GameCardState expectedCardState, bool isFailTest)
{
    CheckStateAndActionResult(preAsicState, preCardState, &action, 1, expectedAsicState, expectedCardState, isFailTest);
}

void CheckStateAndActionResult(AsicSessionState preAsicState, GameCardState preCardState, GcAction* actionList, int actionListLength, AsicSessionState expectedAsicState, GameCardState expectedCardState, bool isFailTest)
{
    NN_LOG("## CheckStateAndActionResult start\n");
    nn::Result result = TransitGcState(preAsicState, preCardState);
    if(result.IsFailure() && isFailTest)
    {
        return;
    }
    ASSERT_TRUE(result.IsSuccess());
    NN_LOG("# TransitGcState to {asic: %s, card: %s} end\n", g_AsicSessionStateName[preAsicState], g_GameCardState[preCardState]);

    for(int i=0; i<actionListLength; i++)
    {
        bool isLastIndex = i == (actionListLength - 1);
        GcAction action = actionList[i];
        result = nn::ResultSuccess();
        NN_LOG("# Action index %d: action %s(%d) start\n", i, g_GcActionName[action], action);

        switch(action)
        {
        case GcAction_None:
            break;
        case GcAction_Wait:
            {
#ifdef NN_BUILD_CONFIG_OS_WIN
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
#else
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
#endif
                break;
            }
        case GcAction_Wait2:
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                break;
            }
        case GcAction_Wait3:
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(200));
                break;
            }
        case GcAction_InsertCard:
        case GcAction_InsertCardWait:
            {
                result = SetDetectPinStateForTest(true, action == GcAction_InsertCard);
                break;
            }
        case GcAction_RemoveCard:
        case GcAction_RemoveCardWait:
            {
                result = SetDetectPinStateForTest(false, action == GcAction_RemoveCard);
                break;
            }
        case GcAction_Activate:
            {
                result = nn::gc::Activate();
                break;
            }
        case GcAction_Deactivate:
            {
                nn::gc::Deactivate();
                break;
            }
        case GcAction_ChangeSecure:
            {
                result = nn::gc::SetCardToSecureMode();
                break;
            }
        case GcAction_PutToSleep:
            {
                result = nn::gc::detail::AsicHandler::GetInstance().OrderWork(nn::gc::detail::AsicWork_PutToSleep);
                break;
            }
        case GcAction_Awaken:
            {
                result = nn::gc::detail::AsicHandler::GetInstance().OrderWork(nn::gc::detail::AsicWork_Awaken);
                break;
            }
        case GcAction_Read:
        case GcAction_ReadNormal:
            {
                result = nn::gc::Read(g_DataBuffer, DataBufferSize, 0, 2);
                break;
            }
        case GcAction_ReadSecure:
            {
                result = nn::gc::Read(g_DataBuffer, DataBufferSize, 0x80000, 2);
                break;
            }
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        bool isTestFailure = result.IsFailure();

        // 最後のアクションの場合
        if(isLastIndex)
        {
            if(result.IsFailure())
            {
                // fail を想定したテストでも Result failure はとりあえず出力する
                NN_LOG("Result%s (Module:%d, Description:%d)\n", isTestFailure ? ":" : " failure:", result.GetModule(), result.GetDescription());
            }
            // 最後のアクションは isFailTest に応じて変更する
            isTestFailure ^= isFailTest;
        }

        if(isTestFailure)
        {
            NN_LOG("Test failure: (Module:%d, Description:%d), expected %s\n", result.GetModule(), result.GetDescription(), isTestFailure ? "failure" : "success");
            ASSERT_TRUE(false);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        NN_LOG("# Action index %d: action %s(%d) end\n", i, g_GcActionName[action], action);
    }

    // 失敗を期待するテストでは最終状態をチェックできない
    if(isFailTest == false)
    {
        // check
        NN_LOG("# Checking ASIC/card conditions: {asic: %s / card: %s} expeced\n", g_AsicSessionStateName[expectedAsicState], g_GameCardState[expectedCardState]);
        ASSERT_TRUE(CheckAsicSessionState(expectedAsicState));
        ASSERT_TRUE(CheckGameCardState(expectedCardState));
    }

    // ここまで来たら TEST 成功とする
    GcTest::DisableOutputLog();
}    // NOLINT(impl/function_size)


void CheckStateAndActionResultForTest(AsicSessionState preAsicState, GameCardState preCardState, GcAction action, AsicSessionState expectedAsicState, GameCardState expectedCardState)
{
    CheckStateAndActionResult(preAsicState, preCardState, action, expectedAsicState, expectedCardState, false);
}


//TEST(GcUnitTest, CardInsertionAndRemoval)
//{
//    for(int i=0; i<100; i++)
//    {
//        NN_LOG("det %i\n", i);
//        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000 * 10));
//        SetDetectPinStateForTest(i % 2 == 0);
//    }
//}


extern "C" void nnMain()
{
#if ! defined(NN_BUILD_CONFIG_OS_WIN)
    nn::spl::InitializeForFs();
    NN_LOG("Read From Partition\n");
    ReadGcCalibrationPartitionToSetInternalKeys();

    nn::fs::FinalizeGameCardDriver();

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();
#endif

    NN_LOG("Gc Test Start.\n");
    InitializeDriver();

    NN_LOG("Test Start\n");
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    auto ret = RUN_ALL_TESTS();

    FinalizeDriver();

    nn::os::FreeMemoryBlock(g_MallocBufferAddress, MemoryHeapSizeForMalloc);
    nn::os::FreeMemoryBlock(g_DataBufferAddress, DataBufferSize);
    nn::os::FreeMemoryBlock(g_WorkBufferAddress, WorkBufferSize);
    NN_LOG("Test End\n");

#if ! defined(NN_BUILD_CONFIG_OS_WIN)
    nn::spl::Finalize();
#endif
    nnt::Exit(ret);
}

