﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/sdmmc/sdmmc_Common.h>
#include <nn/gc/gc.h>
#include <nn/gc/gc_Result.h>
#include <nn/gc/detail/gc_AsicOperation.h>
#include <nn/dd.h>
#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_GameCardForInspection.h>
#include <nn/fs/fs_MmcPrivate.h>

#include <nn/spl/spl_Api.h>

#include "GcCommon.h"

const size_t WorkBufferSize          = nn::util::align_up(nn::gc::GcWorkBufferSize, nn::os::MemoryBlockUnitSize);
const size_t MemoryHeapSize          = 16 * 1024 * 1024;
const size_t DataBufferSize          = MemoryHeapSize / 2;
const size_t MemoryHeapSizeForMalloc = MemoryHeapSize / 4;

char*     g_pWorkBuffer;
uintptr_t g_WorkBufferAddress;
char*     g_pDataBuffer;
uintptr_t g_DataBufferAddress;
uintptr_t g_MallocBufferAddress;
nn::os::EventType g_CardDetectionEvent;

nn::dd::DeviceAddressSpaceType g_Das;
nn::dd::DeviceVirtualAddress g_WorkBufferDeviceVirtualAddress;
nn::dd::DeviceVirtualAddress g_DataBufferDeviceVirtualAddress;

char g_CalKey[] = {
0xd4,0xd0,0x7f,0xb6,0xe0,0xb8,0x63,0x82,0x26,0xfd,0x76,0x88,0x7a,0x86,0x8f,0x03
};

void PrintArray(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    for(size_t i=0; i<bufferLength; i++)
    {
        if(i!=0 && i % 16 == 0)
        {
            NN_LOG("\n");
        }

        NN_LOG("%02X ", buffer[i]);
    }
    NN_LOG("\n");
}

extern "C" void nninitStartup()
{
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");

    // メモリヒープから WorkBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_WorkBufferAddress, WorkBufferSize );
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock0.");

    // メモリヒープから DataBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_DataBufferAddress, DataBufferSize );
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock1.");

    // メモリヒープから malloc で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_MallocBufferAddress, MemoryHeapSizeForMalloc );
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock2.");

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(g_MallocBufferAddress), MemoryHeapSizeForMalloc );
}

// 挿抜があった際に呼ばれるコールバック関数：GC のスレッドをブロックしてしまうため、イベントのシグナルのみにとどめる必要がある
void SignalDetectionEvents(void *pParameter)
{
    nn::os::SignalEvent( &g_CardDetectionEvent );
}

void InitializeDriver()
{
    nn::spl::InitializeForFs();

    NN_LOG("Read From Partition\n");

    ReadGcCalibrationPartitionToSetInternalKeys();

    NN_LOG("Finalize gc driver from fs\n");

    nn::fs::FinalizeGameCardDriver();

    NN_LOG("Initialize gc driver.\n");

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    // 自信で監視するイベントを用意する
    nn::os::InitializeEvent( &g_CardDetectionEvent, false, nn::os::EventClearMode_ManualClear );

    g_pWorkBuffer = reinterpret_cast<char*>(g_WorkBufferAddress);
    g_pDataBuffer = reinterpret_cast<char*>(g_DataBufferAddress);

    SetupDeviceAddressSpace(&g_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);

    g_WorkBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, g_WorkBufferAddress, WorkBufferSize, 0);
    nn::gc::Initialize(g_pWorkBuffer, WorkBufferSize, g_WorkBufferDeviceVirtualAddress);

    g_DataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, g_DataBufferAddress, DataBufferSize, g_WorkBufferDeviceVirtualAddress + WorkBufferSize);
    nn::gc::RegisterDeviceVirtualAddress(g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);

    nn::gc::RegisterDetectionEventCallback( SignalDetectionEvents, nullptr );
}

void FinalizeDriver()
{
    nn::gc::UnregisterDeviceVirtualAddress(g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);
    UnmapDeviceAddressSpaceAligned(&g_Das, g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);

    nn::gc::Finalize();
    UnmapDeviceAddressSpaceAligned(&g_Das, g_WorkBufferAddress, WorkBufferSize, g_WorkBufferDeviceVirtualAddress);

    CleanDeviceAddressSpace(&g_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);

    nn::spl::Finalize();
}


#define NN_GC_RETURN_FALSE_IF_FAILURE(r)        \
    { \
        NN_LOG("$ %s\n", #r); \
        const auto& _nn_result_do_temporary((r)); \
        if(_nn_result_do_temporary.IsFailure()) \
        { \
            NN_LOG("\n@@ Test Failure: (Module:%d, Description:%d)\n\n", _nn_result_do_temporary.GetModule(), _nn_result_do_temporary.GetDescription()); \
            return false; \
        } \
    }

#define NN_GC_RETURN_FALSE_UNLESS_RESULT(r, result)    \
    { \
        NN_LOG("$ %s\n", #r); \
        const auto& _nn_result_do_temporary((r)); \
        if(! result.Includes(_nn_result_do_temporary))    \
        { \
            NN_LOG("\n@@ Test Failure: (Module:%d, Description:%d), Expected (Module:%d, Description:%d)\n\n", \
                   _nn_result_do_temporary.GetModule(), _nn_result_do_temporary.GetDescription(), result.GetModule(), result.GetDescription()); \
            return false; \
        } \
        NN_LOG("\n@ Confirmed '%s'\n\n", #result); \
    }

bool TestNormal()
{
    // NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0, 2));
    // NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::SetCardToSecureMode());
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 2));
    NN_LOG("OK\n");
    return true;
}

// Sequence 1
bool TestCommandCrcError1()
{
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 1));
    NN_LOG("OK\n");
    return true;
}

// Sequence 2
bool TestCommandCrcError3()
{
    NN_GC_RETURN_FALSE_UNLESS_RESULT(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 1), nn::fs::ResultGameCardRetryLimitOut());
    NN_LOG("OK\n");
    return true;
}

// Sequence 3
bool TestFatalError()
{
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 3));
    NN_GC_RETURN_FALSE_UNLESS_RESULT(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 3), nn::fs::ResultGameCardCardFatal());
    NN_LOG("OK\n");
    return true;
}

// Sequence 4
bool TestRefresh()
{
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 3));
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 3));
    NN_LOG("OK\n");
    return true;
}

// Sequence 5
bool TestRefresh3()
{
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 3));
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 3));
    NN_LOG("OK\n");
    return true;
}

// Sequence 6
bool TestDataCrcError1()
{
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 1));
    NN_LOG("OK\n");
    return true;
}

// Sequence 7
bool TestDataCrcError3()
{
    NN_GC_RETURN_FALSE_UNLESS_RESULT(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 1), nn::fs::ResultGameCardRetryLimitOut());
    NN_LOG("OK\n");
    return true;
}

// Sequence 8
bool TestCommandCrcErrorAndRefresh()
{
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 3));
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 1));
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 3));
    NN_LOG("OK\n");
    return true;
}

// Sequence 9
bool TestCommandCrcErrorAndFatalError()
{
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 3));
    NN_GC_RETURN_FALSE_UNLESS_RESULT(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 3), nn::fs::ResultGameCardCardFatal());
    NN_LOG("OK\n");
    return true;
}

// Sequence 10
bool TestRefreshAndDataCrcError()
{
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 3));
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 1));
    NN_LOG("OK\n");
    return true;
}

bool PrepareTest()
{
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::Activate());
    NN_GC_RETURN_FALSE_IF_FAILURE(nn::gc::SetCardToSecureMode());
    return true;
}

static const int NumOfTestCases = 11;

bool IsInList(int* list, int listSize, int index)
{
    for(int i = 0; i < listSize; i++)
    {
        if (list[i] == index)
        {
            return true;
        }
    }
    return false;
}

bool RunGameCardErrorEmulationSequenceImpl(int* execList, int execListSize, bool isConsective)
{
    static bool (*gcTestFunctionList[NumOfTestCases])() = {
        TestNormal,
        TestCommandCrcError1,
        TestCommandCrcError3,
        TestFatalError,
        TestRefresh,
        TestRefresh3,
        TestDataCrcError1,
        TestDataCrcError3,
        TestCommandCrcErrorAndRefresh,
        TestCommandCrcErrorAndFatalError,
        TestRefreshAndDataCrcError
    };
    bool testResultList[NumOfTestCases] = {false};

    bool isErrorOccurred = false;
    for(int i = 0; i < execListSize; i++)
    {
        int index = execList[i];

        NN_LOG("\n\n### Test %d\n", index);
        if(PrepareTest() == false)
        {
            NN_LOG("Error: PrepareTest failed\n");
            testResultList[index] = false;
        }
        else
        {
            testResultList[index] = gcTestFunctionList[index]();
        }
        isErrorOccurred |= (testResultList[index] == false);

        if(isConsective == false)
        {
            nn::gc::Deactivate();
        }
        else
        {
            nn::gc::detail::CardUid cuid;
            nn::gc::detail::AsicOperation::GetInstance().SendCardReadUid(&cuid);
        }
    }

    NN_LOG("\n\n------------------------\n");
    for(int resultFlag = 1; resultFlag >= 0; resultFlag--)
    {
        for(int i = 0; i < execListSize; i++)
        {
            int index = execList[i];

            if (static_cast<int>(testResultList[index]) == resultFlag)
            {
                NN_LOG("Test %2d %s\n", index, testResultList[index] ? "Success" : "Failure");
            }
        }
        if(resultFlag)
        {
            NN_LOG("---\n");
        }
    }
    NN_LOG("------------------------\n");

    return ! isErrorOccurred;
}

bool RunGameCardErrorEmulationSequence(int startIndex, int numTest, int* ignoreList, int ignoreListSize)
{
    int execList[NumOfTestCases] = {0};
    int numExec = 0;
    for(int index = startIndex; index < (startIndex + numTest); index++)
    {
        if( IsInList(ignoreList, ignoreListSize, index) )
        {
            NN_LOG("\n\n# Test %d skipped\n");
            continue;
        }

        execList[numExec++] = index;
    }

    return RunGameCardErrorEmulationSequenceImpl(execList, numExec, false);
}

bool RunGameCardErrorEmulationSequence(int startIndex, int numTest)
{
    return RunGameCardErrorEmulationSequence(startIndex, numTest, nullptr, 0);
}

bool RunGameCardErrorEmulationPresetSequence()
{
    int execList[] = {1, 4, 5, 6, 8, 10, 9};
    return RunGameCardErrorEmulationSequenceImpl(execList, sizeof(execList) / sizeof(int), true);
}


extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    NN_LOG("Setting End.Gc Test Start.\n");

    InitializeDriver();

    while(! nn::gc::IsCardInserted())
    {
        NN_LOG("@ Waiting for Game Card Insert ... \n");
        nn::os::WaitEvent( &g_CardDetectionEvent );
        nn::os::ClearEvent( &g_CardDetectionEvent );
    }

    NN_LOG("\n\nFinished.\n");
    NN_LOG("Start test.\n");

    int startIndex = 1;
    int numTest = NumOfTestCases;
    int ignoreList[NumOfTestCases];
    int ignoreCount = 0;
    bool isPreset = true;

    // argv[0] は実行ファイルのパス
    if (argc > 1)
    {
        isPreset = false;
        startIndex = std::atoi(argv[1]);
        numTest = 1;
    }
    if (argc > 2)
    {
        numTest = std::atoi(argv[2]);
    }
    for(int i = 0; ((i + 3) < argc) && (i < NumOfTestCases); i++)
    {
        NN_LOG("ignore index: %d\n", i + 3);
        ignoreList[i] = std::atoi(argv[i + 3]);
        ignoreCount++;
    }

    NN_LOG("start index: %d, num: %d\n", startIndex, numTest);

    bool isTestSuccess = false;
    if (isPreset)
    {
        isTestSuccess = RunGameCardErrorEmulationPresetSequence();
    }
    else
    {
        isTestSuccess = RunGameCardErrorEmulationSequence(startIndex, numTest, ignoreList, ignoreCount);
    }

    NN_LOG("\n\nFinished. Deactivate.\n");

    nn::gc::Deactivate();
    NN_LOG("\n\nFinalize.\n");
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
    FinalizeDriver();

    nn::os::FreeMemoryBlock(g_MallocBufferAddress, MemoryHeapSizeForMalloc);
    nn::os::FreeMemoryBlock(g_DataBufferAddress, DataBufferSize);
    nn::os::FreeMemoryBlock(g_WorkBufferAddress, WorkBufferSize);
    NN_LOG("Test %s\n", isTestSuccess ? "Success" : "Failure");
}

