﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/sdmmc/sdmmc_Common.h>
#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Util.h>
#include <nn/dd.h>

#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_GameCardForInspection.h>

#include <nn/spl/spl_Api.h>

#include "GcCommon.h"
#include "GcReplaceKeys.h"

const size_t WorkBufferSize          = nn::util::align_up(nn::gc::GcWorkBufferSize, nn::os::MemoryBlockUnitSize);
const size_t MemoryHeapSize          = 64 * 1024 * 1024;
const size_t DataBufferSize          = 8 * 1024 * 1024;
const size_t MemoryHeapSizeForMalloc = 32 * 1024 * 1024;

char*     g_pWorkBuffer;
uintptr_t g_WorkBufferAddress;
char*     g_pDataBuffer;
uintptr_t g_DataBufferAddress;
uintptr_t g_MallocBufferAddress;
nn::os::EventType g_CardDetectionEvent;

nn::dd::DeviceAddressSpaceType g_Das;
nn::dd::DeviceVirtualAddress g_WorkBufferDeviceVirtualAddress;
nn::dd::DeviceVirtualAddress g_DataBufferDeviceVirtualAddress;

void PrintArray(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    for(size_t i=0; i<bufferLength; i++)
    {
        if(i!=0 && i % 16 == 0)
        {
            NN_LOG("\n");
        }

        NN_LOG("%02X ", buffer[i]);
    }
    NN_LOG("\n");
}

extern "C" void nninitStartup()
{
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");

    // メモリヒープから WorkBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_WorkBufferAddress, WorkBufferSize );
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock0.");

    // メモリヒープから DataBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_DataBufferAddress, DataBufferSize );
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock1.");

    // メモリヒープから malloc で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_MallocBufferAddress, MemoryHeapSizeForMalloc );
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock2.");

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(g_MallocBufferAddress), MemoryHeapSizeForMalloc );
}

// 挿抜があった際に呼ばれるコールバック関数：GC のスレッドをブロックしてしまうため、イベントのシグナルのみにとどめる必要がある
void SignalDetectionEvents(void *pParameter)
{
    nn::os::SignalEvent( &g_CardDetectionEvent );
}

void InitializeDriver()
{
    nn::spl::InitializeForFs();

    NN_LOG("Read From Partition\n");

    ReadGcCalibrationPartitionToSetInternalKeys();

    NN_LOG("Finalize gc driver from fs\n");

    nn::fs::FinalizeGameCardDriver();

    NN_LOG("@ Initialize gc driver.\n");

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    g_pWorkBuffer = reinterpret_cast<char*>(g_WorkBufferAddress);
    g_pDataBuffer = reinterpret_cast<char*>(g_DataBufferAddress);

    SetupDeviceAddressSpace(&g_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);

    // 鍵セット
    // LoadFixedDataFromSdCard("@ resource_DevRw");
    // nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );

    // 自信で監視するイベントを用意し、コールバック関数を登録する
    nn::os::InitializeEvent( &g_CardDetectionEvent, false, nn::os::EventClearMode_ManualClear );
    nn::gc::RegisterDetectionEventCallback( SignalDetectionEvents, nullptr );

    g_WorkBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, g_WorkBufferAddress, WorkBufferSize, 0);
    nn::gc::Initialize(g_pWorkBuffer, WorkBufferSize, g_WorkBufferDeviceVirtualAddress);

    g_DataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, g_DataBufferAddress, DataBufferSize, g_WorkBufferDeviceVirtualAddress + WorkBufferSize);
    nn::gc::RegisterDeviceVirtualAddress(g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);
}

void FinalizeDriver()
{
    nn::gc::UnregisterDeviceVirtualAddress(g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);
    UnmapDeviceAddressSpaceAligned(&g_Das, g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);

    nn::gc::Finalize();
    UnmapDeviceAddressSpaceAligned(&g_Das, g_WorkBufferAddress, WorkBufferSize, g_WorkBufferDeviceVirtualAddress);

    CleanDeviceAddressSpace(&g_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);

    nn::spl::Finalize();
}

// #define NN_DETAIL_GC_TEST_SLEEP_ENABLE

#ifdef NN_DETAIL_GC_TEST_SLEEP_ENABLE
#define NN_DETAIL_GC_TEST_SLEEP nn::gc::PutToSleep(); nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) ); nn::gc::Awaken()
#else
#define NN_DETAIL_GC_TEST_SLEEP static_cast<void>(0)
#endif

extern "C" void nnMain()
{
    // 初期化処理
    NN_LOG("@ Hello, I'm gc sample program.\n");

    // GC ドライバの初期化
    InitializeDriver();

    NN_LOG("@ Initialization Finished.\n");

    // テスト用の為、無限ループを行う
    while(true)
    {
        NN_DETAIL_GC_TEST_SLEEP;

        // カード Inserted イベントを待つ：実際にはメニューの人が GPIO イベントを待つ
        NN_LOG("@ Waiting for card accessible event after card insertion ... \n");
        nn::os::WaitEvent( &g_CardDetectionEvent );

        // イベントをクリア
        nn::os::ClearEvent( &g_CardDetectionEvent );

        // 挿入イベントのときのみ処理を行う
        // NOTE: （現在は省略しているが、実際にはイベントを取り損なう可能性があるため、 isLastInserted などを使って管理する必要がある）
        if (! nn::gc::IsCardInserted())
        {
            NN_LOG("@ card is not inserted.\n");
            continue;
        }

        NN_DETAIL_GC_TEST_SLEEP;

        // Activate
        NN_LOG("@ card accessible!\n@ Activating ... \n");
        nn::Result result = nn::gc::Activate();
        if(result.IsFailure())
        {
            NN_LOG("@ Test Failure\n");
            NN_LOG("@ Result failure: (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            // Activate 後の Failure は Deactivate が必須
            nn::gc::Deactivate();
            continue;
        }

        NN_DETAIL_GC_TEST_SLEEP;

        // ゲームカードがノーマルモードの際のページリード
        NN_LOG("@ Finished. 2 page Normal Read Test\n");
        result = nn::gc::Read(g_pDataBuffer, DataBufferSize, 0, 2);
        if(result.IsFailure())
        {
            NN_LOG("@ Test Failure\n");
            NN_LOG("@ Result failure: (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            nn::gc::Deactivate();
            continue;
        }

        // セキュアモードへの変更
        NN_LOG("@ Finished.\n@ Changing card mode to secure ...\n");
        result = nn::gc::SetCardToSecureMode();
        if(result.IsFailure())
        {
            NN_LOG("@ Test Failure\n");
            NN_LOG("@ Result failure: (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            nn::gc::Deactivate();
            continue;
        }

        NN_DETAIL_GC_TEST_SLEEP;

        // ゲームカードがセキュアモードの際のページリード
        NN_LOG("@ Finished. 2 page Secure Read Test\n");
        result = nn::gc::Read(g_pDataBuffer, DataBufferSize, 0x80000, 2);
        if(result.IsFailure())
        {
            NN_LOG("@ Result failure: (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            NN_LOG("@ Test Failure\n");
            nn::gc::Deactivate();
            continue;
        }

#ifdef NN_DETAIL_GC_TEST_SLEEP_ENABLE
        NN_LOG("@ Sleep test\n");
        nn::gc::PutToSleep();
        result = nn::gc::Read(g_pDataBuffer, DataBufferSize, 0, 2);
        NN_LOG("@ Result: (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        result = nn::gc::Read(g_pDataBuffer, DataBufferSize, 0, 2);
        NN_LOG("@ Result: (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        nn::gc::Awaken();
        result = nn::gc::Read(g_pDataBuffer, DataBufferSize, 0, 2);
        NN_LOG("@ Result: (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
#endif

        // デバイスID
        NN_LOG("@ Finished.\n@ Printing card device Id ...\n\n");
        char buf[nn::gc::GcCardDeviceIdSize];
        nn::gc::GetCardDeviceId(buf, sizeof(buf));
        PrintArray(buf, sizeof(buf));

        // ゲームカード情報
        NN_LOG("@ Finished.\n@ Printing game card status ...\n\n");
        nn::gc::GameCardStatus status;
        nn::gc::GetCardStatus(&status);
        PrintArray(reinterpret_cast<char*>(&status), sizeof(status));

        // Deactivate
        NN_LOG("@ done!\n@ Deactivating ... \n");
        nn::gc::Deactivate();
        NN_LOG("@ done!\n");

        // nn::gc::detail::MemoryLogger::GetInstance().OutputLogToConsole();
    }

    FinalizeDriver();
    nn::os::FreeMemoryBlock(g_MallocBufferAddress, MemoryHeapSizeForMalloc);
    nn::os::FreeMemoryBlock(g_DataBufferAddress, DataBufferSize);
    nn::os::FreeMemoryBlock(g_WorkBufferAddress, WorkBufferSize);
    NN_LOG("@ Test Success\n");
}

