﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>

#include <nn/hid/hid_NpadCommon.h>
#include <nn/hid/hid_NpadFullKey.h>
#include <nn/hid/hid_NpadHandheld.h>
#include <nn/hid/hid_NpadJoyDual.h>
#include <nn/hid/hid_NpadJoyLeft.h>
#include <nn/hid/hid_NpadJoyRight.h>
#include <nn/hid/system/hid_NpadSystemExt.h>
#include <nn/hid/system/hid_NpadSystem.h>

#include <nn/os.h>

#include <nnt/nntest.h>

#include "../Common/testGamePad_Common.h"
#include "../Common/testGamePad_Npad.h"
#include "../Common/testGamePad_NpadStateFactorTable.h"

namespace
{

template<typename TState>
void VerifyNpadStates(const TState* const states, int count, int factorCount) NN_NOEXCEPT
{
    const int errorScale = 3000;        // スティックの誤差許容量

    nn::hid::NpadButtonSet buttonMaskVal = nnt::gamepad::GetNpadMaskValue();

    for (auto i = 0; i < count; i++)
    {
        // Button
        ASSERT_TRUE((states[i].buttons & buttonMaskVal) == ::nnt::gamepad::ExpectedFactorTable[factorCount].buttons);

        // analogStickL
        if (::nnt::gamepad::ExpectedFactorTable[factorCount].analogStickL.x == 0
        &&  ::nnt::gamepad::ExpectedFactorTable[factorCount].analogStickL.y == 0)
        {
            ASSERT_EQ(states[i].analogStickL.x, ::nnt::gamepad::ExpectedFactorTable[factorCount].analogStickL.x);
            ASSERT_EQ(states[i].analogStickL.y, ::nnt::gamepad::ExpectedFactorTable[factorCount].analogStickL.y);
        }
        else
        {
            ASSERT_NEAR(states[i].analogStickL.x, ::nnt::gamepad::ExpectedFactorTable[factorCount].analogStickL.x, errorScale);
            ASSERT_NEAR(states[i].analogStickL.y, ::nnt::gamepad::ExpectedFactorTable[factorCount].analogStickL.y, errorScale);
        }
        // analogStickR
        if (::nnt::gamepad::ExpectedFactorTable[factorCount].analogStickR.x == 0
        &&  ::nnt::gamepad::ExpectedFactorTable[factorCount].analogStickR.y == 0)
        {
            ASSERT_EQ(states[i].analogStickR.x, ::nnt::gamepad::ExpectedFactorTable[factorCount].analogStickR.x);
            ASSERT_EQ(states[i].analogStickR.y, ::nnt::gamepad::ExpectedFactorTable[factorCount].analogStickR.y);
        }
        else
        {
            ASSERT_NEAR(states[i].analogStickR.x, ::nnt::gamepad::ExpectedFactorTable[factorCount].analogStickR.x, errorScale);
            ASSERT_NEAR(states[i].analogStickR.y, ::nnt::gamepad::ExpectedFactorTable[factorCount].analogStickR.y, errorScale);
        }
    }
}

template<typename TStyle, typename TState>
void ReadingStateTest(const ::nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    NN_LOG("NpadId : 0x%x, StyleIndex : %x\n", npadId, TStyle::Index);

    // テスト対象の操作形態が有効でない場合はスキップ
    const nn::hid::NpadStyleSet Style = nn::hid::GetNpadStyleSet(npadId);
    if (Style.Test<TStyle>() == false)
    {
        NN_LOG("Skipped, since current style is not supported\n");
        return;
    }

    // 因子数分確認する
    for (auto i = 0; i < nnt::gamepad::factorNum; i++)
    {
        // 入力状態が取得されるまで待つ
        TState states[::nn::hid::NpadStateCountMax];
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nn::hid::NpadStateCountMax * ::nnt::gamepad::GetNpadSamplingInterval().GetMilliSeconds()));

        int count = nn::hid::GetNpadStates(states,
                                            ::nn::hid::NpadStateCountMax,
                                            npadId);

        // 入力状態は指定個数分取得できているか
        ASSERT_GE(::nn::hid::NpadStateCountMax, count);

        // 入力状態が正しく取得できているか
        VerifyNpadStates<TState>(states, count, i);

        // 確認用に出力
        ::nnt::gamepad::PrintNpadState<TState>(states[0]);
        ::nnt::gamepad::PrintNpadButtonState<TState>(states[0]);
    }
}

template<typename TStyle, typename TState>
void ReadingSystemStateTest(const ::nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    NN_LOG("NpadId : 0x%x, StyleIndex : %x\n", npadId, TStyle::Index);

    // テスト対象の操作形態が有効でない場合はスキップ
    const nn::hid::NpadStyleSet Style = nn::hid::GetNpadStyleSet(npadId);
    if (Style.Test<TStyle>() == false)
    {
        NN_LOG("    Skipped, since current style is not supported\n");
        return;
    }

    // 因子数分確認する
    for (auto i = 0; i < nnt::gamepad::factorNum; i++)
    {
        // 入力状態が取得されるまで待つ
        TState states[::nn::hid::NpadStateCountMax];
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nn::hid::NpadStateCountMax * ::nnt::gamepad::GetNpadSamplingInterval().GetMilliSeconds()));

        int count = nn::hid::system::GetNpadStates(states,
                                                   ::nn::hid::NpadStateCountMax,
                                                   npadId);

        // 入力状態は指定個数分取得できているか
        ASSERT_GE(::nn::hid::NpadStateCountMax, count);

        // NpadStyleSystemからは正しくスティック状態が取れません(0しか帰ってこない)ので、除外
        if (Style.Test<::nn::hid::system::NpadStyleSystem>() == false)
        {
            // 入力状態が正しく取得できているか
            VerifyNpadStates<TState>(states, count, i);
        }

        // 確認用に出力
        ::nnt::gamepad::PrintNpadState<TState>(states[0]);
        ::nnt::gamepad::PrintNpadButtonState<TState>(states[0]);
    }
}

} // namespace


//!< Npad の入力状態が正しく取得できるか
TEST(Npad, StateReading)
{
    // テストマシンの構成に応じた Npad の初期化
    nnt::gamepad::Initialize();

    // USB有線通信を無効化
    nnt::gamepad::DisableUsbConnect();

    // コントローラを切断
    nnt::gamepad::DisconnectAll();

    // コントローラを接続
    nnt::gamepad::ConnectAll();

    // 各 NpadId, 操作携帯で入力状態が取得できているかチェック
    for (const auto& id : nnt::gamepad::NpadIds)
    {
        ReadingStateTest<::nn::hid::NpadStyleFullKey, ::nn::hid::NpadFullKeyState>(id);
        ReadingStateTest<::nn::hid::NpadStyleHandheld, ::nn::hid::NpadHandheldState>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyDual, ::nn::hid::NpadJoyDualState>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyLeft, ::nn::hid::NpadJoyLeftState>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyRight, ::nn::hid::NpadJoyRightState>(id);

        ReadingSystemStateTest<::nn::hid::system::NpadStyleSystemExt, ::nn::hid::system::NpadSystemExtState>(id);
        ReadingSystemStateTest<::nn::hid::system::NpadStyleSystem, ::nn::hid::system::NpadSystemState>(id);
    }

    // USB有線通信を有効化
    nnt::gamepad::EnableUsbConnect();

    // コントローラを切断
    nnt::gamepad::DisconnectAll();

    // コントローラを接続
    nnt::gamepad::ConnectAll();

    // 各 NpadId, 操作携帯で入力状態が取得できているかチェック
    for (const auto& id : nnt::gamepad::NpadIds)
    {
        ReadingStateTest<::nn::hid::NpadStyleFullKey, ::nn::hid::NpadFullKeyState>(id);
        ReadingStateTest<::nn::hid::NpadStyleHandheld, ::nn::hid::NpadHandheldState>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyDual, ::nn::hid::NpadJoyDualState>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyLeft, ::nn::hid::NpadJoyLeftState>(id);
        ReadingStateTest<::nn::hid::NpadStyleJoyRight, ::nn::hid::NpadJoyRightState>(id);

        ReadingSystemStateTest<::nn::hid::system::NpadStyleSystemExt, ::nn::hid::system::NpadSystemExtState>(id);
        ReadingSystemStateTest<::nn::hid::system::NpadStyleSystem, ::nn::hid::system::NpadSystemState>(id);
    }

    // コントローラを切断
    nnt::gamepad::DisconnectAll();
}
