﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

// 読み込みテスト
TEST(PowerInterruptionStorageTest, ReadTest)
{
    // 書き込み可能回数
    const int WritableCount = 5;
    const size_t BufferSize = 1024;
    nnt::fs::util::SafeMemoryStorage baseStorage(BufferSize);
    nnt::fs::util::PowerInterruptionStorage storage(
        nn::fs::SubStorage(&baseStorage, 0, BufferSize)
    );
    storage.StartCounting(WritableCount);

    std::unique_ptr<char[]> buffer(new char[BufferSize]);

    // 何回読み込んでも電源断にならないことを確かめる
    for( int i = 0; i < 100; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Read(0, buffer.get(), BufferSize)
        );
    }
    // 終端 1 バイト、先端 1 バイトの読み込み
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Read(BufferSize - 1, buffer.get(), 1)
    );
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Read(1, buffer.get(), BufferSize - 1)
    );
}

// 書き込みテスト
TEST(PowerInterruptionStorageTest, WriteTest)
{
    // 書き込み可能回数
    const int WritableCount = 5;
    const size_t BufferSize = 2048;
    nnt::fs::util::SafeMemoryStorage baseStorage(BufferSize);
    nnt::fs::util::PowerInterruptionStorage storage(
        nn::fs::SubStorage(&baseStorage, 0, BufferSize)
    );
    storage.StartCounting(WritableCount);

    std::unique_ptr<char[]> buffer(new char[BufferSize]);

    // 終端 1 バイト、先端 1 バイトに書き込み
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(BufferSize - 1, buffer.get(), 1));
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(1, buffer.get(), BufferSize - 1));

    // 書き込み可能回数ギリギリまで書き込む
    for( int i = 0; i < WritableCount - 2; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Write(0, buffer.get(), BufferSize)
        );
    }
    // この時点ではまだ読み込めることを確かめる
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, buffer.get(), BufferSize));

    // 書き込み不可になっていることを確かめる
    NNT_ASSERT_RESULT_FAILURE(
        nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred,
        storage.Write(BufferSize - 1, buffer.get(), 1)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred,
        storage.Write(1, buffer.get(), BufferSize - 1)
    );

    // カウントを終了させた後は何度でも書き込めることをたしかめる
    storage.StopCounting();
    for( int i = 0; i < 100; ++i )
    {
        size_t offset = (i * i) % BufferSize;
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Write(offset, buffer.get(), BufferSize - offset)
        );
    }
}

// 読み書き不可状態で読み込みテスト
TEST(PowerInterruptionStorageTest, NotReadableTest)
{
    const size_t BufferSize = 1024;
    const int WritableCount = 5;
    nnt::fs::util::SafeMemoryStorage baseStorage(BufferSize);
    nnt::fs::util::PowerInterruptionStorage storage(
        nn::fs::SubStorage(&baseStorage, 0, BufferSize)
    );
    storage.StartCounting(WritableCount);

    std::unique_ptr<char[]> buffer(new char[BufferSize]);

    // 書き込み可能回数ギリギリまで 0 バイト書き込み
    for( int i = 0; i < WritableCount; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Write(0, buffer.get(), 0)
        );
    }

    // Flush も書き込みとしてカウントされる
    // この Flush により読み書き不可となる
    NNT_ASSERT_RESULT_FAILURE(
        nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred,
        storage.Flush()
    );

    // 読み込みが出来なくなっていることを確かめる
    NNT_ASSERT_RESULT_FAILURE(
        nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred,
        storage.Read(BufferSize - 1, buffer.get(), 1)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred,
        storage.Read(1, buffer.get(), BufferSize - 1)
    );

    // カウントを終了させた後は何度でも書き込めることをたしかめる
    storage.StopCounting();
    for( int i = 0; i < 100; ++i )
    {
        size_t offset = (i * i) % BufferSize;
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Read(offset, buffer.get(), BufferSize - offset)
        );
    }
}

// 4 GB を超えるオフセットのテスト
TEST(PowerInterruptionStorageTest, ReadWriteLargeOffset)
{
    static const size_t BufferSize = 1024;
    static const int64_t StorageSize = nnt::fs::util::LargeOffsetMax + BufferSize;

    nnt::fs::util::VirtualMemoryStorage baseStorage(StorageSize);

    nnt::fs::util::PowerInterruptionStorage storage(nn::fs::SubStorage(&baseStorage, 0, StorageSize));
    storage.StartCounting(nnt::fs::util::LargeOffsetListLength + 1); // 共通テストの書き込み回数 + Flush 回数

    // 共通テストで読み書き
    nnt::fs::util::TestStorageAccessWithLargeOffset(&storage, BufferSize);

    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);

    // まだ読める
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(nnt::fs::util::LargeOffsetMax, buffer.get(), BufferSize));

    // もう書けない
    nnt::fs::util::FillBufferWith32BitCount(buffer.get(), BufferSize, 0);
    NNT_ASSERT_RESULT_FAILURE(
        nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred,
        storage.Write(nnt::fs::util::LargeOffsetMax, buffer.get(), BufferSize));

    // もう読めない
    NNT_ASSERT_RESULT_FAILURE(
        nnt::fs::util::PowerInterruptionStorage::ResultPowerInterruptionOccurred,
        storage.Read(nnt::fs::util::LargeOffsetMax, buffer.get(), BufferSize));
}
