﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

static const int LoopCount = 10;

static const size_t SIZE_TABLE_SMALL[] = {
    1, 15, 1023, 1024, 16 * 1024
};

static const size_t SIZE_TABLE_LARGE[] = {
    1, 15, 1023, 1024, 512 * 1024
};

typedef ::testing::TestWithParam< size_t > SmallMemoryFileTest;
typedef ::testing::TestWithParam< size_t > LargeMemoryFileTest;

// 全領域リードライトしてデータが化けないことを確認します。
TEST_P(SmallMemoryFileTest, ReadWriteAll)
{
    const size_t BufferSize = GetParam();

    std::unique_ptr<nnt::fs::util::MemoryFile> file(new nnt::fs::util::MemoryFile());
    NNT_ASSERT_RESULT_SUCCESS(file->SetSize(BufferSize));
    for( int i = 0; i < LoopCount; ++i )
    {
        std::unique_ptr<uint8_t[]> bufWrite(new uint8_t [BufferSize]);
        std::unique_ptr<uint8_t[]> bufRead(new uint8_t [BufferSize]);
        nnt::fs::util::FillBufferWithRandomValue(bufWrite.get(), BufferSize);
        NNT_EXPECT_RESULT_SUCCESS(file->Write(0, bufWrite.get(), BufferSize, nn::fs::WriteOption()));
        size_t sizeResult;
        NNT_EXPECT_RESULT_SUCCESS(file->Read(&sizeResult, 0, bufRead.get(), BufferSize, nn::fs::ReadOption()));
        ASSERT_EQ(sizeResult, BufferSize);
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(bufWrite.get(), bufRead.get(), BufferSize);
    }
}

// ランダムリードライトしてデータが化けないことを確認します。
TEST_P(LargeMemoryFileTest, ReadWriteRandom)
{
    const size_t BufferSize = GetParam();

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    std::unique_ptr<nnt::fs::util::MemoryFile> file(new nnt::fs::util::MemoryFile());
    NNT_ASSERT_RESULT_SUCCESS(file->SetSize(BufferSize));
    for( int i = 0; i < LoopCount; ++i )
    {
        int64_t offset = std::uniform_int_distribution<int64_t>(0, BufferSize - 1)(mt);
        size_t sizeAccess
            = std::uniform_int_distribution<size_t>(
                1,
                static_cast<size_t>(BufferSize - offset)
              )(mt);
        std::unique_ptr<uint8_t[]> bufWrite(new uint8_t [BufferSize]);
        std::unique_ptr<uint8_t[]> bufRead(new uint8_t [BufferSize]);
        nnt::fs::util::FillBufferWithRandomValue(bufWrite.get(), BufferSize);
        NNT_EXPECT_RESULT_SUCCESS(file->Write(offset, bufWrite.get(), sizeAccess, nn::fs::WriteOption()));
        size_t sizeResult;
        NNT_EXPECT_RESULT_SUCCESS(file->Read(&sizeResult, offset, bufRead.get(), sizeAccess, nn::fs::ReadOption()));
        ASSERT_EQ(sizeResult, sizeAccess);
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(bufWrite.get(), bufRead.get(), sizeAccess);
    }
}

// 範囲外を読み込んでエラーが発生することを確認します。
TEST_P(LargeMemoryFileTest, ReadOutOfRange)
{
    const size_t BufferSize = GetParam();
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    std::unique_ptr<nnt::fs::util::MemoryFile> file(new nnt::fs::util::MemoryFile());
    NNT_ASSERT_RESULT_SUCCESS(file->SetSize(BufferSize));
    for( int i = 0; i < LoopCount; ++i )
    {
        std::unique_ptr<char[]> buf(new char [BufferSize]);
        // オフセットの時点で範囲外になるよう乱数を生成します。
        int64_t offset = std::uniform_int_distribution<int64_t>(BufferSize + 1, BufferSize + 10000)(mt);
        size_t sizeRead = std::uniform_int_distribution<size_t>(1, 10000)(mt);
        size_t sizeResult;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            file->Read(&sizeResult, offset, buf.get(), sizeRead, nn::fs::ReadOption())
        );
    }
}

// 範囲外を含む読み込みでもオフセットが範囲内であればエラーが発生しないことを確認します。
TEST_P(LargeMemoryFileTest, ReadPartial)
{
    const size_t BufferSize = GetParam();
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    std::unique_ptr<nnt::fs::util::MemoryFile> file(new nnt::fs::util::MemoryFile());
    NNT_ASSERT_RESULT_SUCCESS(file->SetSize(BufferSize));
    for( int i = 0; i < LoopCount; ++i )
    {
        // オフセット+サイズが範囲外でもオフセットが範囲内ならエラーにならないことを確認します。
        std::unique_ptr<char[]> buf(new char [BufferSize]);
        size_t offset = std::uniform_int_distribution<size_t>(0, BufferSize - 1)(mt);
        size_t sizeRead = std::uniform_int_distribution<size_t>(BufferSize - offset + 1, BufferSize + 1)(mt);
        size_t sizeResult;
        NNT_ASSERT_RESULT_SUCCESS(file->Read(&sizeResult, offset, buf.get(), sizeRead, nn::fs::ReadOption()));
        EXPECT_GE(sizeRead, sizeResult);
    }
}

// 範囲外に書き込んでエラーが発生することを確認します。
TEST_P(LargeMemoryFileTest, WriteOutOfRange)
{
    const size_t BufferSize = GetParam();
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    std::unique_ptr<nnt::fs::util::MemoryFile> file(new nnt::fs::util::MemoryFile());
    NNT_ASSERT_RESULT_SUCCESS(file->SetSize(BufferSize));
    for( int i = 0; i < LoopCount; ++i )
    {
        std::unique_ptr<char[]> buf(new char [BufferSize]);
        int64_t offset = std::uniform_int_distribution<int64_t>(BufferSize + 1, BufferSize + 10000)(mt);
        size_t sizeWrite = std::uniform_int_distribution<size_t>(1, 10000)(mt);
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            file->Write(offset, buf.get(), sizeWrite, nn::fs::WriteOption())
        );
    }
}

// 範囲外を含む書き込みでもオフセットが範囲内であればエラーが発生しないことを確認します。
TEST_P(LargeMemoryFileTest, WritePartial)
{
    const size_t BufferSize = GetParam();
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    std::unique_ptr<nnt::fs::util::MemoryFile> file(new nnt::fs::util::MemoryFile());
    NNT_ASSERT_RESULT_SUCCESS(file->SetSize(BufferSize));
    for( int i = 0; i < LoopCount; ++i )
    {
        // offset が 0 でも範囲外まで書き込むサイズのバッファを作成
        std::unique_ptr<char[]> buf(new char [BufferSize + 1]);
        size_t offset = std::uniform_int_distribution<size_t>(0, BufferSize - 1)(mt);
        size_t sizeWrite
            = std::uniform_int_distribution<size_t>(BufferSize - offset + 1, BufferSize + 1)(mt);
        NNT_ASSERT_RESULT_SUCCESS(file->Write(offset, buf.get(), sizeWrite, nn::fs::WriteOption()));
    }
}

INSTANTIATE_TEST_CASE_P(
    MemoryFileTest,
    SmallMemoryFileTest,
    ::testing::ValuesIn(SIZE_TABLE_SMALL)
);

INSTANTIATE_TEST_CASE_P(
    MemoryFileTest,
    LargeMemoryFileTest,
    ::testing::ValuesIn(SIZE_TABLE_LARGE)
);
