﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

class AccessCounterStorageTest : public ::testing::Test { };

// リードテスト
TEST_F(AccessCounterStorageTest, TestRead)
{
    const size_t BufferSize = 1024;
    nnt::fs::util::SafeMemoryStorage baseStorage(BufferSize);
    nnt::fs::util::AccessCountWrapperStorage storage(&baseStorage);
    std::unique_ptr<char[]> buf(new char [BufferSize]);
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, buf.get(), BufferSize));
    ASSERT_EQ(1, storage.GetReadTimes());
    ASSERT_EQ(0, storage.GetWriteTimes());
    ASSERT_EQ(0, storage.GetFlushTimes());
}

// 複数回リードテスト
TEST_F(AccessCounterStorageTest, TestReadMultiple)
{
    std::mt19937 mt(1);

    const size_t BufferSize = 1024;
    nnt::fs::util::SafeMemoryStorage baseStorage(BufferSize);
    nnt::fs::util::AccessCountWrapperStorage storage(&baseStorage);
    std::unique_ptr<char[]> buf(new char [BufferSize]);
    const int loopCount = std::uniform_int_distribution<int>(1, 100)(mt);
    for( int loop = 0; loop < loopCount; ++loop )
    {
        int64_t offset = std::uniform_int_distribution<int64_t>(0, BufferSize - 1)(mt);
        size_t sizeAccess = std::uniform_int_distribution<size_t>(1, static_cast<size_t>(BufferSize - offset))(mt);
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, buf.get(), sizeAccess));
        ASSERT_EQ(loop + 1, storage.GetReadTimes());
        ASSERT_EQ(0, storage.GetWriteTimes());
        ASSERT_EQ(0, storage.GetFlushTimes());
    }
}

// ライトテスト
TEST_F(AccessCounterStorageTest, TestWrite)
{
    const size_t BufferSize = 1024;
    nnt::fs::util::SafeMemoryStorage baseStorage(BufferSize);
    nnt::fs::util::AccessCountWrapperStorage storage(&baseStorage);
    std::unique_ptr<char[]> buf(new char [BufferSize]);
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(0, buf.get(), BufferSize));
    ASSERT_EQ(0, storage.GetReadTimes());
    ASSERT_EQ(1, storage.GetWriteTimes());
    ASSERT_EQ(0, storage.GetFlushTimes());
}

// 複数回ライトテスト
TEST_F(AccessCounterStorageTest, TestWriteMultiple)
{
    std::mt19937 mt(1);

    const size_t BufferSize = 1024;
    nnt::fs::util::SafeMemoryStorage baseStorage(BufferSize);
    nnt::fs::util::AccessCountWrapperStorage storage(&baseStorage);
    std::unique_ptr<char[]> buf(new char [BufferSize]);
    const int loopCount = std::uniform_int_distribution<int>(1, 100)(mt);
    for( int loop = 0; loop < loopCount; ++loop )
    {
        int64_t offset = std::uniform_int_distribution<int64_t>(0, BufferSize - 1)(mt);
        size_t sizeAccess = std::uniform_int_distribution<size_t>(1, static_cast<size_t>(BufferSize - offset))(mt);
        NNT_ASSERT_RESULT_SUCCESS(storage.Write(offset, buf.get(), sizeAccess));
        ASSERT_EQ(0, storage.GetReadTimes());
        ASSERT_EQ(loop + 1, storage.GetWriteTimes());
        ASSERT_EQ(0, storage.GetFlushTimes());
    }
}

// フラッシュテスト
TEST_F(AccessCounterStorageTest, TestFlush)
{
    const size_t BufferSize = 1024;
    nnt::fs::util::SafeMemoryStorage baseStorage(BufferSize);
    nnt::fs::util::AccessCountWrapperStorage storage(&baseStorage);
    NNT_ASSERT_RESULT_SUCCESS(storage.Flush());
    ASSERT_EQ(0, storage.GetReadTimes());
    ASSERT_EQ(0, storage.GetWriteTimes());
    ASSERT_EQ(1, storage.GetFlushTimes());
}

// 複数回フラッシュテスト
TEST_F(AccessCounterStorageTest, TestFlushMultiple)
{
    std::mt19937 mt(1);

    const size_t BufferSize = 1024;
    nnt::fs::util::SafeMemoryStorage baseStorage(BufferSize);
    nnt::fs::util::AccessCountWrapperStorage storage(&baseStorage);
    const int loopCount = std::uniform_int_distribution<int>(1, 100)(mt);
    for( int loop = 0; loop < loopCount; ++loop )
    {
        NNT_ASSERT_RESULT_SUCCESS(storage.Flush());
        ASSERT_EQ(0, storage.GetReadTimes());
        ASSERT_EQ(0, storage.GetWriteTimes());
        ASSERT_EQ(loop + 1, storage.GetFlushTimes());
    }
}

