﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/fssrv/fssrv_SaveDataSharedFileStorage.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include "testFs_Unit_DummyFileSystem.h"
#include <nnt/fsUtil/testFs_util_function.h>

namespace {

    const int saveDataOpenCountMax = 10;

    const nn::fs::SaveDataId testSaveDataIds[] =
    {
        0x8000000000008000,
        0x8000000000008001,
        0x8000000000008002,
        0x8000000000008003,
        0x8000000000008004,
        0x8000000000008005,
        0x8000000000008006,
        0x8000000000008007,
        0x8000000000008008,
        0x8000000000008009,
        0x800000000000800a,
    };

}

// OpenSaveData(Mount)のテスト
TEST(SaveDataSharedFileStorageTest, OpenSaveDataApplication)
{
    std::shared_ptr<nn::fs::IStorage> fileStorageApplication1[saveDataOpenCountMax + 1];
    std::shared_ptr<nn::fs::IStorage> fileStorageApplication2[saveDataOpenCountMax + 1];
    std::shared_ptr<nn::fs::IStorage> fileStorageSystem1[saveDataOpenCountMax + 1];
    std::shared_ptr<nn::fs::IStorage> fileStorageSystem2[saveDataOpenCountMax + 1];

    for( auto i = 0; i < saveDataOpenCountMax; ++i)
    {
        std::shared_ptr<nn::fs::detail::DummyFileSystem> fileSystem(new nn::fs::detail::DummyFileSystem());
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssrv::OpenSaveDataStorage(
                &fileStorageApplication1[i],
                std::move(fileSystem),
                testSaveDataIds[i],
                nn::fs::SaveDataType::Account,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write),
                nn::fssrv::SaveDataOpenTypeSetFileStorage::OpenType::Normal));
    }

    // すでに開いているファイルは開けない
    for( auto i = 0; i < saveDataOpenCountMax; ++i)
    {
        std::shared_ptr<nn::fs::detail::DummyFileSystem> fileSystem(new nn::fs::detail::DummyFileSystem());
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultTargetLocked,
            nn::fssrv::OpenSaveDataStorage(
                &fileStorageApplication2[i],
                std::move(fileSystem),
                testSaveDataIds[i],
                nn::fs::SaveDataType::Account,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write),
                nn::fssrv::SaveDataOpenTypeSetFileStorage::OpenType::Normal));
    }

    // Normal で開いているセーブデータも Internal で開ける
    for( auto i = 0; i < saveDataOpenCountMax; ++i)
    {
        std::shared_ptr<nn::fs::detail::DummyFileSystem> fileSystem(new nn::fs::detail::DummyFileSystem());
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssrv::OpenSaveDataStorage(
                &fileStorageSystem1[i],
                std::move(fileSystem),
                testSaveDataIds[i],
                nn::fs::SaveDataType::Account,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write),
                nn::fssrv::SaveDataOpenTypeSetFileStorage::OpenType::Internal));
    }
}

// OpenSaveData(Mount)のテストの逆パターン
TEST(SaveDataSharedFileStorageTest, OpenSaveDataSystem)
{
    std::shared_ptr<nn::fs::IStorage> fileStorageApplication1[saveDataOpenCountMax + 1];
    std::shared_ptr<nn::fs::IStorage> fileStorageApplication2[saveDataOpenCountMax + 1];
    std::shared_ptr<nn::fs::IStorage> fileStorageSystem1[saveDataOpenCountMax + 1];
    std::shared_ptr<nn::fs::IStorage> fileStorageSystem2[saveDataOpenCountMax + 1];

    // saveDataOpenCountLimit までは同時に開ける
    for( auto i = 0; i < saveDataOpenCountMax; ++i)
    {
        std::shared_ptr<nn::fs::detail::DummyFileSystem> fileSystem(new nn::fs::detail::DummyFileSystem());
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssrv::OpenSaveDataStorage(
                &fileStorageSystem1[i],
                std::move(fileSystem),
                testSaveDataIds[i],
                nn::fs::SaveDataType::Account,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write),
                nn::fssrv::SaveDataOpenTypeSetFileStorage::OpenType::Internal));
    }

    // すでに開いているファイルは開けない
    for( auto i = 0; i < saveDataOpenCountMax; ++i)
    {
        std::shared_ptr<nn::fs::detail::DummyFileSystem> fileSystem(new nn::fs::detail::DummyFileSystem());
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultTargetLocked,
            nn::fssrv::OpenSaveDataStorage(
                &fileStorageSystem1[i],
                std::move(fileSystem),
                testSaveDataIds[i],
                nn::fs::SaveDataType::Account,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write),
                nn::fssrv::SaveDataOpenTypeSetFileStorage::OpenType::Internal));
    }

    // Internal で開いているセーブデータも Normal で開ける
    for( auto i = 0; i < saveDataOpenCountMax; ++i)
    {
        std::shared_ptr<nn::fs::detail::DummyFileSystem> fileSystem(new nn::fs::detail::DummyFileSystem());
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssrv::OpenSaveDataStorage(
                &fileStorageApplication1[i],
                std::move(fileSystem),
                testSaveDataIds[i],
                nn::fs::SaveDataType::Account,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write),
                nn::fssrv::SaveDataOpenTypeSetFileStorage::OpenType::Normal));
    }
}

TEST(SaveDataSharedFileStorageTest, FileAccess)
{
    std::shared_ptr<nn::fs::IStorage> fileStorageApplication;
    std::shared_ptr<nn::fs::IStorage> fileStorageSystem;

    {
        std::shared_ptr<nn::fs::detail::DummyFileSystem> fileSystem(new nn::fs::detail::DummyFileSystem());
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssrv::OpenSaveDataStorage(
                &fileStorageApplication,
                std::move(fileSystem),
                testSaveDataIds[0],
                nn::fs::SaveDataType::Account,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write),
                nn::fssrv::SaveDataOpenTypeSetFileStorage::OpenType::Normal));
    }

    // アプリ側アクセステスト
    {
        int64_t size;
        char buffer[64];
        const char buffer2[100] = {};
        NNT_ASSERT_RESULT_SUCCESS(fileStorageApplication->SetSize(64));
        NNT_ASSERT_RESULT_SUCCESS(fileStorageApplication->GetSize(&size));
        NNT_ASSERT_RESULT_SUCCESS(fileStorageApplication->Read(0, buffer, 64));
        NNT_ASSERT_RESULT_SUCCESS(fileStorageApplication->Write(0, buffer, 64));
        NNT_ASSERT_RESULT_SUCCESS(fileStorageApplication->Flush());
        NNT_ASSERT_RESULT_SUCCESS(fileStorageApplication->OperateRange(buffer, 64, nn::fs::OperationId::QueryRange, 0, 0, buffer2, 100));
    }

    {
        std::shared_ptr<nn::fs::detail::DummyFileSystem> fileSystem(new nn::fs::detail::DummyFileSystem());
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssrv::OpenSaveDataStorage(
                &fileStorageSystem,
                std::move(fileSystem),
                testSaveDataIds[0],
                nn::fs::SaveDataType::Account,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write),
                nn::fssrv::SaveDataOpenTypeSetFileStorage::OpenType::Internal));
    }

    // Internal 側アクセステスト
    {
        int64_t size;
        char buffer[64];
        const char buffer2[100] = {};
        NNT_ASSERT_RESULT_SUCCESS(fileStorageSystem->SetSize(64));
        NNT_ASSERT_RESULT_SUCCESS(fileStorageSystem->GetSize(&size));
        NNT_ASSERT_RESULT_SUCCESS(fileStorageSystem->Read(0, buffer, 64));
        NNT_ASSERT_RESULT_SUCCESS(fileStorageSystem->Write(0, buffer, 64));
        NNT_ASSERT_RESULT_SUCCESS(fileStorageSystem->Flush());
        NNT_ASSERT_RESULT_SUCCESS(fileStorageSystem->OperateRange(buffer, 64, nn::fs::OperationId::QueryRange, 0, 0, buffer2, 100));
    }

    // Internal 側をアクセスしてても Normal 側はアクセス出来る
    {
        int64_t size;
        char buffer[64];
        const char buffer2[100] = {};
        NNT_ASSERT_RESULT_SUCCESS(fileStorageApplication->SetSize(64));
        NNT_ASSERT_RESULT_SUCCESS(fileStorageApplication->GetSize(&size));
        NNT_ASSERT_RESULT_SUCCESS(fileStorageApplication->Read(0, buffer, 64));
        NNT_ASSERT_RESULT_SUCCESS(fileStorageApplication->Write(0, buffer, 64));
        NNT_ASSERT_RESULT_SUCCESS(fileStorageApplication->Flush());
        NNT_ASSERT_RESULT_SUCCESS(fileStorageApplication->OperateRange(buffer, 64, nn::fs::OperationId::QueryRange, 0, 0, buffer2, 100));
    }

    // Normal 側をアクセスすると Internal 側はアクセス出来なくなる
    {
        int64_t size;
        char buffer[64];
        const char buffer2[100] = {};
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, fileStorageSystem->SetSize(64));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, fileStorageSystem->GetSize(&size));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, fileStorageSystem->Read(0, buffer, 64));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, fileStorageSystem->Write(0, buffer, 64));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, fileStorageSystem->Flush());
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, fileStorageSystem->OperateRange(buffer, 64, nn::fs::OperationId::QueryRange, 0, 0, buffer2, 100));
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
