﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Allocator.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/kvdb/kvdb_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fssrv/fssrv_SaveDataIndexer.h>
#include <nn/fssrv/fssrv_SaveDataIndexerManager.h>
#include <nn/fssrv/fssrv_SaveDataIndexerTypes.h>
#include <nn/fssrv/fssrv_SaveDataInfoReaderImpl.h>
#include <nn/fssystem/fs_AllocatorUtility.h>

#include "shim/fs_Library.h"

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;

Result PublishSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, nn::fssrv::SaveDataIndexerKey key, bool doVerify)
{
    SaveDataId saveDataId;
    NN_RESULT_DO(saveDataIndexer->Publish(&saveDataId, key));
    if (doVerify)
    {
        nn::fssrv::SaveDataIndexerKey tmpKey;
        NN_RESULT_DO(saveDataIndexer->GetKey(&tmpKey, saveDataId));
        EXPECT_EQ(key, tmpKey);

        // 重複 Put できないことの確認
        if (key.staticSaveDataId != InvalidSystemSaveDataId && key.userId != InvalidUserId)
        {
            NNT_EXPECT_RESULT_FAILURE(ResultAlreadyExists, saveDataIndexer->PutStaticSaveDataIdIndex(key));
        }
    }
    *outValue = saveDataId;
    NN_RESULT_SUCCESS;
}

const int64_t SaveDataSize = 1024;

Result PublishSystemSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, Bit64 applicationId, bool doVerify)
{
    SaveDataSpaceId spaceId = SaveDataSpaceId::System;
    nn::ncm::ProgramId programId = { applicationId };
    nn::fssrv::SaveDataIndexerKey key = nn::fssrv::SaveDataIndexerKey::Make(programId, SaveDataType::System, InvalidUserId, applicationId);
    NN_RESULT_DO(PublishSaveDataId(outValue, saveDataIndexer, key, doVerify));
    NN_RESULT_DO(saveDataIndexer->SetSpaceId(*outValue, spaceId));
    return saveDataIndexer->SetSize(*outValue, SaveDataSize);
}

Result PublishBcatSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, Bit64 applicationId, bool doVerify)
{
    SaveDataSpaceId spaceId = SaveDataSpaceId::User;
    nn::ncm::ProgramId programId = { applicationId };
    nn::fssrv::SaveDataIndexerKey key = nn::fssrv::SaveDataIndexerKey::Make(programId, SaveDataType::Bcat, InvalidUserId, InvalidSystemSaveDataId);
    NN_RESULT_DO(PublishSaveDataId(outValue, saveDataIndexer, key, doVerify));
    NN_RESULT_DO(saveDataIndexer->SetSpaceId(*outValue, spaceId));
    return saveDataIndexer->SetSize(*outValue, SaveDataSize);
}

Result PublishDeviceSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, Bit64 applicationId, bool doVerify)
{
    SaveDataSpaceId spaceId = SaveDataSpaceId::User;
    nn::ncm::ProgramId programId = { applicationId };
    nn::fssrv::SaveDataIndexerKey key = nn::fssrv::SaveDataIndexerKey::Make(programId, SaveDataType::Device, InvalidUserId, InvalidSystemSaveDataId);
    NN_RESULT_DO(PublishSaveDataId(outValue, saveDataIndexer, key, doVerify));
    NN_RESULT_DO(saveDataIndexer->SetSpaceId(*outValue, spaceId));
    return saveDataIndexer->SetSize(*outValue, SaveDataSize);
}

Result PublishAccountSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, Bit64 applicationId, UserId userId, bool doVerify)
{
    SaveDataSpaceId spaceId = SaveDataSpaceId::User;
    nn::ncm::ProgramId programId = { applicationId };
    nn::fssrv::SaveDataIndexerKey key = nn::fssrv::SaveDataIndexerKey::Make(programId, SaveDataType::Account, userId, InvalidSystemSaveDataId);
    NN_RESULT_DO(PublishSaveDataId(outValue, saveDataIndexer, key, doVerify));
    NN_RESULT_DO(saveDataIndexer->SetSpaceId(*outValue, spaceId));
    return saveDataIndexer->SetSize(*outValue, SaveDataSize);
}

Result PublishSystemSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, Bit64 applicationId)
{
    return PublishSystemSaveDataId(outValue, saveDataIndexer, applicationId, false);
}

Result PublishBcatSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, Bit64 applicationId)
{
    return PublishBcatSaveDataId(outValue, saveDataIndexer, applicationId, false);
}

Result PublishDeviceSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, Bit64 applicationId)
{
    return PublishDeviceSaveDataId(outValue, saveDataIndexer, applicationId, false);
}

Result PublishAccountSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, Bit64 applicationId, UserId userId)
{
    return PublishAccountSaveDataId(outValue, saveDataIndexer, applicationId, userId, false);
}

Result GetSystemSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, Bit64 applicationId)
{
    SaveDataSpaceId spaceId = SaveDataSpaceId::System;
    nn::ncm::ProgramId programId = { applicationId };
    nn::fssrv::SaveDataIndexerKey key = nn::fssrv::SaveDataIndexerKey::Make(programId, SaveDataType::System, InvalidUserId, applicationId);
    nn::fssrv::SaveDataIndexerValue value;
    NN_RESULT_DO(saveDataIndexer->Get(&value, key));
    {
        nn::fssrv::SaveDataIndexerValue tmpValue;
        NN_RESULT_DO(saveDataIndexer->GetValue(&tmpValue, value.id));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&value, &tmpValue, sizeof(nn::fssrv::SaveDataIndexerValue));
    }
    EXPECT_EQ(value.spaceId, spaceId);
    *outValue = value.id;
    NN_RESULT_SUCCESS;
}

Result GetBcatSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, Bit64 applicationId)
{
    SaveDataSpaceId spaceId = SaveDataSpaceId::User;
    nn::ncm::ProgramId programId = { applicationId };
    nn::fssrv::SaveDataIndexerKey key = nn::fssrv::SaveDataIndexerKey::Make(programId, SaveDataType::Bcat, InvalidUserId, InvalidSystemSaveDataId);
    nn::fssrv::SaveDataIndexerValue value;
    NN_RESULT_DO(saveDataIndexer->Get(&value, key));
    {
        nn::fssrv::SaveDataIndexerValue tmpValue;
        NN_RESULT_DO(saveDataIndexer->GetValue(&tmpValue, value.id));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&value, &tmpValue, sizeof(nn::fssrv::SaveDataIndexerValue));
    }
    EXPECT_EQ(value.spaceId, spaceId);
    *outValue = value.id;
    NN_RESULT_SUCCESS;
}

Result GetDeviceSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, Bit64 applicationId)
{
    SaveDataSpaceId spaceId = SaveDataSpaceId::User;
    nn::ncm::ProgramId programId = { applicationId };
    nn::fssrv::SaveDataIndexerKey key = nn::fssrv::SaveDataIndexerKey::Make(programId, SaveDataType::Device, InvalidUserId, InvalidSystemSaveDataId);
    nn::fssrv::SaveDataIndexerValue value;
    NN_RESULT_DO(saveDataIndexer->Get(&value, key));
    {
        nn::fssrv::SaveDataIndexerValue tmpValue;
        NN_RESULT_DO(saveDataIndexer->GetValue(&tmpValue, value.id));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&value, &tmpValue, sizeof(nn::fssrv::SaveDataIndexerValue));
    }
    EXPECT_EQ(value.spaceId, spaceId);
    *outValue = value.id;
    NN_RESULT_SUCCESS;
}

Result GetAccountSaveDataId(SaveDataId* outValue, nn::fssrv::ISaveDataIndexer* saveDataIndexer, Bit64 applicationId, UserId userId)
{
    SaveDataSpaceId spaceId = SaveDataSpaceId::User;
    nn::ncm::ProgramId programId = { applicationId };
    nn::fssrv::SaveDataIndexerKey key = nn::fssrv::SaveDataIndexerKey::Make(programId, SaveDataType::Account, userId, InvalidSystemSaveDataId);
    nn::fssrv::SaveDataIndexerValue value;
    NN_RESULT_DO(saveDataIndexer->Get(&value, key));
    {
        nn::fssrv::SaveDataIndexerValue tmpValue;
        NN_RESULT_DO(saveDataIndexer->GetValue(&tmpValue, value.id));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&value, &tmpValue, sizeof(nn::fssrv::SaveDataIndexerValue));
    }
    EXPECT_EQ(value.spaceId, spaceId);
    *outValue = value.id;
    NN_RESULT_SUCCESS;
}

/*
  基本動作テスト
  - 異なった属性を持つエントリを登録できること
  - エントリの 2 重登録ができないこと
  - 登録したエントリが取得できること
  - 登録していないエントリが取得できないこと
*/
TEST(SaveDataIndexer, Basic)
{
    nn::fssrv::SaveDataIndexerAccessor accessor;
    accessor.Initialize(nn::fs::SaveDataSpaceId::System);
    auto saveDataIndexer = accessor.GetInterface();
    saveDataIndexer->Reset(); // TODO: NNT_EXPECT_RESULT_SUCCESS

    const int OwnerNum = 512;
    const int UserNum = 5;

    for (int i = 0; i < OwnerNum; i++)
    {
        Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_SUCCESS(PublishSystemSaveDataId(&id, saveDataIndexer, applicationId));
        NNT_EXPECT_RESULT_SUCCESS(PublishBcatSaveDataId(&id, saveDataIndexer, applicationId));
        NNT_EXPECT_RESULT_SUCCESS(PublishDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_SUCCESS(PublishAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
        }
    }

    NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Commit());

    for (int i = 0; i < OwnerNum; i++)
    {
        nn::Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_FAILURE(ResultAlreadyExists, PublishSystemSaveDataId(&id, saveDataIndexer, applicationId));
        NNT_EXPECT_RESULT_FAILURE(ResultAlreadyExists, PublishBcatSaveDataId(&id, saveDataIndexer, applicationId));
        NNT_EXPECT_RESULT_FAILURE(ResultAlreadyExists, PublishDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_FAILURE(ResultAlreadyExists, PublishAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
        }
    }

    SaveDataId ansId = 1;
    for (int i = 0; i < OwnerNum; i++)
    {
        nn::Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_SUCCESS(GetSystemSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        NNT_EXPECT_RESULT_SUCCESS(GetBcatSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        NNT_EXPECT_RESULT_SUCCESS(GetDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_SUCCESS(GetAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
            EXPECT_EQ(id, ansId++);
        }
    }

    {
        nn::Bit64 applicationId = OwnerNum + 1;
        SaveDataId id;

        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetSystemSaveDataId(&id, saveDataIndexer, applicationId));
        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetBcatSaveDataId(&id, saveDataIndexer, applicationId));
        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
        }
    }
}

/*
  Delete
  - 異なった属性を持つエントリを削除できること
  - 削除されていないエントリが正しく取得できること
  - 削除した後、発行した SaveId が巻き戻らないこと
*/
TEST(SaveDataIndexer, Delete)
{
    nn::fssrv::SaveDataIndexerAccessor accessor;
    accessor.Initialize(nn::fs::SaveDataSpaceId::System);
    auto saveDataIndexer = accessor.GetInterface();
    saveDataIndexer->Reset(); // TODO: NNT_EXPECT_RESULT_SUCCESS

    const int OwnerNum = 512;
    const int UserNum = 5;

    int publishCount = 0;
    for (int i = 0; i < OwnerNum; i++)
    {
        Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_SUCCESS(PublishSystemSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        NNT_EXPECT_RESULT_SUCCESS(PublishBcatSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        NNT_EXPECT_RESULT_SUCCESS(PublishDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_SUCCESS(PublishAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
            publishCount++;
        }
    }

    for (int i = publishCount / 2; i < publishCount; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Delete(static_cast<Bit64>(i + 1)));
    }

    SaveDataId ansId = 1;
    for (int i = 0; i < OwnerNum / 2; i++)
    {
        Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_SUCCESS(GetSystemSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        NNT_EXPECT_RESULT_SUCCESS(GetBcatSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        NNT_EXPECT_RESULT_SUCCESS(GetDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_SUCCESS(GetAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
            EXPECT_EQ(id, ansId++);
        }
    }

    for (int i = OwnerNum / 2; i < OwnerNum; i++)
    {
        nn::Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetSystemSaveDataId(&id, saveDataIndexer, applicationId));
        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetBcatSaveDataId(&id, saveDataIndexer, applicationId));
        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
        }
    }
}

/*
  GetKey
  - 異なった属性を持つエントリで正しく Key の情報が逆引きできること
  - 存在しない ID に対して ResultTargetNotFound が返ること
*/
TEST(SaveDataIndexer, GetKey)
{
    nn::fssrv::SaveDataIndexerAccessor accessor;
    accessor.Initialize(nn::fs::SaveDataSpaceId::System);
    auto saveDataIndexer = accessor.GetInterface();
    saveDataIndexer->Reset(); // TODO: NNT_EXPECT_RESULT_SUCCESS

    const int OwnerNum = 256;
    const int UserNum = 5;

    int publishCount = 0;
    for (int i = 0; i < OwnerNum; i++)
    {
        Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_SUCCESS(PublishSystemSaveDataId(&id, saveDataIndexer, applicationId, true));
        publishCount++;
        NNT_EXPECT_RESULT_SUCCESS(PublishBcatSaveDataId(&id, saveDataIndexer, applicationId, true));
        publishCount++;
        NNT_EXPECT_RESULT_SUCCESS(PublishDeviceSaveDataId(&id, saveDataIndexer, applicationId, true));
        publishCount++;
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_SUCCESS(PublishAccountSaveDataId(&id, saveDataIndexer, applicationId, userId, true));
            publishCount++;
        }
    }

    {
        nn::fssrv::SaveDataIndexerKey tmpKey;
        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, saveDataIndexer->GetKey(&tmpKey, static_cast<SaveDataId>(publishCount + 1)));
    }
}

/*
  Commit/Rollback
  - 複数のエントリの登録がロールバックできること
  - ロールバック前に作成したエントリが正しく取得できること
  - ロールバックした後、発行した SaveId が巻き戻ること
*/
TEST(SaveDataIndexer, CommitAndRollback)
{
    nn::fssrv::SaveDataIndexerAccessor accessor;
    accessor.Initialize(nn::fs::SaveDataSpaceId::System);
    auto saveDataIndexer = accessor.GetInterface();
    saveDataIndexer->Reset(); // TODO: NNT_EXPECT_RESULT_SUCCESS

    const int OwnerNum = 512;
    const int UserNum = 5;

    int publishCount = 0;
    for (int i = 0; i < OwnerNum; i += 2) // applicationId が偶数
    {
        Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_SUCCESS(PublishSystemSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        NNT_EXPECT_RESULT_SUCCESS(PublishBcatSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        NNT_EXPECT_RESULT_SUCCESS(PublishDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_SUCCESS(PublishAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
            publishCount++;
        }
    }

    NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Commit());

    // applicationId が偶数のものを削除
    for (int i = 0; i < publishCount; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Delete(static_cast<Bit64>(i + 1)));
    }

    SaveDataId ansId = publishCount + 1;
    // applicationId が奇数のものを作成
    for (int i = 1; i < OwnerNum; i += 2)
    {
        Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_SUCCESS(PublishSystemSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        NNT_EXPECT_RESULT_SUCCESS(PublishBcatSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        NNT_EXPECT_RESULT_SUCCESS(PublishDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_SUCCESS(PublishAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
            EXPECT_EQ(id, ansId++);
        }
    }

    NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Rollback());

    ansId = 1;
    for (int i = 0; i < OwnerNum; i += 2)
    {
        nn::Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_SUCCESS(GetSystemSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        NNT_EXPECT_RESULT_SUCCESS(GetBcatSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        NNT_EXPECT_RESULT_SUCCESS(GetDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_SUCCESS(GetAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
            EXPECT_EQ(id, ansId++);
        }
    }

    for (int i = 1; i < OwnerNum; i += 2)
    {
        nn::Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetSystemSaveDataId(&id, saveDataIndexer, applicationId));
        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetBcatSaveDataId(&id, saveDataIndexer, applicationId));
        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
        }
    }

    ansId = publishCount + 1;
    // applicationId が奇数のものを作成
    for (int i = 1; i < OwnerNum; i += 2)
    {
        Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_SUCCESS(PublishSystemSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        NNT_EXPECT_RESULT_SUCCESS(PublishBcatSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        NNT_EXPECT_RESULT_SUCCESS(PublishDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        EXPECT_EQ(id, ansId++);
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_SUCCESS(PublishAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
            EXPECT_EQ(id, ansId++);
        }
    }
}

/*
  SaveDataInfoReader
  - 異なった属性を持つエントリの情報を取得できること
  - spaceId で検索できること
  - attribute で検索できること
  - size が正しく設定されていること
*/
TEST(SaveDataIndexer, SaveDataInfoReader)
{
    nn::fssrv::SaveDataIndexerAccessor accessor;
    accessor.Initialize(nn::fs::SaveDataSpaceId::System);
    auto saveDataIndexer = accessor.GetInterface();
    saveDataIndexer->Reset(); // TODO: NNT_EXPECT_RESULT_SUCCESS

    const int OwnerNum = 512;
    const int UserNum = 5;

    int publishCount = 0;
    for (int i = 0; i < OwnerNum; i++)
    {
        Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_SUCCESS(PublishSystemSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        NNT_EXPECT_RESULT_SUCCESS(PublishBcatSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        NNT_EXPECT_RESULT_SUCCESS(PublishDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_SUCCESS(PublishAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
            publishCount++;
        }

        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Commit());
    }

    // all
    {
        std::shared_ptr<nn::fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->OpenSaveDataInfoReader(&reader));

        SaveDataInfo info[4];
        int64_t totalCount = 0;
        while (NN_STATIC_CONDITION(true))
        {
            int64_t count;
            NNT_EXPECT_RESULT_SUCCESS(reader->Read(nn::sf::Out<int64_t>(&count), nn::sf::OutBuffer(reinterpret_cast<char*>(info), sizeof(nn::fs::SaveDataInfo) * 4)));
            if (count == 0)
            {
                break;
            }
            for (int i = 0; i < sizeof(info) / sizeof(info[0]); i++)
            {
                nn::fssrv::SaveDataIndexerValue value;
                nn::fssrv::SaveDataIndexerKey key = nn::fssrv::SaveDataIndexerKey::Make(info[i].applicationId, info[i].saveDataType, info[i].saveDataUserId, info[i].saveDataType == SaveDataType::System ? info[i].systemSaveDataId : InvalidSystemSaveDataId);
                NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Get(&value, key));
                {
                    nn::fssrv::SaveDataIndexerValue tmpValue;
                    NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->GetValue(&tmpValue, value.id));
                    NNT_FS_UTIL_EXPECT_MEMCMPEQ(&value, &tmpValue, sizeof(nn::fssrv::SaveDataIndexerValue));
                }
                EXPECT_EQ(value.id, info[i].saveDataId);
                EXPECT_EQ(info[i].saveDataSize, SaveDataSize);
            }
            totalCount += count;
        }
        EXPECT_EQ(totalCount, publishCount);
    }
}


/*
  IsRemainedReservedOnly
  - 既定のインデックスに達した場合に IsRemainedReservedOnly が true で返ること
  - 上記状態になった後も、ReservedIndexCount 分追加できること
  - 予約分も含んだインデックスの上限値に達すると、以降の追加ではエラーが返ること
*/
TEST(SaveDataIndexer, IsRemainedReservedOnly)
{
    nn::fssrv::SaveDataIndexerAccessor accessor;
    accessor.Initialize(nn::fs::SaveDataSpaceId::System);
    auto saveDataIndexer = accessor.GetInterface();
    saveDataIndexer->Reset(); // TODO: NNT_EXPECT_RESULT_SUCCESS

    const int OwnerNum = 512;
    const int UserNum = 5;

    int publishCount = 0;
    for (int i = 0; i < OwnerNum; i++)
    {
        Bit64 applicationId = i;
        SaveDataId id;

        NNT_EXPECT_RESULT_SUCCESS(PublishSystemSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        NNT_EXPECT_RESULT_SUCCESS(PublishBcatSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        NNT_EXPECT_RESULT_SUCCESS(PublishDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_SUCCESS(PublishAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
            publishCount++;
        }
    }

    NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Commit());

    EXPECT_TRUE(saveDataIndexer->IsRemainedReservedOnly());

    const int ReservedCount = 128;

    for (int i = 0; i < ReservedCount / (UserNum + 3); i++)
    {
        Bit64 applicationId = i + OwnerNum;
        SaveDataId id;

        NNT_EXPECT_RESULT_SUCCESS(PublishSystemSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        NNT_EXPECT_RESULT_SUCCESS(PublishBcatSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        NNT_EXPECT_RESULT_SUCCESS(PublishDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        publishCount++;
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_SUCCESS(PublishAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
            publishCount++;
        }

        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Commit());
    }

    {
        Bit64 applicationId = publishCount;
        SaveDataId id;

        NNT_EXPECT_RESULT_FAILURE(kvdb::ResultOutOfMaxKeySize, PublishSystemSaveDataId(&id, saveDataIndexer, applicationId));
        NNT_EXPECT_RESULT_FAILURE(kvdb::ResultOutOfMaxKeySize, PublishBcatSaveDataId(&id, saveDataIndexer, applicationId));
        NNT_EXPECT_RESULT_FAILURE(kvdb::ResultOutOfMaxKeySize, PublishDeviceSaveDataId(&id, saveDataIndexer, applicationId));
        for (int j = 0; j < UserNum; j++)
        {
            UserId userId = {{ 0x00ull, static_cast<Bit64>(j + 1) }};
            NNT_EXPECT_RESULT_FAILURE(kvdb::ResultOutOfMaxKeySize, PublishAccountSaveDataId(&id, saveDataIndexer, applicationId, userId));
        }
    }
}

#if 0
// db を更新する操作後に既存の Reader が無効化されること
TEST(SaveDataIndexer, InvalidSaveDataInfoReader)
{
    nn::fssrv::SaveDataIndexerAccessor accessor;
    accessor.Initialize(nn::fs::SaveDataSpaceId::System);
    auto saveDataIndexer = accessor.GetInterface();

    const SaveDataId IdA = 0x8000000000004000;

    fssrv::SaveDataIndexerKey keyA = { {0} };
    keyA.staticSaveDataId = IdA;
    keyA.userId           = InvalidUserId;

    std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;

    {
        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Reset());

        auto OpenReader = [&saveDataIndexer](std::shared_ptr<fssrv::SaveDataInfoReaderImpl>* outReader)
        {
            NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->OpenSaveDataInfoReader(outReader));
        };

        auto CheckReaderInvalid = [](std::shared_ptr<fssrv::SaveDataInfoReaderImpl>& reader)
        {
            SaveDataInfo info;
            int64_t count;
            NNT_EXPECT_RESULT_FAILURE(ResultInvalidHandle, reader->Read(nn::sf::Out<int64_t>(&count), nn::sf::OutBuffer(reinterpret_cast<char*>(&info), sizeof(nn::fs::SaveDataInfo) * 1)));
        };

        OpenReader(&reader);
        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->PutStaticSaveDataIdIndex(keyA));
        CheckReaderInvalid(reader);

        OpenReader(&reader);
        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Delete(IdA));
        CheckReaderInvalid(reader);

        OpenReader(&reader);
        SaveDataId id;
        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Publish(&id, keyA));
        CheckReaderInvalid(reader);

        OpenReader(&reader);
        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->SetSize(id, 0));
        CheckReaderInvalid(reader);

        OpenReader(&reader);
        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->SetSpaceId(id, SaveDataSpaceId::User));
        CheckReaderInvalid(reader);

        OpenReader(&reader);
        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->SetState(id, fssrv::SaveDataState::None));
        CheckReaderInvalid(reader);

        OpenReader(&reader);
        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Reset());
        CheckReaderInvalid(reader);

        OpenReader(&reader);
        NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Rollback());
        CheckReaderInvalid(reader);

        {
            NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->PutStaticSaveDataIdIndex(keyA));

            OpenReader(&reader);

            // db を更新しない操作では無効化されないこと
            {
                std::shared_ptr<fssrv::SaveDataInfoReaderImpl> otherReaders;
                NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->OpenSaveDataInfoReader(&otherReaders));

                fssrv::SaveDataIndexerValue value;
                NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->Get(&value, keyA));

                fssrv::SaveDataIndexerKey keyB;
                NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->GetKey(&keyB, IdA));

                NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->GetValue(&value, IdA));
            }

            SaveDataInfo info;
            int64_t count;
            NNT_EXPECT_RESULT_SUCCESS(reader->Read(nn::sf::Out<int64_t>(&count), nn::sf::OutBuffer(reinterpret_cast<char*>(&info), sizeof(nn::fs::SaveDataInfo) * 1)));
        }

    }


}
#endif

namespace
{
    const SaveDataId SaveDataBaseId = 0x8000000000004000;
    const int TestSaveDataIndexerKeyCount = 3;

    void InitTestSaveDataIndexerKey(fssrv::SaveDataIndexerKey* key, int count)
    {
        for (int i = 0; i < count; i++)
        {
            key[i].staticSaveDataId = SaveDataBaseId + i;
            key[i].userId           = InvalidUserId;
        }
    }

    void CheckRead(const std::shared_ptr<fssrv::SaveDataInfoReaderImpl>& reader, const fssrv::SaveDataIndexerKey& key)
    {
        SaveDataInfo info;
        int64_t count;
        NNT_EXPECT_RESULT_SUCCESS(reader->Read(nn::sf::Out<int64_t>(&count), nn::sf::OutBuffer(reinterpret_cast<char*>(&info), sizeof(nn::fs::SaveDataInfo))));
        EXPECT_NE(0, count);
        EXPECT_EQ(key.staticSaveDataId, info.saveDataId);
    }

    void CheckReadEnd(const std::shared_ptr<fssrv::SaveDataInfoReaderImpl>& reader)
    {
        SaveDataInfo info;
        int64_t count;
        NNT_EXPECT_RESULT_SUCCESS(reader->Read(nn::sf::Out<int64_t>(&count), nn::sf::OutBuffer(reinterpret_cast<char*>(&info), sizeof(nn::fs::SaveDataInfo))));
        EXPECT_EQ(0, count);
    }

    class FixSaveDataInfoReaderTest : public ::testing::Test
    {
    protected:
        virtual void SetUp()
        {
            nn::fssrv::SaveDataIndexerAccessor accessor;
            accessor.Initialize(nn::fs::SaveDataSpaceId::System);
            m_Indexer = accessor.GetInterface();
            NNT_EXPECT_RESULT_SUCCESS(m_Indexer->Reset());
        }
        virtual void TearDown()
        {
        }

    public:
        Result OpenReader(std::shared_ptr<fssrv::SaveDataInfoReaderImpl>* outReader)
        {
            return m_Indexer->OpenSaveDataInfoReader(outReader);
        }
        Result Put(const fssrv::SaveDataIndexerKey& key)
        {
            return m_Indexer->PutStaticSaveDataIdIndex(key);
        }
        Result Delete(const fssrv::SaveDataIndexerKey& key)
        {
            return m_Indexer->Delete(key.staticSaveDataId);
        }

    private:
        fssrv::ISaveDataIndexer* m_Indexer;
    };
}

TEST_F(FixSaveDataInfoReaderTest, FixDeletion)
{
    fssrv::SaveDataIndexerKey key[TestSaveDataIndexerKeyCount] = {};
    InitTestSaveDataIndexerKey(key, TestSaveDataIndexerKeyCount);

    NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
    NNT_EXPECT_RESULT_SUCCESS(Put(key[1]));
    NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));

    /* オープン直後に削除 */
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[0]));
        CheckRead(reader, key[1]);
        CheckRead(reader, key[2]);
        CheckReadEnd(reader);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
    }
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[1]));
        CheckRead(reader, key[0]);
        CheckRead(reader, key[2]);
        CheckReadEnd(reader);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[1]));
    }
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[2]));
        CheckRead(reader, key[0]);
        CheckRead(reader, key[1]);
        CheckReadEnd(reader);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));
    }

    /* 走査途中に削除 */
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[0]));
        CheckRead(reader, key[1]);
        CheckRead(reader, key[2]);
        CheckReadEnd(reader);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
    }
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[1]));
        CheckRead(reader, key[2]);
        CheckReadEnd(reader);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[1]));
    }
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[2]));
        CheckRead(reader, key[1]);
        CheckReadEnd(reader);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));
    }

    /* 走査完了後に削除 */
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);
        CheckRead(reader, key[1]);
        CheckRead(reader, key[2]);
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[0]));
        CheckReadEnd(reader);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
    }
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);
        CheckRead(reader, key[1]);
        CheckRead(reader, key[2]);
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[1]));
        CheckReadEnd(reader);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[1]));
    }
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);
        CheckRead(reader, key[1]);
        CheckRead(reader, key[2]);
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[2]));
        CheckReadEnd(reader);
    }
}

TEST_F(FixSaveDataInfoReaderTest, FixInsertion)
{
    fssrv::SaveDataIndexerKey key[TestSaveDataIndexerKeyCount] = {};
    InitTestSaveDataIndexerKey(key, TestSaveDataIndexerKeyCount);

    NNT_EXPECT_RESULT_SUCCESS(Put(key[1]));
    NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));

    /* オープン直後に挿入 */
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
        CheckRead(reader, key[0]);
        CheckRead(reader, key[1]);
        CheckRead(reader, key[2]);
        CheckReadEnd(reader);
    }
    {
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[1]));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        NNT_EXPECT_RESULT_SUCCESS(Put(key[1]));
        CheckRead(reader, key[0]);
        CheckRead(reader, key[1]);
        CheckRead(reader, key[2]);
        CheckReadEnd(reader);
    }
    {
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[2]));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));
        CheckRead(reader, key[0]);
        CheckRead(reader, key[1]);
        CheckRead(reader, key[2]);
        CheckReadEnd(reader);
    }

    /* 走査途中に挿入 */
    {
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[0]));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[1]);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
        CheckRead(reader, key[2]);
        CheckReadEnd(reader);
    }
    {
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[1]));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[1]));
        CheckRead(reader, key[1]);
        CheckRead(reader, key[2]);
        CheckReadEnd(reader);
    }
    {
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[2]));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));
        CheckRead(reader, key[1]);
        CheckRead(reader, key[2]);
        CheckReadEnd(reader);
    }

    /* 走査完了後に挿入 */
    {
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[0]));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[1]);
        CheckRead(reader, key[2]);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
        CheckReadEnd(reader);
    }
    {
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[1]));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);
        CheckRead(reader, key[2]);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[1]));
        CheckReadEnd(reader);
    }
    {
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[2]));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);
        CheckRead(reader, key[1]);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));
        CheckReadEnd(reader);
    }
}

TEST_F(FixSaveDataInfoReaderTest, FixMultipleReader)
{
    fssrv::SaveDataIndexerKey key[TestSaveDataIndexerKeyCount] = {};
    InitTestSaveDataIndexerKey(key, TestSaveDataIndexerKeyCount);

    NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
    NNT_EXPECT_RESULT_SUCCESS(Put(key[1]));
    NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));

    /* 複数オープン中に削除 */
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader1;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader1));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader2;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader2));

        CheckRead(reader1, key[0]);
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[0]));
        CheckRead(reader1, key[1]);
        CheckRead(reader1, key[2]);
        CheckReadEnd(reader1);

        CheckRead(reader2, key[1]);
        CheckRead(reader2, key[2]);
        CheckReadEnd(reader2);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
    }
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader1;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader1));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader2;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader2));

        CheckRead(reader1, key[0]);
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[2]));
        CheckRead(reader1, key[1]);
        CheckReadEnd(reader1);

        CheckRead(reader2, key[0]);
        CheckRead(reader2, key[1]);
        CheckReadEnd(reader2);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));
    }

    /* 複数オープン中に挿入 */
    {
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[0]));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader1;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader1));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader2;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader2));

        CheckRead(reader1, key[1]);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
        CheckRead(reader1, key[2]);
        CheckReadEnd(reader1);

        CheckRead(reader2, key[0]);
        CheckRead(reader2, key[1]);
        CheckRead(reader2, key[2]);
        CheckReadEnd(reader2);
    }
    {
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[2]));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader1;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader1));
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader2;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader2));

        CheckRead(reader1, key[0]);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));
        CheckRead(reader1, key[1]);
        CheckRead(reader1, key[2]);
        CheckReadEnd(reader1);

        CheckRead(reader2, key[0]);
        CheckRead(reader2, key[1]);
        CheckRead(reader2, key[2]);
        CheckReadEnd(reader2);
    }
}

TEST_F(FixSaveDataInfoReaderTest, ExpiredReader)
{
    fssrv::SaveDataIndexerKey key[TestSaveDataIndexerKeyCount] = {};
    InitTestSaveDataIndexerKey(key, TestSaveDataIndexerKeyCount);

    NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
    NNT_EXPECT_RESULT_SUCCESS(Put(key[1]));
    NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));

    /* 途中で寿命を迎えたReaderがあっても問題ないことの確認 */
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader1;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader1));
        CheckRead(reader1, key[0]);
        {
            std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader2;
            NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader2));

            CheckRead(reader2, key[0]);
            NNT_EXPECT_RESULT_SUCCESS(Delete(key[0]));
            CheckRead(reader2, key[1]);
            CheckRead(reader2, key[2]);
            CheckReadEnd(reader2);
        }
        CheckRead(reader1, key[1]);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
        CheckRead(reader1, key[2]);
        CheckReadEnd(reader1);
    }
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader1;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader1));
        CheckRead(reader1, key[0]);
        {
            std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader2;
            NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader2));
            CheckRead(reader2, key[0]);
            NNT_EXPECT_RESULT_SUCCESS(Delete(key[2]));
            CheckRead(reader2, key[1]);
            CheckReadEnd(reader2);
        }
        NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));
        CheckRead(reader1, key[1]);
        CheckRead(reader1, key[2]);
        CheckReadEnd(reader1);
    }
}

TEST_F(FixSaveDataInfoReaderTest, ReuseReader)
{
    fssrv::SaveDataIndexerKey key[TestSaveDataIndexerKeyCount] = {};
    InitTestSaveDataIndexerKey(key, TestSaveDataIndexerKeyCount);

    NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
    NNT_EXPECT_RESULT_SUCCESS(Put(key[1]));
    NNT_EXPECT_RESULT_SUCCESS(Put(key[2]));

    /* 列挙途中で再オープンしたReaderでも問題なく列挙できることの確認 */
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);
        CheckRead(reader, key[1]);

        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));

        CheckRead(reader, key[0]);

        NNT_EXPECT_RESULT_SUCCESS(Delete(key[0]));

        CheckRead(reader, key[1]);
        NNT_EXPECT_RESULT_SUCCESS(Put(key[0]));
        CheckRead(reader, key[2]);
        CheckReadEnd(reader);
    }
    {
        std::shared_ptr<fssrv::SaveDataInfoReaderImpl> reader;
        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);

        NNT_EXPECT_RESULT_SUCCESS(OpenReader(&reader));
        CheckRead(reader, key[0]);
        NNT_EXPECT_RESULT_SUCCESS(Delete(key[2]));
        CheckRead(reader, key[1]);
        CheckReadEnd(reader);
    }
}

TEST(SaveDataIndexer, ShowAllHeavy)
{
    nn::fssrv::SaveDataIndexerAccessor accessor;
    accessor.Initialize(nn::fs::SaveDataSpaceId::System);
    auto saveDataIndexer = accessor.GetInterface();
    std::shared_ptr<nn::fssrv::SaveDataInfoReaderImpl> reader;
    NNT_EXPECT_RESULT_SUCCESS(saveDataIndexer->OpenSaveDataInfoReader(&reader));

    SaveDataInfo info;
    int64_t totalCount = 0;
    while (NN_STATIC_CONDITION(true))
    {
        int64_t count;
        NNT_EXPECT_RESULT_SUCCESS(reader->Read(nn::sf::Out<int64_t>(&count), nn::sf::OutBuffer(reinterpret_cast<char*>(&info), sizeof(nn::fs::SaveDataInfo) * 1)));
        if (count == 0)
        {
            break;
        }

        totalCount++;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));

        NN_LOG("---- entry #%d ----\n", totalCount);
        NN_LOG("saveDataId: 0x%016llx\n", info.saveDataId);
        NN_LOG("saveDataSpaceId: %s\n",
               info.saveDataSpaceId == SaveDataSpaceId::System ? "System" :
               info.saveDataSpaceId == SaveDataSpaceId::User ? "User" : "Unknown" );
        NN_LOG("saveDataType: %s\n",
               info.saveDataType == SaveDataType::System ? "System" :
               info.saveDataType == SaveDataType::Bcat ? "Bcat" :
               info.saveDataType == SaveDataType::Device ? "Device" :
               info.saveDataType == SaveDataType::Account ? "Account" : "Unknown" );
        NN_LOG("saveDataUserId: 0x%016llx_%016llx\n", info.saveDataUserId._data[0], info.saveDataUserId._data[1]);
        NN_LOG("systemSaveDataId: 0x%016llx\n", info.systemSaveDataId);
        NN_LOG("applicationId: 0x%016llx\n", info.applicationId.value);
        NN_LOG("saveDataSize: %d\n", info.saveDataSize);
        NN_LOG("\n");
    }
}

namespace {

    class TestMemoryResource : public nn::MemoryResource
    {
    protected:
        virtual void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(alignment);
            return std::malloc(bytes);
        }
        virtual void do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(bytes);
            NN_UNUSED(alignment);
            return std::free(p);
        }
        virtual bool do_is_equal(const nn::MemoryResource& other) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(other);
            return false;
        }
    };

}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    InitializeFileSystemLibrary();
    nn::fs::SetEnabledAutoAbort(false);
    nn::fssystem::InitializeAllocator(nn::fs::detail::Allocate, nn::fs::detail::Deallocate);

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    class SdHandleManager : public nn::fssrv::IDeviceHandleManager
    {
    public:
        virtual nn::Result GetHandle(nn::fssrv::DeviceHandle* pOutValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *pOutValue = 0;
            NN_RESULT_SUCCESS;
        }
        virtual bool IsValid(nn::fssrv::DeviceHandle handle) NN_NOEXCEPT NN_OVERRIDE
        {
            return true;
        }
    };
    SdHandleManager sdHandleManager;

    SystemSaveDataId TestSaveDataIndexerId = 0x800000000000000fULL;
    TestMemoryResource memoryResource;
    nn::fssrv::InitializeSaveDataIndexerManager(TestSaveDataIndexerId, &memoryResource, &sdHandleManager, false);
#endif

    SetEnabledAutoAbort(false);

    auto ret = RUN_ALL_TESTS();

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    DeleteSaveData(TestSaveDataIndexerId);
#endif

    nnt::Exit(ret);
}
