﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include "testFs_util_CommonStorageTests.h"

namespace {

int VerifyBuffer(const char* buf1, const char* buf2, size_t size)
{
    int result = 0;
    for( size_t i = 0; i < size; i++ )
    {
        if( buf1[i] != buf2[i] )
        {
            NN_SDK_LOG(
                "%04d, %02X <=> %02X\n",
                i,
                static_cast<uint8_t>(buf1[i]),
                static_cast<uint8_t>(buf2[i])
            );
            result = -1;
        }
    }
    return result;
}

}

// 4 GB を超えるオフセットでストレージにアクセスするテスト（読み込みを独自関数で行う）
void TestLargeOffsetAccess(
    nn::fs::IStorage* pStorage,
    size_t bufferSize,
    std::function<void(int64_t, char*, const char*, size_t)> readTestFunc) NN_NOEXCEPT
{
    static const int64_t LargeOffsetList[] = {
        0,
        static_cast<int64_t>(4) * 1024 * 1024 * 1024,
        static_cast<int64_t>(8) * 1024 * 1024 * 1024,
        static_cast<int64_t>(16) * 1024 * 1024 * 1024,
        static_cast<int64_t>(32) * 1024 * 1024 * 1024,
        static_cast<int64_t>(64) * 1024 * 1024 * 1024,
    };

    static const size_t LargeOffsetListLength = sizeof(LargeOffsetList) / sizeof(int64_t);

    static const int64_t LargeOffsetMax = LargeOffsetList[LargeOffsetListLength - 1];

    int64_t storageSize = 0;
    const auto result = pStorage->GetSize(&storageSize);
    if( !nn::fs::ResultNotImplemented::Includes(result) )
    {
        NNT_ASSERT_RESULT_SUCCESS(result);
        ASSERT_GE(storageSize, LargeOffsetMax + static_cast<int64_t>(bufferSize));
    }

    // バッファ初期化
    std::unique_ptr<char> readBuffer(new char[bufferSize]);
    std::unique_ptr<char> writeBufferList[LargeOffsetListLength];
    for( auto& writeBuffer : writeBufferList )
    {
        writeBuffer.reset(new char[bufferSize]);
        nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), bufferSize);
    }

    // 4 GB を超えるオフセットへ書き込み
    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(LargeOffsetList[i], writeBufferList[i].get(), bufferSize));
    }
    NNT_ASSERT_RESULT_SUCCESS(pStorage->Flush());

    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        readTestFunc(LargeOffsetList[i], readBuffer.get(), writeBufferList[i].get(), bufferSize);
    }
}

// 4 GB を超えるオフセットでストレージにアクセスする共通テスト
void TestLargeOffsetAccess(nn::fs::IStorage* pStorage, size_t bufferSize) NN_NOEXCEPT
{
    TestLargeOffsetAccess(
        pStorage,
        bufferSize,
        [=](int64_t offset, char* readBuffer, const char* writeBuffer, size_t bufferSize) NN_NOEXCEPT
        {
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(offset, readBuffer, bufferSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, bufferSize);
        });
}

// 0 バイトのストレージに対するテストを行います。
void StorageTest::TestSizeZero(nn::fs::IStorage* pStorage, bool isAutoExtendSizeOnWrite, bool isReadOnly) NN_NOEXCEPT
{
    char buf[16];

    int64_t sizeStorage;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
    ASSERT_EQ(0, sizeStorage);

    if( !isReadOnly )
    {
        // 0 バイトのストレージに 0 バイトの書き込みます。
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(0, buf, 0));

        // 範囲外の書き込み
        nn::Result result = pStorage->Write(1, buf, 1);
        if( isAutoExtendSizeOnWrite )
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
        }
    }

    // 0 バイトのストレージから 0 バイトの読み込みます。
    NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(0, buf, 0));

    // 範囲外の読み込み
    nn::Result result = pStorage->Read(3, buf, 1);
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
}

// 読み込み可能なストレージに対するリード、ベリファイを行うテストです。
void StorageTest::TestReadVerify(nn::fs::IStorage* pStorage) NN_NOEXCEPT
{
    nnt::fs::util::Vector<char> buf(TestSize);
    char* ptr = &buf[0];

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    int64_t sizeOriginal;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeOriginal));
    int64_t sizeData = sizeOriginal;

    // ランダムなオフセットに対して読み込みを行います。
    for( int32_t i = 0; i < 10; i++ )
    {
        int64_t offset =
            std::uniform_int_distribution<int64_t>(0, sizeData - 1)(mt);
        size_t size =
            std::uniform_int_distribution<size_t>(
                0, static_cast<size_t>(sizeData - offset) - 1)(mt);
        if( offset + static_cast<int64_t>(size) > sizeData )
        {
            size = static_cast<size_t>(sizeData - offset);
        }
        ASSERT_TRUE(size <= TestSize);

        // データを読み込んでベリファイします。
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(offset, ptr, size));
    }
}

// 読み書き可能なストレージに対する読み書き時の範囲チェック、リサイズテストを行います。
void StorageTest::TestBounds(
                      nn::fs::IStorage* pStorage,
                      bool isEnableResize,
                      bool isAutoExtendSizeOnWrite
                  ) NN_NOEXCEPT
{
    nnt::fs::util::Vector<char> writeBuf(TestSize);
    char* pWriteBuf = &writeBuf[0];
    nnt::fs::util::Vector<char> readBuf(TestSize);
    char* pReadBuf = &readBuf[0];

    int64_t sizeStorage;
    int64_t sizeStorageOriginal;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorageOriginal));
    sizeStorage = sizeStorageOriginal;

    // サイズの取得と設定
    if( isEnableResize )
    {
        NNT_ASSERT_RESULT_SUCCESS(pStorage->SetSize(sizeStorageOriginal / 2));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
        ASSERT_EQ(sizeStorageOriginal / 2, sizeStorage);
        NNT_ASSERT_RESULT_SUCCESS(pStorage->SetSize(sizeStorageOriginal));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->SetSize(sizeStorageOriginal + 1));
        //NN_TEST_ASSERT_RESULT_EQUAL(nn::fs::ResultOutOfMemory(), pStorage->SetSize(32*1024));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
        ASSERT_EQ(sizeStorageOriginal + 1, sizeStorage);

        // サイズ設定を再度する
        NNT_ASSERT_RESULT_SUCCESS(pStorage->SetSize(sizeStorageOriginal));

        NNT_ASSERT_RESULT_SUCCESS(pStorage->SetSize(0));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
        ASSERT_EQ(0, sizeStorage);

        NNT_ASSERT_RESULT_SUCCESS(pStorage->SetSize(sizeStorageOriginal));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
        ASSERT_EQ(sizeStorageOriginal, sizeStorage);
    }

    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));

#if 0
    // サイズが変更できないことを検証します。
    if( !isEnableResize )
    {
        // アボートします
        nn::Result result = pStorage->SetSize(16);
        ASSERT_TRUE(nn::fs::ResultPermissionDenied::Includes(result) ||
                    nn::fs::ResultUnsupportedOperation::Includes(result));
        result = pStorage->SetSize(0);
        ASSERT_TRUE(nn::fs::ResultPermissionDenied::Includes(result) ||
                    nn::fs::ResultUnsupportedOperation::Includes(result));
        result = pStorage->SetSize(sizeStorage);
        ASSERT_TRUE(nn::fs::ResultPermissionDenied::Includes(result) ||
                    nn::fs::ResultUnsupportedOperation::Includes(result));
    }
#endif

    // 範囲内アクセス
    {
        // 先頭 1 バイトを書き込む
        if( sizeStorage > 0 )
        {
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(0, pWriteBuf, 1));
        }
        // オフセット 1 で末尾まで書き込む
        if( sizeStorage > 1 )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                pStorage->Write(1, pWriteBuf, static_cast<size_t>(sizeStorage - 1)));
            NNT_ASSERT_RESULT_SUCCESS(
                pStorage->Read(1, pReadBuf, static_cast<size_t>(sizeStorage - 1)));
        }
        // 末尾 2 バイトを書き込む
        if( sizeStorage > 2 )
        {
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(sizeStorage - 2, pWriteBuf, 2));
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(sizeStorage - 2, pReadBuf, 2));
        }
    }

    // 範囲を超えるアクセス
    if( isAutoExtendSizeOnWrite )
    {
        // 自動拡張
        if( sizeStorage > 2 )
        {
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(sizeStorage - 2, pReadBuf, 4));
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(sizeStorage - 2, pWriteBuf, 4));
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(sizeStorage - 2, pReadBuf, 4));
            NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
        }
        NNT_ASSERT_RESULT_SUCCESS(
            pStorage->Read(0, pReadBuf, static_cast<size_t>(sizeStorage)));
        NNT_ASSERT_RESULT_SUCCESS(
            pStorage->Write(0, pWriteBuf, static_cast<size_t>(sizeStorage + 2)));
        NNT_ASSERT_RESULT_SUCCESS(
            pStorage->Read(0, pReadBuf, static_cast<size_t>(sizeStorage + 2)));
    }
    else
    {
        // 自動拡張なし
        if( sizeStorage > 2 )
        {
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(sizeStorage - 2, pReadBuf, 4));
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(sizeStorage - 2, pWriteBuf, 4));
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(sizeStorage - 2, pReadBuf, 4));
        }
        NNT_ASSERT_RESULT_SUCCESS(
            pStorage->Read(0, pReadBuf, static_cast<size_t>(sizeStorage + 2)));
        NNT_ASSERT_RESULT_SUCCESS(
            pStorage->Write(0, pWriteBuf, static_cast<size_t>(sizeStorage + 2 + 12)));
        NNT_ASSERT_RESULT_SUCCESS(
            pStorage->Read(0, pReadBuf, static_cast<size_t>(sizeStorage + 2 + 12)));
    }

    nn::Result result;
    // 完全に範囲外のアクセス
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
    if( isAutoExtendSizeOnWrite )
    {
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(sizeStorage + 1, pWriteBuf, 1));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(sizeStorage + 2, pWriteBuf, 1));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(sizeStorage + 1, pReadBuf, 1));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(sizeStorage + 2, pReadBuf, 1));
    }
    else
    {
        result = pStorage->Write(sizeStorage + 1, pWriteBuf, 1);
        ASSERT_TRUE(nn::fs::ResultInvalidOffset::Includes(result) ||
                    nn::fs::ResultInvalidSize::Includes(result));
        result = pStorage->Write(sizeStorage + 2, pWriteBuf, 1);
        ASSERT_TRUE(nn::fs::ResultInvalidOffset::Includes(result) ||
                    nn::fs::ResultInvalidSize::Includes(result));
        result = pStorage->Read(sizeStorage + 1, pReadBuf, 1);
        ASSERT_TRUE(nn::fs::ResultInvalidOffset::Includes(result) ||
                    nn::fs::ResultInvalidSize::Includes(result));
        result = pStorage->Read(sizeStorage + 2, pReadBuf, 1);
        ASSERT_TRUE(nn::fs::ResultInvalidOffset::Includes(result) ||
                    nn::fs::ResultInvalidSize::Includes(result));
    }

    // サイズの取得と設定
    if( isEnableResize )
    {
        NNT_ASSERT_RESULT_SUCCESS(pStorage->SetSize(sizeStorageOriginal));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
        ASSERT_EQ(sizeStorageOriginal, sizeStorage);
    }

    // 自動伸長しつつ、データを書き込みます。
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
    if( isAutoExtendSizeOnWrite )
    {
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(4 + sizeStorageOriginal, pWriteBuf, 8));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
        ASSERT_EQ(12 + sizeStorageOriginal, sizeStorage);
        NNT_ASSERT_RESULT_SUCCESS(pStorage->SetSize(16 + sizeStorageOriginal));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
        ASSERT_EQ(16 + sizeStorageOriginal, sizeStorage);
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(0 + sizeStorageOriginal, pWriteBuf, 16));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(8 + sizeStorageOriginal, pWriteBuf, 8));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(8 + sizeStorageOriginal, pWriteBuf + 8, 16));
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
        ASSERT_EQ(24 + sizeStorageOriginal, sizeStorage);
    }
} // NOLINT(impl/function_size)

// 読み書き可能なストレージに対する1バイト単位のリード、ベリファイを行うテストです。
void StorageTest::TestReadWrite1(
                      nn::fs::IStorage* pStorage,
                      bool isAutoExtendSizeOnWrite
                  ) NN_NOEXCEPT
{
    nnt::fs::util::Vector<char> buf1(TestSize);
    nnt::fs::util::Vector<char> buf2(TestSize);
    nnt::fs::util::Vector<char> buf3(TestSize);

    char* ptr1 = &buf1[0];
    char* ptr2 = &buf2[0];
    char* ptr3 = &buf3[0];

    int64_t sizeStorage;
    int64_t sizeStorageOrg;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorageOrg));
    sizeStorage = sizeStorageOrg;

    // テスト幅は、TestSizeに抑えます
    int64_t sizeMax = sizeStorage;
    if( sizeMax > TestSize )
    {
        sizeMax = TestSize;
    }

    std::memset(ptr1, 0, TestSize);
    std::memset(ptr2, 0xFF, TestSize);
    std::memset(ptr3, 0, TestSize);
    NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(0, ptr2, static_cast<size_t>(sizeMax)));
    NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(0, ptr3, TestSize));

    // 自動伸長、範囲外アクセスの確認
    if( isAutoExtendSizeOnWrite )
    {
        // 拡張を許すストレージの場合、サイズが増えます
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(sizeStorage, ptr1, 1));
        ++sizeMax;
        ++sizeStorage;

        int64_t sizeNewStorage = 0;
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeNewStorage));
        ASSERT_EQ(sizeNewStorage, sizeStorage);

        // 末尾を越えて書き込む場合でもサイズが増えます
        nn::Result result = pStorage->Write(sizeStorage + 1, ptr1, 1);
        NNT_ASSERT_RESULT_SUCCESS(result);
        sizeMax += 2;
        sizeStorage += 2;

        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeNewStorage));
        ASSERT_EQ(sizeNewStorage, sizeStorage);

    }
    else
    {
        // 拡張を許さないストレージの場合、サイズは増えません
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(sizeStorage, ptr1, 1));

        int64_t sizeNewStorage;
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeNewStorage));
        ASSERT_EQ(sizeNewStorage, sizeStorage);

        // 末尾を越えて書き込むのは禁止です
        nn::Result result = pStorage->Write(sizeStorage + 1, ptr1, 1);
        ASSERT_TRUE(nn::fs::ResultInvalidOffset::Includes(result) ||
            nn::fs::ResultInvalidSize::Includes(result));
    }

    // データ比較
    NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(0, ptr2, TestSize));
    ASSERT_EQ(0, VerifyBuffer(ptr3, ptr2, static_cast<size_t>(sizeStorageOrg)));

    if( sizeMax < TestSize )
    {
        ASSERT_EQ(0xFF, static_cast<uint8_t>(ptr2[sizeMax]));
        if( isAutoExtendSizeOnWrite )
        {
            // 拡張時に書き換えた部分の中身をチェックする
            ASSERT_EQ(0, static_cast<uint8_t>(ptr2[sizeStorageOrg]));
            ASSERT_EQ(0, static_cast<uint8_t>(ptr2[sizeStorageOrg + 2]));
        }
    }

#if 0
    char* pBaseBuf = &baseBuf[0];
    char* pFillBuf = &fillBuf[0];
    NN_TEST_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&size));

    // 書き込まれているデータを生成します
    for( n = 0; n < size; n++ )
    {
        pFillBuf[n] = (m + n) & 0xFF;
    }

    // ストレージを後ろから読み取ります。
    for( n = 0; n < size; n++ )
    {
        NN_TEST_ASSERT_RESULT_SUCCESS(
            pStorage[m]->Read(size - n - 1, &pBaseBuf[size - n - 1], 1));
    }
    NN_TEST_ASSERT_EQUAL(0, std::memcmp(pBaseBuf, pFillBuf, size));

    // ストレージを前から読み取ります。
    for( n = 0; n < size; n++ )
    {
        NN_TEST_ASSERT_RESULT_SUCCESS(pStorage->Read(n, &pBaseBuf[n], 1));
    }
    NN_TEST_ASSERT_EQUAL(0, std::memcmp(pBaseBuf, pFillBuf, size));

    // ストレージを一括で読み込みます。
    NN_TEST_ASSERT_RESULT_SUCCESS(pStorage->Read(0, pBaseBuf, size));
    NN_TEST_ASSERT_EQUAL(0, std::memcmp(pBaseBuf, pFillBuf, size));
#endif

    // 自動伸長、範囲外アクセスの確認
    if( isAutoExtendSizeOnWrite )
    {
        NNT_ASSERT_RESULT_SUCCESS(pStorage->SetSize(sizeStorageOrg));
    }
}

// 読み書き可能なストレージに対するランダム読み書き、ベリファイテストを行います。
void StorageTest::TestReadWrite2(nn::fs::IStorage* pStorage) NN_NOEXCEPT
{
    static const int Skip = 263;

    nnt::fs::util::Vector<char> writeBuf(TestSize);
    char* pWriteBuf = &writeBuf[0];
    nnt::fs::util::Vector<char> readBuf(TestSize);
    char* pReadBuf = &readBuf[0];
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // ストレージのサイズを求めます
    int64_t sizeStorage;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));

    // バイト単位で書き込みます
    for( int64_t i = 0; i < sizeStorage; i += Skip )
    {
        // テストデータを書き込みます
        const char ch = (i / Skip) & 0xFF;
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(i, &ch, 1));
    }

    // バイト単位で読み込みます
    for( int32_t i = 0; i < sizeStorage; i += Skip )
    {
        char ch1;
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(i, &ch1, 1));

        const char ch2 = (i / Skip) & 0xFF;
        ASSERT_EQ(ch1, ch2);
    }

    // ランダムなオフセット、サイズで読み書きを行います
    for( int32_t i = 0; i < 10; ++i )
    {
        // 書き込みオフセットとサイズを決定します
        int64_t offset =
            std::uniform_int_distribution<int64_t>(0, sizeStorage - 1)(mt);
        size_t size =
            std::uniform_int_distribution<size_t>(
                0, static_cast<size_t>(sizeStorage - offset) - 1)(mt);

        for( size_t j = 0; j < size; j++ )
        {
            pWriteBuf[j] = (i + j) & 0xFF;
        }

        // データを書き込みます。
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(offset, pWriteBuf, size));

        if( std::uniform_int_distribution<>(0, 7)(mt) == 0 )
        {
            // データを読み込んでベリファイします。
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(offset, pReadBuf, size));
            ASSERT_EQ(0, VerifyBuffer(pReadBuf, pWriteBuf, size));
        }
    }
}

// テストデータを書き込んでおきます。
void StorageTest::TestWriteTestData(
                      int64_t* pSizeStorage,
                      nn::fs::IStorage* pStorage
                  ) NN_NOEXCEPT
{
    // ストレージのサイズを求めます
    int64_t sizeStorage;
    NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage));
    *pSizeStorage = sizeStorage;

    if( sizeStorage >= 8 )
    {
        // ストレージの前後に特定のデータを書き込んで起きます
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(0x00000000, &sizeStorage, 8));

        if( sizeStorage >= 16 )
        {
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(sizeStorage - 8, "abcdefgh", 8));
        }
    }
}

// テストデータが適切に書き込まれているかを比較します。
void StorageTest::TestVerifyTestData(
                      nn::fs::IStorage* pStorage,
                      int64_t sizeStorage
                  ) NN_NOEXCEPT
{
    int64_t sizeStorageRead;
    char buf[8];

    {
        // ストレージのサイズを求めます。
        int64_t sizeStorage0;
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&sizeStorage0));
        ASSERT_EQ(sizeStorage, sizeStorage0);
    }

    if( sizeStorage >= 8 )
    {
        // ストレージの前後に特定のデータが書き込まれていることを確認します。
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(0x00000000, &sizeStorageRead, 8));
        ASSERT_EQ(sizeStorageRead, sizeStorage);

        if( sizeStorage >= 16 )
        {
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(sizeStorage - 8, buf, 8));
            ASSERT_EQ(0, std::memcmp(buf, "abcdefgh", 8));
        }
    }
}
