﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <limits>
#include <random>
#include <algorithm>
#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_IStorage.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonStorageTests.h"

/*!
@brief アクセスカウント付きラッパーストレージオブジェクトです。
*/
template<typename TBase>
class AccessCountWrapperStorage : public TBase
{
    typedef TBase Base;      //!< 基底クラス

public:
    //! アクセスカウント構造体
    struct AccessCountData
    {
        int32_t     readTimes;     //!< 読み込み回数
        int32_t     writeTimes;    //!< 書き込み回数
        int32_t     flushTimes;    //!< フラッシュ回数
    };

public:
    AccessCountWrapperStorage() NN_NOEXCEPT
        : m_IsRandomError(false),
          m_IsLatencyEmulated(false),
          m_bVerbose(false)
    {
    }

    /*!
    @brief 下位ストレージの offset から size バイト読み込み buffer にコピーします。

    @param[in] offset 読み込み開始位置
    @param[in,out] buffer 読み込んだ内容をコピーするバッファ
    @param[in] size 読み込むデータサイズ

    @return 関数の処理結果を返します。
    */
    virtual nn::Result Read(
                           int64_t offset,
                           void* buffer,
                           size_t size
                       ) NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_IsRandomError)
        {
            if (m_Random.Get<>(5000) < 10)
            {
                if (m_bVerbose)
                {
                    NN_SDK_LOG("\n!READ ERROR!\n");
                }
                return GetRandomReadErrorResult();
            }
        }

        if( m_IsLatencyEmulated )
        {
            nn::os::SleepThread(m_Latency);
        }

        if (size && m_bVerbose)
        {
            NN_SDK_LOG("Read ofs:%6llX sz:%6d\n", offset, size );
        }

        // 読み込み回数カウントアップ
        ++m_AccessCountData.readTimes;

        return Base::Read(offset, buffer, size);
    }

    /*!
    @brief 下位ストレージの offset 以降に buffer を size バイト分コピーします。

    @param[in] offset 書き込み開始位置
    @param[in,out] buffer 書き込むデータ
    @param[in] size 書き込むデータサイズ

    @return 関数の処理結果を返します。
    */
    virtual nn::Result Write(
                           int64_t offset,
                           const void* buffer,
                           size_t size
                       ) NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_IsRandomError)
        {
            if (m_Random.Get<>(5000) < 10)
            {
                if (m_bVerbose)
                {
                    NN_SDK_LOG("\n!WRITE ERROR!\n");
                }
                return GetRandomWriteErrorResult();
            }
        }

        if( m_IsLatencyEmulated )
        {
            nn::os::SleepThread(m_Latency);
        }

        if (size > 0)
        {
            // 書き込み回数カウントアップ
            ++m_AccessCountData.writeTimes;
        }

        if (size && m_bVerbose)
        {
            NN_SDK_LOG("Writ ofs:%6llX sz:%6d\n", offset, size );
        }

        return Base::Write(offset, buffer, size);
    }

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        // フラッシュ指定書き込み回数カウントアップ
        ++m_AccessCountData.flushTimes;
        return Base::Flush();
    }

    /*!
    @brief アクセスカウントデータを取得します。

    @param[out] accessCountData アクセスカウントデータ

    @return 関数の処理結果を返します。
    */
    nn::Result GetAccessCounter(AccessCountData* accessCountData) const NN_NOEXCEPT
    {
        if (accessCountData != nullptr)
        {
            *accessCountData = m_AccessCountData;
        }
        NN_RESULT_SUCCESS;
    }

    /*!
    @brief アクセスカウントデータをリセットします。
    */
    void ResetAccessCounter() NN_NOEXCEPT
    {
        std::memset(&m_AccessCountData, 0, sizeof(m_AccessCountData));
    }

    /*!
    @brief 現在の読み込み回数を取得します。

    @return 読み込み回数を返します。
    */
    int32_t GetReadTimes() const NN_NOEXCEPT
    {
        return m_AccessCountData.readTimes;
    }

    /*!
    @brief 現在の書き込み回数を取得します。

    @return 書き込み回数を返します。
    */
    int32_t GetWriteTimes() const NN_NOEXCEPT
    {
        return m_AccessCountData.writeTimes;
    }

    /*!
    @brief 現在のフラッシュ回数を取得します。

    @return フラッシュ回数を返します。
    */
    int32_t GetFlushTimes() const NN_NOEXCEPT
    {
        return m_AccessCountData.flushTimes;
    }

    void SetRandomError(bool isEnable) NN_NOEXCEPT
    {
        m_IsRandomError = isEnable;
    }

    static nn::Result GetRandomReadErrorResult() NN_NOEXCEPT
    {
        // SaveDataFileSystem では返しそうもないエラーを返しときます。
        return nn::fs::ResultSdCardAccessFailed();
    }

    static nn::Result GetRandomWriteErrorResult() NN_NOEXCEPT
    {
        // SaveDataFileSystem では返しそうもないエラーを返しときます。
        return nn::fs::ResultSdCardAccessFailed();
    }

    void SetLatency(const nn::TimeSpan& latency) NN_NOEXCEPT
    {
        m_Latency = latency;
        m_IsLatencyEmulated = true;
    }

    void SetVerbose(bool bVerbose) NN_NOEXCEPT
    {
        m_bVerbose = bVerbose;
    }

private:
    AccessCountData m_AccessCountData;      //!< アクセスカウントデータ
    bool m_IsRandomError;                   //!< ランダムなタイミングでエラーを発生させるかどうか
    bool m_IsLatencyEmulated;               //!< 読み書きを遅延させるかどうか
    bool m_bVerbose;
    char reserved[1];
    nn::TimeSpan m_Latency;
    Random m_Random;
};

/*!
@brief アクセスカウント付きメモリファイルオブジェクトです。
*/
typedef AccessCountWrapperStorage<nnt::fs::util::SafeMemoryStorage> AccessCountedMemoryStorage;
