﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <random>
#include <numeric>
#include <thread>
#include <algorithm>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fsa/fs_IFile.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/save/fs_UnionStorage.h>

namespace nn { namespace fssystem { namespace save {

using namespace nnt::fs::util;

class UnionStorageTest : public ::testing::Test
{
protected:
    static const auto BlockSize = 16 * 1024;
    static const auto OriginalStorageSize = 8 * 1024 * 1024;
    static const auto LogStorageSize = OriginalStorageSize * 2 + BlockSize;

protected:
    UnionStorageTest() NN_NOEXCEPT
    : m_BaseStorage(),
      m_OriginalStorage(),
      m_LogStorage(),
      m_UnionStorage()
    {
        m_BaseStorage.Initialize(OriginalStorageSize + LogStorageSize);
        m_BaseStorage.OperateRange(nn::fs::OperationId::FillZero, 0, m_BaseStorage.GetSize());
        m_OriginalStorage = fs::SubStorage(&m_BaseStorage, 0, OriginalStorageSize);
        m_LogStorage = fs::SubStorage(&m_BaseStorage, OriginalStorageSize, LogStorageSize);
        UnionStorage::Format(m_LogStorage, BlockSize);
        m_UnionStorage.Initialize(m_OriginalStorage, m_LogStorage, BlockSize);
    }

    UnionStorage& GetUnionStorage() NN_NOEXCEPT
    {
        return m_UnionStorage;
    }

    fs::IStorage& GetOriginalStorage() NN_NOEXCEPT
    {
        return m_OriginalStorage;
    }

    fs::IStorage& GetLogStorage() NN_NOEXCEPT
    {
        return m_LogStorage;
    }

private:
    nnt::fs::util::Vector<char> m_StorageBuffer;
    SafeMemoryStorage m_BaseStorage;
    fs::SubStorage m_OriginalStorage;
    fs::SubStorage m_LogStorage;
    UnionStorage m_UnionStorage;
};

// 基本的な動作を確認するテストです
TEST_F(UnionStorageTest, Basic)
{
    nn::Result result;

    const size_t bufferSize = BlockSize * 16;
    nnt::fs::util::Vector<char> readBuffer(bufferSize, 0);
    nnt::fs::util::Vector<char> writeBuffer(bufferSize, 0);
    nnt::fs::util::Vector<char> zeroBuffer(bufferSize, 0);

    result = GetOriginalStorage().Write(0, writeBuffer.data(), writeBuffer.size());
    NNT_ASSERT_RESULT_SUCCESS(result);

    // 書き込める
    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');
    result = GetUnionStorage().Write(0, writeBuffer.data(), writeBuffer.size());
    NNT_ASSERT_RESULT_SUCCESS(result);

    // 読み込める
    result = GetUnionStorage().Read(0, readBuffer.data(), readBuffer.size());
    NNT_ASSERT_RESULT_SUCCESS(result);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.data(), readBuffer.data(), bufferSize);

    // もとのストレージには書き込まない
    result = GetOriginalStorage().Read(0, readBuffer.data(), readBuffer.size());
    NNT_ASSERT_RESULT_SUCCESS(result);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(zeroBuffer.data(), readBuffer.data(), bufferSize);

    // コミットすると書き込まれる
    result = GetUnionStorage().Commit();
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = GetOriginalStorage().Read(0, readBuffer.data(), readBuffer.size());
    NNT_ASSERT_RESULT_SUCCESS(result);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.data(), readBuffer.data(), bufferSize);
}

// 境界値テストです
TEST_F(UnionStorageTest, Boundary)
{
    nn::Result result;

    nnt::fs::util::Vector<char> buffer(OriginalStorageSize + 1);

    // 初期化
    {
        const size_t blockSize = 512;
        const size_t originalStorageSize = 8 * 1024;
        const auto logStorageSize = originalStorageSize + blockSize;
        SafeMemoryStorage baseStorage(originalStorageSize + logStorageSize);
        fs::SubStorage originalStorage(&baseStorage, 0, originalStorageSize);
        fs::SubStorage logStorage(&baseStorage, originalStorageSize, logStorageSize);
        UnionStorage unionStorage;

        result = UnionStorage::Format(
            fs::SubStorage(&logStorage, 0, sizeof(int64_t) * 2 - 1), blockSize);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);
        result = UnionStorage::Format(logStorage, blockSize);
        NNT_EXPECT_RESULT_SUCCESS(result);

        result = unionStorage.Initialize(originalStorage, logStorage, blockSize - 1);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidLogBlockSize, result);
        result = unionStorage.Initialize(originalStorage, logStorage, blockSize + 1);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidLogBlockSize, result);

        int64_t logData[2] = { 0 };

        logData[0] = -512;
        logData[1] = -1;
        result = logStorage.Write(0, logData, sizeof(logData));
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = unionStorage.Initialize(originalStorage, logStorage, blockSize);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidLogBlockSize, result);
        logData[0] = 0;
        logData[1] = -1;
        result = logStorage.Write(0, logData, sizeof(logData));
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = unionStorage.Initialize(originalStorage, logStorage, blockSize);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidLogBlockSize, result);
        logData[0] = 1;
        logData[1] = -1;
        result = logStorage.Write(0, logData, sizeof(logData));
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = unionStorage.Initialize(originalStorage, logStorage, blockSize);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidLogBlockSize, result);
        logData[0] = 2;
        logData[1] = blockSize - 1;
        result = logStorage.Write(0, logData, sizeof(logData));
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = unionStorage.Initialize(originalStorage, logStorage, blockSize);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidLogBlockSize, result);
        logData[0] = blockSize;
        logData[1] = blockSize + 1;
        result = logStorage.Write(0, logData, sizeof(logData));
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = unionStorage.Initialize(originalStorage, logStorage, blockSize);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidLogOffset, result);
    }

    // 読み書き
    for( int count = 0; count < 3; ++count )
    {
        result = GetUnionStorage().Read(-1, buffer.data(), OriginalStorageSize);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);
        result = GetUnionStorage().Read(0, buffer.data(), OriginalStorageSize);
        NNT_EXPECT_RESULT_SUCCESS(result);
        result = GetUnionStorage().Read(1, buffer.data(), OriginalStorageSize);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);
        result = GetUnionStorage().Read(0, buffer.data(), OriginalStorageSize + 1);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);
        result = GetUnionStorage().Read(OriginalStorageSize - 1, buffer.data(), 1);
        NNT_EXPECT_RESULT_SUCCESS(result);
        result = GetUnionStorage().Read(OriginalStorageSize, buffer.data(), 1);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);
        result = GetUnionStorage().Read(OriginalStorageSize - 1, buffer.data(), 2);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);

        result = GetUnionStorage().Write(-1, buffer.data(), OriginalStorageSize);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);
        result = GetUnionStorage().Write(0, buffer.data(), OriginalStorageSize);
        NNT_EXPECT_RESULT_SUCCESS(result);
        result = GetUnionStorage().Write(1, buffer.data(), OriginalStorageSize);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);
        result = GetUnionStorage().Write(0, buffer.data(), OriginalStorageSize + 1);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);
        result = GetUnionStorage().Write(OriginalStorageSize - 1, buffer.data(), 1);
        NNT_EXPECT_RESULT_SUCCESS(result);
        result = GetUnionStorage().Write(OriginalStorageSize, buffer.data(), 1);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);
        result = GetUnionStorage().Write(OriginalStorageSize - 1, buffer.data(), 2);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);
    }
}

// ログ破壊で正しくエラーが返ることを確認するテストです
TEST_F(UnionStorageTest, CorruptLog)
{
    nn::Result result;
    nnt::fs::util::Vector<char> buffer(BlockSize * 2);
    const int64_t sentinel = -1;

    std::iota(buffer.begin(), buffer.end(), '\x0');
    result = GetUnionStorage().Write(0, buffer.data(), buffer.size());
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = GetLogStorage().Write(sizeof(int64_t), &sentinel, sizeof(int64_t));
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = GetUnionStorage().Read(BlockSize, buffer.data(), BlockSize);
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultLogNotFound, result);
    result = GetUnionStorage().Write(BlockSize * 2, buffer.data(), BlockSize * 2);
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultLogNotFound, result);
    result = GetUnionStorage().Commit();
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultUnexpectedEndOfLog, result);
}

// ログ枯渇で正しくエラーが返ることを確認するテストです
TEST_F(UnionStorageTest, ExhaustLog)
{
    nn::Result result;
    nnt::fs::util::Vector<char> buffer(BlockSize);
    int blockIndex = 0;

    std::iota(buffer.begin(), buffer.end(), '\x0');

    do
    {
        result = GetUnionStorage().Write(BlockSize * blockIndex, buffer.data(), buffer.size());
        ++blockIndex;
    } while( result.IsSuccess() );

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, result);
}

// コミットが正しくできることを確認するテストです
TEST_F(UnionStorageTest, CommitHeavy)
{
    nn::Result result;

    static const size_t BufferSize = BlockSize * 16;
    nnt::fs::util::Vector<char> originalBuffer(BufferSize, 0);
    nnt::fs::util::Vector<char> readBuffer(BufferSize, 0);
    nnt::fs::util::Vector<char> writeBuffer(BufferSize, 0);

    for( int count = 0; count < 256; ++count )
    {
        // 書き込んだら読み込めるが、オリジナルは変化しない
        std::iota(writeBuffer.begin(), writeBuffer.end(), static_cast<char>(count));
        result = GetOriginalStorage().Read(0, originalBuffer.data(), originalBuffer.size());
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = GetUnionStorage().Write(0, writeBuffer.data(), writeBuffer.size());
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = GetUnionStorage().Read(0, readBuffer.data(), readBuffer.size());
        NNT_ASSERT_RESULT_SUCCESS(result);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.data(), readBuffer.data(), BufferSize);
        result = GetOriginalStorage().Read(0, readBuffer.data(), readBuffer.size());
        NNT_ASSERT_RESULT_SUCCESS(result);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(originalBuffer.data(), readBuffer.data(), BufferSize);

        // コミットで何度でもオリジナルを上書きできる
        for( int commitCount = 0; commitCount < 64; ++commitCount )
        {
            std::iota(originalBuffer.begin(), originalBuffer.end(), static_cast<char>(commitCount));
            result = GetOriginalStorage().Write(0, originalBuffer.data(), originalBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = GetUnionStorage().Commit();
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = GetOriginalStorage().Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.data(), readBuffer.data(), BufferSize);
        }
    }
}

// ストレージの再マウントができることを確認するテストです
TEST_F(UnionStorageTest, Remount)
{
    nn::Result result;

    static const size_t BufferSize = BlockSize * 16;
    nnt::fs::util::Vector<char> originalBuffer(BufferSize, 0);
    nnt::fs::util::Vector<char> readBuffer(BufferSize, 0);
    nnt::fs::util::Vector<char> writeBuffer(BufferSize, 0);

    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');
    result = GetUnionStorage().Write(0, writeBuffer.data(), writeBuffer.size());
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = GetUnionStorage().Freeze();
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = GetUnionStorage().Flush();
    NNT_ASSERT_RESULT_SUCCESS(result);

    // 番兵を書き込んだログを再マウントできる
    for( int commitCount = 0; commitCount < 64; ++commitCount )
    {
        UnionStorage unionStorage;

        std::iota(originalBuffer.begin(), originalBuffer.end(), static_cast<char>(commitCount));
        result = GetOriginalStorage().Write(0, originalBuffer.data(), originalBuffer.size());
        NNT_ASSERT_RESULT_SUCCESS(result);

        result = unionStorage.Initialize(
                     fs::SubStorage(&GetOriginalStorage(), 0, OriginalStorageSize),
                     fs::SubStorage(&GetLogStorage(), 0, LogStorageSize),
                     BlockSize
                 );
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = unionStorage.Commit();
        NNT_ASSERT_RESULT_SUCCESS(result);

        result = GetOriginalStorage().Read(0, readBuffer.data(), readBuffer.size());
        NNT_ASSERT_RESULT_SUCCESS(result);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.data(), readBuffer.data(), BufferSize);
    }
}

// 読み書きおよびコミットを規則的に繰り返すテストです
TEST_F(UnionStorageTest, Repeat)
{
    static const size_t DataSize = BlockSize * 4;
    static const size_t BufferSize = 1234;
    nnt::fs::util::Vector<char> originalBuffer(BufferSize, 0);
    nnt::fs::util::Vector<char> readBuffer(BufferSize, 0);
    nnt::fs::util::Vector<char> writeBuffer(BufferSize, 0);

    auto check = [&](int64_t offset, size_t size)
    {
        std::iota(writeBuffer.begin(), writeBuffer.begin() + size, static_cast<char>(offset));

        nn::Result result;

        result = GetOriginalStorage().Read(offset, originalBuffer.data(), size);
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = GetUnionStorage().Write(offset, writeBuffer.data(), size);
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = GetUnionStorage().Read(offset, readBuffer.data(), size);
        NNT_ASSERT_RESULT_SUCCESS(result);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.data(), readBuffer.data(), BufferSize);
        result = GetOriginalStorage().Read(offset, readBuffer.data(), size);
        NNT_ASSERT_RESULT_SUCCESS(result);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(originalBuffer.data(), readBuffer.data(), BufferSize);
        result = GetUnionStorage().Commit();
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = GetOriginalStorage().Read(offset, readBuffer.data(), size);
        NNT_ASSERT_RESULT_SUCCESS(result);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.data(), readBuffer.data(), BufferSize);

    };

    for( int count = 0; count < 1024; ++count )
    {
        for( int64_t offset = DataSize - BufferSize; 0 <= offset; offset -= BufferSize )
        {
            check(offset, BufferSize);
        }

        for( int64_t offset = 0; offset < DataSize; offset += BufferSize )
        {
            check(offset, BufferSize);
        }

        for( int64_t offset = 0; offset < DataSize; offset += BufferSize * 2 )
        {
            check(offset, BufferSize);
        }

        for( int64_t offset = BufferSize; offset < DataSize; offset += BufferSize * 2 )
        {
            check(offset, BufferSize);
        }
    }
}

// 読み込みを並列で実行するテストです
TEST_F(UnionStorageTest, ConcurrentRead)
{
    static const int CountMax = 256;

    // 事前にストレージに書き込んでおきます
    {
        nnt::fs::util::Vector<char> buffer(BlockSize);

        std::iota(buffer.begin(), buffer.end(), '\x0');

        for( auto index = 0; index < CountMax; index += 16 )
        {
            const auto result = GetUnionStorage().Write(
                                    buffer.size() * index,
                                    buffer.data(),
                                    buffer.size()
                                );
            NNT_ASSERT_RESULT_SUCCESS(result);
        }
    }

    // 並列で読み込みます
    static const auto ThreadCount = 8;
    nn::os::ThreadType threads[ThreadCount];

    static const auto StackSize = 32 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK static char s_ThreadStack[StackSize * ThreadCount] = {};

    struct Argument
    {
        UnionStorage* pStorage;
        nn::Result result;
    };
    Argument args[ThreadCount] = {};

    const auto ReadFunc = [](void* pArg)
    {
        NN_SDK_REQUIRES_NOT_NULL(pArg);
        NN_SDK_REQUIRES_NOT_NULL(reinterpret_cast<Argument*>(pArg)->pStorage);

        const auto pStorage = reinterpret_cast<Argument*>(pArg)->pStorage;
        auto& result = reinterpret_cast<Argument*>(pArg)->result;

        nnt::fs::util::Vector<char> buffer(BlockSize);

        std::iota(buffer.begin(), buffer.end(), '\x1');

        for( int count = 0; count < CountMax; ++count )
        {
            result = pStorage->Read(BlockSize * count, buffer.data(), buffer.size());

            if( result.IsFailure() )
            {
                break;
            }
        }
    };

    for( auto& thread : threads )
    {
        const auto index = &thread - threads;
        args[index].pStorage = &GetUnionStorage();
        args[index].result = nn::ResultSuccess();
        const auto result = nn::os::CreateThread(
                                &thread,
                                ReadFunc,
                                args + index,
                                s_ThreadStack + StackSize * index,
                                StackSize,
                                nn::os::DefaultThreadPriority
                            );
        NNT_ASSERT_RESULT_SUCCESS(result);
        nn::os::StartThread(&thread);
    }

    for( auto& thread : threads )
    {
        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);
        NNT_EXPECT_RESULT_SUCCESS(args[&thread - threads].result);
    }
}

// 書き込みとコミットを並列で実行するテストです
TEST_F(UnionStorageTest, ConcurrentCommit)
{
    nn::os::ThreadType writeThread;
    nn::os::ThreadType commitThread;

    static const auto StackSize = 32 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK static char s_WriteThreadStack[StackSize] = {};
    NN_OS_ALIGNAS_THREAD_STACK static char s_CommitThreadStack[StackSize] = {};

    struct Argument
    {
        UnionStorage* pStorage;
        nn::Result result;
    };
    Argument writeArg = { &GetUnionStorage(), nn::ResultSuccess() };
    Argument commitArg = { &GetUnionStorage(), nn::ResultSuccess() };

    static const int CountMax = 256;

    const auto WriteFunc = [](void* pArg)
    {
        NN_SDK_REQUIRES_NOT_NULL(pArg);
        NN_SDK_REQUIRES_NOT_NULL(reinterpret_cast<Argument*>(pArg)->pStorage);

        const auto pStorage = reinterpret_cast<Argument*>(pArg)->pStorage;
        auto& result = reinterpret_cast<Argument*>(pArg)->result;

        nnt::fs::util::Vector<char> buffer(BlockSize);

        std::iota(buffer.begin(), buffer.end(), '\x0');

        for( int count = 0; count < CountMax; ++count )
        {
            result = pStorage->Write(BlockSize * count, buffer.data(), buffer.size());

            if( result.IsFailure() )
            {
                break;
            }
        }
    };

    const auto CommitFunc = [](void* pArg)
    {
        NN_SDK_REQUIRES_NOT_NULL(pArg);
        NN_SDK_REQUIRES_NOT_NULL(reinterpret_cast<Argument*>(pArg)->pStorage);

        const auto pStorage = reinterpret_cast<Argument*>(pArg)->pStorage;
        auto& result = reinterpret_cast<Argument*>(pArg)->result;

        for( int count = 0; count < CountMax; ++count )
        {
            result = pStorage->Freeze();

            if( result.IsFailure() )
            {
                break;
            }

            result = pStorage->Commit();

            if( result.IsFailure() )
            {
                break;
            }
        }
    };

    nn::Result result;
    result = nn::os::CreateThread(
                 &writeThread,
                 WriteFunc,
                 &writeArg,
                 s_WriteThreadStack,
                 StackSize,
                 nn::os::DefaultThreadPriority
             );
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = nn::os::CreateThread(
                 &commitThread,
                 CommitFunc,
                 &commitArg,
                 s_CommitThreadStack,
                 StackSize,
                 nn::os::DefaultThreadPriority
             );
    NNT_ASSERT_RESULT_SUCCESS(result);

    nn::os::StartThread(&writeThread);
    nn::os::StartThread(&commitThread);
    nn::os::WaitThread(&writeThread);
    nn::os::WaitThread(&commitThread);
    nn::os::DestroyThread(&writeThread);
    nn::os::DestroyThread(&commitThread);

    NNT_EXPECT_RESULT_SUCCESS(writeArg.result);
    NNT_EXPECT_RESULT_SUCCESS(commitArg.result);
}

// 無作為に操作してみるテストです
TEST_F(UnionStorageTest, RandomHeavy)
{
    nn::Result result;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    static const size_t BufferSize = 64 * BlockSize;
    nnt::fs::util::Vector<char> originalBuffer(BufferSize, 0);
    nnt::fs::util::Vector<char> readBuffer(BufferSize, 0);
    nnt::fs::util::Vector<char> writeBuffer(BufferSize, 0);

    for( int count = 0; count < 4096; ++count )
    {
        switch( std::uniform_int_distribution<int>(0, 2)(mt) )
        {
        case 0:
            // 読み込んで検証する
            {
                const auto offset = std::uniform_int_distribution<size_t>(0, BufferSize - 1)(mt);
                const auto size
                    = std::uniform_int_distribution<size_t>(1, BufferSize - offset)(mt);

                result = GetUnionStorage().Read(offset, readBuffer.data(), size);
                NNT_ASSERT_RESULT_SUCCESS(result);
                NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.data() + offset, readBuffer.data(), size);
            }
            break;

        case 1:
            // 書き込んでもオリジナルが変化しないことを確認する
            {
                const auto offset = std::uniform_int_distribution<size_t>(0, BufferSize - 1)(mt);
                const auto size
                    = std::uniform_int_distribution<size_t>(1, BufferSize - offset)(mt);

                const auto beginPosition = writeBuffer.begin() + offset;
                const auto endPosition = writeBuffer.begin() + offset + size;
                std::generate(beginPosition, endPosition, [&mt]()
                {
                    return static_cast<char>(std::uniform_int_distribution<>(
                               std::numeric_limits<char>::min(),
                               std::numeric_limits<char>::max()
                           )(mt));
                });

                result = GetOriginalStorage().Read(offset, originalBuffer.data(), size);
                NNT_ASSERT_RESULT_SUCCESS(result);
                result = GetUnionStorage().Write(offset, writeBuffer.data() + offset, size);
                NNT_ASSERT_RESULT_SUCCESS(result);
                result = GetOriginalStorage().Read(offset, readBuffer.data(), size);
                NNT_ASSERT_RESULT_SUCCESS(result);
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(originalBuffer.data(), readBuffer.data(), size);
            }
            break;

        case 2:
            // コミットでオリジナルに反映されることを確認する
            {
                result = GetUnionStorage().Commit();
                NNT_ASSERT_RESULT_SUCCESS(result);
                result = GetOriginalStorage().Read(0, readBuffer.data(), BufferSize);
                NNT_ASSERT_RESULT_SUCCESS(result);
                NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.data(), readBuffer.data(), BufferSize);
            }
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

#ifndef NN_SDK_BUILD_RELEASE
// 事前条件違反で正しく落ちることを確認するテストです
TEST(UnionStorageDeathTest, Precondition)
{
    nn::Result result;

    static const size_t BlockSize = 512;
    static const size_t OriginalStorageSize = 8 * 1024;
    static const auto LogStorageSize = OriginalStorageSize + BlockSize;
    SafeMemoryStorage baseStorage(OriginalStorageSize + LogStorageSize);
    fs::SubStorage originalStorage(&baseStorage, 0, OriginalStorageSize);
    fs::SubStorage logStorage(&baseStorage, OriginalStorageSize, LogStorageSize);
    UnionStorage unionStorage;

    // 未初期化での API 呼び出し
    {
        nnt::fs::util::Vector<char> buffer(1024);
        int64_t size = 0;

        EXPECT_DEATH_IF_SUPPORTED(unionStorage.Freeze(), "");
        EXPECT_DEATH_IF_SUPPORTED(unionStorage.Commit(), "");
        EXPECT_DEATH_IF_SUPPORTED(unionStorage.Read(0, buffer.data(), buffer.size()), "");
        EXPECT_DEATH_IF_SUPPORTED(unionStorage.Write(0, buffer.data(), buffer.size()), "");
        EXPECT_DEATH_IF_SUPPORTED(unionStorage.Flush(), "");
        NN_ASSERT(unionStorage.SetSize(2048).IsFailure());  // SetSize 未実装のため IStorage でエラー Result が帰る
        EXPECT_DEATH_IF_SUPPORTED(unionStorage.GetSize(&size), "");
    }

    // Format
    {
        EXPECT_DEATH_IF_SUPPORTED(UnionStorage::Format(logStorage, -1), "");
        EXPECT_DEATH_IF_SUPPORTED(UnionStorage::Format(logStorage, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(UnionStorage::Format(logStorage, 1), "");
        EXPECT_DEATH_IF_SUPPORTED(UnionStorage::Format(logStorage, 255), "");
        EXPECT_DEATH_IF_SUPPORTED(UnionStorage::Format(logStorage, -1), "");
        result = UnionStorage::Format(logStorage, BlockSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // Initialize
    {
        result = unionStorage.Initialize(
                     fs::SubStorage(&originalStorage, 0, OriginalStorageSize),
                     fs::SubStorage(&logStorage, 0, LogStorageSize),
                     BlockSize
                 );
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_DEATH_IF_SUPPORTED(unionStorage.Initialize(
            fs::SubStorage(&originalStorage, 0, OriginalStorageSize),
            fs::SubStorage(&logStorage, 0, LogStorageSize),
            BlockSize
        ), "");
    }
}
#endif

// 4 GB を超えるオフセットのテスト
TEST(UnionStorageLargeTest, ReadWrite)
{
    static const int64_t BlockSize = 16 * 1024;
    static const size_t AccessSize = BlockSize;
    static const int64_t StorageSize = nnt::fs::util::LargeOffsetMax + AccessSize;

    nnt::fs::util::VirtualMemoryStorage originalStorage(StorageSize);
    nnt::fs::util::VirtualMemoryStorage logStorage(StorageSize);
    NNT_ASSERT_RESULT_SUCCESS(nn::fssystem::save::UnionStorage::Format(
        nn::fs::SubStorage(&logStorage, 0, StorageSize),
        BlockSize));

    nn::fssystem::save::UnionStorage unionStorage;
    NNT_ASSERT_RESULT_SUCCESS(unionStorage.Initialize(
        nn::fs::SubStorage(&originalStorage, 0, StorageSize),
        nn::fs::SubStorage(&logStorage, 0, StorageSize),
        BlockSize));

    // バッファ初期化
    std::unique_ptr<char> readBuffer(new char[AccessSize]);
    std::unique_ptr<char> writeBufferList[nnt::fs::util::LargeOffsetListLength];
    for( auto& writeBuffer : writeBufferList )
    {
        writeBuffer.reset(new char[AccessSize]);
        FillBufferWithRandomValue(writeBuffer.get(), AccessSize);
    }

    // 書き込み
    for( size_t i = 0; i < nnt::fs::util::LargeOffsetListLength; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(unionStorage.Write(nnt::fs::util::LargeOffsetList[i], writeBufferList[i].get(), AccessSize));
    }

    // コミット前の読み込み
    for( size_t i = 0; i < nnt::fs::util::LargeOffsetListLength; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(unionStorage.Read(nnt::fs::util::LargeOffsetList[i], readBuffer.get(), AccessSize));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList[i].get(), readBuffer.get(), AccessSize);

        // オリジナルには反映されない
        NNT_ASSERT_RESULT_SUCCESS(originalStorage.Read(nnt::fs::util::LargeOffsetList[i], readBuffer.get(), AccessSize));
        EXPECT_NE(0, std::memcmp(writeBufferList[i].get(), readBuffer.get(), AccessSize));
    }

    // コミット
    NNT_ASSERT_RESULT_SUCCESS(unionStorage.Commit());

    // コミット後の読み込み
    for( size_t i = 0; i < nnt::fs::util::LargeOffsetListLength; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(unionStorage.Read(nnt::fs::util::LargeOffsetList[i], readBuffer.get(), AccessSize));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList[i].get(), readBuffer.get(), AccessSize);

        // オリジナルに反映される
        NNT_ASSERT_RESULT_SUCCESS(originalStorage.Read(nnt::fs::util::LargeOffsetList[i], readBuffer.get(), AccessSize));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList[i].get(), readBuffer.get(), AccessSize);
    }
}

}}}
