﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <random>
#include <nn/fssystem/save/fs_MappingTable.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

// マッピングテーブルのメタデータの計算テスト
TEST(MappingTableTest, QueryMappingMetaSize)
{
    int64_t tableSize;
    int64_t bitmapSizeUpdatedPhysical;
    int64_t bitmapSizeUpdatedVirtual;
    int64_t bitmapSizeUnassigned;

    nn::fssystem::save::MappingTable::QueryMappingMetaSize(
        &tableSize,
        &bitmapSizeUpdatedPhysical,
        &bitmapSizeUpdatedVirtual,
        &bitmapSizeUnassigned,
        0,
        0
    );
    EXPECT_EQ(16, sizeof(nn::fssystem::save::MappingTable::ControlArea) +
                  tableSize +
                  bitmapSizeUpdatedPhysical +
                  bitmapSizeUpdatedVirtual +
                  bitmapSizeUnassigned);

    nn::fssystem::save::MappingTable::QueryMappingMetaSize(
        &tableSize,
        &bitmapSizeUpdatedPhysical,
        &bitmapSizeUpdatedVirtual,
        &bitmapSizeUnassigned,
        1,
        1
    );
    EXPECT_EQ(36, sizeof(nn::fssystem::save::MappingTable::ControlArea) +
                  tableSize +
                  bitmapSizeUpdatedPhysical +
                  bitmapSizeUpdatedVirtual +
                  bitmapSizeUnassigned);

    nn::fssystem::save::MappingTable::QueryMappingMetaSize(
        &tableSize,
        &bitmapSizeUpdatedPhysical,
        &bitmapSizeUpdatedVirtual,
        &bitmapSizeUnassigned,
        16,
        1
    );
    EXPECT_EQ(156, sizeof(nn::fssystem::save::MappingTable::ControlArea) +
                   tableSize +
                   bitmapSizeUpdatedPhysical +
                   bitmapSizeUpdatedVirtual +
                   bitmapSizeUnassigned);

    nn::fssystem::save::MappingTable::QueryMappingMetaSize(
        &tableSize,
        &bitmapSizeUpdatedPhysical,
        &bitmapSizeUpdatedVirtual,
        &bitmapSizeUnassigned,
        16,
        4
    );
    EXPECT_EQ(156, sizeof(nn::fssystem::save::MappingTable::ControlArea) +
                   tableSize +
                   bitmapSizeUpdatedPhysical +
                   bitmapSizeUpdatedVirtual +
                   bitmapSizeUnassigned);

    nn::fssystem::save::MappingTable::QueryMappingMetaSize(
        &tableSize,
        &bitmapSizeUpdatedPhysical,
        &bitmapSizeUpdatedVirtual,
        &bitmapSizeUnassigned,
        1024,
        64
    );
    EXPECT_EQ(8608, sizeof(nn::fssystem::save::MappingTable::ControlArea) +
                    tableSize +
                    bitmapSizeUpdatedPhysical +
                    bitmapSizeUpdatedVirtual +
                    bitmapSizeUnassigned);

    nn::fssystem::save::MappingTable::QueryMappingMetaSize(
        &tableSize,
        &bitmapSizeUpdatedPhysical,
        &bitmapSizeUpdatedVirtual,
        &bitmapSizeUnassigned,
        4096,
        1024
    );
    EXPECT_EQ(34576, sizeof(nn::fssystem::save::MappingTable::ControlArea) +
                     tableSize +
                     bitmapSizeUpdatedPhysical +
                     bitmapSizeUpdatedVirtual +
                     bitmapSizeUnassigned);

    nn::fssystem::save::MappingTable::QueryMappingMetaSize(
        &tableSize,
        &bitmapSizeUpdatedPhysical,
        &bitmapSizeUpdatedVirtual,
        &bitmapSizeUnassigned,
        65536,
        16
    );
    EXPECT_EQ(548888, sizeof(nn::fssystem::save::MappingTable::ControlArea) +
                      tableSize +
                      bitmapSizeUpdatedPhysical +
                      bitmapSizeUpdatedVirtual +
                      bitmapSizeUnassigned);
}



namespace {

struct FormatTestParam
{
    int64_t offset;
    uint32_t countMapEntry;
    uint32_t countReserved;
};

nnt::fs::util::Vector<FormatTestParam> MakeFormatTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<FormatTestParam> result;

    {
        FormatTestParam data;
        data.offset = 0;
        data.countMapEntry = 1;
        data.countReserved = 1;
        result.push_back(data);
    }

    {
        FormatTestParam data;
        data.offset = 0;
        data.countMapEntry = 128;
        data.countReserved = 32;
        result.push_back(data);
    }

    {
        FormatTestParam data;
        data.offset = 32;
        data.countMapEntry = 160;
        data.countReserved = 42;
        result.push_back(data);
    }

    {
        FormatTestParam data;
        data.offset = 1442;
        data.countMapEntry = 192;
        data.countReserved = 48;
        result.push_back(data);
    }

    return result;
}

}

class MappingTableFormatTest : public ::testing::TestWithParam< FormatTestParam >
{
};

// メタデータ領域のフォーマットテスト
TEST_P(MappingTableFormatTest, Format)
{
    static const int64_t SizeControlArea = sizeof(nn::fssystem::save::MappingTable::ControlArea);

    const FormatTestParam& param = GetParam();
    const uint32_t countMapEntry = param.countMapEntry;
    const uint32_t countReserved = param.countReserved;
    const size_t offset = static_cast<size_t>(param.offset);

    int64_t sizeTable;
    int64_t sizeBitmapUpdatedPhysical;
    int64_t sizeBitmapUpdatedVirtual;
    int64_t sizeBitmapUnassigned;
    nn::fssystem::save::MappingTable::QueryMappingMetaSize(
        &sizeTable,
        &sizeBitmapUpdatedPhysical,
        &sizeBitmapUpdatedVirtual,
        &sizeBitmapUnassigned,
        countMapEntry,
        countReserved
    );
    nnt::fs::util::SafeMemoryStorage
        storageControlArea(SizeControlArea + offset);
    nnt::fs::util::SafeMemoryStorage
        storageTable(sizeTable + offset);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUpdatedPhysical(sizeBitmapUpdatedPhysical + offset);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUpdatedVirtual(sizeBitmapUpdatedVirtual + offset);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUnassigned(sizeBitmapUnassigned + offset);

    // フォーマットが関係ない範囲に書き込まないかを確認するための
    // データを書き込みます。
    if( offset > 0 )
    {
        storageControlArea.FillBuffer(0, offset, 0x12);
        storageTable.FillBuffer(0, offset, 0x12);
        storageBitmapUpdatedPhysical.FillBuffer(0, offset, 0x12);
        storageBitmapUpdatedVirtual.FillBuffer(0, offset, 0x12);
        storageBitmapUnassigned.FillBuffer(0, offset, 0x12);
    }

    nn::fs::SubStorage subStorageControlArea(
        &storageControlArea,
        offset,
        SizeControlArea
    );
    nn::fs::SubStorage subStorageTable(
        &storageTable,
        offset,
        sizeTable
    );
    nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
        &storageBitmapUpdatedPhysical,
        offset,
        sizeBitmapUpdatedPhysical
    );
    nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
        &storageBitmapUpdatedVirtual,
        offset,
        sizeBitmapUpdatedVirtual
    );
    nn::fs::SubStorage subStorageBitmapUnassigned(
        &storageBitmapUnassigned,
        offset,
        sizeBitmapUnassigned
    );

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::MappingTable::Format(
            subStorageControlArea,
            subStorageTable,
            subStorageBitmapUpdatedPhysical,
            subStorageBitmapUpdatedVirtual,
            subStorageBitmapUnassigned,
            countMapEntry,
            countReserved
        )
    );

    // フォーマットが関係ない範囲に書き込みを行なっていないか確認します。
    if( offset > 0 )
    {
        ASSERT_EQ(true, storageControlArea.VerifyBuffer(0, offset, 0x12));
        ASSERT_EQ(true, storageTable.VerifyBuffer(0, offset, 0x12));
        ASSERT_EQ(true, storageBitmapUpdatedPhysical.VerifyBuffer(0, offset, 0x12));
        ASSERT_EQ(true, storageBitmapUpdatedVirtual.VerifyBuffer(0, offset, 0x12));
        ASSERT_EQ(true, storageBitmapUnassigned.VerifyBuffer(0, offset, 0x12));
    }
}

// マッピングテーブルのマウントテスト
TEST_P(MappingTableFormatTest, Initialize)
{
    static const int64_t SizeControlArea = sizeof(nn::fssystem::save::MappingTable::ControlArea);

    const FormatTestParam& param = GetParam();
    const uint32_t countMapEntry = param.countMapEntry;
    const uint32_t countReserved = param.countReserved;
    const size_t offset = static_cast<size_t>(param.offset);

    int64_t sizeTable;
    int64_t sizeBitmapUpdatedPhysical;
    int64_t sizeBitmapUpdatedVirtual;
    int64_t sizeBitmapUnassigned;
    nn::fssystem::save::MappingTable::QueryMappingMetaSize(
        &sizeTable,
        &sizeBitmapUpdatedPhysical,
        &sizeBitmapUpdatedVirtual,
        &sizeBitmapUnassigned,
        countMapEntry,
        countReserved
    );
    nnt::fs::util::SafeMemoryStorage
        storageControlArea(SizeControlArea + offset);
    nnt::fs::util::SafeMemoryStorage
        storageTable(sizeTable + offset);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUpdatedPhysical(sizeBitmapUpdatedPhysical + offset);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUpdatedVirtual(sizeBitmapUpdatedVirtual + offset);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUnassigned(sizeBitmapUnassigned + offset);

    // フォーマット/マウントが関係ない範囲に書き込まないかを確認するための
    // データを書き込みます。
    if( offset > 0 )
    {
        storageControlArea.FillBuffer(0, offset, 0x66);
        storageTable.FillBuffer(0, offset, 0x66);
        storageBitmapUpdatedPhysical.FillBuffer(0, offset, 0x66);
        storageBitmapUpdatedVirtual.FillBuffer(0, offset, 0x66);
        storageBitmapUnassigned.FillBuffer(0, offset, 0x66);
    }

    nn::fs::SubStorage subStorageControlArea(
        &storageControlArea,
        offset,
        SizeControlArea
    );
    nn::fs::SubStorage subStorageTable(
        &storageTable,
        offset,
        sizeTable
    );
    nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
        &storageBitmapUpdatedPhysical,
        offset,
        sizeBitmapUpdatedPhysical
    );
    nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
        &storageBitmapUpdatedVirtual,
        offset,
        sizeBitmapUpdatedVirtual
    );
    nn::fs::SubStorage subStorageBitmapUnassigned(
        &storageBitmapUnassigned,
        offset,
        sizeBitmapUnassigned
    );

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::MappingTable::Format(
            subStorageControlArea,
            subStorageTable,
            subStorageBitmapUpdatedPhysical,
            subStorageBitmapUpdatedVirtual,
            subStorageBitmapUnassigned,
            countMapEntry,
            countReserved
        )
    );
    for( int i = 0; i < 10; ++i )
    {
        nn::fssystem::save::MappingTable table;
        NNT_ASSERT_RESULT_SUCCESS(
            table.Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned
            )
        );
    }

    // フォーマット/マウントが関係ない範囲に書き込みを行なっていないか確認します。
    if( offset > 0 )
    {
        ASSERT_EQ(true, storageControlArea.VerifyBuffer(0, offset, 0x66));
        ASSERT_EQ(true, storageTable.VerifyBuffer(0, offset, 0x66));
        ASSERT_EQ(true, storageBitmapUpdatedPhysical.VerifyBuffer(0, offset, 0x66));
        ASSERT_EQ(true, storageBitmapUpdatedVirtual.VerifyBuffer(0, offset, 0x66));
        ASSERT_EQ(true, storageBitmapUnassigned.VerifyBuffer(0, offset, 0x66));
    }
}

INSTANTIATE_TEST_CASE_P(
    MappingTableTest,
    MappingTableFormatTest,
    ::testing::ValuesIn(MakeFormatTestParam())
);



namespace {

struct MarkUpdateTestParam
{
    int64_t offset;
    uint32_t countMapEntry;
    uint32_t countReserved;
};

nnt::fs::util::Vector<MarkUpdateTestParam> MakeMarkUpdateTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<MarkUpdateTestParam> result;

    {
        MarkUpdateTestParam data;
        data.offset = 0;
        data.countMapEntry = 128;
        data.countReserved = 32;
        result.push_back(data);
    }

    {
        MarkUpdateTestParam data;
        data.offset = 32;
        data.countMapEntry = 160;
        data.countReserved = 42;
        result.push_back(data);
    }

    {
        MarkUpdateTestParam data;
        data.offset = 1442;
        data.countMapEntry = 192;
        data.countReserved = 48;
        result.push_back(data);
    }

    return result;
}

}

class MappingTableMarkUpdateTest : public ::testing::TestWithParam< MarkUpdateTestParam >
{
protected:
    // 物理インデックスを取得します。
    static uint32_t PhysicalIndex(uint32_t index) NN_NOEXCEPT
    {
        return nn::fssystem::save::MappingTable::PhysicalIndex(index);
    }

    // エントリ数を取得します。
    uint32_t GetEntryCount() const NN_NOEXCEPT
    {
        return m_CountMapEntry;
    }

    // 予約エントリ数を取得します。
    uint32_t GetReservedCount() const NN_NOEXCEPT
    {
        return m_CountReserved;
    }

    // セットアップします。
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        static const int64_t SizeControlArea =
            sizeof(nn::fssystem::save::MappingTable::ControlArea);

        const MarkUpdateTestParam& param = GetParam();

        m_Offset = param.offset;
        m_CountMapEntry = param.countMapEntry;
        m_CountReserved = param.countReserved;

        NN_ASSERT(GetEntryCount() >= 64);
        NN_ASSERT(GetEntryCount() <= 192);
        NN_ASSERT(GetReservedCount() >= 32);
        NN_ASSERT(GetReservedCount() <= 48);

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::MappingTable::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetEntryCount(),
            GetReservedCount()
        );
        m_StorageControlArea.Initialize(SizeControlArea + m_Offset);
        m_StorageTable.Initialize(sizeTable + m_Offset);
        m_StorageBitmapUpdatedPhysical.Initialize(sizeBitmapUpdatedPhysical + m_Offset);
        m_StorageBitmapUpdatedVirtual.Initialize(sizeBitmapUpdatedVirtual + m_Offset);
        m_StorageBitmapUnassigned.Initialize(sizeBitmapUnassigned + m_Offset);

        // テスト中に関係ない範囲に書き込まないかを確認するためのデータを書き込みます。
        if( m_Offset > 0 )
        {
            m_StorageControlArea.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageTable.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUpdatedPhysical.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUpdatedVirtual.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUnassigned.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
        }

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_Offset,
            SizeControlArea
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_Offset,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_Offset,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_Offset,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_Offset,
            sizeBitmapUnassigned
        );

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::MappingTable::Format(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                GetEntryCount(),
                GetReservedCount()
            )
        );
        m_Table.reset(new nn::fssystem::save::MappingTable());
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned
            )
        );
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 関係ない範囲に書き込みを行なっていないか確認します。
        if( m_Offset > 0 )
        {
            ASSERT_EQ(
                true,
                m_StorageControlArea.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageTable.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUpdatedPhysical.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUpdatedVirtual.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUnassigned.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
        }

        EXPECT_EQ(true, m_StorageControlArea.CheckValid());
        EXPECT_EQ(true, m_StorageTable.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedPhysical.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedVirtual.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUnassigned.CheckValid());
    }

protected:
    std::unique_ptr<nn::fssystem::save::MappingTable> m_Table;

private:
    int64_t m_Offset;
    uint32_t m_CountMapEntry;
    uint32_t m_CountReserved;
    nnt::fs::util::SafeMemoryStorage m_StorageControlArea;
    nnt::fs::util::SafeMemoryStorage m_StorageTable;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedPhysical;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedVirtual;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUnassigned;
};

// 初期状態では物理/論理インデックスは一致することを確認
TEST_P(MappingTableMarkUpdateTest, GetPhysicalIndex)
{
    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        uint32_t indexPhysical;
        NNT_ASSERT_RESULT_SUCCESS(m_Table->GetPhysicalIndex(&indexPhysical, index));
        ASSERT_EQ(index, indexPhysical);
    }
}

// MarkUpdate 後の物理/論理インデックス対応を確認
TEST_P(MappingTableMarkUpdateTest, MarkUpdate)
{
    // 論理/物理インデックスが一致することを確認します。
    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        uint32_t indexPhysical;
        NNT_ASSERT_RESULT_SUCCESS(m_Table->GetPhysicalIndex(&indexPhysical, index));
        ASSERT_EQ(index, indexPhysical);
    }

    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // 4 ブロック目から 2 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            3,
            2
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(3), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount()), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(4), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 1), indexTailDst);

    // 反転した領域の物理インデックスが変化していることを確認します。
    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        uint32_t indexPhysical;
        NNT_ASSERT_RESULT_SUCCESS(m_Table->GetPhysicalIndex(&indexPhysical, index));
        if( (3 <= index) && (index < 3 + 2) )
        {
            ASSERT_EQ(GetEntryCount() + index - 3, indexPhysical);
        }
        else
        {
            ASSERT_EQ(index, indexPhysical);
        }
    }

    // 11 ブロック目から 12 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            10,
            12
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(10), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 2), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(21), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 13), indexTailDst);

    // 反転した領域の物理インデックスが変化していることを確認します。
    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        uint32_t indexPhysical;
        NNT_ASSERT_RESULT_SUCCESS(m_Table->GetPhysicalIndex(&indexPhysical, index));
        if( (3 <= index) && (index < 3 + 2) )
        {
            ASSERT_EQ(GetEntryCount() + index - 3, indexPhysical);
        }
        else if( (10 <= index) && (index < 10 + 12) )
        {
            ASSERT_EQ(GetEntryCount() + 2 + index - 10, indexPhysical);
        }
        else
        {
            ASSERT_EQ(index, indexPhysical);
        }
    }

    // 65 ブロック目から 1 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            64,
            1
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(64), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 14), indexHeadDst);
    // 1 ブロックだけ割当てする場合は末尾の確保は不要です。
    EXPECT_EQ(false, isNeedCopyTail);
    EXPECT_EQ(0, indexTailSrc);
    EXPECT_EQ(0, indexTailDst);
}

// 最後の 1 ブロックに対して MarkUpdate を行ないます。
TEST_P(MappingTableMarkUpdateTest, MarkUpdateFinal)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // 最後のブロックに対して割当てを行います。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            GetEntryCount() - 1,
            1
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() - 1), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount()), indexHeadDst);
    EXPECT_EQ(false, isNeedCopyTail);
    EXPECT_EQ(0, indexTailSrc);
    EXPECT_EQ(0, indexTailDst);

    // 最後のブロックを含むように割当てを行います。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            GetEntryCount() - 4,
            4
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() - 4), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 1), indexHeadDst);
    EXPECT_EQ(false, isNeedCopyTail);
    EXPECT_EQ(0, indexTailSrc);
    EXPECT_EQ(0, indexTailDst);
}

// 最後の 1 ブロックを含むように連続ブロックに対して MarkUpdate を行ないます。
TEST_P(MappingTableMarkUpdateTest, MarkUpdateFinal2)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // 最後のブロックを含むように割当てを行います。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            GetEntryCount() - 4,
            4
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() - 4), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount()), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() - 1), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 3), indexTailDst);
}

// index に範囲外の値を指定して MarkUpdate した時の挙動を確認
TEST_P(MappingTableMarkUpdateTest, MarkUpdateIndexOutOfRange)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // 最後よりも一つ後のブロックを割当てます。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultOutOfRange,
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            GetEntryCount(),
            1
        )
    );
}

// すでに MarkUpdate 済みの領域を覆うように MarkUpdate するテスト
TEST_P(MappingTableMarkUpdateTest, MarkUpdateOverwrap)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // 4 ブロック目から 22 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            3,
            22
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(3), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount()), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(24), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 21), indexTailDst);

    // 17 ブロック目から 4 ブロックを割当てます。
    // (すでに反転済み)
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            16,
            4
        )
    );
    EXPECT_EQ(false, isNeedCopyHead);
    EXPECT_EQ(0, indexHeadSrc);
    EXPECT_EQ(0, indexHeadDst);
    EXPECT_EQ(false, isNeedCopyTail);
    EXPECT_EQ(0, indexTailSrc);
    EXPECT_EQ(0, indexTailDst);

    // 21 ブロック目から 8 ブロックを割当てます。
    // (前半部分については割当て済み)
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            20,
            8
        )
    );
    EXPECT_EQ(false, isNeedCopyHead);
    EXPECT_EQ(0, indexHeadSrc);
    EXPECT_EQ(0, indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(27), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 24), indexTailDst);
}

// 1回の MarkUpdate で予約領域を枯渇させるテスト
TEST_P(MappingTableMarkUpdateTest, SingleMarkUpdateEmpty)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // 100 ブロックを割当てます。
    // (予約領域を超えたブロック数)
    const uint32_t index = 4;
    const uint32_t countAssign = 100;
    NN_SDK_REQUIRES(countAssign > GetReservedCount());
    NN_SDK_REQUIRES(index + countAssign < GetEntryCount());

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultMappingTableFull,
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            index,
            countAssign
        )
    );
}

// 複数回の MarkUpdate で予約領域を枯渇させるテスト
TEST_P(MappingTableMarkUpdateTest, MultipleMarkUpdateEmpty)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // 5 ブロック目から 20 ブロックを割当ます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            4,
            20
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(4), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount()), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(23), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 19), indexTailDst);

    // 65 ブロック目から 50 ブロックを割当てます。
    // (使用可能範囲内かつ予約領域超え)
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultMappingTableFull,
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            64,
            50
        )
    );

    // 未割当領域枯渇のため、これ以上の MarkUpdate は
    // 失敗することを確認します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultMappingTableFull,
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            0,
            3
        )
    );

    // 割り当て済みの領域に対しては MarkUpdate が成功することを確認します。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            10,
            4
        )
    );
    EXPECT_EQ(false, isNeedCopyHead);
    EXPECT_EQ(false, isNeedCopyTail);
}

// 予約領域の限界まで MarkUpdate を実行するテスト
TEST_P(MappingTableMarkUpdateTest, MarkUpdateLimit)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    for( uint32_t i = 0; i < GetReservedCount(); ++i )
    {
        // 1 ブロックずつ割当てます。
        const uint32_t index = 4 + i * 3;
        NN_ASSERT(index < GetEntryCount());
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->MarkUpdate(
                &isNeedCopyHead,
                &indexHeadSrc,
                &indexHeadDst,
                &isNeedCopyTail,
                &indexTailSrc,
                &indexTailDst,
                index,
                1
            )
        );
        EXPECT_EQ(true, isNeedCopyHead);
        EXPECT_EQ(PhysicalIndex(4 + i * 3), indexHeadSrc);
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + i), indexHeadDst);
        EXPECT_EQ(false, isNeedCopyTail);
        EXPECT_EQ(0, indexTailSrc);
        EXPECT_EQ(0, indexTailDst);
    }

    // これ以上反転できないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultMappingTableFull,
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            GetEntryCount() - 2,
            1
        )
    );
}

// MarkUpdate を繰り返して、容量不足になっても意図しない Result を返さないことを確認します。
TEST_P(MappingTableMarkUpdateTest, MarkUpdateResult)
{
    nn::Result result;
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int loop = 0; loop < 10; ++loop )
    {
        const uint32_t index = std::uniform_int_distribution<uint32_t>(
            0, GetEntryCount() - 1)(mt);
        const uint32_t count = std::uniform_int_distribution<uint32_t>(
            1, std::min(GetReservedCount() - 1, GetEntryCount() - index))(mt);

        result = m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            index,
            count
        );
        if( !result.IsSuccess() )
        {
            if( !nn::fs::ResultMappingTableFull::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(result)
                    << "loop = " << loop
                    << ", index = " << index
                    << ", count = " << count;
            }
        }
    }
}

INSTANTIATE_TEST_CASE_P(
    MappingTableTest,
    MappingTableMarkUpdateTest,
    ::testing::ValuesIn(MakeMarkUpdateTestParam())
);

namespace {

struct ReadEntryTestParam
{
    int64_t offset;
    uint32_t countMapEntry;
    uint32_t countReserved;
};

nnt::fs::util::Vector<ReadEntryTestParam> MakeReadEntryTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<ReadEntryTestParam> result;

    {
        ReadEntryTestParam data;
        data.offset = 0;
        data.countMapEntry = 128;
        data.countReserved = 32;
        result.push_back(data);
    }

    {
        ReadEntryTestParam data;
        data.offset = 32;
        data.countMapEntry = 160;
        data.countReserved = 42;
        result.push_back(data);
    }

    {
        ReadEntryTestParam data;
        data.offset = 1442;
        data.countMapEntry = 192;
        data.countReserved = 48;
        result.push_back(data);
    }

    return result;
}

}

class MappingTableReadEntryTest : public ::testing::TestWithParam< ReadEntryTestParam >
{
protected:
    typedef nn::fssystem::save::MappingTable::MappingEntry MappingEntry;

protected:
    // 物理インデックスを取得します。
    static uint32_t PhysicalIndex(uint32_t index) NN_NOEXCEPT
    {
        return nn::fssystem::save::MappingTable::PhysicalIndex(index);
    }

    // エントリ数を取得します。
    uint32_t GetEntryCount() const NN_NOEXCEPT
    {
        return m_CountMapEntry;
    }

    // 予約エントリ数を取得します。
    uint32_t GetReservedCount() const NN_NOEXCEPT
    {
        return m_CountReserved;
    }

    // セットアップします。
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        static const int64_t SizeControlArea =
            sizeof(nn::fssystem::save::MappingTable::ControlArea);

        const ReadEntryTestParam& param = GetParam();

        m_Offset = param.offset;
        m_CountMapEntry = param.countMapEntry;
        m_CountReserved = param.countReserved;

        NN_ASSERT(GetEntryCount() >= 64);
        NN_ASSERT(GetEntryCount() <= 192);
        NN_ASSERT(GetReservedCount() >= 32);
        NN_ASSERT(GetReservedCount() <= 48);

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::MappingTable::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetEntryCount(),
            GetReservedCount()
        );
        m_StorageControlArea.Initialize(SizeControlArea + m_Offset);
        m_StorageTable.Initialize(sizeTable + m_Offset);
        m_StorageBitmapUpdatedPhysical.Initialize(sizeBitmapUpdatedPhysical + m_Offset);
        m_StorageBitmapUpdatedVirtual.Initialize(sizeBitmapUpdatedVirtual + m_Offset);
        m_StorageBitmapUnassigned.Initialize(sizeBitmapUnassigned + m_Offset);

        // テスト中に関係ない範囲に書き込まないかを確認するためのデータを書き込みます。
        if( m_Offset > 0 )
        {
            m_StorageControlArea.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageTable.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUpdatedPhysical.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUpdatedVirtual.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUnassigned.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
        }

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_Offset,
            SizeControlArea
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_Offset,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_Offset,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_Offset,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_Offset,
            sizeBitmapUnassigned
        );

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::MappingTable::Format(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                GetEntryCount(),
                GetReservedCount()
            )
        );
        m_Table.reset(new nn::fssystem::save::MappingTable());
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned
            )
        );
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 関係ない範囲に書き込みを行なっていないか確認します。
        if( m_Offset > 0 )
        {
            ASSERT_EQ(
                true,
                m_StorageControlArea.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageTable.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUpdatedPhysical.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUpdatedVirtual.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUnassigned.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
        }

        EXPECT_EQ(true, m_StorageControlArea.CheckValid());
        EXPECT_EQ(true, m_StorageTable.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedPhysical.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedVirtual.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUnassigned.CheckValid());
    }

    nn::Result ReadMappingEntry(MappingEntry* outValue, uint32_t index) NN_NOEXCEPT
    {
        return m_Table->ReadMappingEntry(outValue, index);
    }

    nn::Result ReadMappingEntries(
        MappingEntry* outValue, size_t bufferCount, uint32_t index, size_t count
    ) NN_NOEXCEPT
    {
        return m_Table->ReadMappingEntries(outValue, bufferCount, index, count);
    }

    nn::Result GetPhysicalIndex(
        uint32_t* outValue, uint32_t index, const MappingEntry& entry
    ) NN_NOEXCEPT
    {
        if( nn::fssystem::save::MappingTable::IsPhysicalIndex(index) )
        {
            *outValue = nn::fssystem::save::MappingTable::GetValidIndex(index);
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(m_Table->CheckPhysicalIndex(entry.indexPhysical));

        *outValue = nn::fssystem::save::MappingTable::GetValidIndex(entry.indexPhysical);
        NN_RESULT_SUCCESS;
    }

protected:
    std::unique_ptr<nn::fssystem::save::MappingTable> m_Table;

private:
    int64_t m_Offset;
    uint32_t m_CountMapEntry;
    uint32_t m_CountReserved;
    nnt::fs::util::SafeMemoryStorage m_StorageControlArea;
    nnt::fs::util::SafeMemoryStorage m_StorageTable;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedPhysical;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedVirtual;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUnassigned;
};

// 初期状態の MappingEntry をテストします。
TEST_P(MappingTableReadEntryTest, ReadSetUp)
{
    // ReadMappingEntry と ReadMappingEntries の読み込み内容が一致することをテストします。
    auto buffer = nnt::fs::util::AllocateBuffer(sizeof(MappingEntry) * GetEntryCount());
    MappingEntry* entryBuffer = reinterpret_cast<MappingEntry*>(buffer.get());
    NNT_ASSERT_RESULT_SUCCESS(
        ReadMappingEntries(entryBuffer, GetEntryCount(), 0, GetEntryCount()));

    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        MappingEntry entry;
        NNT_ASSERT_RESULT_SUCCESS(ReadMappingEntry(&entry, index));

        uint32_t indexPhysical = 0;
        NNT_ASSERT_RESULT_SUCCESS(GetPhysicalIndex(&indexPhysical, index, entryBuffer[index]));
        ASSERT_EQ(index, indexPhysical);

        ASSERT_EQ(0, memcmp(&entry, &entryBuffer[index], sizeof(MappingEntry)));
    }
}

// MarkUpdate 後の物理/論理インデックス対応を MappingEntry からテストします。
TEST_P(MappingTableReadEntryTest, ReadMarkUpdate)
{
    auto buffer = nnt::fs::util::AllocateBuffer(sizeof(MappingEntry) * GetEntryCount());
    MappingEntry* entryBuffer = reinterpret_cast<MappingEntry*>(buffer.get());

    // 論理/物理インデックスが一致することを確認します。
    NNT_ASSERT_RESULT_SUCCESS(
        ReadMappingEntries(entryBuffer, GetEntryCount(), 0, GetEntryCount()));
    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        uint32_t indexPhysical = 0;
        NNT_ASSERT_RESULT_SUCCESS(GetPhysicalIndex(&indexPhysical, index, entryBuffer[index]));
        ASSERT_EQ(index, indexPhysical);
    }

    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // 4 ブロック目から 2 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            3,
            2
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(3), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount()), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(4), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 1), indexTailDst);

    // 反転した領域の物理インデックスが変化していることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(
        ReadMappingEntries(entryBuffer, GetEntryCount(), 0, GetEntryCount()));
    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        uint32_t indexPhysical = 0;
        NNT_ASSERT_RESULT_SUCCESS(GetPhysicalIndex(&indexPhysical, index, entryBuffer[index]));
        if( (3 <= index) && (index < 3 + 2) )
        {
            ASSERT_EQ(GetEntryCount() + index - 3, indexPhysical);
        }
        else
        {
            ASSERT_EQ(index, indexPhysical);
        }
    }

    // 11 ブロック目から 12 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            10,
            12
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(10), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 2), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(21), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 13), indexTailDst);

    // 反転した領域の物理インデックスが変化していることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(
        ReadMappingEntries(entryBuffer, GetEntryCount(), 0, GetEntryCount()));
    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        uint32_t indexPhysical = 0;
        NNT_ASSERT_RESULT_SUCCESS(GetPhysicalIndex(&indexPhysical, index, entryBuffer[index]));
        if( (3 <= index) && (index < 3 + 2) )
        {
            ASSERT_EQ(GetEntryCount() + index - 3, indexPhysical);
        }
        else if( (10 <= index) && (index < 10 + 12) )
        {
            ASSERT_EQ(GetEntryCount() + 2 + index - 10, indexPhysical);
        }
        else
        {
            ASSERT_EQ(index, indexPhysical);
        }
    }

    // 65 ブロック目から 1 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            64,
            1
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(64), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 14), indexHeadDst);
    // 1 ブロックだけ割当てする場合は末尾の確保は不要です。
    EXPECT_EQ(false, isNeedCopyTail);
    EXPECT_EQ(0, indexTailSrc);
    EXPECT_EQ(0, indexTailDst);

    // ReadMappingEntry と ReadMappingEntries の読み込み内容が一致することをテストします。
    NNT_ASSERT_RESULT_SUCCESS(
        ReadMappingEntries(entryBuffer, GetEntryCount(), 0, GetEntryCount()));
    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        MappingEntry entry;
        NNT_ASSERT_RESULT_SUCCESS(ReadMappingEntry(&entry, index));

        ASSERT_EQ(0, memcmp(&entry, &entryBuffer[index], sizeof(MappingEntry)));
    }
} // NOLINT(impl/function_size)

// 予約領域の限界まで MarkUpdate した後の MappingEntry をテストします。
TEST_P(MappingTableReadEntryTest, ReadMarkUpdateLimit)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    for( uint32_t i = 0; i < GetReservedCount(); ++i )
    {
        // 1 ブロックずつ割当てます。
        const uint32_t index = 4 + i * 3;
        NN_ASSERT(index < GetEntryCount());
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->MarkUpdate(
                &isNeedCopyHead,
                &indexHeadSrc,
                &indexHeadDst,
                &isNeedCopyTail,
                &indexTailSrc,
                &indexTailDst,
                index,
                1
            )
        );
        EXPECT_EQ(true, isNeedCopyHead);
        EXPECT_EQ(PhysicalIndex(4 + i * 3), indexHeadSrc);
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + i), indexHeadDst);
        EXPECT_EQ(false, isNeedCopyTail);
        EXPECT_EQ(0, indexTailSrc);
        EXPECT_EQ(0, indexTailDst);
    }

    // これ以上反転できないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultMappingTableFull,
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            GetEntryCount() - 2,
            1
        )
    );

    // ReadMappingEntry と ReadMappingEntries の読み込み内容が一致することをテストします。
    auto buffer = nnt::fs::util::AllocateBuffer(sizeof(MappingEntry) * GetEntryCount());
    MappingEntry* entryBuffer = reinterpret_cast<MappingEntry*>(buffer.get());
    NNT_ASSERT_RESULT_SUCCESS(
        ReadMappingEntries(entryBuffer, GetEntryCount(), 0, GetEntryCount()));

    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        MappingEntry entry;
        NNT_ASSERT_RESULT_SUCCESS(ReadMappingEntry(&entry, index));

        ASSERT_EQ(0, memcmp(&entry, &entryBuffer[index], sizeof(MappingEntry)));
    }
}

INSTANTIATE_TEST_CASE_P(
    MappingTableTest,
    MappingTableReadEntryTest,
    ::testing::ValuesIn(MakeReadEntryTestParam())
);



namespace {

struct CommitTestParam
{
    int64_t offset;
    uint32_t countMapEntry;
    uint32_t countReserved;
};

nnt::fs::util::Vector<CommitTestParam> MakeCommitTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<CommitTestParam> result;

    {
        CommitTestParam data;
        data.offset = 0;
        data.countMapEntry = 128;
        data.countReserved = 32;
        result.push_back(data);
    }

    {
        CommitTestParam data;
        data.offset = 192;
        data.countMapEntry = 163;
        data.countReserved = 35;
        result.push_back(data);
    }

    return result;
}

}

// Commit 後の論理/物理インデックスの変化を検証するテストです。
// MappingTable の論理/物理インデックス割当てアルゴリズムに強く依存しています。
// 割当てアルゴリズム側を変更した場合、テストのパラメータを変える必要がありえます。
class MappingTableCommitTest : public ::testing::TestWithParam< CommitTestParam >
{
protected:
    // 物理インデックスを取得します。
    static uint32_t PhysicalIndex(uint32_t index) NN_NOEXCEPT
    {
        return nn::fssystem::save::MappingTable::PhysicalIndex(index);
    }

    // エントリ数を取得します。
    uint32_t GetEntryCount() const NN_NOEXCEPT
    {
        return m_CountMapEntry;
    }

    // 予約エントリ数を取得します。
    uint32_t GetReservedCount() const NN_NOEXCEPT
    {
        return m_CountReserved;
    }

    // セットアップします。
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        static const int64_t SizeControlArea =
            sizeof(nn::fssystem::save::MappingTable::ControlArea);

        const CommitTestParam& param = GetParam();

        m_Offset = param.offset;
        m_CountMapEntry = param.countMapEntry;
        m_CountReserved = param.countReserved;

        NN_ASSERT(GetEntryCount() >= 64);
        NN_ASSERT(GetEntryCount() <= 192);
        NN_ASSERT(GetReservedCount() >= 32);
        NN_ASSERT(GetReservedCount() <= 40);

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::MappingTable::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetEntryCount(),
            GetReservedCount()
        );

        m_StorageControlArea.Initialize(SizeControlArea + m_Offset);
        m_StorageTable.Initialize(sizeTable + m_Offset);
        m_StorageBitmapUpdatedPhysical.Initialize(sizeBitmapUpdatedPhysical + m_Offset);
        m_StorageBitmapUpdatedVirtual.Initialize(sizeBitmapUpdatedVirtual + m_Offset);
        m_StorageBitmapUnassigned.Initialize(sizeBitmapUnassigned + m_Offset);

        // テスト中に関係ない範囲に書き込まないかを確認するためのデータを書き込みます。
        if( m_Offset > 0 )
        {
            m_StorageControlArea.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageTable.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUpdatedPhysical.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUpdatedVirtual.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUnassigned.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
        }

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_Offset,
            SizeControlArea
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_Offset,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_Offset,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_Offset,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_Offset,
            sizeBitmapUnassigned
        );

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::MappingTable::Format(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                GetEntryCount(),
                GetReservedCount()
            )
        );

        m_Table.reset(new nn::fssystem::save::MappingTable());
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned
            )
        );
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 関係ない範囲に書き込みを行なっていないか確認します。
        if( m_Offset > 0 )
        {
            ASSERT_EQ(
                true,
                m_StorageControlArea.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageTable.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUpdatedPhysical.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUpdatedVirtual.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUnassigned.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
        }

        EXPECT_EQ(true, m_StorageControlArea.CheckValid());
        EXPECT_EQ(true, m_StorageTable.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedPhysical.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedVirtual.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUnassigned.CheckValid());
    }

    // 再マウントを行ないます。
    nn::Result Remount() NN_NOEXCEPT
    {
        static const int64_t SizeControlArea =
            sizeof(nn::fssystem::save::MappingTable::ControlArea);

        m_Table.reset(new nn::fssystem::save::MappingTable());

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::MappingTable::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetEntryCount(),
            GetReservedCount()
        );

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_Offset,
            SizeControlArea
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_Offset,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_Offset,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_Offset,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_Offset,
            sizeBitmapUnassigned
        );

        NN_RESULT_DO(
            m_Table->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned
            )
        );
        NN_RESULT_SUCCESS;
    }

protected:
    std::unique_ptr<nn::fssystem::save::MappingTable> m_Table;

private:
    int64_t m_Offset;
    uint32_t m_CountMapEntry;
    uint32_t m_CountReserved;
    nnt::fs::util::SafeMemoryStorage m_StorageControlArea;
    nnt::fs::util::SafeMemoryStorage m_StorageTable;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedPhysical;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedVirtual;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUnassigned;
};

// MarkUpdate、Commit、MarkUpdate を実行するテスト1
TEST_P(MappingTableCommitTest, SimpleCommit)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // 6 ブロック目から 4 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            5,
            4
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(5), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount()), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(8), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 3), indexTailDst);

    // コミットします。
    NNT_ASSERT_RESULT_SUCCESS(m_Table->Commit());

    // 16 ブロック目から 4 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            15,
            4
        )
    );

    // 最初の MarkUpdate で使用中になっていた物理ブロックが
    // 再利用されて割当たることを確認します。
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(15), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(5), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(18), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(8), indexTailDst);

    // 31 ブロック目から 12 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            30,
            12
        )
    );

    // Commit 前に割当たっていた予約領域以降が割当たることを確認します。
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(30), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 4), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(41), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 15), indexTailDst);
}

// MarkUpdate、Commit、MarkUpdate を実行するテスト2
TEST_P(MappingTableCommitTest, DevideCommit)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // NOTE:
    // 分断が発生するような形に MarkUpdate を実行します。
    //
    // 1. ReservedSize の半分に近いブロック数割当てを行なう
    // 2. Commit を行なう
    // 3. 小さなブロック数の割当てを行なう
    // 4. ReservedSize の半分を超えるブロック数割り当てを行なう

    // 6 ブロック目から 12 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            5,
            12
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(5), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount()), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(16), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 11), indexTailDst);

    // 26 ブロック目から 7 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            26,
            5
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(26), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 12), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(30), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 16), indexTailDst);

    // コミットします。
    NNT_ASSERT_RESULT_SUCCESS(m_Table->Commit());

    // 最初の MarkUpdate で使用中になっていた物理ブロックが
    // 再利用されて割当たることを確認します。

    // 4 ブロック目から 7 ブロックを割当てます。
    // ・前半は一度も割当てが行なわれていない範囲を含めます
    // ・後半は一度割当たったことのある範囲を含めます
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            3,
            7
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(3), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(5), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 4), indexTailSrc);
    EXPECT_EQ(PhysicalIndex(11), indexTailDst);

    // 32 ブロック目から 24 ブロックを割当てます。
    NNT_ASSERT_RESULT_SUCCESS(
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            31,
            24
        )
    );
    EXPECT_EQ(true, isNeedCopyHead);
    EXPECT_EQ(PhysicalIndex(31), indexHeadSrc);
    EXPECT_EQ(PhysicalIndex(GetEntryCount() + 17), indexHeadDst);
    EXPECT_EQ(true, isNeedCopyTail);
    EXPECT_EQ(PhysicalIndex(54), indexTailSrc);
    EXPECT_GT(PhysicalIndex(GetEntryCount()), indexTailDst);
}

// 限界まで MarkUpdate、Commit、限界まで MarkUpdate を実行するテスト
TEST_P(MappingTableCommitTest, MarkUpdateLimitAndCommit)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // 予約領域を 1 ブロックずつ割当てます。
    for( uint32_t i = 0; i < GetReservedCount(); ++i )
    {
        const uint32_t index = 4 + i * 3;
        NN_ASSERT(index < GetEntryCount());
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->MarkUpdate(
                &isNeedCopyHead,
                &indexHeadSrc,
                &indexHeadDst,
                &isNeedCopyTail,
                &indexTailSrc,
                &indexTailDst,
                index,
                1
            )
        );
        EXPECT_EQ(true, isNeedCopyHead);
        EXPECT_EQ(PhysicalIndex(4 + i * 3), indexHeadSrc);
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + i), indexHeadDst);
        EXPECT_EQ(false, isNeedCopyTail);
        EXPECT_EQ(0, indexTailSrc);
        EXPECT_EQ(0, indexTailDst);
    }

    // コミットします。
    NNT_ASSERT_RESULT_SUCCESS(m_Table->Commit());

    // 再度予約領域を 1 ブロックずつ割当てます。
    for( uint32_t i = 0; i < GetReservedCount(); ++i )
    {
        const uint32_t index = 4 + i * 3;
        NN_ASSERT(index < GetEntryCount());
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->MarkUpdate(
                &isNeedCopyHead,
                &indexHeadSrc,
                &indexHeadDst,
                &isNeedCopyTail,
                &indexTailSrc,
                &indexTailDst,
                index,
                1
            )
        );
        EXPECT_EQ(true, isNeedCopyHead);
        // はじめに割当てした時と物理インデックスが逆転することを確認します。
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + i), indexHeadSrc);
        EXPECT_EQ(PhysicalIndex(4 + i * 3), indexHeadDst);
        EXPECT_EQ(false, isNeedCopyTail);
        EXPECT_EQ(0, indexTailSrc);
        EXPECT_EQ(0, indexTailDst);
    }
}

// Commit せずに再マウントした時の挙動を確認するテスト
TEST_P(MappingTableCommitTest, RemountWithoutCommit)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // 予約領域を 1 ブロックずつ割当てます。
    for( uint32_t i = 0; i < GetReservedCount(); ++i )
    {
        const uint32_t index = 4 + i * 3;
        NN_ASSERT(index < GetEntryCount());
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->MarkUpdate(
                &isNeedCopyHead,
                &indexHeadSrc,
                &indexHeadDst,
                &isNeedCopyTail,
                &indexTailSrc,
                &indexTailDst,
                index,
                1
            )
        );
        EXPECT_EQ(true, isNeedCopyHead);
        EXPECT_EQ(PhysicalIndex(4 + i * 3), indexHeadSrc);
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + i), indexHeadDst);
        EXPECT_EQ(false, isNeedCopyTail);
        EXPECT_EQ(0, indexTailSrc);
        EXPECT_EQ(0, indexTailDst);
    }

    // これ以上ブロックを割当てできないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultMappingTableFull,
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            1,
            1
        )
    );

    // 再マウントします。
    NNT_ASSERT_RESULT_SUCCESS(Remount());

    // 再マウントしてもブロックを割当てできないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultMappingTableFull,
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            1,
            1
        )
    );

    // コミットします。
    NNT_ASSERT_RESULT_SUCCESS(m_Table->Commit());

    // 再度予約領域を 1 ブロックずつ割当てます。
    for( uint32_t i = 0; i < GetReservedCount(); ++i )
    {
        const uint32_t index = 4 + i * 3;
        NN_ASSERT(index < GetEntryCount());
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->MarkUpdate(
                &isNeedCopyHead,
                &indexHeadSrc,
                &indexHeadDst,
                &isNeedCopyTail,
                &indexTailSrc,
                &indexTailDst,
                index,
                1
            )
        );
        EXPECT_EQ(true, isNeedCopyHead);
        // はじめに割当てした時と物理インデックスが逆転することを確認します。
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + i), indexHeadSrc);
        EXPECT_EQ(PhysicalIndex(4 + i * 3), indexHeadDst);
        EXPECT_EQ(false, isNeedCopyTail);
        EXPECT_EQ(0, indexTailSrc);
        EXPECT_EQ(0, indexTailDst);
    }
}

INSTANTIATE_TEST_CASE_P(
    MappingTableTest,
    MappingTableCommitTest,
    ::testing::ValuesIn(MakeCommitTestParam())
);



namespace {

struct IteratorTestParam
{
    int64_t offset;
    uint32_t countMapEntry;
    uint32_t countReserved;
};

nnt::fs::util::Vector<IteratorTestParam> MakeIteratorTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<IteratorTestParam> result;

    {
        IteratorTestParam data;
        data.offset = 0;
        data.countMapEntry = 128;
        data.countReserved = 32;
        result.push_back(data);
    }

    {
        IteratorTestParam data;
        data.offset = 192;
        data.countMapEntry = 163;
        data.countReserved = 35;
        result.push_back(data);
    }

    return result;
}

}

// イテレーション時の連続ブロック数を検証するテストです。
// MappingTable の論理/物理インデックス割当てアルゴリズムに強く依存しています。
// 割当てアルゴリズム側を変更した場合、テストのパラメータを変える必要がありえます。
class MappingTableIteratorTest : public ::testing::TestWithParam< IteratorTestParam >
{
protected:
    typedef nn::fssystem::save::MappingTable::Iterator MappingTableIterator;

protected:
    // 物理インデックスを取得します。
    static uint32_t PhysicalIndex(uint32_t index) NN_NOEXCEPT
    {
        return nn::fssystem::save::MappingTable::PhysicalIndex(index);
    }

    // エントリ数を取得します。
    uint32_t GetEntryCount() const NN_NOEXCEPT
    {
        return m_CountMapEntry;
    }

    // 予約エントリ数を取得します。
    uint32_t GetReservedCount() const NN_NOEXCEPT
    {
        return m_CountReserved;
    }

    // セットアップします。
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        static const int64_t SizeControlArea =
            sizeof(nn::fssystem::save::MappingTable::ControlArea);

        const IteratorTestParam& param = GetParam();

        m_Offset = param.offset;
        m_CountMapEntry = param.countMapEntry;
        m_CountReserved = param.countReserved;

        NN_ASSERT(GetEntryCount() >= 64);
        NN_ASSERT(GetEntryCount() <= 192);
        NN_ASSERT(GetReservedCount() >= 32);
        NN_ASSERT(GetReservedCount() <= 40);

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::MappingTable::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetEntryCount(),
            GetReservedCount()
        );

        m_StorageControlArea.Initialize(SizeControlArea + m_Offset);
        m_StorageTable.Initialize(sizeTable + m_Offset);
        m_StorageBitmapUpdatedPhysical.Initialize(sizeBitmapUpdatedPhysical + m_Offset);
        m_StorageBitmapUpdatedVirtual.Initialize(sizeBitmapUpdatedVirtual + m_Offset);
        m_StorageBitmapUnassigned.Initialize(sizeBitmapUnassigned + m_Offset);

        // テスト中に関係ない範囲に書き込まないかを確認するためのデータを書き込みます。
        if( m_Offset > 0 )
        {
            m_StorageControlArea.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageTable.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUpdatedPhysical.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUpdatedVirtual.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUnassigned.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
        }

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_Offset,
            SizeControlArea
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_Offset,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_Offset,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_Offset,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_Offset,
            sizeBitmapUnassigned
        );

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::MappingTable::Format(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                GetEntryCount(),
                GetReservedCount()
            )
        );

        m_Table.reset(new nn::fssystem::save::MappingTable());
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned
            )
        );
    }

    // 破棄します。
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 関係ない範囲に書き込みを行なっていないか確認します。
        if( m_Offset > 0 )
        {
            ASSERT_EQ(
                true,
                m_StorageControlArea.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageTable.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUpdatedPhysical.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUpdatedVirtual.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUnassigned.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
        }

        EXPECT_EQ(true, m_StorageControlArea.CheckValid());
        EXPECT_EQ(true, m_StorageTable.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedPhysical.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedVirtual.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUnassigned.CheckValid());
    }

    // 再マウントを行ないます。
    nn::Result Remount() NN_NOEXCEPT
    {
        static const int64_t SizeControlArea =
            sizeof(nn::fssystem::save::MappingTable::ControlArea);

        m_Table.reset(new nn::fssystem::save::MappingTable());

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::MappingTable::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetEntryCount(),
            GetReservedCount()
        );

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_Offset,
            SizeControlArea
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_Offset,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_Offset,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_Offset,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_Offset,
            sizeBitmapUnassigned
        );

        NN_RESULT_DO(
            m_Table->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned
            )
        );

        NN_RESULT_SUCCESS;
    }

protected:
    std::unique_ptr<nn::fssystem::save::MappingTable> m_Table;

private:
    int64_t m_Offset;
    uint32_t m_CountMapEntry;
    uint32_t m_CountReserved;
    nnt::fs::util::SafeMemoryStorage m_StorageControlArea;
    nnt::fs::util::SafeMemoryStorage m_StorageTable;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedPhysical;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedVirtual;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUnassigned;
};

// MarkUpdate 後の先頭からのイテレーションの挙動を確認するテスト
TEST_P(MappingTableIteratorTest, IterateSimple)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    // 6 ブロック目から 4 ブロックを割当てます。
    {
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->MarkUpdate(
                &isNeedCopyHead,
                &indexHeadSrc,
                &indexHeadDst,
                &isNeedCopyTail,
                &indexTailSrc,
                &indexTailDst,
                5,
                4
            )
        );
        EXPECT_EQ(true, isNeedCopyHead);
        EXPECT_EQ(PhysicalIndex(5), indexHeadSrc);
        EXPECT_EQ(PhysicalIndex(GetEntryCount()), indexHeadDst);
        EXPECT_EQ(true, isNeedCopyTail);
        EXPECT_EQ(PhysicalIndex(8), indexTailSrc);
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + 3), indexTailDst);
    }

    // イテレーターの挙動を確認します。
    {
        MappingTableIterator it;
        NNT_ASSERT_RESULT_SUCCESS(m_Table->MakeIterator(&it, 5, 4));
        EXPECT_EQ(PhysicalIndex(GetEntryCount()), it.GetPhysicalIndex());
        EXPECT_EQ(4, it.GetBlockCount());
    }

    // 16 ブロック目から 4 ブロックを割当てます。
    {
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->MarkUpdate(
                &isNeedCopyHead,
                &indexHeadSrc,
                &indexHeadDst,
                &isNeedCopyTail,
                &indexTailSrc,
                &indexTailDst,
                15,
                4
            )
        );
        // 最初の MarkUpdate で使用中になっていた物理ブロックが
        // 再利用されて割当たることを確認します。
        EXPECT_EQ(true, isNeedCopyHead);
        EXPECT_EQ(PhysicalIndex(15), indexHeadSrc);
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + 4), indexHeadDst);
        EXPECT_EQ(true, isNeedCopyTail);
        EXPECT_EQ(PhysicalIndex(18), indexTailSrc);
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + 7), indexTailDst);
    }

    // イテレーターの挙動を確認します。
    {
        MappingTableIterator it;
        NNT_ASSERT_RESULT_SUCCESS(m_Table->MakeIterator(&it, 5, 4));
        EXPECT_EQ(PhysicalIndex(GetEntryCount()), it.GetPhysicalIndex());
        EXPECT_EQ(4, it.GetBlockCount());
    }
    {
        MappingTableIterator it;
        NNT_ASSERT_RESULT_SUCCESS(m_Table->MakeIterator(&it, 15, 4));
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + 4), it.GetPhysicalIndex());
        EXPECT_EQ(4, it.GetBlockCount());
    }
}

INSTANTIATE_TEST_CASE_P(
    MappingTableTest,
    MappingTableIteratorTest,
    ::testing::ValuesIn(MakeIteratorTestParam())
);



namespace {

struct ExpandTestParam
{
    int64_t offset;
    uint32_t countMapEntry;
    uint32_t countReserved;
    uint32_t countMapEntryExpandable;
    uint32_t countReservedExpandable;
};

nnt::fs::util::Vector<ExpandTestParam> MakeExpandTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<ExpandTestParam> result;

    {
        ExpandTestParam data;
        data.offset = 0;
        data.countMapEntry = 128;
        data.countReserved = 32;
        data.countMapEntryExpandable = 16;
        data.countReservedExpandable = 0;
        result.push_back(data);
    }

    {
        ExpandTestParam data;
        data.offset = 32;
        data.countMapEntry = 160;
        data.countReserved = 42;
        data.countMapEntryExpandable = 32;
        data.countReservedExpandable = 5;
        result.push_back(data);
    }

    return result;
}

}

class MappingTableExpandTest : public ::testing::TestWithParam< ExpandTestParam >
{
protected:
    // 物理インデックスを取得します。
    static uint32_t PhysicalIndex(uint32_t index) NN_NOEXCEPT
    {
        return nn::fssystem::save::MappingTable::PhysicalIndex(index);
    }

    // エントリ数を取得します。
    uint32_t GetEntryCount() const NN_NOEXCEPT
    {
        return m_CountMapEntry;
    }

    // 予約エントリ数を取得します。
    uint32_t GetReservedCount() const NN_NOEXCEPT
    {
        return m_CountReserved;
    }

    // テストで拡大可能なエントリ数を取得します。
    uint32_t GetExpandableEntryCount() const NN_NOEXCEPT
    {
        return m_CountMaxExpandableEntry;
    }

    // テストで拡大可能なエントリ数を取得します。
    uint32_t GetExpandableReservedCount() const NN_NOEXCEPT
    {
        return m_CountMaxExpandableReserved;
    }

    // セットアップします。
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        static const int64_t SizeControlArea =
            sizeof(nn::fssystem::save::MappingTable::ControlArea);

        const ExpandTestParam& param = GetParam();
        m_Offset = param.offset;
        m_CountMapEntry = param.countMapEntry;
        m_CountReserved = param.countReserved;
        m_CountMaxExpandableEntry = param.countMapEntryExpandable;
        m_CountMaxExpandableReserved = param.countReservedExpandable;

        NN_ASSERT(GetEntryCount() >= 64);
        NN_ASSERT(GetEntryCount() <= 192);
        NN_ASSERT(GetReservedCount() >= 32);
        NN_ASSERT(GetReservedCount() <= 48);

        int64_t sizeTableMax;
        int64_t sizeBitmapMaxUpdatedPhysical;
        int64_t sizeBitmapMaxUpdatedVirtual;
        int64_t sizeBitmapMaxUnassigned;
        nn::fssystem::save::MappingTable::QueryMappingMetaSize(
            &sizeTableMax,
            &sizeBitmapMaxUpdatedPhysical,
            &sizeBitmapMaxUpdatedVirtual,
            &sizeBitmapMaxUnassigned,
            GetEntryCount() + GetExpandableEntryCount(),
            GetReservedCount() + GetExpandableReservedCount()
        );

        m_StorageControlArea.Initialize(SizeControlArea + m_Offset);
        m_StorageTable.Initialize(sizeTableMax + m_Offset);
        m_StorageBitmapUpdatedPhysical.Initialize(sizeBitmapMaxUpdatedPhysical + m_Offset);
        m_StorageBitmapUpdatedVirtual.Initialize(sizeBitmapMaxUpdatedVirtual + m_Offset);
        m_StorageBitmapUnassigned.Initialize(sizeBitmapMaxUnassigned + m_Offset);

        // テスト中に関係ない範囲に書き込まないかを確認するためのデータを書き込みます。
        if( m_Offset > 0 )
        {
            m_StorageControlArea.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageTable.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUpdatedPhysical.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUpdatedVirtual.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
            m_StorageBitmapUnassigned.FillBuffer(0, static_cast<size_t>(m_Offset), 0x66);
        }

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::MappingTable::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetEntryCount(),
            GetReservedCount()
        );

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_Offset,
            SizeControlArea
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_Offset,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_Offset,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_Offset,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_Offset,
            sizeBitmapUnassigned
        );

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::MappingTable::Format(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                GetEntryCount(),
                GetReservedCount()
            )
        );

        m_Table.reset(new nn::fssystem::save::MappingTable());
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned
            )
        );
    }

    // 破棄します。
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 関係ない範囲に書き込みを行なっていないか確認します。
        if( m_Offset > 0 )
        {
            ASSERT_EQ(
                true,
                m_StorageControlArea.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageTable.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUpdatedPhysical.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUpdatedVirtual.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUnassigned.VerifyBuffer(0, static_cast<size_t>(m_Offset), 0x66)
            );
        }

        EXPECT_EQ(true, m_StorageControlArea.CheckValid());
        EXPECT_EQ(true, m_StorageTable.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedPhysical.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedVirtual.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUnassigned.CheckValid());
    }

    // マッピングテーブルを拡張します。
    void Expand(uint32_t countEntryAdd, uint32_t countReservedAdd) NN_NOEXCEPT
    {
        static const int64_t SizeControlArea =
            sizeof(nn::fssystem::save::MappingTable::ControlArea);

        NN_SDK_ASSERT(countEntryAdd <= GetExpandableEntryCount());
        NN_SDK_ASSERT(countReservedAdd <= GetExpandableReservedCount());

        m_Table.reset(new nn::fssystem::save::MappingTable());

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::MappingTable::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetEntryCount() + countEntryAdd,
            GetReservedCount() + countReservedAdd
        );

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_Offset,
            SizeControlArea
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_Offset,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_Offset,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_Offset,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_Offset,
            sizeBitmapUnassigned
        );

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::MappingTable::Expand(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                GetEntryCount() + countEntryAdd,
                GetReservedCount() + countReservedAdd
            )
        );

        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned
            )
        );
    }

protected:
    std::unique_ptr<nn::fssystem::save::MappingTable> m_Table;

private:
    int64_t m_Offset;
    uint32_t m_CountMapEntry;
    uint32_t m_CountReserved;
    uint32_t m_CountMaxExpandableEntry;
    uint32_t m_CountMaxExpandableReserved;
    nnt::fs::util::SafeMemoryStorage m_StorageControlArea;
    nnt::fs::util::SafeMemoryStorage m_StorageTable;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedPhysical;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedVirtual;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUnassigned;
};

// 拡張後の物理/論理インデックスの対応を確認します。
TEST_P(MappingTableExpandTest, GetPhysicalIndex)
{
    NN_SDK_ASSERT(GetExpandableEntryCount() >= 15);

    uint32_t indexPhysical;

    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        NNT_ASSERT_RESULT_SUCCESS(m_Table->GetPhysicalIndex(&indexPhysical, index));
        ASSERT_EQ(index, indexPhysical);
    }

    // 範囲外アクセスできないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultOutOfRange,
        m_Table->GetPhysicalIndex(&indexPhysical, GetEntryCount())
    );

    // エントリーを拡張します。
    Expand(10, 0);

    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        NNT_ASSERT_RESULT_SUCCESS(m_Table->GetPhysicalIndex(&indexPhysical, index));
        ASSERT_EQ(index, indexPhysical);
    }

    // 拡張したエントリーの物理インデックスは拡張した領域になっていることを確認します。
    for( uint32_t index = 0; index < 10; ++index )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->GetPhysicalIndex(&indexPhysical, index + GetEntryCount()));
        ASSERT_EQ(index + GetEntryCount() + GetReservedCount(), indexPhysical);
    }

    // 範囲外アクセスできないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultOutOfRange,
        m_Table->GetPhysicalIndex(&indexPhysical, GetEntryCount() + 10)
    );

    // エントリーを拡張します。
    Expand(15, 0);

    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        NNT_ASSERT_RESULT_SUCCESS(m_Table->GetPhysicalIndex(&indexPhysical, index));
        ASSERT_EQ(index, indexPhysical);
    }

    // 拡張したエントリーの物理インデックスは拡張した領域になっていることを確認します。
    for( uint32_t index = 0; index < 15; ++index )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->GetPhysicalIndex(&indexPhysical, index + GetEntryCount()));
        ASSERT_EQ(index + GetEntryCount() + GetReservedCount(), indexPhysical);
    }

    // 範囲外アクセスできないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultOutOfRange,
        m_Table->GetPhysicalIndex(&indexPhysical, GetEntryCount() + 15)
    );
}

// 予約領域の限界まで MarkUpdate を実行するテスト
TEST_P(MappingTableExpandTest, MarkUpdateLimit)
{
    NN_SDK_ASSERT(GetExpandableEntryCount() >= 5);

    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    for( uint32_t i = 0; i < GetReservedCount(); ++i )
    {
        // 1 ブロックずつ割当てます。
        const uint32_t index = 4 + i * 3;
        NN_ASSERT(index < GetEntryCount());
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->MarkUpdate(
                &isNeedCopyHead,
                &indexHeadSrc,
                &indexHeadDst,
                &isNeedCopyTail,
                &indexTailSrc,
                &indexTailDst,
                index,
                1
            )
        );
        EXPECT_EQ(true, isNeedCopyHead);
        EXPECT_EQ(PhysicalIndex(4 + i * 3), indexHeadSrc);
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + i), indexHeadDst);
        EXPECT_EQ(false, isNeedCopyTail);
        EXPECT_EQ(0, indexTailSrc);
        EXPECT_EQ(0, indexTailDst);
    }

    // これ以上反転できないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultMappingTableFull,
        m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            GetEntryCount() - 2,
            1
        )
    );

    // 未割当領域を拡張します。
    if( (GetExpandableEntryCount() >= 5) && (GetExpandableReservedCount() > 0) )
    {
        Expand(5, 1);

        // 拡張したことでさらに反転できるようになっていることを確認します。
        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->MarkUpdate(
                &isNeedCopyHead,
                &indexHeadSrc,
                &indexHeadDst,
                &isNeedCopyTail,
                &indexTailSrc,
                &indexTailDst,
                GetEntryCount() + 3,
                1
            )
        );
        EXPECT_EQ(true, isNeedCopyHead);
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + GetReservedCount() + 3), indexHeadSrc);
        EXPECT_EQ(PhysicalIndex(GetEntryCount() + GetReservedCount() + 5), indexHeadDst);
        EXPECT_EQ(false, isNeedCopyTail);
        EXPECT_EQ(0, indexTailSrc);
        EXPECT_EQ(0, indexTailDst);
    }
}

INSTANTIATE_TEST_CASE_P(
    MappingTableTest,
    MappingTableExpandTest,
    ::testing::ValuesIn(MakeExpandTestParam())
);



namespace {

struct AgingTestParam
{
    int64_t offset;
    uint32_t countMapEntry;
    uint32_t countReserved;
};

nnt::fs::util::Vector<AgingTestParam> MakeAgingTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<AgingTestParam> result;

    {
        AgingTestParam data;
        data.offset = 0;
        data.countMapEntry = 128;
        data.countReserved = 32;
        result.push_back(data);
    }

    return result;
}

}

class MappingTableAgingTest : public ::testing::TestWithParam< AgingTestParam >
{
protected:
    // 物理インデックスを取得します。
    static uint32_t PhysicalIndex(uint32_t index) NN_NOEXCEPT
    {
        return nn::fssystem::save::MappingTable::PhysicalIndex(index);
    }

    // エントリ数を取得します。
    uint32_t GetEntryCount() const NN_NOEXCEPT
    {
        return m_CountMapEntry;
    }

    // 予約エントリ数を取得します。
    uint32_t GetReservedCount() const NN_NOEXCEPT
    {
        return m_CountReserved;
    }

    // セットアップします。
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        static const int64_t SizeControlArea =
            sizeof(nn::fssystem::save::MappingTable::ControlArea);

        const AgingTestParam& param = GetParam();
        m_Offset = param.offset;
        m_CountMapEntry = param.countMapEntry;
        m_CountReserved = param.countReserved;

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::MappingTable::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetEntryCount(),
            GetReservedCount()
        );

        m_StorageControlArea.Initialize(SizeControlArea + m_Offset);
        m_StorageTable.Initialize(sizeTable + m_Offset);
        m_StorageBitmapUpdatedPhysical.Initialize(sizeBitmapUpdatedPhysical + m_Offset);
        m_StorageBitmapUpdatedVirtual.Initialize(sizeBitmapUpdatedVirtual + m_Offset);
        m_StorageBitmapUnassigned.Initialize(sizeBitmapUnassigned + m_Offset);

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_Offset,
            SizeControlArea
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_Offset,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_Offset,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_Offset,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_Offset,
            sizeBitmapUnassigned
        );

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::MappingTable::Format(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                GetEntryCount(),
                GetReservedCount()
            )
        );

        m_Table.reset(new nn::fssystem::save::MappingTable());

        NNT_ASSERT_RESULT_SUCCESS(
            m_Table->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned
            )
        );
    }

    // 破棄します。
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        EXPECT_EQ(true, m_StorageControlArea.CheckValid());
        EXPECT_EQ(true, m_StorageTable.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedPhysical.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedVirtual.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUnassigned.CheckValid());
    }

    // 再マウントを行ないます。
    nn::Result Remount() NN_NOEXCEPT
    {
        static const int64_t SizeControlArea =
            sizeof(nn::fssystem::save::MappingTable::ControlArea);

        m_Table.reset(new nn::fssystem::save::MappingTable());

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::MappingTable::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetEntryCount(),
            GetReservedCount()
        );

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_Offset,
            SizeControlArea
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_Offset,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_Offset,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_Offset,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_Offset,
            sizeBitmapUnassigned
        );

        NN_RESULT_DO(
            m_Table->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned
            )
        );

        NN_RESULT_SUCCESS;
    }

protected:
    std::unique_ptr<nn::fssystem::save::MappingTable> m_Table;

private:
    int64_t m_Offset;
    uint32_t m_CountMapEntry;
    uint32_t m_CountReserved;
    nnt::fs::util::SafeMemoryStorage m_StorageControlArea;
    nnt::fs::util::SafeMemoryStorage m_StorageTable;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedPhysical;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedVirtual;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUnassigned;
};

// MarkUpdate/Commit を繰り返して落ちないことを確認します。
TEST_P(MappingTableAgingTest, MarkUpdateWithRandomCommit)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int loop = 0; loop < 10000; ++loop )
    {
        const uint32_t index = std::uniform_int_distribution<uint32_t>(
            0, GetEntryCount() - 1)(mt);
        const uint32_t count = std::uniform_int_distribution<uint32_t>(
            1, std::min(GetReservedCount() - 1, GetEntryCount() - index))(mt);

        nn::Result result = m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            index,
            count
        );
        if( !result.IsSuccess() )
        {
            if( nn::fs::ResultMappingTableFull::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(m_Table->Commit());
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(result)
                    << "loop = " << loop
                    << ", index = " << index
                    << ", count = " << count;
            }
        }

        // ランダムなタイミングで Commit を実行します。
        if( std::uniform_int_distribution<uint32_t>(0, 100)(mt) < 10 )
        {
            NNT_ASSERT_RESULT_SUCCESS(m_Table->Commit());
        }
    }

    // 物理インデックス取得しても落ちないこと、
    // インデックスが範囲内に収まっていることを確認します。
    for( uint32_t index = 0; index < GetEntryCount(); ++index )
    {
        uint32_t indexPhysical;
        NNT_ASSERT_RESULT_SUCCESS(m_Table->GetPhysicalIndex(&indexPhysical, index));
        ASSERT_GE(indexPhysical, static_cast<uint32_t>(0));
        ASSERT_GT(GetEntryCount() + GetReservedCount(), indexPhysical);
    }
}

// ランダムな操作を繰り返して落ちないことを確認します。
TEST_P(MappingTableAgingTest, RandomOperation)
{
    bool isNeedCopyHead;
    uint32_t indexHeadSrc;
    uint32_t indexHeadDst;
    bool isNeedCopyTail;
    uint32_t indexTailSrc;
    uint32_t indexTailDst;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int loop = 0; loop < 10000; ++loop )
    {
        const uint32_t index = std::uniform_int_distribution<uint32_t>(
            0, GetEntryCount() - 1)(mt);
        const uint32_t count = std::uniform_int_distribution<uint32_t>(
            1, std::min(GetReservedCount() - 1, GetEntryCount() - index))(mt);

        nn::Result result = m_Table->MarkUpdate(
            &isNeedCopyHead,
            &indexHeadSrc,
            &indexHeadDst,
            &isNeedCopyTail,
            &indexTailSrc,
            &indexTailDst,
            index,
            count
        );
        if( !result.IsSuccess() )
        {
            if( !nn::fs::ResultMappingTableFull::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(result)
                    << "loop = " << loop
                    << ", index = " << index
                    << ", count = " << count;
            }
        }

        // ランダムなタイミングで Commit を実行します。
        if( std::uniform_int_distribution<uint32_t>(0, 100)(mt) < 10 )
        {
            NNT_ASSERT_RESULT_SUCCESS(m_Table->Commit());
        }

        // ランダムなタイミングで物理インデックスの範囲チェックを行ないます。
        if( std::uniform_int_distribution<uint32_t>(0, 100)(mt) < 10 )
        {
            for( uint32_t indexEntry = 0; indexEntry < GetEntryCount(); ++indexEntry )
            {
                uint32_t indexPhysical;
                NNT_ASSERT_RESULT_SUCCESS(m_Table->GetPhysicalIndex(&indexPhysical, indexEntry));
                ASSERT_GE(indexPhysical, static_cast<uint32_t>(0));
                ASSERT_GT(GetEntryCount() + GetReservedCount(), indexPhysical);
            }
        }

        // ランダムなタイミングで再マウントします。
        if( std::uniform_int_distribution<uint32_t>(0, 100)(mt) < 5 )
        {
            NNT_ASSERT_RESULT_SUCCESS(Remount());
        }
    }
}

INSTANTIATE_TEST_CASE_P(
    MappingTableTest,
    MappingTableAgingTest,
    ::testing::ValuesIn(MakeAgingTestParam())
);
