﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/fssystem/save/fs_IntegrityVerificationStorage.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonStorageTests.h"

class FsIntegrityVerificationStorageTest : public ::testing::TestWithParam<bool>
{
};

// 完全性のテスト
TEST_P(FsIntegrityVerificationStorageTest, Simple)
{
    static const size_t TestSize = 8 * 1024 * 1024;
    static const int TestLoop = 20;
    static const int CacheCountMax = 1024;
    static const size_t CacheBufferSize = 256 * 1024;
    static const size_t CacheBlockSize = 4 * 1024;

    const bool isRealDataSign = GetParam();

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> buf1(TestSize);
    nnt::fs::util::Vector<char> buf2(TestSize);
    nnt::fs::util::Vector<char> buf3(TestSize);
    nnt::fs::util::Vector<char> cache(CacheBufferSize);
    char* ptr1 = &buf1[0];
    char* ptr2 = &buf2[0];
    char* ptr3 = &buf3[0];

    nn::fssystem::FileSystemBufferManager cacheBuffer;
    NNT_ASSERT_RESULT_SUCCESS(
        cacheBuffer.Initialize(
            CacheCountMax,
            reinterpret_cast<uintptr_t>(&cache[0]),
            CacheBufferSize,
            CacheBlockSize
        )
    );

    for( size_t sizeMax = 1; sizeMax <= TestSize; )
    {
        NN_SDK_LOG("%d...", sizeMax);

        // 1ブロックの検証サイズは256～MaxBlockSizeの間で可変です。
        size_t sizeBlock = GenerateRandomBlockSize(256, MaxBlockSize);
        const size_t sizeHash = ((sizeMax + sizeBlock - 1) / sizeBlock) *
                                nn::fssystem::save::IntegrityVerificationStorage::HashSize;
        const size_t sizeMaxAligned = nn::util::align_up(sizeMax, sizeBlock);

        // ハッシュのデータ
        nnt::fs::util::SafeMemoryStorage baseStorageHash;
        baseStorageHash.Initialize(sizeHash);

        // ファイルボディ
        nnt::fs::util::SafeMemoryStorage baseStorageBody;
        baseStorageBody.Initialize(sizeMax);

        nn::fs::SubStorage storageHash(&baseStorageHash, 0, sizeHash);
        nn::fs::SubStorage storageBody(&baseStorageBody, 0, sizeMax);

        nn::fssystem::save::IntegrityVerificationStorage storageVerify;
        {
            nn::fs::HashSalt salt = {};
            std::generate(std::begin(salt.value), std::end(salt.value), [&]() NN_NOEXCEPT
            {
                return static_cast<char>(std::uniform_int_distribution<>(0, 255)(mt) - 128);
            });
            storageVerify.Initialize(
                storageHash,
                storageBody,
                sizeBlock,
                32,
                &cacheBuffer,
                salt,
                isRealDataSign,
                nn::fs::StorageType::StorageType_SaveData
            );
        }

        // データを書き込みます。
        // ブロックサイズで書き込んだときは必ずその値が戻ってきます。
        // (sizeMaxではなく)
        std::memset(ptr1, 0x00, TestSize);
        //std::memset(ptr2, 0x00, TestSize);
        std::memset(ptr1, 0xFF, sizeMax);
        NNT_ASSERT_RESULT_SUCCESS(
            storageVerify.Write(0, ptr1, sizeMaxAligned));

        // データを読み込んでベリファイテスト
        // ブロックサイズで読み込んだときは必ずその値が戻ってきます。
        // (sizeMaxではなく)
        NNT_ASSERT_RESULT_SUCCESS(storageVerify.Read(0, ptr2, sizeMaxAligned));
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, sizeMax));

        // ランダムなデータを書き込み、読み込んでベリファイします。
        for( size_t i = 0; i < sizeMax; ++i )
        {
            ptr1[i] = static_cast<char>(std::uniform_int_distribution<>(0, 255)(mt));
        }
        NNT_ASSERT_RESULT_SUCCESS(storageVerify.Write(0, ptr1, sizeMaxAligned));
        NNT_ASSERT_RESULT_SUCCESS(storageVerify.Read(0, ptr2, sizeMaxAligned));
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, sizeMaxAligned));

        // SHAの結果を比較します。
        {
            int64_t offsetHash = 0;
            for( size_t offset = 0; offset < sizeMax; offset += sizeBlock )
            {
                // 署名データを読み込みます。
                NNT_ASSERT_RESULT_SUCCESS(
                    baseStorageHash.Read(
                        offsetHash,
                        ptr3,
                        nn::fssystem::save::IntegrityVerificationStorage::HashSize
                    )
                );
                offsetHash += nn::fssystem::save::IntegrityVerificationStorage::HashSize;

                // 署名を計算します。
                char sign[nn::fssystem::save::IntegrityVerificationStorage::HashSize];
                nn::crypto::GenerateSha256Hash(sign, sizeof(sign), &ptr1[offset], sizeBlock);

                // 手元で計算したものとの合致を確認します。
                ASSERT_EQ(0, std::memcmp(sign, ptr3,
                                    nn::fssystem::save::IntegrityVerificationStorage::HashSize));
            }
        }

        for( int loop = 0; loop < TestLoop; ++loop )
        {
            int64_t offsetBreak = std::uniform_int_distribution<int64_t>(0, sizeMax - 1)(mt);

            // 検証の仕組みをバイパスしてファイルデータの一部分を書き換えます。
            // 署名が合わなくなるため、正しくリードできなくなります。
            char ch;
            NNT_ASSERT_RESULT_SUCCESS(baseStorageBody.Read(offsetBreak, &ch, 1));
            // 1バイトだけ書き換えます。
            char ch2 = ch ^ (1 << std::uniform_int_distribution<>(0, 7)(mt));
            NNT_ASSERT_RESULT_SUCCESS(baseStorageBody.Write(offsetBreak, &ch2, 1));

            int64_t offsetBreakBlock = (offsetBreak / sizeBlock) * sizeBlock;
            for( size_t offset = 0; offset < sizeMax; offset += sizeBlock )
            {
                // そのブロックのみ、正常にデータが読み込めなくなっていることを確認します。
                if( static_cast<int64_t>(offset) == offsetBreakBlock )
                {
                    if( isRealDataSign )
                    {
                        NNT_ASSERT_RESULT_FAILURE(
                            nn::fs::ResultIntegrityVerificationStorageCorrupted,
                            storageVerify.Read(offset, ptr2, sizeBlock)
                        );
                    }
                    else
                    {
                        NNT_ASSERT_RESULT_FAILURE(
                            nn::fs::ResultIntegrityVerificationStorageCorrupted,
                            storageVerify.Read(offset, ptr2, sizeBlock)
                        );
                    }
                    // 適当に書き込んで整合性が取れるようにします。
                    NNT_ASSERT_RESULT_SUCCESS(storageVerify.Write(offset, ptr2, sizeBlock));
                }
                else
                {
                    NNT_ASSERT_RESULT_SUCCESS(storageVerify.Read(offset, ptr2, sizeBlock));
                }
            }
        }

        NN_SDK_LOG("\n");

        sizeMax = (sizeMax * 2) + std::uniform_int_distribution<>(0, 255)(mt);
    }
} // NOLINT(impl/function_size)

// ベリファイのテスト
TEST_P(FsIntegrityVerificationStorageTest, Verify)
{
    static const size_t TestSize = 8 * 1024 * 1024;
    static const int TestLoop = 20;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4096;
    static const size_t CacheBufferSize = 1024 * 1024;

    const bool isRealDataSign = GetParam();

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> buf1(TestSize);
    nnt::fs::util::Vector<char> buf2(TestSize);
    nnt::fs::util::Vector<char> buf3(TestSize);
    nnt::fs::util::Vector<char> cache(CacheBufferSize);

    char* ptr1 = &buf1[0];
    char* ptr2 = &buf2[0];
    char* ptr3 = &buf3[0];

    for( size_t sizeMax = 1; sizeMax <= TestSize; )
    {
        NN_SDK_LOG("%d...", sizeMax);

        // 1ブロックの検証サイズは256～MaxBlockSizeの間で可変です。
        const size_t sizeBlock = GenerateRandomBlockSize(256, MaxBlockSize);
        const size_t sizeHash = ((sizeMax + sizeBlock - 1) / sizeBlock) *
                                nn::fssystem::save::IntegrityVerificationStorage::HashSize;
        const size_t sizeMaxAligned = nn::util::align_up(sizeMax, sizeBlock);

        // ハッシュのデータ
        nnt::fs::util::SafeMemoryStorage baseStorageHash;
        baseStorageHash.Initialize(sizeHash);

        // ファイルボディ
        nnt::fs::util::SafeMemoryStorage baseStorageBody;
        baseStorageBody.Initialize(sizeMax);

        // キャッシュ
        nn::fssystem::FileSystemBufferManager cacheBuffer;
        NNT_ASSERT_RESULT_SUCCESS(
            cacheBuffer.Initialize(
                CacheCountMax,
                reinterpret_cast<uintptr_t>(&cache[0]),
                CacheBufferSize,
                CacheBlockSize
            )
        );

        nn::os::Mutex locker(true);
        nn::fs::SubStorage storageHash(&baseStorageHash, 0, sizeHash);
        nn::fs::SubStorage storageBody(&baseStorageBody, 0, sizeMax);

        // 完全性検証
        nn::fssystem::save::IntegrityVerificationStorage storageVerify;
        {
            nn::fs::HashSalt salt = {};
            std::generate(std::begin(salt.value), std::end(salt.value), [&]() NN_NOEXCEPT
            {
                return static_cast<char>(std::uniform_int_distribution<>(0, 255)(mt) - 128);
            });
            storageVerify.Initialize(
                storageHash,
                storageBody,
                sizeBlock,
                nn::fssystem::save::IntegrityVerificationStorage::HashSize,
                &cacheBuffer,
                salt,
                isRealDataSign,
                nn::fs::StorageType::StorageType_SaveData
            );
        }

        // バッファー
        nn::fssystem::save::BlockCacheBufferedStorage storageBuffer;
        NNT_ASSERT_RESULT_SUCCESS(
            storageBuffer.Initialize(
                &cacheBuffer,
                &locker,
                &storageVerify,
                sizeMax,
                sizeBlock,
                storageBuffer.DefaultMaxCacheEntryCount,
                false,          // isReadDataCahce
                0x00,           // bufferLevel
                false,          // bKeepBurstMode
                nn::fs::StorageType::StorageType_SaveData
            )
        );

        std::memset(ptr1, 0x00, TestSize);
        std::memset(ptr2, 0x00, TestSize);
        std::memset(ptr3, 0x00, TestSize);

        // データを書き込みます。
        std::memset(ptr1, 0xFF, sizeMax);
        NNT_ASSERT_RESULT_SUCCESS(storageBuffer.Write(0, ptr1, sizeMaxAligned));

        // データを読み込んでベリファイテスト
        NNT_ASSERT_RESULT_SUCCESS(storageBuffer.Read(0, ptr2, sizeMaxAligned));
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, sizeMax));

        // ランダムなデータを書き込み、読み込んでベリファイします。
        for( size_t i = 0; i < sizeMax; ++i )
        {
            ptr1[i] = static_cast<char>(std::uniform_int_distribution<>(0, 255)(mt));
        }
        NNT_ASSERT_RESULT_SUCCESS(storageBuffer.Write(0, ptr1, sizeMaxAligned));
        NNT_ASSERT_RESULT_SUCCESS(storageBuffer.Read(0, ptr2, sizeMaxAligned));
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, sizeMax));

        std::memcpy(ptr3, ptr2, sizeMax);

        // アドレス、サイズともにランダムに書き込みます。
        for( size_t i = 0; i < TestLoop; ++i )
        {
            int64_t offset = std::uniform_int_distribution<int64_t>(0, sizeMax - 1)(mt);
            size_t size = std::uniform_int_distribution<size_t>(
                            0, sizeMax - static_cast<size_t>(offset) - 1)(mt);

            std::memset(ptr1, i & 0xFF, size);
            std::memcpy(ptr3 + offset, ptr1, size);
            NNT_ASSERT_RESULT_SUCCESS(storageBuffer.Write(offset, ptr1, size));
        }

        NNT_ASSERT_RESULT_SUCCESS(storageBuffer.Read(0, ptr1, TestSize));
        ASSERT_EQ(0, std::memcmp(ptr1, ptr3, sizeMax));

        NN_SDK_LOG("\n");

        sizeMax = (sizeMax * 2) + std::uniform_int_distribution<>(0, 255)(mt);
    }
} // NOLINT(impl/function_size)

// フラッシュのテスト
TEST_F(FsIntegrityVerificationStorageTest, Flush)
{
    static const size_t TestSize = 10 * 1024 * 1024;
    static const int TestLoop = 30;
    static const size_t CacheBufferSize = 256 * 1024;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4096;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> buf(TestSize);
    char* ptr = &buf[0];

    nnt::fs::util::Vector<char> shaShared;
    nnt::fs::util::Vector<char> bufCache(CacheBufferSize);
    nn::os::Mutex locker(true);

    for( int loop = 0; loop < TestLoop; ++loop )
    {
        size_t sizeStorage = std::uniform_int_distribution<size_t>(1, TestSize)(mt);
        size_t offsetBase = std::uniform_int_distribution<size_t>(0, 127)(mt);

        // 1ブロックの検証サイズは256～MaxBlockSizeの間で可変です。
        const size_t sizeHashBlock = GenerateRandomBlockSize(256, MaxBlockSize);
        const size_t sizeHash = ((sizeStorage + sizeHashBlock - 1) / sizeHashBlock) *
                                nn::fssystem::save::IntegrityVerificationStorage::HashSize;

        sizeStorage = nn::util::align_up(sizeStorage, sizeHashBlock);

        // SHAバッファサイズ
        const size_t sizeShaSharedBufSize = GenerateRandomBlockSize(32, MaxBlockSize);
        shaShared.resize(sizeShaSharedBufSize);

        // ハッシュのデータ
        nnt::fs::util::AccessCountedMemoryStorage baseStorageHash;
        baseStorageHash.Initialize(static_cast<size_t>(sizeHash + offsetBase * 2));

        // ファイルボディ
        nnt::fs::util::AccessCountedMemoryStorage baseStorageBody;
        baseStorageBody.Initialize(static_cast<size_t>(sizeStorage + offsetBase * 2));

        // キャッシュ
        nn::fssystem::FileSystemBufferManager cacheBuffer;
        NNT_ASSERT_RESULT_SUCCESS(
            cacheBuffer.Initialize(
                CacheCountMax,
                reinterpret_cast<uintptr_t>(&bufCache[0]),
                CacheBufferSize,
                CacheBlockSize
            )
        );

        nn::fs::SubStorage storageHash(&baseStorageHash, offsetBase, sizeHash);
        nn::fs::SubStorage storageBody(&baseStorageBody, offsetBase, sizeStorage);

        // 完全性検証
        nn::fs::HashSalt salt = {};
        std::generate(std::begin(salt.value), std::end(salt.value), [&]() NN_NOEXCEPT
        {
            return static_cast<char>(std::uniform_int_distribution<>(0, 255)(mt) - 128);
        });

        nn::fssystem::save::IntegrityVerificationStorage storageVerify;
        storageVerify.Initialize(
            storageHash,
            storageBody,
            sizeHashBlock,
            nn::fssystem::save::IntegrityVerificationStorage::HashSize,
            &cacheBuffer,
            salt,
            false,
            nn::fs::StorageType::StorageType_SaveData
        );

        // バッファー
        nn::fssystem::save::BlockCacheBufferedStorage storageBuffer;
        NNT_ASSERT_RESULT_SUCCESS(
            storageBuffer.Initialize(
                &cacheBuffer,
                &locker,
                &storageVerify,
                sizeStorage,
                sizeHashBlock,
                storageBuffer.DefaultMaxCacheEntryCount,
                false,                // isReadDataCache
                0x00,                 // bufferLevel
                false,                // bKeepBurstMode
                nn::fs::StorageType::StorageType_SaveData
            )
        );

        // アクセス回数リセット
        baseStorageHash.ResetAccessCounter();
        baseStorageBody.ResetAccessCounter();

        // 適当な位置に書き込み
        for( int loop3 = 0; loop3 <= 3; ++loop3 )
        {
            const int64_t offset = nn::util::align_down(
                std::uniform_int_distribution<int64_t>(0, sizeStorage - 1)(mt), sizeHashBlock);
            const size_t size = nn::util::align_up(
                std::uniform_int_distribution<size_t>(
                    0, sizeStorage - static_cast<size_t>(offset) - 1)(mt), sizeHashBlock);

            std::memset(ptr, loop3, size);

            NNT_ASSERT_RESULT_SUCCESS(storageVerify.Write(offset, ptr, size));
        }

        // フラッシュは発生していません。
        ASSERT_EQ(0, baseStorageHash.GetFlushTimes());
        ASSERT_EQ(0, baseStorageBody.GetFlushTimes());

        // アクセス回数リセット
        baseStorageHash.ResetAccessCounter();
        baseStorageBody.ResetAccessCounter();

        // 0 バイト書き込み
        for( int loop3 = 0; loop3 <= 3; ++loop3 )
        {
            size_t offset = nn::util::align_down(
                std::uniform_int_distribution<size_t>(0, sizeStorage - 1)(mt), sizeHashBlock);

            // アクセス回数リセット
            baseStorageHash.ResetAccessCounter();
            baseStorageBody.ResetAccessCounter();

            NNT_ASSERT_RESULT_SUCCESS(storageVerify.Write(offset, ptr, 0));

            ASSERT_EQ(0, baseStorageHash.GetFlushTimes());
            ASSERT_EQ(0, baseStorageBody.GetFlushTimes());
        }

        // アクセス回数リセット
        baseStorageHash.ResetAccessCounter();
        baseStorageBody.ResetAccessCounter();

        // 範囲外に書き込み
        for( size_t loop3 = 0; loop3 <= 3; loop3++ )
        {
            int64_t offset = nn::util::align_up(
                std::uniform_int_distribution<int64_t>(sizeStorage + 1, sizeStorage * 2)(mt),
                sizeHashBlock
            );
            size_t size = nn::util::align_up(
                std::uniform_int_distribution<size_t>(1, sizeStorage)(mt),
                sizeHashBlock
            );

            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOffset,
                storageVerify.Write(offset, ptr, size)
            );
        }

        // 範囲外エラー発生時に書き込みは行われない
        ASSERT_EQ(0, baseStorageHash.GetWriteTimes());
        ASSERT_EQ(0, baseStorageBody.GetWriteTimes());
    }
} // NOLINT(impl/function_size)

// キャッシュ無効化のテスト
void InvalidateTest(nn::fs::StorageType storageType)
{
    // ストレージを準備する
    static const auto BlockSize = 256;
    static const auto BodyStorageSize = 16 * 1024;
    static const auto HashStorageSize
        = BodyStorageSize * nn::fssystem::save::IntegrityVerificationStorage::HashSize;

    static const auto CacheBufferSize = 256 * 1024;
    static const auto CacheCountMax = 1024;
    static const auto CacheBlockSize = 4096;

    nnt::fs::util::AccessCountedMemoryStorage hashStorage;
    hashStorage.Initialize(HashStorageSize);

    nnt::fs::util::AccessCountedMemoryStorage bodyStorage;
    bodyStorage.Initialize(BodyStorageSize);

    nn::fs::HashSalt salt = {};

    nnt::fs::util::Vector<char> cacheBuffer(CacheBufferSize);
    nn::fssystem::FileSystemBufferManager bufferManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufferManager.Initialize(
            CacheCountMax,
            reinterpret_cast<uintptr_t>(cacheBuffer.data()),
            CacheBufferSize,
            CacheBlockSize
        )
    );

    nn::fssystem::save::IntegrityVerificationStorage storage;
    storage.Initialize(
        nn::fs::SubStorage(&hashStorage, 0, HashStorageSize),
        nn::fs::SubStorage(&bodyStorage, 0, BodyStorageSize),
        BlockSize,
        nn::fssystem::save::IntegrityVerificationStorage::HashSize,
        &bufferManager,
        salt,
        false,
        storageType);

    if( storageType == nn::fs::StorageType::StorageType_SaveData )
    {
        // StorageType_SaveDataではInvalidateしてはいけない
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultUnsupportedOperation,
            storage.OperateRange(
            nn::fs::OperationId::Invalidate,
            0,
            BodyStorageSize));
    }
    else
    {
        // キャッシュ無効化要求が下位ストレージに到達するはず
        NNT_EXPECT_RESULT_SUCCESS(storage.OperateRange(
            nn::fs::OperationId::Invalidate,
            0,
            BodyStorageSize));
        EXPECT_GT(hashStorage.GetInvalidateTimes(), 0);
        EXPECT_GT(bodyStorage.GetInvalidateTimes(), 0);
    }
}

TEST_F(FsIntegrityVerificationStorageTest, Invalidate)
{
    InvalidateTest(nn::fs::StorageType::StorageType_RomFs);
    InvalidateTest(nn::fs::StorageType::StorageType_SaveData);
}

// ゼロフィルテスト
TEST_F(FsIntegrityVerificationStorageTest, FillZero)
{
    static const size_t TestSize = 8 * 1024 * 1024;
    static const size_t CacheBufferSize = 256 * 1024;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4096;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> bufWrite(TestSize, -1);
    nnt::fs::util::Vector<char> bufRead(TestSize, 0);
    nnt::fs::util::Vector<char> cache(CacheBufferSize);

    nn::fssystem::FileSystemBufferManager cacheBuffer;
    NNT_ASSERT_RESULT_SUCCESS(
        cacheBuffer.Initialize(
            CacheCountMax,
            reinterpret_cast<uintptr_t>(cache.data()),
            CacheBufferSize,
            CacheBlockSize
        )
    );

    for( size_t sizeMax = 1 << 15; sizeMax <= TestSize; sizeMax *= 2 )
    {
        // 1ブロックの検証サイズは256-32768の間で可変です。
        const int64_t sizeBlock = GenerateRandomBlockSize(32, 32768);
        const int64_t countBlock = sizeMax / sizeBlock;
        const int64_t sizeHash =
            countBlock * nn::fssystem::save::IntegrityVerificationStorage::HashSize;

        // ハッシュのデータ
        nnt::fs::util::SafeMemoryStorage baseStorageHash;
        baseStorageHash.Initialize(sizeHash);

        // ファイルボディ
        nnt::fs::util::SafeMemoryStorage baseStorageBody;
        baseStorageBody.Initialize(sizeMax);

        nn::fs::HashSalt salt = {};
        std::generate(std::begin(salt.value), std::end(salt.value), [&mt]() NN_NOEXCEPT
        {
            return static_cast<char>(
                       std::uniform_int_distribution<>(
                           std::numeric_limits<char>::min(),
                           std::numeric_limits<char>::max()
                       )(mt)
                   );
        });

        nn::fssystem::save::IntegrityVerificationStorage storageVerify;
        storageVerify.Initialize(
            nn::fs::SubStorage(&baseStorageHash, 0, sizeHash),
            nn::fs::SubStorage(&baseStorageBody, 0, sizeMax),
            sizeBlock,
            32,
            &cacheBuffer,
            salt,
            true,
            nn::fs::StorageType::StorageType_SaveData
        );

        // ゼロ埋めしてから読み込めば全てゼロが読み込めるはず
        nn::Result result = storageVerify.OperateRange(
            nn::fs::OperationId::FillZero, 0, sizeMax);
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = storageVerify.Read(0, bufRead.data(), sizeMax);
        NNT_ASSERT_RESULT_SUCCESS(result);
        for( size_t index = 0; index < sizeMax; ++index )
        {
            EXPECT_EQ(0, bufRead[index]);
        }

        // 書き込んでから読み込めば書いた値が読めるはず
        result = storageVerify.Write(0, bufWrite.data(), sizeMax);
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = storageVerify.Read(0, bufRead.data(), sizeMax);
        NNT_ASSERT_RESULT_SUCCESS(result);
        for( size_t index = 0; index < sizeMax; ++index )
        {
            EXPECT_EQ(static_cast<char>(-1), bufRead[index]);
        }

        // 一部をゼロ埋めすればそこからはゼロが、それ以外からはもともと書かれていた値が読み込めるはず
        {
            const int64_t indexFill =
                std::uniform_int_distribution<int64_t>(0, countBlock - 1)(mt);
            const int64_t countFill =
                std::uniform_int_distribution<int64_t>(1, countBlock - indexFill)(mt);

            result = storageVerify.OperateRange(
                nn::fs::OperationId::FillZero,
                indexFill * sizeBlock,
                countFill * sizeBlock
            );
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = storageVerify.Read(0, bufRead.data(), sizeMax);
            NNT_ASSERT_RESULT_SUCCESS(result);

            for( size_t index = 0; index < sizeMax; ++index )
            {
                if( indexFill * sizeBlock <= static_cast<int64_t>(index) &&
                    static_cast<int64_t>(index) < (indexFill + countFill) * sizeBlock )
                {
                    EXPECT_EQ(0, bufRead[index]);
                }
                else
                {
                    EXPECT_EQ(static_cast<char>(-1), bufRead[index]);
                }
            }
        }

        // 完全に範囲外のゼロ埋め要求は失敗するはず
        result = storageVerify.OperateRange(
            nn::fs::OperationId::FillZero, sizeMax + sizeBlock, sizeBlock);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
        result = storageVerify.OperateRange(
            nn::fs::OperationId::FillZero, -sizeBlock, sizeBlock);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);

        // 部分的に範囲外のゼロ埋め要求は成功するはず
        {
            const int64_t indexFill =
                std::uniform_int_distribution<int64_t>(0, countBlock - 1)(mt);
            const int64_t countFill =
                countBlock - indexFill + std::uniform_int_distribution<int64_t>(1, 64)(mt);

            result = storageVerify.Write(0, bufWrite.data(), sizeMax);
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = storageVerify.OperateRange(
                nn::fs::OperationId::FillZero,
                indexFill * sizeBlock,
                countFill * sizeBlock
            );
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = storageVerify.Read(0, bufRead.data(), sizeMax);
            NNT_ASSERT_RESULT_SUCCESS(result);

            for( size_t index = 0; index < sizeMax; ++index )
            {
                if( indexFill * sizeBlock <= static_cast<int64_t>(index) )
                {
                    EXPECT_EQ(0, bufRead[index]);
                }
                else
                {
                    EXPECT_EQ(static_cast<char>(-1), bufRead[index]);
                }
            }
        }
    }
} // NOLINT(impl/function_size)

// メモリ破壊を監視するストレージです。
TEST_F(FsIntegrityVerificationStorageTest, DestroySignature)
{
    static const size_t TestSize = 8 * 1024 * 1024;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4096;
    static const size_t CacheBufferSize = 256 * 1024;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> bufWrite(TestSize, -1);
    nnt::fs::util::Vector<char> cache(CacheBufferSize);

    nn::fssystem::FileSystemBufferManager cacheBuffer;
    NNT_ASSERT_RESULT_SUCCESS(
        cacheBuffer.Initialize(
            CacheCountMax,
            reinterpret_cast<uintptr_t>(cache.data()),
            CacheBufferSize,
            CacheBlockSize
        )
    );

    for( size_t sizeMax = 1 << 15; sizeMax <= TestSize; sizeMax *= 2 )
    {
        // 1ブロックの検証サイズは256-32768の間で可変です。
        const int sizeBlock = static_cast<int>(GenerateRandomBlockSize(32, 32768));
        const int64_t countBlock = (sizeMax + sizeBlock - 1) / sizeBlock;
        const int64_t sizeHash =
            countBlock * nn::fssystem::save::IntegrityVerificationStorage::HashSize;

        // ハッシュのデータ
        nnt::fs::util::SafeMemoryStorage baseStorageHash;
        baseStorageHash.Initialize(sizeHash);

        // ファイルボディ
        nnt::fs::util::SafeMemoryStorage baseStorageBody;
        baseStorageBody.Initialize(sizeMax);

        nn::fs::HashSalt salt = {};
        std::generate(std::begin(salt.value), std::end(salt.value), [&mt]() NN_NOEXCEPT
        {
            return static_cast<char>(
                       std::uniform_int_distribution<>(
                           std::numeric_limits<char>::min(),
                           std::numeric_limits<char>::max()
                       )(mt)
                   );
        });

        nn::fssystem::save::IntegrityVerificationStorage storageVerify;
        storageVerify.Initialize(
            nn::fs::SubStorage(&baseStorageHash, 0, sizeHash),
            nn::fs::SubStorage(&baseStorageBody, 0, sizeMax),
            sizeBlock,
            32,
            &cacheBuffer,
            salt,
            true,
            nn::fs::StorageType::StorageType_SaveData
        );

        nn::Result result;

        // 署名を破壊した領域の読み込みはエラーになるが、それ以外の領域の読み込みは成功するはず
        {
            const int64_t indexDestroy =
                std::uniform_int_distribution<int64_t>(0, countBlock - 1)(mt);
            const int64_t countDestroy =
                std::uniform_int_distribution<int64_t>(1, countBlock - indexDestroy)(mt);

            result = storageVerify.Write(0, bufWrite.data(), sizeMax);
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = storageVerify.OperateRange(
                nn::fs::OperationId::DestroySignature,
                indexDestroy * sizeBlock,
                countDestroy * sizeBlock
            );
            NNT_ASSERT_RESULT_SUCCESS(result);

            nnt::fs::util::Vector<char> bufRead(sizeBlock, 0);
            for( int64_t index = 0; index < countBlock; ++index )
            {
                result = storageVerify.Read(index * sizeBlock, bufRead.data(), sizeBlock);
                if( index < indexDestroy || indexDestroy + countDestroy <= index )
                {
                    NNT_EXPECT_RESULT_SUCCESS(result);
                }
                else
                {
                    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultIntegrityVerificationStorageCorrupted, result);
                }
            }
        }

        // 完全に範囲外の署名破壊要求は失敗するはず
        result = storageVerify.OperateRange(
            nn::fs::OperationId::DestroySignature,
            sizeMax + sizeBlock,
            sizeBlock
        );
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
        result = storageVerify.OperateRange(
            nn::fs::OperationId::DestroySignature,
            -sizeBlock,
            sizeBlock
        );
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);

        // 部分的に範囲外の署名破壊要求は成功するはず
        {
            const int64_t indexDestroy =
                std::uniform_int_distribution<int64_t>(0, countBlock - 1)(mt);
            const int64_t countDestroy =
                countBlock - indexDestroy + std::uniform_int_distribution<int64_t>(1, 64)(mt);

            result = storageVerify.Write(0, bufWrite.data(), sizeMax);
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = storageVerify.OperateRange(
                nn::fs::OperationId::DestroySignature,
                indexDestroy * sizeBlock,
                countDestroy * sizeBlock
            );
            NNT_ASSERT_RESULT_SUCCESS(result);

            nnt::fs::util::Vector<char> bufRead(sizeBlock, 0);
            for( int64_t index = 0; index < countBlock; ++index )
            {
                result = storageVerify.Read(index * sizeBlock, bufRead.data(), sizeBlock);
                if( index < indexDestroy )
                {
                    NNT_EXPECT_RESULT_SUCCESS(result);
                }
                else
                {
                    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultIntegrityVerificationStorageCorrupted, result);
                }
            }
        }
    }
} // NOLINT(impl/function_size)

// 不明な操作のテスト
TEST_F(FsIntegrityVerificationStorageTest, UnknownOperation)
{
    static const size_t TestSize = 8 * 1024 * 1024;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4096;
    static const size_t CacheBufferSize = 256 * 1024;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> cache(CacheBufferSize);

    nn::fssystem::FileSystemBufferManager cacheBuffer;
    NNT_ASSERT_RESULT_SUCCESS(
        cacheBuffer.Initialize(
            CacheCountMax,
            reinterpret_cast<uintptr_t>(cache.data()),
            CacheBufferSize,
            CacheBlockSize
        )
    );

    for( size_t sizeMax = 1 << 15; sizeMax <= TestSize; sizeMax *= 2 )
    {
        // 1ブロックの検証サイズは256-32768の間で可変です。
        const int sizeBlock = static_cast<int>(GenerateRandomBlockSize(32, 32768));
        const int64_t countBlock = (sizeMax + sizeBlock - 1) / sizeBlock;
        const int64_t sizeHash =
            countBlock * nn::fssystem::save::IntegrityVerificationStorage::HashSize;

        // ハッシュのデータ
        nnt::fs::util::SafeMemoryStorage baseStorageHash;
        baseStorageHash.Initialize(sizeHash);

        // ファイルボディ
        nnt::fs::util::SafeMemoryStorage baseStorageBody;
        baseStorageBody.Initialize(static_cast<size_t>(sizeMax));

        nn::fs::HashSalt salt = {};
        std::generate(std::begin(salt.value), std::end(salt.value), [&mt]() NN_NOEXCEPT
        {
            return static_cast<char>(
                       std::uniform_int_distribution<>(
                           std::numeric_limits<char>::min(),
                           std::numeric_limits<char>::max()
                       )(mt)
                   );
        });

        nn::fssystem::save::IntegrityVerificationStorage storageVerify;
        storageVerify.Initialize(
            nn::fs::SubStorage(&baseStorageHash, 0, sizeHash),
            nn::fs::SubStorage(&baseStorageBody, 0, sizeMax),
            sizeBlock,
            32,
            &cacheBuffer,
            salt,
            true,
            nn::fs::StorageType::StorageType_SaveData
        );

        // 不明な操作は失敗するはず
        const auto indexOperate =
            std::uniform_int_distribution<int64_t>(0, countBlock - 1)(mt);
        const auto countOperate =
            std::uniform_int_distribution<int64_t>(1, countBlock - indexOperate)(mt);
        const auto result = storageVerify.OperateRange(
            static_cast<nn::fs::OperationId>(-1),
            indexOperate * sizeBlock,
            countOperate * sizeBlock
        );
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation, result);
    }
}

class IntegrityVerificationStorageTestSetup : public TestStorageSetup
{
public:
    // コンストラクタです。
    IntegrityVerificationStorageTestSetup() NN_NOEXCEPT
        : m_Lock(true)
    {
    }

    // ファイルオブジェクトをフォーマットします。
    virtual void Format(
                     int64_t sizeStorage,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        static const int CacheCountMax = 1024;
        static const size_t CacheBlockSize = 4096;
        static const int32_t CacheBufferSize = 256 * 1024;

        NN_UNUSED(sizeBlock);

        // 1ブロックの検証サイズは256～MaxBlockSizeの間で可変です。
        m_SizeHashBlock = GenerateRandomBlockSize(256, MaxBlockSize);
        m_SizeHash = static_cast<size_t>(
            ((sizeStorage + m_SizeHashBlock - 1) / m_SizeHashBlock) *
            nn::fssystem::save::IntegrityVerificationStorage::HashSize);

        // SHAバッファサイズ
        const size_t sizeShaSharedBufSize = GenerateRandomBlockSize(32, MaxBlockSize);
        m_ShaShared.resize(sizeShaSharedBufSize);

        // ハッシュのデータ
        m_BaseStorageHash.reset(new nnt::fs::util::SafeMemoryStorage);
        m_BaseStorageHash->Initialize(m_SizeHash + offset * 2);

        // ファイルボディ
        m_BaseStorageBody.reset(new nnt::fs::util::SafeMemoryStorage);
        m_BaseStorageBody->Initialize(sizeStorage + offset * 2);

        // キャッシュ
        m_BufCache.resize(CacheBufferSize);
        NNT_ASSERT_RESULT_SUCCESS(
            m_CacheBuffer.Initialize(
                CacheCountMax,
                reinterpret_cast<uintptr_t>(&m_BufCache[0]),
                CacheBufferSize,
                CacheBlockSize
            )
        );
    }

    // ファイルオブジェクトを初期化します。
    virtual void Initialize(
                     int64_t sizeStorage,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(sizeBlock);

        // 完全性検証
        m_StorageVerify.reset(new nn::fssystem::save::IntegrityVerificationStorage());
        nn::fs::SubStorage storageHash(m_BaseStorageHash.get(), offset, m_SizeHash);
        nn::fs::SubStorage storageBody(m_BaseStorageBody.get(), offset, sizeStorage);
        {
            nn::fs::HashSalt salt = {};
            std::generate(std::begin(salt.value), std::end(salt.value), [this]() NN_NOEXCEPT
            {
                return static_cast<char>(m_Random.Get<>(255) - 128);
            });
            m_StorageVerify->Initialize(
                storageHash,
                storageBody,
                m_SizeHashBlock,
                nn::fssystem::save::IntegrityVerificationStorage::HashSize,
                &m_CacheBuffer,
                salt,
                false,
                nn::fs::StorageType::StorageType_SaveData
            );
        }

        // バッファー
        m_StorageBuffer.reset(new nn::fssystem::save::BlockCacheBufferedStorage());
        NNT_ASSERT_RESULT_SUCCESS(
            m_StorageBuffer->Initialize(
                &m_CacheBuffer,
                &m_Lock,
                m_StorageVerify.get(),
                sizeStorage,
                m_SizeHashBlock,
                nn::fssystem::save::BlockCacheBufferedStorage::DefaultMaxCacheEntryCount,
                false,             // isReadDataCache
                0x00,              // bufferLevel
                false,             // bKeepBurstMode
                nn::fs::StorageType::StorageType_SaveData
            )
        );

        SetStorage(m_StorageVerify.get());
    }

    // ファイルオブジェクトをアンマウントします。
    virtual void Unmount() NN_NOEXCEPT NN_OVERRIDE
    {
        m_StorageBuffer.reset(nullptr);
        m_StorageVerify.reset(nullptr);
    }

    // ファイルオブジェクトを破棄します。
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        m_CacheBuffer.Finalize();
    }

    // 読み込み専用かどうか
    virtual bool IsReadOnly() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // ファイルへのデータ書き込み時に自動伸長するかどうか
    virtual bool IsAutoExtendSizeOnWrite() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // ファイルのサイズ変更が可能かどうか
    virtual bool IsEnableResize() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // サイズ 0 のファイルを作成可能かどうか
    virtual bool IsAcceptsSizeZero() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

private:
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_BaseStorageHash;
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_BaseStorageBody;
    std::unique_ptr<nn::fssystem::save::IntegrityVerificationStorage> m_StorageVerify;
    std::unique_ptr<nn::fssystem::save::BlockCacheBufferedStorage> m_StorageBuffer;
    nnt::fs::util::Vector<char> m_ShaShared;

    nn::os::Mutex m_Lock;

    nnt::fs::util::Vector<char> m_BufCache;
    nn::fssystem::FileSystemBufferManager m_CacheBuffer;

    size_t m_SizeHashBlock;
    size_t m_SizeHash;
};

// 共通テストを行います。
TEST_F(FsIntegrityVerificationStorageTest, StorageCommonTest)
{
    //StorageTest::RunTest<IntegrityVerificationStorageTestSetup>();
}

TEST_F(FsIntegrityVerificationStorageTest, Large)
{
    static const size_t SizeBlock = 16 * 1024;
    static const int64_t StorageSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024 + SizeBlock;
    static const int CacheCountMax = 1024;
    static const size_t CacheBufferSize = 256 * 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const int64_t SizeHash = ((StorageSize + SizeBlock - 1) / SizeBlock) *
        nn::fssystem::save::IntegrityVerificationStorage::HashSize;

    auto cache = nnt::fs::util::AllocateBuffer(CacheBufferSize);

    nn::fssystem::FileSystemBufferManager cacheBuffer;
    NNT_ASSERT_RESULT_SUCCESS(
        cacheBuffer.Initialize(
            CacheCountMax,
            reinterpret_cast<uintptr_t>(cache.get()),
            CacheBufferSize,
            CacheBlockSize));

    // ハッシュのデータ
    nnt::fs::util::VirtualMemoryStorage baseStorageHash;
    baseStorageHash.Initialize(SizeHash);

    // ファイルボディ
    nnt::fs::util::VirtualMemoryStorage baseStorageBody;
    baseStorageBody.Initialize(StorageSize);

    nn::fs::SubStorage storageHash(&baseStorageHash, 0, SizeHash);
    nn::fs::SubStorage storageBody(&baseStorageBody, 0, StorageSize);

    nn::fssystem::save::IntegrityVerificationStorage storageVerify;
    {
        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        nn::fs::HashSalt salt = {};
        std::generate(
            std::begin(salt.value),
            std::end(salt.value),
            [&]() NN_NOEXCEPT
            {
                return static_cast<char>(std::uniform_int_distribution<>(0, 255)(mt) - 128);
            });
        storageVerify.Initialize(
            storageHash,
            storageBody,
            SizeBlock,
            32,
            &cacheBuffer,
            salt,
            true,
            nn::fs::StorageType::StorageType_SaveData);
    }

    TestLargeOffsetAccess(&storageVerify, SizeBlock);
}
